/*----------------------------------------------------------------------------*/
// (C) 2021 FCNT LIMITED
/*----------------------------------------------------------------------------*/
// SPDX-License-Identifier: GPL-2.0

//==============================================================================
// include file
//==============================================================================
#include <linux/init.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/custom_mode.h>
#include <linux/err.h>
#include <linux/io.h>
#include <linux/soc/qcom/smem.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/nonvolatile_common.h>
#include <linux/of.h>
#include <linux/of_gpio.h>
#include <linux/gpio.h>

//==============================================================================
// define
//==============================================================================

//==============================================================================
// const data
//==============================================================================
#define TCXO_LED_CONTROL_ENABLE     (1)

#if TCXO_LED_CONTROL_ENABLE
#define LED_R_GPIO		(98)
enum {
	TCXO_STATE_OFF = 0,
	TCXO_STATE_RPM_APSS,
	TCXO_STATE_RPM_MODEM,
	TCXO_STATE_RPM_APSS_ADSP,
	TCXO_STATE_RPM_APSS_WCNSS,
	TCXO_STATE_RPM_APSS_MODEM,
	TCXO_STATE_RPM_APSS_TZ,
	TCXO_STATE_RPM_ADSP_MODEM,
	TCXO_STATE_MAX,
} tcxo_state_type;
#endif /* TCXO_LED_CONTROL_ENABLE */

//==============================================================================
// global valuable
//==============================================================================
int custom_boot_mode = CUSTOM_MODE_NORMAL;
EXPORT_SYMBOL(custom_boot_mode);
int charging_mode = 0;
EXPORT_SYMBOL(charging_mode);
int makercmd_mode = 0;
EXPORT_SYMBOL(makercmd_mode);
int custom_recovery_mode = 0;
EXPORT_SYMBOL(custom_recovery_mode);

//==============================================================================
// private valuable
//==============================================================================

//==============================================================================
// static functions prototype
//==============================================================================

//==============================================================================
// functions
//==============================================================================
static int sdram_id_show(struct seq_file *m, void *unused)
{
	int ret = 0;
	uint32_t *smem_p;

	smem_p = (uint32_t*)smem_alloc_vendor0(SMEM_OEM_V0_003);
	if (smem_p == NULL) {
		ret = -EFAULT;
		printk(KERN_ERR "smem_alloc_vendor0 failed.\n");
	} else {
		seq_printf(m, "0x%08x\n", *smem_p);
	}
	return ret;
}

static int sdram_id_open(struct inode *inode, struct file *file)
{
	return single_open(file, sdram_id_show, NULL);
}

static const struct file_operations sdram_id_fops = {
	.owner = THIS_MODULE,
	.open = sdram_id_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

static int __init sdram_id_init(void)
{
	proc_create("sdram_id", S_IRUGO, NULL, &sdram_id_fops);
	return 0;
}
module_init(sdram_id_init);

static int system_rev_show(struct seq_file *m, void *unused)
{
	int ret = 0;
	uint32_t *smem_p;

	smem_p = (uint32_t *)smem_alloc_vendor0(SMEM_OEM_V0_004);
	if (smem_p == NULL) {
		ret = -EFAULT;
		printk(KERN_ERR "smem_alloc_vendor0 failed.\n");
	} else {
		seq_printf(m, "Revision\t: %04x\n", *smem_p);
	}
	return ret;
}

static int system_rev_open(struct inode *inode, struct file *file)
{
	return single_open(file, system_rev_show, NULL);
}

static const struct file_operations system_rev_fops = {
	.owner = THIS_MODULE,
	.open = system_rev_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

static int __init system_rev_init(void)
{
	proc_create("system_rev", S_IRUGO, NULL, &system_rev_fops);
	return 0;
}
module_init(system_rev_init);

static int parts_id_show(struct seq_file *m, void *unused)
{
	int ret = 0;
	uint32_t *smem_p;

	smem_p = (uint32_t *)smem_alloc_vendor0(SMEM_OEM_V0_014);
	if (smem_p == NULL) {
		ret = -EFAULT;
		printk(KERN_ERR "smem_alloc_vendor0 failed.\n");
	} else {
		seq_printf(m, "0x%08x\n", *smem_p);
	}
	return ret;
}

static int parts_id_open(struct inode *inode, struct file *file)
{
	return single_open(file, parts_id_show, NULL);
}

static const struct file_operations parts_id_fops = {
	.owner = THIS_MODULE,
	.open = parts_id_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

static int __init parts_id_init(void)
{
	proc_create("parts_id", S_IRUGO, NULL, &parts_id_fops);
	return 0;
}
module_init(parts_id_init);

static int variant_info_show(struct seq_file *m, void *unused)
{
	int ret = 0;
	uint32_t *smem_p;

	smem_p = (uint32_t *)smem_alloc_vendor0(SMEM_OEM_V0_016);
	if (smem_p == NULL) {
		ret = -EFAULT;
		printk(KERN_ERR "smem_alloc_vendor0 failed.\n");
	} else {
		seq_printf(m, "0x%08x\n", *smem_p);
	}
	return ret;
}

static int variant_info_open(struct inode *inode, struct file *file)
{
	return single_open(file, variant_info_show, NULL);
}

static const struct file_operations variant_info_fops = {
	.owner = THIS_MODULE,
	.open = variant_info_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

static int __init variant_info_init(void)
{
	proc_create("variant_info", S_IRUGO, NULL, &variant_info_fops);
	return 0;
}
module_init(variant_info_init);

static int __init androidboot_mode_enable(char *str)
{
	if (strcmp(str, "charger") == 0) {
		charging_mode = 1;
		custom_boot_mode = CUSTOM_MODE_OFF_CHARGE;
	} else if (strcmp(str, "makermode") == 0) {
		makercmd_mode = 1;
		custom_boot_mode = CUSTOM_MODE_MAKER_MODE;
	} else if (strcmp(str, "kernelmode") == 0) {
		makercmd_mode = 1;
		custom_boot_mode = CUSTOM_MODE_KERNEL_MODE;
	} else if (!strcmp(str, "recoverymenu")) {
		custom_recovery_mode = 1;
		custom_boot_mode = CUSTOM_MODE_SP_MODE;
	} else if (!strcmp(str, "osupdatemode")) {
		custom_recovery_mode = 1;
		custom_boot_mode = CUSTOM_MODE_SD_DOWNLOADER;
	} else if (!strcmp(str, "masterclear")) {
		custom_recovery_mode = 1;
		custom_boot_mode = CUSTOM_MODE_MASTER_CLEAR;
	}

	printk(KERN_INFO "custom_boot_mode = %d\n", custom_boot_mode);
	printk(KERN_INFO "chargemode: %d\n", charging_mode);
	printk(KERN_INFO "recoverymode: %d\n", custom_recovery_mode);
	printk(KERN_INFO "makermode : %d\n", makercmd_mode);

	return 0;
}
early_param("androidboot.mode", androidboot_mode_enable);

#define APNV_AUTO_POWER_OFF_F 48021
#define APNV_SHIP_MODE_I 49264

int ship_mode_stdn = 0;
EXPORT_SYMBOL(ship_mode_stdn);

static int __init ship_mode_setting(void)
{
	unsigned char p_buf, s_buf;
	int result = 0;

	result = get_nonvolatile((uint8_t*)&p_buf, APNV_AUTO_POWER_OFF_F, 1);
	if (unlikely(result < 0)) {
		return 0;
	}
	result = get_nonvolatile((uint8_t*)&s_buf, APNV_SHIP_MODE_I, 1);
	if (unlikely(result < 0)) {
		return 0;
	}
	if ((p_buf == 1) && (s_buf == 1)) {
		ship_mode_stdn = 1;
		printk(KERN_INFO "ship mode shutdown\n");
	}
	return 0;
}
late_initcall(ship_mode_setting);

static int tcxo_led_gpio = -1;
static void fcnt_tcxo_led_init(void)
{
#if TCXO_LED_CONTROL_ENABLE
	struct device_node *node;

	if (!of_have_populated_dt()) {
		printk(KERN_ERR "tcxo_led_init: device tree is not initialized\n");
		return;
	}

	node = of_find_compatible_node(NULL, NULL, "fcnt,tcxo_led");
	if (node) {
		tcxo_led_gpio = of_get_named_gpio(node, "led-red-gpio", 0);
	} else {
		printk(KERN_INFO "tcxo_led node does not exist.\n");
	}

#endif /* TCXO_LED_CONTROL_ENABLE */
	return;
}

static void fcnt_tcxo_led_switch(bool value)
{
#if TCXO_LED_CONTROL_ENABLE
	uint32_t * smem_p;

	if (!gpio_is_valid(tcxo_led_gpio)) {
		printk(KERN_INFO "tcxo_led_gpio is not valid.\n");
		return;
	}

	do {
		smem_p = smem_alloc_vendor0(SMEM_OEM_V0_005);
		if (!smem_p)
			break;

		switch (*smem_p) {
		case TCXO_STATE_RPM_APSS:
		case TCXO_STATE_RPM_APSS_ADSP:
		case TCXO_STATE_RPM_APSS_WCNSS:
		case TCXO_STATE_RPM_APSS_MODEM:
		case TCXO_STATE_RPM_APSS_TZ:
			printk("tcxo led %d\n",value);
			gpio_set_value(tcxo_led_gpio, (value ? 1 : 0));
			break;
		default:
			/* nop */
			break;
		}
	} while(0);

#endif /* TCXO_LED_CONTROL_ENABLE */
	return;
}

void fcnt_tcxo_led_ctrl(bool value)
{
	fcnt_tcxo_led_switch(value);
	return;
}

static int __init fcnt_tcxo_led_on(void)
{
	fcnt_tcxo_led_init();
	fcnt_tcxo_led_switch(true);
	return 0;
}
late_initcall(fcnt_tcxo_led_on);

#define APLNV_DATABU_ADB_ENABLE_I    (49294)

static int force_adb_off_show(struct seq_file *m, void *unused)
{
	unsigned char buf;
	int ret = 0;

	ret = get_nonvolatile((uint8_t*)&buf, APLNV_DATABU_ADB_ENABLE_I, 1);
	if (unlikely(ret < 0)) {
		ret = -EFAULT;
		return ret;
	}

	if (ret == sizeof(buf)) {
		seq_printf(m, "%d\n", buf);
	} else {
		seq_printf(m, "%d\n", 0);
	}

	return 0;
}

static ssize_t force_adb_off_write(struct file *file, const char __user *userbuf,
			     size_t count, loff_t *data)
{
	unsigned char buf = 0;

	set_nonvolatile((uint8_t*)&buf, APLNV_DATABU_ADB_ENABLE_I, 1);
	return count;
}

static int force_adb_off_open(struct inode *inode, struct file *file)
{
	return single_open(file, force_adb_off_show, NULL);
}

static const struct file_operations force_adb_off_fops = {
	.owner = THIS_MODULE,
	.open = force_adb_off_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
	.write		= force_adb_off_write,
};

static int __init force_adb_off_init(void)
{
	proc_create("force_adb_off", 0600, NULL, &force_adb_off_fops);
	return 0;
}
module_init(force_adb_off_init);

MODULE_LICENSE("GPL v2");
