/*----------------------------------------------------------------------------*/
// (C) 2021 FCNT LIMITED 
/*----------------------------------------------------------------------------*/
/*
 * DRV260X haptics driver family
 *
 * Author: Dan Murphy <dmurphy@ti.com>
 *
 * Copyright:   (C) 2014 Texas Instruments, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 */

#include <linux/i2c.h>
#include <linux/input.h>
#include <linux/module.h>
#include <linux/regmap.h>
#include <linux/slab.h>
#include <linux/delay.h>
#include <linux/gpio/consumer.h>
#include <linux/regulator/consumer.h>

#include <dt-bindings/input/ti-drv260x.h>

#include <linux/hrtimer.h>
#include <linux/leds.h>
#include <linux/nonvolatile_common.h>

#define DRV260X_STATUS		0x0
#define DRV260X_MODE		0x1
#define DRV260X_RT_PB_IN	0x2
#define DRV260X_LIB_SEL		0x3
#define DRV260X_WV_SEQ_1	0x4
#define DRV260X_WV_SEQ_2	0x5
#define DRV260X_WV_SEQ_3	0x6
#define DRV260X_WV_SEQ_4	0x7
#define DRV260X_WV_SEQ_5	0x8
#define DRV260X_WV_SEQ_6	0x9
#define DRV260X_WV_SEQ_7	0xa
#define DRV260X_WV_SEQ_8	0xb
#define DRV260X_GO				0xc
#define DRV260X_OVERDRIVE_OFF	0xd
#define DRV260X_SUSTAIN_P_OFF	0xe
#define DRV260X_SUSTAIN_N_OFF	0xf
#define DRV260X_BRAKE_OFF		0x10
#define DRV260X_A_TO_V_CTRL		0x11
#define DRV260X_A_TO_V_MIN_INPUT	0x12
#define DRV260X_A_TO_V_MAX_INPUT	0x13
#define DRV260X_A_TO_V_MIN_OUT	0x14
#define DRV260X_A_TO_V_MAX_OUT	0x15
#define DRV260X_RATED_VOLT		0x16
#define DRV260X_OD_CLAMP_VOLT	0x17
#define DRV260X_CAL_COMP		0x18
#define DRV260X_CAL_BACK_EMF	0x19
#define DRV260X_FEEDBACK_CTRL	0x1a
#define DRV260X_CTRL1			0x1b
#define DRV260X_CTRL2			0x1c
#define DRV260X_CTRL3			0x1d
#define DRV260X_CTRL4			0x1e
#define DRV260X_CTRL5			0x1f
#define DRV260X_LRA_LOOP_PERIOD	0x20
#define DRV260X_VBAT_MON		0x21
#define DRV260X_LRA_RES_PERIOD	0x22
#define DRV260X_MAX_REG			0x23
#define DRV260X_RAM_ADDRESS_UPPER	0xFD
#define DRV260X_RAM_ADDRESS_LOWER	0xFE
#define DRV260X_RAM_DATA		0xFF

#define DRV260X_GO_BIT				0x01

/* Library Selection */
#define DRV260X_LIB_SEL_MASK		0x07
#define DRV260X_LIB_SEL_RAM			0x0
#define DRV260X_LIB_SEL_OD			0x1
#define DRV260X_LIB_SEL_40_60		0x2
#define DRV260X_LIB_SEL_60_80		0x3
#define DRV260X_LIB_SEL_100_140		0x4
#define DRV260X_LIB_SEL_140_PLUS	0x5

#define DRV260X_LIB_SEL_HIZ_MASK	0x10
#define DRV260X_LIB_SEL_HIZ_EN		0x01
#define DRV260X_LIB_SEL_HIZ_DIS		0

/* Mode register */
#define DRV260X_STANDBY				(1 << 6)
#define DRV260X_STANDBY_MASK		0x40
#define DRV260X_INTERNAL_TRIGGER	0x00
#define DRV260X_EXT_TRIGGER_EDGE	0x01
#define DRV260X_EXT_TRIGGER_LEVEL	0x02
#define DRV260X_PWM_ANALOG_IN		0x03
#define DRV260X_AUDIOHAPTIC			0x04
#define DRV260X_RT_PLAYBACK			0x05
#define DRV260X_DIAGNOSTICS			0x06
#define DRV260X_AUTO_CAL			0x07

/* Audio to Haptics Control */
#define DRV260X_AUDIO_HAPTICS_PEAK_10MS		(0 << 2)
#define DRV260X_AUDIO_HAPTICS_PEAK_20MS		(1 << 2)
#define DRV260X_AUDIO_HAPTICS_PEAK_30MS		(2 << 2)
#define DRV260X_AUDIO_HAPTICS_PEAK_40MS		(3 << 2)

#define DRV260X_AUDIO_HAPTICS_FILTER_100HZ	0x00
#define DRV260X_AUDIO_HAPTICS_FILTER_125HZ	0x01
#define DRV260X_AUDIO_HAPTICS_FILTER_150HZ	0x02
#define DRV260X_AUDIO_HAPTICS_FILTER_200HZ	0x03

/* Min/Max Input/Output Voltages */
#define DRV260X_AUDIO_HAPTICS_MIN_IN_VOLT	0x19
#define DRV260X_AUDIO_HAPTICS_MAX_IN_VOLT	0x64
#define DRV260X_AUDIO_HAPTICS_MIN_OUT_VOLT	0x19
#define DRV260X_AUDIO_HAPTICS_MAX_OUT_VOLT	0xFF

/* Feedback register */
#define DRV260X_FB_REG_ERM_MODE			0x7f
#define DRV260X_FB_REG_LRA_MODE			(1 << 7)

#define DRV260X_BRAKE_FACTOR_MASK	0x1f
#define DRV260X_BRAKE_FACTOR_2X		(1 << 0)
#define DRV260X_BRAKE_FACTOR_3X		(2 << 4)
#define DRV260X_BRAKE_FACTOR_4X		(3 << 4)
#define DRV260X_BRAKE_FACTOR_6X		(4 << 4)
#define DRV260X_BRAKE_FACTOR_8X		(5 << 4)
#define DRV260X_BRAKE_FACTOR_16		(6 << 4)
#define DRV260X_BRAKE_FACTOR_DIS	(7 << 4)

#define DRV260X_LOOP_GAIN_LOW		0xf3
#define DRV260X_LOOP_GAIN_MED		(1 << 2)
#define DRV260X_LOOP_GAIN_HIGH		(2 << 2)
#define DRV260X_LOOP_GAIN_VERY_HIGH	(3 << 2)

#define DRV260X_BEMF_GAIN_0			0xfc
#define DRV260X_BEMF_GAIN_1		(1 << 0)
#define DRV260X_BEMF_GAIN_2		(2 << 0)
#define DRV260X_BEMF_GAIN_3		(3 << 0)

/* Control 1 register */
#define DRV260X_AC_CPLE_EN			(1 << 5)
#define DRV260X_STARTUP_BOOST		(1 << 7)

/* Control 2 register */

#define DRV260X_IDISS_TIME_45		0
#define DRV260X_IDISS_TIME_75		(1 << 0)
#define DRV260X_IDISS_TIME_150		(1 << 1)
#define DRV260X_IDISS_TIME_225		0x03

#define DRV260X_BLANK_TIME_45	(0 << 2)
#define DRV260X_BLANK_TIME_75	(1 << 2)
#define DRV260X_BLANK_TIME_150	(2 << 2)
#define DRV260X_BLANK_TIME_225	(3 << 2)

#define DRV260X_SAMP_TIME_150	(0 << 4)
#define DRV260X_SAMP_TIME_200	(1 << 4)
#define DRV260X_SAMP_TIME_250	(2 << 4)
#define DRV260X_SAMP_TIME_300	(3 << 4)

#define DRV260X_BRAKE_STABILIZER	(1 << 6)
#define DRV260X_UNIDIR_IN			(0 << 7)
#define DRV260X_BIDIR_IN			(1 << 7)

/* Control 3 Register */
#define DRV260X_LRA_OPEN_LOOP		(1 << 0)
#define DRV260X_ANANLOG_IN			(1 << 1)
#define DRV260X_LRA_DRV_MODE		(1 << 2)
#define DRV260X_RTP_UNSIGNED_DATA	(1 << 3)
#define DRV260X_SUPPLY_COMP_DIS		(1 << 4)
#define DRV260X_ERM_OPEN_LOOP		(1 << 5)
#define DRV260X_NG_THRESH_0			(0 << 6)
#define DRV260X_NG_THRESH_2			(1 << 6)
#define DRV260X_NG_THRESH_4			(2 << 6)
#define DRV260X_NG_THRESH_8			(3 << 6)

/* Control 4 Register */
#define DRV260X_AUTOCAL_TIME_150MS		(0 << 4)
#define DRV260X_AUTOCAL_TIME_250MS		(1 << 4)
#define DRV260X_AUTOCAL_TIME_500MS		(2 << 4)
#define DRV260X_AUTOCAL_TIME_1000MS		(3 << 4)
#define DRV260X_FEADBACK_TIME_THRESH	50

#define APNV_VIB_MAGNITUDE		49265
#define APNV_VIB_MAGNITUDE_SIZE		2
#define APNV_VIB_MAGNITUDE_LO		0
#define APNV_VIB_MAGNITUDE_HI		1

#define APNV_VIB_FREQ			49268
#define APNV_VIB_FREQ_SIZE		3
#define APNV_VIB_FREQ_LO		0
#define APNV_VIB_FREQ_HI		1
#define APNV_VIB_FREQ_OFFSET		2

#define OD_CLAMP_FEADBACK_VALUE_DRV2604L	0x66
#define OD_CLAMP_FEADBACK_VALUE_DRV2604		0x5E
#define RATED_VOLT_FEADBACK_VALUE			0x52
#define BEMF_GAIN_VALUE						0x02
#define CAL_COMP_VALUE						0x0C
#define CAL_BEMF_VALUE						0xB3
#define OD_CLAMP_NORMAL_VALUE		0x40
#define RATED_VOLTAGE_NORMAL_VALUE	0x38
#define OL_LRA_PERIOD				0x36
#define AMPLITUDEI_LEVEL			255
#define DRV260X_VOLT_CALC(lv,volt) \
	(volt / 2) + ((volt / 2 * 1000 * lv / 255) / 1000)
#define DRV260X_OD_CALM_CALC(lv,volt) \
	((volt / 2 * 1000) + (volt / 2 * 1000 * lv / 255)) * 1147 / 1000000
#define APNV_BEMF_GAIN			49447
#define APNV_BEMF_GAIN_SIZE		1
#define APNV_BEMF_GAIN_LO		0

#define APNV_A_CAL_COMP			49445
#define APNV_A_CAL_COMP_SIZE	1
#define APNV_A_CAL_COMP_LO		0

#define APNV_A_CAL_BEMF			49446
#define APNV_A_CAL_BEMF_SIZE	1
#define APNV_A_CAL_BEMF_LO		0

#define CALIBRATION_PASS			0
#define CALIBRATION_FAIL			1
#define I2C_ERROR					2
#define SOFT_ERROR					3
#define CALIBRATION_NOT_EXECUTED	4

/**
 * struct drv260x_data -
 * @input_dev - Pointer to the input device
 * @client - Pointer to the I2C client
 * @regmap - Register map of the device
 * @work - Work item used to off load the enable/disable of the vibration
 * @enable_gpio - Pointer to the gpio used for enable/disabling
 * @regulator - Pointer to the regulator for the IC
 * @magnitude - Magnitude of the vibration event
 * @mode - The operating mode of the IC (LRA_NO_CAL, ERM or LRA)
 * @library - The vibration library to be used
 * @rated_voltage - The rated_voltage of the actuator
 * @overdriver_voltage - The over drive voltage of the actuator
**/
struct drv260x_data {
	struct input_dev *input_dev;
	struct i2c_client *client;
	struct regmap *regmap;
	struct work_struct work;
	struct gpio_desc *enable_gpio;
	struct regulator *regulator;
	u32 magnitude;
	u32 mode;
	u32 library;
	int rated_voltage;
	int overdrive_voltage;
	struct gpio_desc *hap_pon;
	struct led_classdev cdev;
	struct hrtimer  stop_timer;
	u32 play_time_ms;
	int module_en;
	struct gpio_desc *trigger_gpio;
	u32 od_clam_feedback_value;
	u32 od_clam_normal_value;
	u32 rated_voltage_normal_value;
	u32 lra_loop_period_value;
	u32 lra_res_period_value;
	u32 rated_voltage_normal_value_nv;
	struct mutex voltage_mutex;
	u32 feedback_value;
	u32 cal_comp_value;
	u32 cal_back_emf_value;
	u32 calib_result;
	u32 reg_read_addr;
	int retry_recovery;
	int is_drv2604l;
};

#if 0
static const struct reg_default drv260x_reg_defs[] = {
	{ DRV260X_STATUS, 0xe0 },
	{ DRV260X_MODE, 0x40 },
	{ DRV260X_RT_PB_IN, 0x00 },
	{ DRV260X_LIB_SEL, 0x00 },
	{ DRV260X_WV_SEQ_1, 0x01 },
	{ DRV260X_WV_SEQ_2, 0x00 },
	{ DRV260X_WV_SEQ_3, 0x00 },
	{ DRV260X_WV_SEQ_4, 0x00 },
	{ DRV260X_WV_SEQ_5, 0x00 },
	{ DRV260X_WV_SEQ_6, 0x00 },
	{ DRV260X_WV_SEQ_7, 0x00 },
	{ DRV260X_WV_SEQ_8, 0x00 },
	{ DRV260X_GO, 0x00 },
	{ DRV260X_OVERDRIVE_OFF, 0x00 },
	{ DRV260X_SUSTAIN_P_OFF, 0x00 },
	{ DRV260X_SUSTAIN_N_OFF, 0x00 },
	{ DRV260X_BRAKE_OFF, 0x00 },
	{ DRV260X_A_TO_V_CTRL, 0x05 },
	{ DRV260X_A_TO_V_MIN_INPUT, 0x19 },
	{ DRV260X_A_TO_V_MAX_INPUT, 0xff },
	{ DRV260X_A_TO_V_MIN_OUT, 0x19 },
	{ DRV260X_A_TO_V_MAX_OUT, 0xff },
	{ DRV260X_RATED_VOLT, 0x3e },
	{ DRV260X_OD_CLAMP_VOLT, 0x8c },
	{ DRV260X_CAL_COMP, 0x0c },
	{ DRV260X_CAL_BACK_EMF, 0x6c },
	{ DRV260X_FEEDBACK_CTRL, 0x36 },
	{ DRV260X_CTRL1, 0x93 },
	{ DRV260X_CTRL2, 0xfa },
	{ DRV260X_CTRL3, 0xa0 },
	{ DRV260X_CTRL4, 0x20 },
	{ DRV260X_CTRL5, 0x80 },
	{ DRV260X_LRA_LOOP_PERIOD, 0x33 },
	{ DRV260X_VBAT_MON, 0x00 },
	{ DRV260X_LRA_RES_PERIOD, 0x00 },
};
#endif
#if 0
static const struct reg_sequence drv260x_reg_init[] = {
	{ DRV260X_STATUS, 0xC0 },
	{ DRV260X_MODE, 0x00 },
	{ DRV260X_RT_PB_IN, 0x7F },
	{ DRV260X_LIB_SEL, 0x00 },
	{ DRV260X_WV_SEQ_1, 0x01 },
	{ DRV260X_WV_SEQ_2, 0x00 },
	{ DRV260X_WV_SEQ_3, 0x00 },
	{ DRV260X_WV_SEQ_4, 0x00 },
	{ DRV260X_WV_SEQ_5, 0x00 },
	{ DRV260X_WV_SEQ_6, 0x00 },
	{ DRV260X_WV_SEQ_7, 0x00 },
	{ DRV260X_WV_SEQ_8, 0x00 },
	{ DRV260X_GO, 0x00 },
	{ DRV260X_OVERDRIVE_OFF, 0x00 },
	{ DRV260X_SUSTAIN_P_OFF, 0x00 },
	{ DRV260X_SUSTAIN_N_OFF, 0x00 },
	{ DRV260X_BRAKE_OFF, 0x00 },
	/*{ DRV260X_A_TO_V_CTRL, 0x05 },
	{ DRV260X_A_TO_V_MIN_INPUT, 0x19 },
	{ DRV260X_A_TO_V_MAX_INPUT, 0xff },
	{ DRV260X_A_TO_V_MIN_OUT, 0x19 },
	{ DRV260X_A_TO_V_MAX_OUT, 0xff },*/
	{ DRV260X_RATED_VOLT, 0x38 },
	{ DRV260X_OD_CLAMP_VOLT, 0x40 },
	{ DRV260X_CAL_COMP, 0x0c },
	{ DRV260X_CAL_BACK_EMF, 0xB3 },
	{ DRV260X_FEEDBACK_CTRL, 0xA6 },
	{ DRV260X_CTRL1, 0x95 },
	{ DRV260X_CTRL2, 0xf5 },
	{ DRV260X_CTRL3, 0x81 },
	{ DRV260X_CTRL4, 0x20 },
	{ DRV260X_CTRL5, 0x40 },
	{ DRV260X_LRA_LOOP_PERIOD, 0x36 },
	{ DRV260X_VBAT_MON, 0xE6 },
	{ DRV260X_LRA_RES_PERIOD, 0x36 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x00 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x01 },
	{ DRV260X_RAM_DATA, 0x01 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x00 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x02 },
	{ DRV260X_RAM_DATA, 0x00 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x00 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x03 },
	{ DRV260X_RAM_DATA, 0x04 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x01 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x00 },
	{ DRV260X_RAM_DATA, 0x3F },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x01 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x01 },
	{ DRV260X_RAM_DATA, 0x02 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x01 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x02 },
	{ DRV260X_RAM_DATA, 0x00 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x01 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x03 },
	{ DRV260X_RAM_DATA, 0x08 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x00 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x04 },
	{ DRV260X_RAM_DATA, 0x02 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x00 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x05 },
	{ DRV260X_RAM_DATA, 0x00 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x00 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x06 },
	{ DRV260X_RAM_DATA, 0x04 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x02 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x00 },
	{ DRV260X_RAM_DATA, 0x12 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x02 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x01 },
	{ DRV260X_RAM_DATA, 0x02 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x02 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x02 },
	{ DRV260X_RAM_DATA, 0x00 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x02 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x03 },
	{ DRV260X_RAM_DATA, 0x08 },
};

static const struct reg_sequence drv260x_vib_on[] = {
	{ DRV260X_MODE, 0x05 },
	{ DRV260X_WV_SEQ_1, 0x01 },
	{ DRV260X_CTRL3, 0x80 },
//	{ DRV260X_FEEDBACK_CTRL, 0xA6 },
//	{ DRV260X_RATED_VOLT, 0x52},
//	{ DRV260X_OD_CLAMP_VOLT, 0x5E},
};
#endif
static const struct reg_sequence drv2604_reg_init[] = {
	{ DRV260X_STATUS, 0x80 },
	{ DRV260X_MODE, 0x00 },
	{ DRV260X_RT_PB_IN, 0x7F },
	{ DRV260X_LIB_SEL, 0x00 },
	{ DRV260X_WV_SEQ_1, 0x01 },
	{ DRV260X_WV_SEQ_2, 0x00 },
	{ DRV260X_WV_SEQ_3, 0x00 },
	{ DRV260X_WV_SEQ_4, 0x00 },
	{ DRV260X_WV_SEQ_5, 0x00 },
	{ DRV260X_WV_SEQ_6, 0x00 },
	{ DRV260X_WV_SEQ_7, 0x00 },
	{ DRV260X_WV_SEQ_8, 0x00 },
	{ DRV260X_GO, 0x00 },
	{ DRV260X_OVERDRIVE_OFF, 0x00 },
	{ DRV260X_SUSTAIN_P_OFF, 0x00 },
	{ DRV260X_SUSTAIN_N_OFF, 0x00 },
	{ DRV260X_BRAKE_OFF, 0x00 },
	{ DRV260X_RATED_VOLT, 0x38 },
	{ DRV260X_OD_CLAMP_VOLT, 0x40 },
	{ DRV260X_CAL_COMP, 0x0C },
	{ DRV260X_CAL_BACK_EMF, 0xB3 },
	{ DRV260X_FEEDBACK_CTRL, 0xF4 },
	{ DRV260X_CTRL1, 0x95 },
	{ DRV260X_CTRL2, 0xFD },
	{ DRV260X_CTRL3, 0x80 },
	{ DRV260X_CTRL4, 0x20 },
	{ DRV260X_LRA_RES_PERIOD, 0x36 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x00 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x01 },
	{ DRV260X_RAM_DATA, 0x01 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x00 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x02 },
	{ DRV260X_RAM_DATA, 0x00 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x00 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x03 },
	{ DRV260X_RAM_DATA, 0x04 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x01 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x00 },
	{ DRV260X_RAM_DATA, 0x3F },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x01 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x01 },
	{ DRV260X_RAM_DATA, 0x02 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x01 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x02 },
	{ DRV260X_RAM_DATA, 0x00 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x01 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x03 },
	{ DRV260X_RAM_DATA, 0x08 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x00 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x04 },
	{ DRV260X_RAM_DATA, 0x02 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x00 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x05 },
	{ DRV260X_RAM_DATA, 0x00 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x00 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x06 },
	{ DRV260X_RAM_DATA, 0x04 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x02 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x00 },
	{ DRV260X_RAM_DATA, 0x12 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x02 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x01 },
	{ DRV260X_RAM_DATA, 0x02 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x02 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x02 },
	{ DRV260X_RAM_DATA, 0x00 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x02 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x03 },
	{ DRV260X_RAM_DATA, 0x08 },
};

static const struct reg_sequence drv2604l_reg_init[] = {
	{ DRV260X_STATUS, 0xC0 },
	{ DRV260X_MODE, 0x00 },
	{ DRV260X_RT_PB_IN, 0x7F },
	{ DRV260X_LIB_SEL, 0x00 },
	{ DRV260X_WV_SEQ_1, 0x01 },
	{ DRV260X_WV_SEQ_2, 0x00 },
	{ DRV260X_WV_SEQ_3, 0x00 },
	{ DRV260X_WV_SEQ_4, 0x00 },
	{ DRV260X_WV_SEQ_5, 0x00 },
	{ DRV260X_WV_SEQ_6, 0x00 },
	{ DRV260X_WV_SEQ_7, 0x00 },
	{ DRV260X_WV_SEQ_8, 0x00 },
	{ DRV260X_GO, 0x00 },
	{ DRV260X_OVERDRIVE_OFF, 0x00 },
	{ DRV260X_SUSTAIN_P_OFF, 0x00 },
	{ DRV260X_SUSTAIN_N_OFF, 0x00 },
	{ DRV260X_BRAKE_OFF, 0x00 },
	{ DRV260X_RATED_VOLT, 0x38 },
	{ DRV260X_OD_CLAMP_VOLT, 0x40 },
	{ DRV260X_CAL_COMP, 0x0C },
	{ DRV260X_CAL_BACK_EMF, 0xB3 },
	{ DRV260X_FEEDBACK_CTRL, 0xF6 },
	{ DRV260X_CTRL1, 0x95 },
	{ DRV260X_CTRL2, 0xF5 },
	{ DRV260X_CTRL3, 0x80 },
	{ DRV260X_CTRL4, 0x20 },
	{ DRV260X_CTRL5, 0x40 },
	{ DRV260X_LRA_LOOP_PERIOD, 0x36 },
	{ DRV260X_LRA_RES_PERIOD, 0x36 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x00 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x01 },
	{ DRV260X_RAM_DATA, 0x01 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x00 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x02 },
	{ DRV260X_RAM_DATA, 0x00 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x00 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x03 },
	{ DRV260X_RAM_DATA, 0x04 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x01 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x00 },
	{ DRV260X_RAM_DATA, 0x3F },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x01 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x01 },
	{ DRV260X_RAM_DATA, 0x02 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x01 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x02 },
	{ DRV260X_RAM_DATA, 0x00 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x01 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x03 },
	{ DRV260X_RAM_DATA, 0x08 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x00 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x04 },
	{ DRV260X_RAM_DATA, 0x02 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x00 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x05 },
	{ DRV260X_RAM_DATA, 0x00 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x00 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x06 },
	{ DRV260X_RAM_DATA, 0x04 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x02 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x00 },
	{ DRV260X_RAM_DATA, 0x12 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x02 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x01 },
	{ DRV260X_RAM_DATA, 0x02 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x02 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x02 },
	{ DRV260X_RAM_DATA, 0x00 },
	{ DRV260X_RAM_ADDRESS_UPPER, 0x02 },
	{ DRV260X_RAM_ADDRESS_LOWER, 0x03 },
	{ DRV260X_RAM_DATA, 0x08 },
};

static const struct reg_sequence drv2604_vib_on[] = {
	{ DRV260X_MODE, 0x05 },
	{ DRV260X_WV_SEQ_1, 0x01 },
	{ DRV260X_FEEDBACK_CTRL, 0xF4 },
	{ DRV260X_CTRL3, 0x80 },
};

static const struct reg_sequence drv2604l_vib_on[] = {
	{ DRV260X_MODE, 0x05 },
	{ DRV260X_WV_SEQ_1, 0x01 },
	{ DRV260X_CTRL3, 0x80 },
};

static const struct reg_sequence drv260x_vib_off[] = {
	{ DRV260X_MODE, 0x00 },
};

#if 0
static const struct reg_sequence drv260x_vib_standby[] = {
	{ DRV260X_MODE, 0x01 },
	{ DRV260X_CTRL3, 0x81 },
};
#endif

#if 0
static const struct reg_sequence drv260x_vib_touch_feedback[] = {
	{ DRV260X_MODE, 0x01 },
	{ DRV260X_WV_SEQ_1, 0x01 },
//	{ DRV260X_FEEDBACK_CTRL, 0xA6 },
	{ DRV260X_CTRL3, 0x81 },
};

static const struct reg_sequence drv260x_vib_exlider_feedback[] = {
	{ DRV260X_MODE, 0x01 },
	{ DRV260X_WV_SEQ_1, 0x02 },
//	{ DRV260X_FEEDBACK_CTRL, 0xA6 },
	{ DRV260X_CTRL3, 0x81 },
};

static const struct reg_sequence drv260x_vib_calibration[] = {
	{ DRV260X_MODE, 0x07 },
	{ DRV260X_RATED_VOLT, 0x52 },
	{ DRV260X_OD_CLAMP_VOLT, 0x5E },
	{ DRV260X_FEEDBACK_CTRL, 0xA6 },
	{ DRV260X_CTRL3, 0x80 },
};
#endif
static const struct reg_sequence drv2604_vib_touch_feedback[] = {
	{ DRV260X_MODE, 0x01 },
	{ DRV260X_WV_SEQ_1, 0x01 },
	{ DRV260X_FEEDBACK_CTRL, 0xF5 },
	{ DRV260X_CTRL3, 0x80 },
};

static const struct reg_sequence drv2604l_vib_touch_feedback[] = {
	{ DRV260X_MODE, 0x01 },
	{ DRV260X_WV_SEQ_1, 0x01 },
	{ DRV260X_CTRL3, 0x81 },
};

static const struct reg_sequence drv2604_vib_exlider_feedback[] = {
	{ DRV260X_MODE, 0x01 },
	{ DRV260X_WV_SEQ_1, 0x02 },
	{ DRV260X_FEEDBACK_CTRL, 0xF5 },
	{ DRV260X_CTRL3, 0x80 },
};

static const struct reg_sequence drv2604l_vib_exlider_feedback[] = {
	{ DRV260X_MODE, 0x01 },
	{ DRV260X_WV_SEQ_1, 0x02 },
	{ DRV260X_CTRL3, 0x81 },
};

static const struct reg_sequence drv2604_vib_calibration[] = {
	{ DRV260X_MODE, 0x07 },
	{ DRV260X_RATED_VOLT, 0x52 },
	{ DRV260X_OD_CLAMP_VOLT, 0x5E },
	{ DRV260X_FEEDBACK_CTRL, 0xF5 },
	{ DRV260X_CTRL2, 0xFD },
	{ DRV260X_CTRL3, 0x80 },
};

static const struct reg_sequence drv2604l_vib_calibration[] = {
	{ DRV260X_MODE, 0x07 },
	{ DRV260X_RATED_VOLT, 0x52 },
	{ DRV260X_OD_CLAMP_VOLT, 0x5E },
	{ DRV260X_FEEDBACK_CTRL, 0xA6 },
	{ DRV260X_CTRL2, 0xF5 },
	{ DRV260X_CTRL3, 0x80 },
};

#define DRV260X_DEF_RATED_VOLT		0x90
#define DRV260X_DEF_OD_CLAMP_VOLT	0x90

static enum hrtimer_restart hap_stop_timer(struct hrtimer *timer)
{
	struct drv260x_data *haptics = container_of(timer, struct drv260x_data,
					stop_timer);

	haptics->module_en = 0;
	schedule_work(&haptics->work);

	return HRTIMER_NORESTART;
}

/**
 * Rated and Overdriver Voltages:
 * Calculated using the formula r = v * 255 / 5.6
 * where r is what will be written to the register
 * and v is the rated or overdriver voltage of the actuator
 **/
static int drv260x_calculate_voltage(unsigned int voltage)
{
	return (voltage * 255 / 5600);
}

static int drv260x_get_NV_data(struct drv260x_data *haptics)
{
	int error = 0;
	unsigned char hap_getNvData[APNV_VIB_MAGNITUDE_SIZE];
	unsigned char hap_getNvData2[APNV_VIB_FREQ_SIZE];
	unsigned char hap_getNvData3[APNV_BEMF_GAIN_SIZE];
	unsigned char hap_getNvData4[APNV_A_CAL_COMP_SIZE];
	unsigned char hap_getNvData5[APNV_A_CAL_BEMF_SIZE];

	memset(hap_getNvData, 0x00, sizeof(hap_getNvData));
	error = get_nonvolatile(hap_getNvData, APNV_VIB_MAGNITUDE, APNV_VIB_MAGNITUDE_SIZE);
	if (unlikely(error < 0)) {
		dev_info(&haptics->client->dev,"%s: get_nonvolatile(APNV_VIB_MAGNITUDE) fail use defualt value\n", __func__);
		if (haptics->is_drv2604l == 1) {
			haptics->od_clam_feedback_value = OD_CLAMP_FEADBACK_VALUE_DRV2604L;
		} else {
			haptics->od_clam_feedback_value = OD_CLAMP_FEADBACK_VALUE_DRV2604;
		}
		haptics->rated_voltage_normal_value = RATED_VOLTAGE_NORMAL_VALUE;
	} else {
		dev_info(&haptics->client->dev,"%s: get_nonvolatile(APNV_VIB_MAGNITUDE) success \n", __func__);
		if (haptics->is_drv2604l == 1) {
			haptics->od_clam_feedback_value = hap_getNvData[APNV_VIB_MAGNITUDE_HI];
		} else {
			haptics->od_clam_feedback_value = OD_CLAMP_FEADBACK_VALUE_DRV2604;
		}

		haptics->rated_voltage_normal_value = hap_getNvData[APNV_VIB_MAGNITUDE_LO];
		error = regmap_write(haptics->regmap, DRV260X_RATED_VOLT,
			haptics->rated_voltage_normal_value);
		if (error) {
			dev_err(&haptics->client->dev,
				"Failed to write set DRV260X_RATED_VOLT,%d\n", error);
		}
	}
	haptics->rated_voltage_normal_value_nv = haptics->rated_voltage_normal_value;
	haptics->od_clam_normal_value = haptics->rated_voltage_normal_value_nv * 1147 / 1000;

	memset(hap_getNvData2, 0x00, sizeof(hap_getNvData2));
	error = get_nonvolatile(hap_getNvData2, APNV_VIB_FREQ, APNV_VIB_FREQ_SIZE);
	if (unlikely(error < 0)) {
		dev_info(&haptics->client->dev,"%s: get_nonvolatile(APNV_VIB_FREQ) fail use defualt value\n", __func__);
		haptics->lra_loop_period_value = OL_LRA_PERIOD;
		haptics->lra_res_period_value = OL_LRA_PERIOD;
	} else {
		dev_info(&haptics->client->dev,"%s: get_nonvolatile(APNV_VIB_FREQ)success \n", __func__);
		haptics->lra_loop_period_value = hap_getNvData2[APNV_VIB_FREQ_LO];
		haptics->lra_res_period_value = hap_getNvData2[APNV_VIB_FREQ_LO];
		if (haptics->is_drv2604l == 1) {
			error = regmap_write(haptics->regmap, DRV260X_LRA_LOOP_PERIOD,
						haptics->lra_loop_period_value);
			if (error) {
				dev_err(&haptics->client->dev,
					"Failed to write set DRV260X_LRA_LOOP_PERIOD,%d\n", error);
			}
		}

		error = regmap_write(haptics->regmap, DRV260X_LRA_RES_PERIOD,
					haptics->lra_res_period_value);
		if (error) {
			dev_err(&haptics->client->dev,
				"Failed to write set DRV260X_LRA_RES_PERIOD ,%d\n", error);
		}
	}

	memset(hap_getNvData3, 0x00, sizeof(hap_getNvData3));
	 error = get_nonvolatile(hap_getNvData3, APNV_BEMF_GAIN, APNV_BEMF_GAIN_SIZE);
	if (unlikely(error < 0)) {
		dev_info(&haptics->client->dev,"%s: get_nonvolatile(APNV_BEMF_GAIN) fail use defualt value\n", __func__);
		haptics->feedback_value = BEMF_GAIN_VALUE;
	} else  {
		dev_info(&haptics->client->dev,"%s: get_nonvolatile(APNV_BEMF_GAIN)success \n", __func__);
		haptics->feedback_value = (hap_getNvData3[APNV_BEMF_GAIN_LO] & 0x03);
		if (haptics->is_drv2604l == 1) {
			error = regmap_write(haptics->regmap, DRV260X_FEEDBACK_CTRL,
						(haptics->feedback_value  | (0xF4 & 0xFC)));
		} else {
			error = regmap_write(haptics->regmap, DRV260X_FEEDBACK_CTRL, 0xF4);
		}
		if (error)
			dev_err(&haptics->client->dev,
				"Failed to write set DRV260X_FEEDBACK_CTRL ,%d\n", error);
	}

	memset(hap_getNvData4, 0x00, sizeof(hap_getNvData4));
	error = get_nonvolatile(hap_getNvData4, APNV_A_CAL_COMP, APNV_A_CAL_COMP_SIZE);
	if (unlikely(error < 0)) {
		dev_info(&haptics->client->dev,"%s: get_nonvolatile(APNV_A_CAL_COMP) fail use defualt value\n", __func__);
		haptics->cal_comp_value = CAL_COMP_VALUE;
	} else {
		dev_info(&haptics->client->dev,"%s: get_nonvolatile(APNV_A_CAL_COMP)success \n", __func__);
		haptics->cal_comp_value = hap_getNvData4[APNV_A_CAL_COMP_LO];
		error = regmap_write(haptics->regmap, DRV260X_CAL_COMP,
					haptics->cal_comp_value);
		if (error)
			dev_err(&haptics->client->dev,
				"Failed to write set DRV260X_CAL_COMP, %d\n", error);
	}

	memset(hap_getNvData5, 0x00, sizeof(hap_getNvData5));
	error = get_nonvolatile(hap_getNvData5, APNV_A_CAL_BEMF, APNV_A_CAL_BEMF_SIZE);
	if (unlikely(error < 0)) {
		dev_info(&haptics->client->dev,"%s: get_nonvolatile(APNV_A_CAL_BEMF) fail use defualt value\n", __func__);
		haptics->cal_back_emf_value = CAL_BEMF_VALUE;
	} else {
		dev_info(&haptics->client->dev,"%s: get_nonvolatile(APNV_A_CAL_BEMF)success \n", __func__);
		haptics->cal_back_emf_value = hap_getNvData5[APNV_A_CAL_BEMF_LO];
		error = regmap_write(haptics->regmap, DRV260X_CAL_BACK_EMF,
					haptics->cal_back_emf_value);
		if (error)
			dev_err(&haptics->client->dev,
				"Failed to write set DRV260X_CAL_BACK_EMF,%d\n", error);
	}

	return 0;
}

static inline int drv260x_feedback_mode(struct drv260x_data *haptics)
{
	int error;

	if (haptics->play_time_ms == 45) {
		dev_dbg(&haptics->client->dev,"%s: vibrator start exlider feedback mode .\n",__func__);
		if (haptics->is_drv2604l == 1) {
			error = regmap_multi_reg_write(haptics->regmap,
						drv2604l_vib_exlider_feedback,
						ARRAY_SIZE(drv2604l_vib_exlider_feedback));
		} else {
			error = regmap_multi_reg_write(haptics->regmap,
						drv2604_vib_exlider_feedback,
						ARRAY_SIZE(drv2604_vib_exlider_feedback));
		}
		if (error)
			dev_dbg(&haptics->client->dev,"Failed to write set exlider feedback mode: %d\n", error);
	} else {
		dev_dbg(&haptics->client->dev,"%s: vibrator start touch feedback mode .\n",__func__);
		if (haptics->is_drv2604l == 1) {
			error = regmap_multi_reg_write(haptics->regmap,
						drv2604l_vib_touch_feedback,
						ARRAY_SIZE(drv2604l_vib_touch_feedback));
		} else {
			error = regmap_multi_reg_write(haptics->regmap,
						drv2604_vib_touch_feedback,
						ARRAY_SIZE(drv2604_vib_touch_feedback));
		}
		if (error)
			dev_err(&haptics->client->dev,"Failed to write set touch feedback mode: %d\n", error);
	}

	mutex_lock(&haptics->voltage_mutex);
	error = regmap_write(haptics->regmap, DRV260X_OD_CLAMP_VOLT, haptics->od_clam_feedback_value);
	if (error) {
		dev_err(&haptics->client->dev,
			"Failed to write set voltage: %d\n", error);
	}
	if (haptics->is_drv2604l != 1) {
		error = regmap_write(haptics->regmap, DRV260X_RATED_VOLT, RATED_VOLT_FEADBACK_VALUE);
		if (error) {
			dev_err(&haptics->client->dev,
				"Failed to write set rakuraku mode register RV: %d\n", error);
		}
	}
	mutex_unlock(&haptics->voltage_mutex);

	gpiod_set_value(haptics->trigger_gpio, 1);
	udelay(2);
	gpiod_set_value(haptics->trigger_gpio, 0);

	return 0;
}

static void drv260x_worker(struct work_struct *work)
{
	struct drv260x_data *haptics = container_of(work, struct drv260x_data, work);
	int error;
	int time_ms = haptics->play_time_ms;

	gpiod_set_value(haptics->enable_gpio, 1);
	/* Data sheet says to wait 250us before trying to communicate */
	udelay(250);

	dev_dbg(&haptics->client->dev,"%s: vibrator will into status =%d and work time = %d .\n",__func__, haptics->module_en, time_ms);
	if (haptics->module_en > 0) {
		if (haptics->is_drv2604l == 1) {
			error = regmap_multi_reg_write(haptics->regmap,
											drv2604l_vib_on,
											ARRAY_SIZE(drv2604l_vib_on));
		} else {
			error = regmap_multi_reg_write(haptics->regmap,
											drv2604_vib_on,
											ARRAY_SIZE(drv2604_vib_on));
		}
		if (error) {
			dev_err(&haptics->client->dev,
				"Failed to write set mode: %d\n", error);
		}

		mutex_lock(&haptics->voltage_mutex);
		error = regmap_write(haptics->regmap, DRV260X_RATED_VOLT,
					haptics->rated_voltage_normal_value);
		if (error) {
			dev_err(&haptics->client->dev,
				"Failed to write set DRV260X_RATED_VOLT,%d\n", error);
		}

		error = regmap_write(haptics->regmap, DRV260X_OD_CLAMP_VOLT,
					haptics->od_clam_normal_value);

		if (error) {
			dev_err(&haptics->client->dev,
				"Failed to write set DRV260X_OD_CLAMP_VOLT,%d\n", error);
		}
		mutex_unlock(&haptics->voltage_mutex);

		hrtimer_start(&haptics->stop_timer,
			ktime_set(time_ms / MSEC_PER_SEC,
			(time_ms % MSEC_PER_SEC) * NSEC_PER_MSEC),
			HRTIMER_MODE_REL);
	} else {
		 error = regmap_multi_reg_write(haptics->regmap,
										drv260x_vib_off,
										ARRAY_SIZE(drv260x_vib_off));
		if (error) {
			dev_err(&haptics->client->dev,
				"Failed to write set vib off: %d\n", error);
		}
		udelay(10);
#if 0
		error = regmap_multi_reg_write(haptics->regmap,
										drv260x_vib_standby,
										ARRAY_SIZE(drv260x_vib_standby));
		if (error){
			dev_err(&haptics->client->dev,
				"Failed to write set vib standby: %d\n", error);
		}
#endif
	}
}

#if 0
static int drv260x_haptics_play(struct input_dev *input, void *data,
				struct ff_effect *effect)
{
	struct drv260x_data *haptics = input_get_drvdata(input);

	haptics->mode = DRV260X_LRA_NO_CAL_MODE;

	if (effect->u.rumble.strong_magnitude > 0)
		haptics->magnitude = effect->u.rumble.strong_magnitude;
	else if (effect->u.rumble.weak_magnitude > 0)
		haptics->magnitude = effect->u.rumble.weak_magnitude;
	else
		haptics->magnitude = 0;

	schedule_work(&haptics->work);

	return 0;
}
#endif

static void drv260x_close(struct input_dev *input)
{
#if 0
	struct drv260x_data *haptics = input_get_drvdata(input);
	int error;

	cancel_work_sync(&haptics->work);

	error = regmap_write(haptics->regmap, DRV260X_MODE, DRV260X_STANDBY);
	if (error)
		dev_err(&haptics->client->dev,
			"Failed to enter standby mode: %d\n", error);

	gpiod_set_value(haptics->enable_gpio, 0);
#endif
}

static ssize_t drv260x_haptics_show_state(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct led_classdev *cdev = dev_get_drvdata(dev);
	struct drv260x_data *haptics = container_of(cdev, struct drv260x_data, cdev);

	return snprintf(buf, PAGE_SIZE, "%d\n", haptics->module_en);
}

static ssize_t drv260x_haptics_store_state(struct device *dev,
                struct device_attribute *attr, const char *buf, size_t count)
{
	/* At present, nothing to do with setting state */
	return count;
}

static ssize_t drv260x_haptics_show_duration(struct device *dev,
                struct device_attribute *attr, char *buf)
{
	struct led_classdev *cdev = dev_get_drvdata(dev);
	struct drv260x_data *haptics = container_of(cdev, struct drv260x_data, cdev);
	ktime_t time_rem;
	s64 time_us = 0;

	if (hrtimer_active(&haptics->stop_timer)) {
		time_rem = hrtimer_get_remaining(&haptics->stop_timer);
		time_us = ktime_to_us(time_rem);
	}

	return snprintf(buf, PAGE_SIZE, "%lld\n", div_s64(time_us, 1000));
}

static ssize_t drv260x_haptics_store_duration(struct device *dev,
                struct device_attribute *attr, const char *buf, size_t count)
{
	struct led_classdev *cdev = dev_get_drvdata(dev);
	struct drv260x_data *haptics = container_of(cdev, struct drv260x_data, cdev);
	u32 val;
	int rc;

	rc = kstrtouint(buf, 0, &val);
	if (rc < 0) {
		dev_err(&haptics->client->dev,"get duration value fail. \n");
		return rc;
	}
	haptics->play_time_ms = val;
	return count;
}

static ssize_t drv260x_haptics_show_activate(struct device *dev,
                struct device_attribute *attr, char *buf)
{
	/* For now nothing to show */
	return snprintf(buf, PAGE_SIZE, "%d\n", 0);
}

static ssize_t drv260x_haptics_store_activate(struct device *dev,
                struct device_attribute *attr, const char *buf, size_t count)
{
	struct led_classdev *cdev = dev_get_drvdata(dev);
	struct drv260x_data *haptics = container_of(cdev, struct drv260x_data, cdev);
	u32 val;
	int rc;

	rc = kstrtouint(buf, 0, &val);
	if (rc < 0) {
		dev_err(&haptics->client->dev,"get activate value fail. \n");
		return rc;
	}

	if (val != 0 && val != 1) {
		dev_err(&haptics->client->dev," The activate value is setted error ,return \n");
		return count;
	}
	haptics->module_en = val;
	hrtimer_cancel(&haptics->stop_timer);
	cancel_work_sync(&haptics->work);
	if( haptics->module_en && haptics->play_time_ms < DRV260X_FEADBACK_TIME_THRESH) {
		rc = drv260x_feedback_mode(haptics);
	}
	else {
		schedule_work(&haptics->work);
	}
	return count;
}

static ssize_t drv260x_haptics_show_vmax(struct device *dev,
                struct device_attribute *attr, char *buf)
{
	//struct led_classdev *cdev = dev_get_drvdata(dev);
	return snprintf(buf, PAGE_SIZE, "%d\n", 0);
}

static ssize_t drv260x_haptics_store_vmax(struct device *dev,
                struct device_attribute *attr, const char *buf, size_t count)
{
	struct led_classdev *cdev = dev_get_drvdata(dev);
	struct drv260x_data *haptics = container_of(cdev, struct drv260x_data, cdev);
	u32 val;
	int rc;
	int error;

	rc = kstrtouint(buf, 0, &val);
	if (rc < 0) {
		dev_err(&haptics->client->dev,"get vmax value fail. \n");
		return rc;
	}

	if (val < 1 || val > AMPLITUDEI_LEVEL) {
		dev_err(&haptics->client->dev," the value is error ,return \n");
		return count;
	}

	mutex_lock(&haptics->voltage_mutex);
	haptics->rated_voltage_normal_value =
		DRV260X_VOLT_CALC(val, haptics->rated_voltage_normal_value_nv);
	haptics->od_clam_normal_value =
		DRV260X_OD_CALM_CALC(val, haptics->rated_voltage_normal_value_nv);
	dev_dbg(&haptics->client->dev,"%s:Amplitude:%d RateVolt:0x%x OdClamVolt:0x%x\n"
				,__func__, val, haptics->rated_voltage_normal_value ,haptics->od_clam_normal_value);
	if (gpiod_get_value(haptics->enable_gpio)) {
		error = regmap_write(haptics->regmap, DRV260X_RATED_VOLT,
					haptics->rated_voltage_normal_value);
		if (error) {
			dev_err(&haptics->client->dev,
				"Failed to write set DRV260X_RATED_VOLT,%d\n", error);
		}

		error = regmap_write(haptics->regmap, DRV260X_OD_CLAMP_VOLT,
					haptics->od_clam_normal_value);

		if (error) {
			dev_err(&haptics->client->dev,
				"Failed to write set DRV260X_OD_CLAMP_VOLT,%d\n", error);
		}
	}
	mutex_unlock(&haptics->voltage_mutex);
	return count;
}

static ssize_t drv260x_haptics_show_freq(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	//struct led_classdev *cdev = dev_get_drvdata(dev);
	return snprintf(buf, PAGE_SIZE, "%d\n", 0);
}

static ssize_t drv260x_haptics_store_freq(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
	//struct led_classdev *cdev = dev_get_drvdata(dev);
	return count;
}

static void drv260x_set_caldata(struct drv260x_data *haptics)
{
	int error;
	u32 cal_val;
	unsigned char hap_getNvData[APNV_BEMF_GAIN_SIZE];

	error = regmap_read(haptics->regmap, DRV260X_STATUS, &cal_val );
	if (error) {
		dev_err(&haptics->client->dev,"Failed to read DRV260X_STATUS:%d\n", error);
		haptics->calib_result = I2C_ERROR;
		return;
	}
	cal_val = cal_val & 0x08;
	if (cal_val) {
		dev_err(&haptics->client->dev,"Auto-calibration failed\n");
		haptics->calib_result = CALIBRATION_FAIL;
		return;
	}

	error = get_nonvolatile(hap_getNvData,
				APNV_BEMF_GAIN, APNV_BEMF_GAIN_SIZE);
	if (unlikely(error < 0)) {
		haptics->feedback_value = 2;
	} else {
		haptics->feedback_value = hap_getNvData[APNV_BEMF_GAIN_LO];
	}

	error = regmap_read(haptics->regmap, DRV260X_FEEDBACK_CTRL, &cal_val );
	if (error) {
		dev_err(&haptics->client->dev,"Failed to read DRV260X_FEEDBACK_CTRL: %d\n", error);
		haptics->calib_result = I2C_ERROR;
		return;
	}

	haptics->feedback_value = cal_val & 0x03;
	error = set_nonvolatile((uint8_t*)&haptics->feedback_value,
				APNV_BEMF_GAIN, sizeof(haptics->feedback_value));
	if (error) {
		dev_err(&haptics->client->dev,"Failed to send feedback_value to NV: %d\n", error);
		haptics->calib_result = SOFT_ERROR;
		return;
	}

	error = regmap_read(haptics->regmap, DRV260X_CAL_COMP, &cal_val );
	if (error) {
		dev_err(&haptics->client->dev,"Failed to read DRV260X_CAL_COMP :%d\n", error);
		haptics->calib_result = I2C_ERROR;
		return;
	}
	haptics->cal_comp_value = cal_val;
	error = set_nonvolatile((uint8_t*)&haptics->cal_comp_value,
				APNV_A_CAL_COMP, sizeof(haptics->cal_comp_value));
	if (error) {
		dev_err(&haptics->client->dev,"Failed to send cal_comp_value to NV: %d\n", error);
		haptics->calib_result = SOFT_ERROR;
		return;
	}

	error = regmap_read(haptics->regmap, DRV260X_CAL_BACK_EMF, &cal_val );
	if (error) {
		dev_err(&haptics->client->dev,"Failed to read DRV260X_CAL_BACK_EMF :%d\n", error);
		haptics->calib_result = I2C_ERROR;
		return;
	}
	haptics->cal_back_emf_value = cal_val;
	error = set_nonvolatile((uint8_t*)&haptics->cal_back_emf_value,
				APNV_A_CAL_BEMF, sizeof(haptics->cal_back_emf_value));
	if (error) {
		dev_err(&haptics->client->dev,"Failed to send cal_back_emf_value to NV: %d\n", error);
		haptics->calib_result = SOFT_ERROR;
		return;
	}
	haptics->calib_result = CALIBRATION_PASS;

	return;
}

static const char hap_calibration_str_on[] = "on";
static ssize_t drv260x_haptics_store_calibration(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
	int state, error;
	struct led_classdev *cdev = dev_get_drvdata(dev);
	struct drv260x_data *haptics = container_of(cdev, struct drv260x_data, cdev);

	if (strncmp(buf, hap_calibration_str_on, sizeof(hap_calibration_str_on)) != 0) {
		dev_err(&haptics->client->dev,"Calibration Param Error!\n");
		return count;
	}

	dev_dbg(&haptics->client->dev,"%s: vibrator start calibration. \n",__func__);
	if (haptics->is_drv2604l == 1) {
		error = regmap_multi_reg_write(haptics->regmap,
						drv2604l_vib_calibration,
						ARRAY_SIZE(drv2604l_vib_calibration));
	} else {
		error = regmap_multi_reg_write(haptics->regmap,
						drv2604_vib_calibration,
						ARRAY_SIZE(drv2604_vib_calibration));
	}
	if (error) {
		dev_err(&haptics->client->dev,"Failed to write set mode: %d\n", error);
		haptics->calib_result = I2C_ERROR;
		goto end;
	}

	error = regmap_write(haptics->regmap, DRV260X_GO ,DRV260X_GO_BIT);
	if (error) {
		dev_err(&haptics->client->dev,"Failed to write set GO bot : %d\n", error);
		haptics->calib_result = I2C_ERROR;
		goto end;
	}

	mdelay(1000);

	error = regmap_read(haptics->regmap,DRV260X_GO, &state);
	if (error) {
		dev_err(&haptics->client->dev,"DRV260X_GO value is read error : %d", error);
		haptics->calib_result = I2C_ERROR;
		goto end;
	}

	if (state != 0) {
		dev_err(&haptics->client->dev,"DRV260X_GO value is error ,and calibration error.\n");
		haptics->calib_result = CALIBRATION_FAIL;
		goto end;
	}

	drv260x_set_caldata(haptics);

end:
	dev_info(&haptics->client->dev,"DRV260X calibration store 0x%x,0x%x,0x%x,0x%x\n",
			haptics->calib_result, haptics->feedback_value,
			haptics->cal_comp_value, haptics->cal_back_emf_value);

	return haptics->calib_result;
}

static ssize_t drv260x_haptics_show_calibration(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct led_classdev *cdev = dev_get_drvdata(dev);
	struct drv260x_data *haptics = container_of(cdev, struct drv260x_data, cdev);

	dev_info(&haptics->client->dev,"DRV260X calibration show 0x%x,0x%x,0x%x,0x%x\n",
			haptics->calib_result, haptics->feedback_value,
			haptics->cal_comp_value, haptics->cal_back_emf_value);

	return snprintf(buf, PAGE_SIZE, "%x,%x,%x\n", haptics->feedback_value,
				haptics->cal_comp_value, haptics->cal_back_emf_value);
}

static ssize_t drv260x_haptics_store_reg(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
	int ret;
	unsigned long value;
	struct led_classdev *cdev = dev_get_drvdata(dev);
	struct drv260x_data *haptics = container_of(cdev, struct drv260x_data, cdev);

	ret = kstrtoul(buf, 16, &value);
	if (ret < 0 || (value > 0xFF)) {
		dev_err(dev, "Register Address is Invalid\n");
		return count;
	}
	haptics->reg_read_addr = (u32)value;

	return count;
}

static ssize_t drv260x_haptics_show_reg(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct led_classdev *cdev = dev_get_drvdata(dev);
	struct drv260x_data *haptics = container_of(cdev, struct drv260x_data, cdev);
	int state, error;

	if (haptics->reg_read_addr > 0xFF) {
		dev_err(&haptics->client->dev,"DRV260X register address error\n");
		return -EINVAL;
	}
	error = regmap_read(haptics->regmap, haptics->reg_read_addr, &state);
	if (error) {
		dev_err(&haptics->client->dev,"DRV260X register read error : %d\n", error);
		return -EIO;
	}

	return snprintf(buf, PAGE_SIZE, "Reg Val:0x%x\n", state);
}

static DEVICE_ATTR(state, 0664, drv260x_haptics_show_state, drv260x_haptics_store_state);
static DEVICE_ATTR(duration, 0664, drv260x_haptics_show_duration, drv260x_haptics_store_duration);
static DEVICE_ATTR(activate, 0664, drv260x_haptics_show_activate, drv260x_haptics_store_activate);
static DEVICE_ATTR(vmax_mv, 0664, drv260x_haptics_show_vmax, drv260x_haptics_store_vmax);
static DEVICE_ATTR(frequency, 0664,drv260x_haptics_show_freq, drv260x_haptics_store_freq);
static DEVICE_ATTR(calibration, 0664,drv260x_haptics_show_calibration, drv260x_haptics_store_calibration);
static DEVICE_ATTR(reg_read, 0664,drv260x_haptics_show_reg, drv260x_haptics_store_reg);

static struct attribute *attributes[] = {
	&dev_attr_state.attr,
	&dev_attr_duration.attr,
	&dev_attr_activate.attr,
	&dev_attr_vmax_mv.attr,
	&dev_attr_frequency.attr,
	&dev_attr_calibration.attr,
	&dev_attr_reg_read.attr,
	NULL
};

static const struct attribute_group attribute_group = {
	.attrs = attributes,
};

static enum led_brightness drv260x_brightness_get(struct led_classdev *cdev)
{
	return 0;
}

static void drv260x_brightness_set(struct led_classdev *cdev,
				enum led_brightness level)
{
}

#if 0
static const struct reg_sequence drv260x_lra_cal_regs[] = {
	{ DRV260X_MODE, DRV260X_AUTO_CAL },
	{ DRV260X_CTRL3, DRV260X_NG_THRESH_2 },
	{ DRV260X_FEEDBACK_CTRL, DRV260X_FB_REG_LRA_MODE |
		DRV260X_BRAKE_FACTOR_4X | DRV260X_LOOP_GAIN_HIGH },
};

static const struct reg_sequence drv260x_lra_init_regs[] = {
	{ DRV260X_MODE, DRV260X_RT_PLAYBACK },
	{ DRV260X_A_TO_V_CTRL, DRV260X_AUDIO_HAPTICS_PEAK_20MS |
		DRV260X_AUDIO_HAPTICS_FILTER_125HZ },
	{ DRV260X_A_TO_V_MIN_INPUT, DRV260X_AUDIO_HAPTICS_MIN_IN_VOLT },
	{ DRV260X_A_TO_V_MAX_INPUT, DRV260X_AUDIO_HAPTICS_MAX_IN_VOLT },
	{ DRV260X_A_TO_V_MIN_OUT, DRV260X_AUDIO_HAPTICS_MIN_OUT_VOLT },
	{ DRV260X_A_TO_V_MAX_OUT, DRV260X_AUDIO_HAPTICS_MAX_OUT_VOLT },
	{ DRV260X_FEEDBACK_CTRL, DRV260X_FB_REG_LRA_MODE |
		DRV260X_BRAKE_FACTOR_2X | DRV260X_LOOP_GAIN_MED |
		DRV260X_BEMF_GAIN_3 },
	{ DRV260X_CTRL1, DRV260X_STARTUP_BOOST },
	{ DRV260X_CTRL2, DRV260X_SAMP_TIME_250 },
	{ DRV260X_CTRL3, DRV260X_NG_THRESH_2 | DRV260X_ANANLOG_IN },
	{ DRV260X_CTRL4, DRV260X_AUTOCAL_TIME_500MS },
};

static const struct reg_sequence drv260x_erm_cal_regs[] = {
	{ DRV260X_MODE, DRV260X_AUTO_CAL },
	{ DRV260X_A_TO_V_MIN_INPUT, DRV260X_AUDIO_HAPTICS_MIN_IN_VOLT },
	{ DRV260X_A_TO_V_MAX_INPUT, DRV260X_AUDIO_HAPTICS_MAX_IN_VOLT },
	{ DRV260X_A_TO_V_MIN_OUT, DRV260X_AUDIO_HAPTICS_MIN_OUT_VOLT },
	{ DRV260X_A_TO_V_MAX_OUT, DRV260X_AUDIO_HAPTICS_MAX_OUT_VOLT },
	{ DRV260X_FEEDBACK_CTRL, DRV260X_BRAKE_FACTOR_3X |
		DRV260X_LOOP_GAIN_MED | DRV260X_BEMF_GAIN_2 },
	{ DRV260X_CTRL1, DRV260X_STARTUP_BOOST },
	{ DRV260X_CTRL2, DRV260X_SAMP_TIME_250 | DRV260X_BLANK_TIME_75 |
		DRV260X_IDISS_TIME_75 },
	{ DRV260X_CTRL3, DRV260X_NG_THRESH_2 | DRV260X_ERM_OPEN_LOOP },
	{ DRV260X_CTRL4, DRV260X_AUTOCAL_TIME_500MS },
};
#endif

static int drv260x_init(struct drv260x_data *haptics)
{
	int error;

	if (haptics->is_drv2604l == 1) {
		error = regmap_register_patch(haptics->regmap,
						drv2604l_reg_init,
						ARRAY_SIZE(drv2604l_reg_init));
	} else {
		error = regmap_register_patch(haptics->regmap,
						drv2604_reg_init,
						ARRAY_SIZE(drv2604_reg_init));
	}
	if (error) {
		dev_err(&haptics->client->dev,"Failed to init: %d\n",error);
		return error;
	}

	return 0;
}

#include <linux/i2c-algo-bit.h>
static int drv260x_reinit(struct drv260x_data *haptics)
{
	int error = 0;
	struct i2c_algo_bit_data *bit_data;

	dev_info(&haptics->client->dev, "%s :start hw recovery\n", __func__);
	bit_data = haptics->client->adapter->algo_data;
	if (!bit_data) {
		dev_info(&haptics->client->dev,"%s :drv260x recovery algo data is null\n",__func__);
		goto out;
	}
	/* Reset Haptics IC */
	gpiod_set_value(haptics->trigger_gpio, 0);
	gpiod_set_value(haptics->enable_gpio, 0);
	bit_data->setscl(bit_data->data, 0);
	bit_data->setsda(bit_data->data, 0);
	gpiod_set_value(haptics->hap_pon, 0);
	mdelay(1200);
	gpiod_set_value(haptics->hap_pon, 1);
	udelay(100);
	bit_data->setscl(bit_data->data, 1);
	bit_data->setsda(bit_data->data, 1);
	udelay(100);
	gpiod_set_value(haptics->enable_gpio, 1);
	udelay(250);

	/* Reinit Setting */
	if (haptics->is_drv2604l == 1) {
		error = regmap_multi_reg_write(haptics->regmap,
										drv2604l_reg_init,
										ARRAY_SIZE(drv2604l_reg_init));
	} else {
		error = regmap_multi_reg_write(haptics->regmap,
										drv2604_reg_init,
										ARRAY_SIZE(drv2604_reg_init));
	}
	if (error) {
		dev_err(&haptics->client->dev,"Failed to init: %d\n",error);
		goto out;
	}

	error = regmap_write(haptics->regmap, DRV260X_RATED_VOLT,
		haptics->rated_voltage_normal_value);
	if (error) {
		dev_err(&haptics->client->dev,
			"Failed to write set DRV260X_RATED_VOLT,%d\n", error);
		goto out;
	}

	if (haptics->is_drv2604l == 1) {
		error = regmap_write(haptics->regmap, DRV260X_LRA_LOOP_PERIOD,
					haptics->lra_loop_period_value);
		if (error) {
			dev_err(&haptics->client->dev,
				"Failed to write set DRV260X_LRA_LOOP_PERIOD,%d\n", error);
			goto out;
		}
	}

	error = regmap_write(haptics->regmap, DRV260X_LRA_RES_PERIOD,
				haptics->lra_res_period_value);
	if (error) {
		dev_err(&haptics->client->dev,
			"Failed to write set DRV260X_LRA_RES_PERIOD ,%d\n", error);
		goto out;
	}

	if (haptics->is_drv2604l == 1) {
		error = regmap_write(haptics->regmap, DRV260X_FEEDBACK_CTRL,
					(haptics->feedback_value  | (0xF4 & 0xFC)));
	} else {
		error = regmap_write(haptics->regmap, DRV260X_FEEDBACK_CTRL, 0xF4);
	}
	if (error) {
		dev_err(&haptics->client->dev,
			"Failed to write set DRV260X_FEEDBACK_CTRL ,%d\n", error);
		goto out;
	}

	error = regmap_write(haptics->regmap, DRV260X_CAL_COMP,
				haptics->cal_comp_value);
	if (error) {
		dev_err(&haptics->client->dev,
			"Failed to write set DRV260X_CAL_COMP, %d\n", error);
		goto out;
	}

	error = regmap_write(haptics->regmap, DRV260X_CAL_BACK_EMF,
				haptics->cal_back_emf_value);
	if (error) {
		dev_err(&haptics->client->dev,
			"Failed to write set DRV260X_CAL_BACK_EMF,%d\n", error);
		goto out;
	}

	dev_info(&haptics->client->dev,"%s :hw recovery success\n",__func__);

out:
	return error;
}

static const struct regmap_config drv260x_regmap_config = {
	.reg_bits = 8,
	.val_bits = 8,

#if 0
	.max_register = DRV260X_MAX_REG,
	.reg_defaults = drv260x_reg_defs,
	.num_reg_defaults = ARRAY_SIZE(drv260x_reg_defs),
#endif
	.cache_type = REGCACHE_NONE,
};

static int drv260x_probe(struct i2c_client *client,
			 const struct i2c_device_id *id)
{
	struct device *dev = &client->dev;
	struct drv260x_data *haptics;
	u32 voltage;
	int error;

	haptics = devm_kzalloc(dev, sizeof(*haptics), GFP_KERNEL);
	if (!haptics)
		return -ENOMEM;

	mutex_init(&haptics->voltage_mutex);
	error = device_property_read_u32(dev, "mode", &haptics->mode);
	if (error) {
		dev_err(dev, "Can't fetch 'mode' property: %d\n", error);
		return error;
	}

	if (haptics->mode < DRV260X_LRA_MODE ||
	    haptics->mode > DRV260X_ERM_MODE) {
		dev_err(dev, "Vibrator mode is invalid: %i\n", haptics->mode);
		return -EINVAL;
	}

	error = device_property_read_u32(dev, "library-sel", &haptics->library);
	if (error) {
		dev_err(dev, "Can't fetch 'library-sel' property: %d\n", error);
		return error;
	}

	if (haptics->library < DRV260X_LIB_EMPTY ||
	    haptics->library > DRV260X_ERM_LIB_F) {
		dev_err(dev,
			"Library value is invalid: %i\n", haptics->library);
		return -EINVAL;
	}

	if (haptics->mode == DRV260X_LRA_MODE &&
	    haptics->library != DRV260X_LIB_EMPTY &&
	    haptics->library != DRV260X_LIB_LRA) {
		dev_err(dev, "LRA Mode with ERM Library mismatch\n");
		return -EINVAL;
	}

	if (haptics->mode == DRV260X_ERM_MODE &&
	    (haptics->library == DRV260X_LIB_EMPTY ||
	     haptics->library == DRV260X_LIB_LRA)) {
		dev_err(dev, "ERM Mode with LRA Library mismatch\n");
		return -EINVAL;
	}

	error = device_property_read_u32(dev, "vib-rated-mv", &voltage);
	haptics->rated_voltage = error ? DRV260X_DEF_RATED_VOLT :
					 drv260x_calculate_voltage(voltage);

	error = device_property_read_u32(dev, "vib-overdrive-mv", &voltage);
	haptics->overdrive_voltage = error ? DRV260X_DEF_OD_CLAMP_VOLT :
					     drv260x_calculate_voltage(voltage);

	haptics->regulator = devm_regulator_get(dev, "vbat");
	if (IS_ERR(haptics->regulator)) {
		error = PTR_ERR(haptics->regulator);
		dev_err(dev, "unable to get regulator, error: %d\n", error);
		return error;
	}

	haptics->hap_pon =devm_gpiod_get_optional(dev, "pon",
							GPIOD_OUT_HIGH);
	if (IS_ERR(haptics->hap_pon))
		return PTR_ERR(haptics->hap_pon);

	haptics->trigger_gpio = devm_gpiod_get_optional(dev,"trigger",
							GPIOD_OUT_LOW);
	if (IS_ERR(haptics->trigger_gpio))
		return PTR_ERR(haptics->trigger_gpio);

	haptics->enable_gpio = devm_gpiod_get_optional(dev, "enable",
							GPIOD_OUT_LOW);
	if (IS_ERR(haptics->enable_gpio))
		return PTR_ERR(haptics->enable_gpio);

	haptics->input_dev = devm_input_allocate_device(dev);
	if (!haptics->input_dev) {
		dev_err(dev, "Failed to allocate input device\n");
		return -ENOMEM;
	}

	haptics->input_dev->name = "drv260x:haptics";
	haptics->input_dev->close = drv260x_close;
	input_set_drvdata(haptics->input_dev, haptics);
#if 0
	input_set_capability(haptics->input_dev, EV_FF, FF_RUMBLE);

	error = input_ff_create_memless(haptics->input_dev, NULL,
					drv260x_haptics_play);
	if (error) {
		dev_err(dev, "input_ff_create() failed: %d\n", error);
		return error;
	}
#endif

	INIT_WORK(&haptics->work, drv260x_worker);

	haptics->client = client;
	i2c_set_clientdata(client, haptics);
	hrtimer_init(&haptics->stop_timer, CLOCK_MONOTONIC, HRTIMER_MODE_REL);
	haptics->stop_timer.function = hap_stop_timer;

	haptics->cdev.name = "vibrator";
	haptics->cdev.brightness_get = drv260x_brightness_get;
	haptics->cdev.brightness_set = drv260x_brightness_set;
	haptics->cdev.max_brightness = 100;
	error = devm_led_classdev_register(&client->dev, &haptics->cdev);
	if (error) {
		dev_err(&client->dev, "Error in registering led class device, error=%d\n",error);
		goto led_reg_fail;
	}

	error = sysfs_create_group(&haptics->cdev.dev->kobj, &attribute_group);
	if (error < 0) {
		dev_err(&client->dev, "Error in creating sysfs file, rc=%d\n",error);
		goto led_reg_fail;
	}

	gpiod_set_value(haptics->hap_pon, 1);
	udelay(100);
	gpiod_set_value(haptics->enable_gpio, 1);
	udelay(250);
	haptics->regmap = devm_regmap_init_i2c(client, &drv260x_regmap_config);
	if (IS_ERR(haptics->regmap)) {
		error = PTR_ERR(haptics->regmap);
		dev_err(dev, "Failed to allocate register map: %d\n", error);
		goto dev_init_fail; 
	}

	{
		u32 device_type = 0;

		error = regmap_read(haptics->regmap, DRV260X_STATUS, &device_type);
		if (error) {
			dev_err(&haptics->client->dev,"Failed to read DRV260X_STATUS:%d\n", error);
			return error;
		}
		device_type = (device_type & 0x000000E0) >> 5;
		if (device_type == 6) {
			haptics->is_drv2604l = 1;
			dev_info(&haptics->client->dev, "Haptics IC is DRV2604L\n");
		} else {
			haptics->is_drv2604l = 0;
			dev_info(&haptics->client->dev, "Haptics IC is DRV2604\n");
		}
	}

	error = drv260x_init(haptics);
	if (error) {
		dev_err(dev, "Device init failed: %d\n", error);
		goto dev_init_fail; 
	}

	error = drv260x_get_NV_data(haptics);
	if (error) {
		dev_err(dev, "get NV data failed: %d\n", error);
	}

	error = input_register_device(haptics->input_dev);
	if (error) {
		dev_err(dev, "couldn't register input device: %d\n", error);
		goto dev_init_fail; 
	}

	haptics->calib_result = CALIBRATION_NOT_EXECUTED;

	return 0;

dev_init_fail:
	sysfs_remove_group(&haptics->cdev.dev->kobj, &attribute_group);
led_reg_fail:
	cancel_work_sync(&haptics->work);
	hrtimer_cancel(&haptics->stop_timer);
	i2c_set_clientdata(client, NULL);

	return error;
}

static int drv260x_remove(struct i2c_client *client)
{
	struct drv260x_data *haptics = i2c_get_clientdata(client);
	int  ret;

	dev_info(&client->dev,"%s :Enter into drv260x remove.\n",__func__);

	if (!haptics) {
		dev_info(&client->dev,"%s :data pointer is null.\n",__func__);
		return 0;
	}
	sysfs_remove_group(&haptics->cdev.dev->kobj, &attribute_group);
	cancel_work_sync(&haptics->work);
	hrtimer_cancel(&haptics->stop_timer);
	devm_led_classdev_unregister(&client->dev, &haptics->cdev);
	gpiod_set_value(haptics->enable_gpio, 0);
	gpiod_set_value(haptics->hap_pon, 0);

	ret = regulator_disable(haptics->regulator);
	if (ret) {
		dev_err(&client->dev, "Failed to disable regulator\n");
		regmap_update_bits(haptics->regmap,
					DRV260X_MODE,
					DRV260X_STANDBY_MASK, 0);
	}
	mutex_destroy(&haptics->voltage_mutex);
	i2c_set_clientdata(client, NULL);

	return 0;
}

static void drv260x_shutdown(struct i2c_client *client)
{
	dev_info(&client->dev,"%s :Enter into drv260x shutdown.\n",__func__);

	drv260x_remove(client);
}

static int __maybe_unused drv260x_suspend(struct device *dev)
{
	struct drv260x_data *haptics = dev_get_drvdata(dev);
	int ret = 0;

	mutex_lock(&haptics->input_dev->mutex);

	if (gpiod_get_value(haptics->enable_gpio)) {
		dev_info(dev,"%s :set standby mode\n",__func__);
		ret = regmap_write(haptics->regmap, DRV260X_MODE, 0x40);
		if (ret) {
			dev_err(dev, "Failed to set standby mode\n");
			ret = drv260x_reinit(haptics);
			if (ret) {
				dev_err(dev, "hw recovery fail retry later\n");
				haptics->retry_recovery = 1;
				ret = 0;
			} else {
				haptics->retry_recovery = 0;
				regmap_write(haptics->regmap, DRV260X_MODE, 0x40);
			}
		}

		gpiod_set_value(haptics->enable_gpio, 0);

		ret = regulator_disable(haptics->regulator);
		if (ret) {
			dev_err(dev, "Failed to disable regulator\n");
			gpiod_set_value(haptics->enable_gpio, 1);
			udelay(250);
			regmap_write(haptics->regmap, DRV260X_MODE, 0x00);
		}
	}
	mutex_unlock(&haptics->input_dev->mutex);
	return ret;
}

static int __maybe_unused drv260x_resume(struct device *dev)
{
	struct drv260x_data *haptics = dev_get_drvdata(dev);
	int ret = 0;

	mutex_lock(&haptics->input_dev->mutex);

	if (!gpiod_get_value(haptics->enable_gpio)) {
		dev_info(dev,"%s :set normal mode\n",__func__);
		ret = regulator_enable(haptics->regulator);
		if (ret) {
			dev_err(dev, "Failed to enable regulator\n");
			goto out;
		}

		if (haptics->retry_recovery == 1) {
			ret = drv260x_reinit(haptics);
			if (ret) {
				dev_err(dev, "hw recovery fail retry later\n");
				haptics->retry_recovery = 1;
				ret = 0;
			} else {
				haptics->retry_recovery = 0;
			}
			goto out;
		}

		gpiod_set_value(haptics->enable_gpio, 1);
		udelay(250);

		ret = regmap_write(haptics->regmap, DRV260X_MODE, 0x00);
		if (ret) {
			dev_err(dev, "Failed to unset standby mode\n");
			ret = drv260x_reinit(haptics);
			if (ret) {
				dev_err(dev, "hw recovery fail retry later\n");
				haptics->retry_recovery = 1;
				ret = 0;
			} else {
				haptics->retry_recovery = 0;
			}
			goto out;
		}
	}

out:
	mutex_unlock(&haptics->input_dev->mutex);
	return ret;
}

static SIMPLE_DEV_PM_OPS(drv260x_pm_ops, drv260x_suspend, drv260x_resume);

static const struct i2c_device_id drv260x_id[] = {
	{ "drv2605l", 0 },
	{ }
};
MODULE_DEVICE_TABLE(i2c, drv260x_id);

static const struct of_device_id drv260x_of_match[] = {
	{ .compatible = "ti,drv2604", },
	{ .compatible = "ti,drv2604l", },
	{ .compatible = "ti,drv2605", },
	{ .compatible = "ti,drv2605l", },
	{ }
};
MODULE_DEVICE_TABLE(of, drv260x_of_match);

static struct i2c_driver drv260x_driver = {
	.probe		= drv260x_probe,
	.remove		= drv260x_remove,
	.shutdown	= drv260x_shutdown,
	.driver		= {
		.name	= "drv260x-haptics",
		.of_match_table = drv260x_of_match,
		.pm	= &drv260x_pm_ops,
	},
	.id_table = drv260x_id,
};
module_i2c_driver(drv260x_driver);

MODULE_DESCRIPTION("TI DRV260x haptics driver");
MODULE_LICENSE("GPL");
MODULE_AUTHOR("Dan Murphy <dmurphy@ti.com>");
