/*
 * (C) 2021 FCNT LIMITED
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
//==============================================================================
// include file
//==============================================================================
#include <linux/gpio.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/slab.h>
#include <linux/sched.h>
#include <linux/mutex.h>
#include <linux/power_supply.h>
#include <linux/of.h>
#include <linux/of_gpio.h>
#include <linux/printk.h>
#include <linux/moduleparam.h>

#include <linux/debugfs.h>
#include <linux/fs.h>
#include <linux/seq_file.h>
#include <linux/pm_wakeup.h>

#include "custom_det.h"
#include <linux/nonvolatile_common.h>		
#ifdef FEATURE_OEM_CHARGE
#include <linux/mfd/oem_charger.h>
#endif /* FEATURE_OEM_CHARGE */

#ifdef FEATURE_CUSTOM_DET_DRIVER

//==============================================================================
// define & enum
//==============================================================================
/*
 * OEM Chager dummy
 */
#ifndef FEATURE_OEM_CHARGE
	#define OEM_CHG_ERROR_OVP_NONE				/* dummy */(0)
	#define OEM_CHG_ERROR_OVP_DETECT				/* dummy */(1)
	#define OEM_CHG_ERROR_USB_HOT_DETECT			/* dummy */(2)
	#define OEM_CHG_ERROR_USB_HOT_NONE			/* dummy */(3)
	#define OEM_CHG_USB_CURRENT					/* dummy */(1)
	#define OEM_CHG_OFF_CURRENT					/* dummy */(0)
	#define POWER_SUPPLY_PROP_USB_ID_CODE		/* dummy */(0)
	#define POWER_SUPPLY_PROP_USB_ID_PHYS		/* dummy */(1)
	#define oem_chg_notify_error(type) 			/* nop */
	#define oem_chg_other_vbus_draw(type)		/* nop */
	#define oem_chg_mhl_vbus_draw(type)			/* nop */
	#define oem_chg_holder_vbus_draw(type)		/* nop */
	#define oem_chg_apsd_vbus_draw(type)			/* nop */
#endif /* FEATURE_OEM_CHARGE */

/*
 * Interrupt
 */
typedef enum {
	
	CUSTOM_DET_IRQ_ADP_SAFE = 0,
	CUSTOM_DET_IRQ_USB_HOT,
	
	CUSTOM_DET_IRQ_TOTAL,
} CUSTOM_DET_IRQ_TYPE;

typedef enum {
	CUSTOM_DET_IRQ_FREE = 0,
	CUSTOM_DET_IRQ_ENABLE,
	CUSTOM_DET_IRQ_DISABLE
} CUSTOM_DET_IRQ_OPE;

/*
 * GPIO
 */
typedef enum {
	
	CUSTOM_DET_GPIO_ADP_SAFE = 0,
	CUSTOM_DET_GPIO_USB_HOT,
	CUSTOM_DET_GPIO_USB_THSEL0,
	CUSTOM_DET_GPIO_USB_THSEL1,
	
	CUSTOM_DET_GPIO_OTG_EN,
	
	CUSTOM_DET_GPIO_VBDET,
	
	CUSTOM_DET_GPIO_TOTAL,
} CUSTOM_DET_GPIO_TYPE;

typedef enum {
	CUSTOM_DET_GPIO_FREE = 0,
	CUSTOM_DET_GPIO_IN,
	CUSTOM_DET_GPIO_OUT_L,
	CUSTOM_DET_GPIO_OUT_H,
	CUSTOM_DET_GPIO_GET
} CUSTOM_DET_GPIO_OPE;

#define CUSTOM_DET_GPIO(p, type)			((p)->gpios[(type)])
#define CUSTOM_DET_GPIO_IS_VALID(p, type)	gpio_is_valid((p)->gpios[(type)])

/*
 * wake lock
 */
typedef enum {
	CUSTOM_DET_WAKELOCK_USBIN_OV = 0,
	CUSTOM_DET_WAKELOCK_DCIN_OV,
	
	CUSTOM_DET_WAKELOCK_ADP_SAFE,
	CUSTOM_DET_WAKELOCK_USB_HOT,
	
	
	CUSTOM_DET_WAKELOCK_VBDET,
	
	CUSTOM_DET_WAKELOCK_TOTAL,
} CUSTOM_DET_WAKELOCK_TYPE;

#define CUSTOM_DET_WAKE_LOCK(p, type) __pm_stay_awake((p)->wake_src[(type)])
#define CUSTOM_DET_WAKE_LOCK_TIMEOUT(p, type, time_ms) __pm_wakeup_event((p)->wake_src[(type)], time_ms)
#define CUSTOM_DET_WAKE_UNLOCK(p, type) __pm_relax((p)->wake_src[(type)])
#define CUSTOM_DET_WAKE_ACTIVE(p, type) ((p)->wake_src[(type)]->active)

/*
 * mutex
 */
typedef enum {
	CUSTOM_DET_MUTEX_OVP_STATE = 0,
	CUSTOM_DET_MUTEX_SAFETY_STATE,	
	CUSTOM_DET_MUTEX_TOTAL,
} CUSTOM_DET_MUTEX_TYPE;

#define CUSTOM_DET_MUTEX_LOCK(p, type)			mutex_lock(&(p)->mutexs[(type)])
#define CUSTOM_DET_MUTEX_UNLOCK(p, type)		mutex_unlock(&(p)->mutexs[(type)])
#define CUSTOM_DET_MUTEX_INIT(p, type)			mutex_init(&(p)->mutexs[(type)])

/*
 * ovp
 */
typedef enum {
	CUSTOM_DET_OVP_NONE = 0,
	CUSTOM_DET_OVP_DETECT,
	CUSTOM_DET_OVP_INIT = -1,
} CUSTOM_DET_OVP_STATUS;

typedef enum {
	CUSTOM_DET_OVP_USB = 0,
	CUSTOM_DET_OVP_DC,
} CUSTOM_DET_OVP_TYPE;

/*
 * customdet_state
 */
/* for charging mode */
#define CUSTOM_DET_DEVICE_CHATTERING		(255)
#define CUSTOM_DET_DEVICE_PROBE_END			(254)
#define CUSTOM_DET_DEVICE_CHATTER_END		(253)

/*
 * Chattering
 */
#define CUSTOM_DET_CHATTERING_COUNT_UP(now_value, pre_value, chatt_cnt) \
	if (pre_value == now_value) {	\
		chatt_cnt++;				\
	} else {						\
		pre_value = now_value;		\
		chatt_cnt = 1;				\
	}

#define CUSTOM_DET_CHATTERING_COMP_CHECK(chatt_cnt, detectcnt) \
	(chatt_cnt >= detectcnt) ? true : false

#define CUSTOM_DET_DETECT_COUNT				(30)
#define CUSTOM_DET_DETECT_COUNT_MAX			(600)
#define CUSTOM_DET_DETECT_WAIT_MS			(25)
#define CUSTOM_DET_USB_HOT_DETECT_COUNT		(1)
#define CUSTOM_DET_VBDET_DETECT_COUNT		(5)

#define CUSTOM_DET_CHATT			(-1)

#define CUSTOM_DET_CHECK_DCSTS		BIT(0)
#define CUSTOM_DET_CHECK_USBSTS		BIT(1)

#define CUSTOM_DET_CHECK_ADPSAFE	BIT(2)
#define CUSTOM_DET_CHECK_USBHOT		BIT(3)
#define CUSTOM_DET_CHECK_VBDET		BIT(4)

#define CUSTOM_DET_CHECK_INIT		( \
								CUSTOM_DET_CHECK_DCSTS   | \
								CUSTOM_DET_CHECK_VBDET   | \
								CUSTOM_DET_CHECK_USBSTS  | \
								CUSTOM_DET_CHECK_ADPSAFE | \
								CUSTOM_DET_CHECK_USBHOT \
								)
#define CUSTOM_DET_BIT_CHK(a, bit)	((bit) == ((a) & (bit)))

/*
 * USB Safety
 */
typedef enum {
	CUSTOM_DET_USB_SAFETY_ADP_SAFE = 0,
	CUSTOM_DET_USB_SAFETY_USB_TEMP,
	CUSTOM_DET_USB_SAFETY_MAX
} CUSTOM_DET_USB_SAFETY;

typedef enum {
	CUSTOM_DET_USB_TEMP_ERROR = 0,
	CUSTOM_DET_USB_TEMP_NORMAL,
	CUSTOM_DET_USB_TEMP_MAX
} CUSTOM_DET_USB_TEMP_FUNC;

#define OEMDET_USB_THSEL_PATTERN_NUM		(60)

#define OEMDET_USB_THSEL_WAIT_US			(10)  
#define OEMDET_USB_THSEL_WAIT_US_MAX		(255)
#define OEMDET_USB_THSEL_AFTER_WAIT_US	(1000)
#define OEMDET_USB_THSEL_PATTERN1_P1		(0xC92492492492491)
#define OEMDET_USB_THSEL_PATTERN1_P2		(0x9C0E001C0038E3B)
#define OEMDET_USB_THSEL_PATTERN2_P1		(0xC92492492492491)
#define OEMDET_USB_THSEL_PATTERN2_P2		(0x9C0E001C000003B)

/*
 * CUSTOM_DET Feature Data
 */
#define OEMDET_VBDET_EN "custom,vbdet_en"
#define OEMDET_VCDET_EN "custom,vcdet_en"

static int vb_status = -1;

#define DEMO_JYUDEN_70             49283 
#define OEMDET_FLG_VCDET            0x01
#define OEMDET_FLG_ADP_SAFE         0x02
#define OEMDET_FLG_USB_HOT          0x04
#define OEMDET_FLG_USB_THSEL        0x08
#define OEMDET_FLG_VBDET            0x10
#define OEMDET_FLG_USBID_CHG        0x20
#define OEMDET_FLG_USBID_R10K       0x40

#define CUSTOM_DET_FEATURE_CHK(p, type) CUSTOM_DET_BIT_CHK((p)->feature_check, OEMDET_FLG_##type)

#define OEMDET_DEFAULT_POWER_ROLE   POWER_SUPPLY_TYPEC_PR_SINK

typedef enum {
	CUSTOM_DET_POWER_ROLE_NONE = 0,
	CUSTOM_DET_POWER_ROLE_DRP,
	CUSTOM_DET_POWER_ROLE_UFP,
	CUSTOM_DET_POWER_ROLE_DFP,
	CUSTOM_DET_POWER_ROLE_TOTAL,
} CUSTOM_DET_POWER_ROLE;

//==============================================================================
// struct
//==============================================================================
/*
 * Device Status Information
 */
struct status_info {
	CUSTOM_DET_OVP_STATUS ovp_state_dc;
	CUSTOM_DET_OVP_STATUS ovp_state_usb;
	
	int adp_safe;
	int usb_hot;
	
	
	int vb_det;
	
};

/*
 * Interrupt Information
 */
struct custom_det_irq_info {
	int irq_no;
	CUSTOM_DET_IRQ_OPE ope;
};

/*
 * GPIO Setting Information
 */
struct custom_det_gpio_setup_info {
	char * devtree_name;
	char * name;
	CUSTOM_DET_GPIO_OPE setting;
};

/*
 * PowerRole Information
 */
struct custom_det_power_role_info {
	char * power_role_name;
	int  power_role_val;
};


/*
 * CUSTOM_DET_DRIVER Platform Data
 */
struct custom_det_platform_data {
	void *_chip;						/* smbchg_chip */
	struct smb_charger *chg;			/* chg device */
	u8		ovp_detect;					/* over voltage detect */
	int		usb_safety_err_detect;		/* usb safety error detect */	
	u8		feature_check;				/* usb detect feature check flg */
										/* GPIO No */
	int		gpios[CUSTOM_DET_GPIO_TOTAL];
										/* irq No */
	struct custom_det_irq_info irq_info[CUSTOM_DET_IRQ_TOTAL];
										/* wakelock */
	struct wakeup_source *wake_src[CUSTOM_DET_WAKELOCK_TOTAL];
										/* mutex */
	struct mutex mutexs[CUSTOM_DET_MUTEX_TOTAL];
#ifdef CONFIG_DEBUG_FS
	struct dentry *customdet_debugfs_root;
	bool		debug_otg_disable;
#endif
};

//==============================================================================
// const
//==============================================================================
static const char * custom_det_irq_name[CUSTOM_DET_IRQ_TOTAL] = {
	"custom_adp_safe",
	"custom_usb_hot",
};

static const struct custom_det_gpio_setup_info custom_det_gpio_setup[CUSTOM_DET_GPIO_TOTAL] = {
	
	{"custom,adp_safe-gpio",	"OEMDET_ADP_SAFE",	CUSTOM_DET_GPIO_IN},
	{"custom,usb_hot-gpio",		"OEMDET_USB_HOT",	CUSTOM_DET_GPIO_IN},
	{"custom,usb_thsel0-gpio",	"OEMDET_USB_THSEL0",	CUSTOM_DET_GPIO_OUT_L},
	{"custom,usb_thsel1-gpio",	"OEMDET_USB_THSEL1",	CUSTOM_DET_GPIO_OUT_H},
	
	{"custom,otg_en-gpio",		"OEMDET_OTG_EN",		CUSTOM_DET_GPIO_OUT_L},
	{"custom,vb_det-gpio",		"OEMDET_VBDET",		CUSTOM_DET_GPIO_IN},
};

static const char * custom_det_wakelock_setup[CUSTOM_DET_WAKELOCK_TOTAL] = {
	"custom_usbin_ov_wake_lock",
	"custom_dcin_ov_wake_lock",
	
	"custom_adp_safe_wake_lock",
	"custom_usb_hot_wake_lock",
	
};

static const struct custom_det_power_role_info custom_det_power_role[CUSTOM_DET_POWER_ROLE_TOTAL] = {
	{"none", POWER_SUPPLY_TYPEC_PR_NONE}, /* CUSTOM_DET_POWER_ROLE_NONE */
	{"drp",  POWER_SUPPLY_TYPEC_PR_DUAL}, /* CUSTOM_DET_POWER_ROLE_DRP */
	{"ufp",  POWER_SUPPLY_TYPEC_PR_SINK}, /* CUSTOM_DET_POWER_ROLE_UFP */
	{"dfp",  POWER_SUPPLY_TYPEC_PR_SOURCE}, /* CUSTOM_DET_POWER_ROLE_DFP */
};

//==============================================================================
// static valuable
//==============================================================================
static struct custom_det_platform_data custom_platform_data;
static unsigned int customdet_state = CUSTOM_DET_DEVICE_CHATTERING;
static bool   chg_req_state = false;
static bool chg_notify_err_delay_flg = false; 

static unsigned int usb_temp = 0;	
static unsigned int usb_temp_wait_us = OEMDET_USB_THSEL_WAIT_US;

static int power_role= OEMDET_DEFAULT_POWER_ROLE;
static int default_power_role= OEMDET_DEFAULT_POWER_ROLE;

//==============================================================================
// external reference
//==============================================================================
#ifdef CONFIG_DEBUG_FS
extern void smb5_debug_interrupt_statistics(struct seq_file *file, void *data);
#endif

//==============================================================================
// functions
//==============================================================================
static void _custom_det_irq_control_init(struct custom_det_platform_data *custom_pdata, CUSTOM_DET_IRQ_TYPE type, int irq_no)
{
	custom_pdata->irq_info[type].irq_no = irq_no;
	custom_pdata->irq_info[type].ope = CUSTOM_DET_IRQ_ENABLE;
}

static void _custom_det_irq_control(struct custom_det_platform_data *custom_pdata, CUSTOM_DET_IRQ_TYPE type, CUSTOM_DET_IRQ_OPE ope)
{
	switch (ope) {
	case CUSTOM_DET_IRQ_ENABLE:
		if (custom_pdata->irq_info[type].ope == CUSTOM_DET_IRQ_DISABLE) {
			enable_irq(custom_pdata->irq_info[type].irq_no);
			custom_pdata->irq_info[type].ope = ope;
		}
		break;
	case CUSTOM_DET_IRQ_DISABLE:
		if (custom_pdata->irq_info[type].ope == CUSTOM_DET_IRQ_ENABLE) {
			disable_irq(custom_pdata->irq_info[type].irq_no);
			custom_pdata->irq_info[type].ope = ope;
		}
		break;
	case CUSTOM_DET_IRQ_FREE:
		if (custom_pdata->irq_info[type].ope != CUSTOM_DET_IRQ_FREE) {
			free_irq(custom_pdata->irq_info[type].irq_no, NULL);
			custom_pdata->irq_info[type].ope = ope;
		}
		break;
	}
}

static bool _custom_det_gpio_control_init(struct device_node *node, struct custom_det_platform_data *custom_pdata, CUSTOM_DET_GPIO_TYPE type, char* dt_name, char* gpio_name)
{
	bool ret = false;
	int rc;

	do {
		custom_pdata->gpios[type] = of_get_named_gpio_flags(node, dt_name, 0, NULL);

		if (!CUSTOM_DET_GPIO_IS_VALID(custom_pdata, type)) {
			CUSTOM_DET_LOG("%s invalid gpio\n", dt_name);
			break;
		}
		rc = gpio_request(custom_pdata->gpios[type], gpio_name);
		if (rc != 0) {
			CUSTOM_DET_LOG("gpio_request err %s rc=%d\n", gpio_name, rc);
			custom_pdata->gpios[type] = -1;
			break;
		}
		ret = true;
	} while (0);

	return ret;
}

static int _custom_det_gpio_control(struct custom_det_platform_data *custom_pdata, CUSTOM_DET_GPIO_TYPE type, CUSTOM_DET_GPIO_OPE ope)
{
	int ret = -1;

	if (CUSTOM_DET_GPIO_IS_VALID(custom_pdata, type)) {
		ret = 0;
		switch (ope) {
		case CUSTOM_DET_GPIO_FREE:
			gpio_free(custom_pdata->gpios[type]);
			custom_pdata->gpios[type] = -1;
			break;
		case CUSTOM_DET_GPIO_IN:
			gpio_direction_input(custom_pdata->gpios[type]);
			break;
		case CUSTOM_DET_GPIO_OUT_L:
			gpio_direction_output(custom_pdata->gpios[type], 0);
			break;
		case CUSTOM_DET_GPIO_OUT_H:
			gpio_direction_output(custom_pdata->gpios[type], 1);
			break;
		case CUSTOM_DET_GPIO_GET:
			ret = gpio_get_value(custom_pdata->gpios[type]);
		default:
			break;
		}
	}

	return ret;
}

static void _custom_det_feature_check(struct custom_det_platform_data *custom_pdata)
{
	int result = 0;
	u8 val;

#if 0
	if (of_property_read_bool(custom_pdata->chg->dev->of_node, OEMDET_VBDET_EN))
		custom_pdata->feature_check |= OEMDET_FLG_VBDET;
#else
	if (CUSTOM_DET_GPIO_IS_VALID(custom_pdata, CUSTOM_DET_GPIO_VBDET))
		custom_pdata->feature_check |= OEMDET_FLG_VBDET;
#endif

	if (of_property_read_bool(custom_pdata->chg->dev->of_node, OEMDET_VCDET_EN))
		custom_pdata->feature_check |= OEMDET_FLG_VCDET;

	if (CUSTOM_DET_GPIO_IS_VALID(custom_pdata, CUSTOM_DET_GPIO_ADP_SAFE)) {
		result = get_nonvolatile(&val, DEMO_JYUDEN_70, 1);
		if (result < 0)
			val = 0x00;

		if (val == 0x00)
			custom_pdata->feature_check |= OEMDET_FLG_ADP_SAFE;
	}

	if (CUSTOM_DET_GPIO_IS_VALID(custom_pdata, CUSTOM_DET_GPIO_USB_HOT))
		custom_pdata->feature_check |= OEMDET_FLG_USB_HOT;

	if (CUSTOM_DET_GPIO_IS_VALID(custom_pdata, CUSTOM_DET_GPIO_USB_THSEL0) && CUSTOM_DET_GPIO_IS_VALID(custom_pdata, CUSTOM_DET_GPIO_USB_THSEL1))
		custom_pdata->feature_check |= OEMDET_FLG_USB_THSEL;

	CUSTOM_DET_LOG("feature_check value = %u\n", custom_pdata->feature_check);
}

static int _custom_det_ov_status_get(struct custom_det_platform_data *custom_pdata, CUSTOM_DET_OVP_TYPE type, CUSTOM_DET_OVP_STATUS *value)
{
	u8 rt_sts;
	int rc = 0;
	u16 reg;
	u8 rt_mask;

	if (type == CUSTOM_DET_OVP_DC) {
		reg = DCIN_BASE + INT_RT_STS_OFFSET;
		rt_mask = USE_DCIN_BIT;
	} else {
		reg = USBIN_BASE + INT_RT_STS_OFFSET;
		rt_mask = USBIN_OV_RT_STS_BIT;
	}

	do {
		rc = smblib_read(custom_pdata->chg, reg, &rt_sts);
		if (rc < 0) {
			CUSTOM_DET_LOG("Couldn't read 0x%x rt status rc = %d\n", reg, rc);
			break;
		}

		if ((rt_sts & rt_mask) == rt_mask)
			*value = CUSTOM_DET_OVP_DETECT;
		else
			*value = CUSTOM_DET_OVP_NONE;

	} while (0);

	return rc;
}

static void _custom_det_status_chatt(struct custom_det_platform_data *custom_pdata, int check, struct status_info *status)
{
	struct status_info now_value;
	int chatt_cnt_ovp_dc = 0;
	int chatt_cnt_ovp_usb = 0;

	
	int chatt_cnt_adp_safe = 0;
	int chatt_cnt_usb_hot = 0;
	
	
	int chatt_cnt_det_usb = 0;
	
	bool loopbreak = false;
	int total = 0;
	int ret;
	/* bit      3 |       2 |      1 |     0  */
	/*     usbhot | adpsafe | usbsts | dcsts  */
	int detect = 0;

	
	status->vb_det = -1;
	
	status->ovp_state_dc = CUSTOM_DET_OVP_INIT;
	status->ovp_state_usb = CUSTOM_DET_OVP_INIT;
	
	status->adp_safe = -1;
	status->usb_hot = -1;
	

	while (1) {

		if (CUSTOM_DET_BIT_CHK(check, CUSTOM_DET_CHECK_VBDET) && !CUSTOM_DET_BIT_CHK(detect, CUSTOM_DET_CHECK_VBDET)) {
			if (CUSTOM_DET_FEATURE_CHK(custom_pdata, VBDET)) {
				now_value.vb_det = _custom_det_gpio_control(custom_pdata, CUSTOM_DET_GPIO_VBDET, CUSTOM_DET_GPIO_GET);
				if (now_value.vb_det >= 0) {
					CUSTOM_DET_CHATTERING_COUNT_UP(now_value.vb_det, status->vb_det, chatt_cnt_det_usb);
				}
				if (CUSTOM_DET_CHATTERING_COMP_CHECK(chatt_cnt_det_usb, CUSTOM_DET_VBDET_DETECT_COUNT)) {
					detect |= CUSTOM_DET_CHECK_VBDET;
				}
			} else {
					detect |= CUSTOM_DET_CHECK_VBDET;
			}
		}

		if (CUSTOM_DET_BIT_CHK(check, CUSTOM_DET_CHECK_DCSTS) && !CUSTOM_DET_BIT_CHK(detect, CUSTOM_DET_CHECK_DCSTS)) {
			if (CUSTOM_DET_FEATURE_CHK(custom_pdata, VCDET)) {
				ret = _custom_det_ov_status_get(custom_pdata, CUSTOM_DET_OVP_DC, &now_value.ovp_state_dc);
				if (ret == 0) {
					CUSTOM_DET_CHATTERING_COUNT_UP(now_value.ovp_state_dc, status->ovp_state_dc, chatt_cnt_ovp_dc);
				}
				if (CUSTOM_DET_CHATTERING_COMP_CHECK(chatt_cnt_ovp_dc, CUSTOM_DET_DETECT_COUNT)) {
					detect |= CUSTOM_DET_CHECK_DCSTS;
				}
			} else {
				detect |= CUSTOM_DET_CHECK_DCSTS;
			}
		}

		if (CUSTOM_DET_BIT_CHK(check, CUSTOM_DET_CHECK_USBSTS) && !CUSTOM_DET_BIT_CHK(detect, CUSTOM_DET_CHECK_USBSTS)) {
			ret = _custom_det_ov_status_get(custom_pdata, CUSTOM_DET_OVP_USB, &now_value.ovp_state_usb);
			if (ret == 0) {
				CUSTOM_DET_CHATTERING_COUNT_UP(now_value.ovp_state_usb, status->ovp_state_usb, chatt_cnt_ovp_usb);
			}
			if (CUSTOM_DET_CHATTERING_COMP_CHECK(chatt_cnt_ovp_usb, CUSTOM_DET_DETECT_COUNT)) {
				detect |= CUSTOM_DET_CHECK_USBSTS;
			}
		}

		
		if (CUSTOM_DET_BIT_CHK(check, CUSTOM_DET_CHECK_ADPSAFE) && !CUSTOM_DET_BIT_CHK(detect, CUSTOM_DET_CHECK_ADPSAFE)) {
			if (CUSTOM_DET_FEATURE_CHK(custom_pdata, ADP_SAFE)) {
				now_value.adp_safe = _custom_det_gpio_control(custom_pdata, CUSTOM_DET_GPIO_ADP_SAFE, CUSTOM_DET_GPIO_GET);
				if (now_value.adp_safe >= 0) {
					CUSTOM_DET_CHATTERING_COUNT_UP(now_value.adp_safe, status->adp_safe, chatt_cnt_adp_safe);
				}
				if (CUSTOM_DET_CHATTERING_COMP_CHECK(chatt_cnt_adp_safe, CUSTOM_DET_DETECT_COUNT)) {
					detect |= CUSTOM_DET_CHECK_ADPSAFE;
				}
			} else {
				detect |= CUSTOM_DET_CHECK_ADPSAFE;
			}
		}
		if (CUSTOM_DET_BIT_CHK(check, CUSTOM_DET_CHECK_USBHOT) && !CUSTOM_DET_BIT_CHK(detect, CUSTOM_DET_CHECK_USBHOT)) {
			if (CUSTOM_DET_FEATURE_CHK(custom_pdata, USB_HOT)) {
				now_value.usb_hot = _custom_det_gpio_control(custom_pdata, CUSTOM_DET_GPIO_USB_HOT, CUSTOM_DET_GPIO_GET);
				if (now_value.usb_hot >= 0) {
					CUSTOM_DET_CHATTERING_COUNT_UP(now_value.usb_hot, status->usb_hot, chatt_cnt_usb_hot);
				}
				if (CUSTOM_DET_CHATTERING_COMP_CHECK(chatt_cnt_usb_hot, CUSTOM_DET_USB_HOT_DETECT_COUNT)) {
					detect |= CUSTOM_DET_CHECK_USBHOT;
				}
			} else {
				detect |= CUSTOM_DET_CHECK_USBHOT;
			}
		}
		

		if (CUSTOM_DET_BIT_CHK(detect, check)) {
			
			
			CUSTOM_DET_LOG("[DC]ovp_state=%d [USB]ovp_state=%d value=%d adp_safe=%d usb_hot=%d count=%d\n",
					status->ovp_state_dc, status->ovp_state_usb, status->vb_det, status->adp_safe, status->usb_hot, total);
			
			
			break;
		}

		total++;
		if (total >= CUSTOM_DET_DETECT_COUNT_MAX) {
			
			if (CUSTOM_DET_BIT_CHK(check, CUSTOM_DET_CHECK_VBDET) && !CUSTOM_DET_BIT_CHK(detect, CUSTOM_DET_CHECK_VBDET)) {
				CUSTOM_DET_RECLOG("chattering 600 count over usb_dc=[USB]\n");
				if (check == CUSTOM_DET_CHECK_INIT) {
					status->vb_det = 1;
					loopbreak = true;
				}
				total = 0;
			}
			
			
			if (CUSTOM_DET_BIT_CHK(check, CUSTOM_DET_CHECK_DCSTS) && !CUSTOM_DET_BIT_CHK(detect, CUSTOM_DET_CHECK_DCSTS)) {
				status->ovp_state_dc = CUSTOM_DET_OVP_DETECT;
				loopbreak = true;
				CUSTOM_DET_RECLOG("ovp chattering count over -> ovp detect usb_dc=[DC]\n");
			}
			if (CUSTOM_DET_BIT_CHK(check, CUSTOM_DET_CHECK_USBSTS) && !CUSTOM_DET_BIT_CHK(detect, CUSTOM_DET_CHECK_USBSTS)) {
				status->ovp_state_usb = CUSTOM_DET_OVP_DETECT;
				loopbreak = true;
				CUSTOM_DET_RECLOG("ovp chattering count over -> ovp detect usb_dc=[USB]\n");
			}
			if (CUSTOM_DET_BIT_CHK(check, CUSTOM_DET_CHECK_ADPSAFE) && !CUSTOM_DET_BIT_CHK(detect, CUSTOM_DET_CHECK_ADPSAFE)) {
				status->adp_safe = 0;
				loopbreak = true;
				CUSTOM_DET_RECLOG("adp safe chattering count over\n");
			}
			if (loopbreak) {
				break;
			}
			
		}

		msleep(CUSTOM_DET_DETECT_WAIT_MS);
	}
}

static void _custom_det_ovp_notify_proc(struct custom_det_platform_data *custom_pdata, CUSTOM_DET_OVP_STATUS ovp_detect, CUSTOM_DET_OVP_TYPE usb_dc)
{
	CUSTOM_DET_MUTEX_LOCK(custom_pdata, CUSTOM_DET_MUTEX_OVP_STATE);

	if (ovp_detect == CUSTOM_DET_OVP_DETECT) {
		if (!custom_pdata->ovp_detect) {
			CUSTOM_DET_RECLOG("over voltage detect Call API (ERROR DETECT)\n");
			oem_chg_notify_error(OEM_CHG_ERROR_OVP_DETECT);
		}
		custom_pdata->ovp_detect |= (1 << usb_dc);
	} else {
		if (custom_pdata->ovp_detect) {
			custom_pdata->ovp_detect &= ~(1 << usb_dc);
			if (!custom_pdata->ovp_detect) {
				CUSTOM_DET_RECLOG("over voltage cancel Call API (NO ERROR)\n");
				oem_chg_notify_error(OEM_CHG_ERROR_OVP_NONE);
			}
		}
	}

	CUSTOM_DET_MUTEX_UNLOCK(custom_pdata, CUSTOM_DET_MUTEX_OVP_STATE);
}

static void _custom_det_usb_safety_notify_proc(struct custom_det_platform_data *custom_pdata, int err_detect, CUSTOM_DET_USB_SAFETY type)
{
	CUSTOM_DET_LOG("type:%d:detect%d err_detect:0x%x\n", type, err_detect, custom_pdata->usb_safety_err_detect);

	CUSTOM_DET_MUTEX_LOCK(custom_pdata, CUSTOM_DET_MUTEX_SAFETY_STATE);

	if (!err_detect) {

		CUSTOM_DET_RECLOG("safety error usbhot=%d short=%d\n",
					((custom_pdata->usb_safety_err_detect | (1 << type)) & (1 << CUSTOM_DET_USB_SAFETY_USB_TEMP)) ? 1:0,
					((custom_pdata->usb_safety_err_detect | (1 << type)) & (1 << CUSTOM_DET_USB_SAFETY_ADP_SAFE)) ? 1:0);

		if (!custom_pdata->usb_safety_err_detect) {

			CUSTOM_DET_LOG("USB_HOT_DETECT\n");
			if (custom_pdata->chg) {
				custom_det_otg_en_set(false);
				smblib_masked_write(custom_pdata->chg, DCDC_CMD_OTG_REG,
						OTG_EN_BIT, 0);
			}

			if (!chg_notify_err_delay_flg)
				oem_chg_notify_error(OEM_CHG_ERROR_USB_HOT_DETECT);
			else
				chg_notify_err_delay_flg = false;
		}
		custom_pdata->usb_safety_err_detect |= (1 << type);
	} else {
		if (custom_pdata->usb_safety_err_detect) {

			CUSTOM_DET_RECLOG("safety error usbhot=%d short=%d\n",
						((custom_pdata->usb_safety_err_detect & ~(1 << type)) & (1 << CUSTOM_DET_USB_SAFETY_USB_TEMP)) ? 1:0,
						((custom_pdata->usb_safety_err_detect & ~(1 << type)) & (1 << CUSTOM_DET_USB_SAFETY_ADP_SAFE)) ? 1:0);

			custom_pdata->usb_safety_err_detect &= ~(1 << type);

			if (!custom_pdata->usb_safety_err_detect) {

				CUSTOM_DET_LOG("USB_HOT_NONE\n");
				if ((custom_pdata->chg->custom_typec_present) == 0) {
					CUSTOM_DET_LOG("CHG_NOTIFY_ERR_USB_HOT_NONE\n");
					oem_chg_notify_error(OEM_CHG_ERROR_USB_HOT_NONE);
				} else {
					CUSTOM_DET_LOG("CHG_NOTIFY_ERR_USB_HOT_NONE_DELAY\n");
					chg_notify_err_delay_flg = true;
				}
			}
		}
	}

	CUSTOM_DET_MUTEX_UNLOCK(custom_pdata, CUSTOM_DET_MUTEX_SAFETY_STATE);
}

static int _custom_det_set_customdet_state(const char *val, const struct kernel_param *kp)
{
	return 0;
}

static int _custom_det_get_customdet_state(char *val, const struct kernel_param *kp)
{
	return sprintf(val, "%d", customdet_state);
}

MODULE_PARM_DESC(customdet_state, "OEMDET State");
module_param_call(customdet_state, _custom_det_set_customdet_state, _custom_det_get_customdet_state, &customdet_state, 0440);


static int _custom_det_get_vb_status(char *val, const struct kernel_param *kp)
{
	struct custom_det_platform_data *custom_pdata = &custom_platform_data;
	struct status_info status;

	CUSTOM_DET_WAKE_LOCK(custom_pdata, CUSTOM_DET_WAKELOCK_VBDET);

	_custom_det_status_chatt(custom_pdata, CUSTOM_DET_CHECK_VBDET, &status);

	CUSTOM_DET_WAKE_UNLOCK(custom_pdata, CUSTOM_DET_WAKELOCK_VBDET);

	return sprintf(val, "%d", status.vb_det);
}

MODULE_PARM_DESC(vb_status, "VBDET status");
module_param_call(vb_status, NULL, _custom_det_get_vb_status, &vb_status, 0444);

static int _custom_det_usb_temp_write(CUSTOM_DET_USB_TEMP_FUNC type)
{
	struct custom_det_platform_data *custom_pdata = &custom_platform_data;
	uint64_t mask = 1;
	uint64_t pin1;
	uint64_t pin2;
	int i;

	if (!CUSTOM_DET_FEATURE_CHK(custom_pdata, USB_THSEL)) {
		CUSTOM_DET_LOG("%s not support\n", __func__);
		return -1;
	}

	mask <<= (OEMDET_USB_THSEL_PATTERN_NUM - 1);
	if (type == CUSTOM_DET_USB_TEMP_NORMAL) {
		CUSTOM_DET_LOG("%s Pattern2\n", __func__);

		pin1 = OEMDET_USB_THSEL_PATTERN2_P1;
		pin2 = OEMDET_USB_THSEL_PATTERN2_P2;
	} else if (type == CUSTOM_DET_USB_TEMP_ERROR) {
		CUSTOM_DET_LOG("%s Pattern1\n", __func__);

		pin1 = OEMDET_USB_THSEL_PATTERN1_P1;
		pin2 = OEMDET_USB_THSEL_PATTERN1_P2;
	} else {
		CUSTOM_DET_LOG("%s parameter error %d\n", __func__, type);
		return -1;
	}

	for (i = 0; i < OEMDET_USB_THSEL_PATTERN_NUM; i++) {

		if (pin1 & mask) {
			_custom_det_gpio_control(custom_pdata, CUSTOM_DET_GPIO_USB_THSEL0, CUSTOM_DET_GPIO_OUT_H);
		} else {
			_custom_det_gpio_control(custom_pdata, CUSTOM_DET_GPIO_USB_THSEL0, CUSTOM_DET_GPIO_OUT_L);
		}

		if (pin2 & mask) {
			_custom_det_gpio_control(custom_pdata, CUSTOM_DET_GPIO_USB_THSEL1, CUSTOM_DET_GPIO_OUT_L);
		} else {
			_custom_det_gpio_control(custom_pdata, CUSTOM_DET_GPIO_USB_THSEL1, CUSTOM_DET_GPIO_OUT_H);
		}

		udelay(usb_temp_wait_us); 
		mask >>= 1;
	}
	udelay(OEMDET_USB_THSEL_AFTER_WAIT_US);

	return 0;
}

static int _custom_det_set_usbtemp(const char *val, const struct kernel_param *kp)
{
	int ret;

	ret = param_set_uint(val, kp);
	if (ret) {
		return ret;
	}
	CUSTOM_DET_LOG("%s value:%d usb_temp_wait_us value:%d\n", __func__, usb_temp, usb_temp_wait_us); 

	_custom_det_usb_temp_write(usb_temp);
	return ret;
}

static int _custom_det_get_usbtemp(char *val, const struct kernel_param *kp)
{
	struct custom_det_platform_data *custom_pdata = &custom_platform_data;
	struct status_info status;

	CUSTOM_DET_WAKE_LOCK(custom_pdata, CUSTOM_DET_WAKELOCK_USB_HOT);

	_custom_det_status_chatt(custom_pdata, CUSTOM_DET_CHECK_USBHOT, &status);

	CUSTOM_DET_WAKE_UNLOCK(custom_pdata, CUSTOM_DET_WAKELOCK_USB_HOT);

	return sprintf(val, "%d", status.usb_hot);
}

MODULE_PARM_DESC(usb_temp, "USB temperature state");
module_param_call(usb_temp, _custom_det_set_usbtemp, _custom_det_get_usbtemp, &usb_temp, 0644);

static int _custom_det_set_usbtemp_wait_us(const char *val, const struct kernel_param *kp)
{
	int ret;

	ret = param_set_uint(val, kp);
	if (ret) {
		return ret;
	}

	if (usb_temp_wait_us > OEMDET_USB_THSEL_WAIT_US_MAX) {
		usb_temp_wait_us = OEMDET_USB_THSEL_WAIT_US;
		return -EINVAL;
	}

	CUSTOM_DET_LOG("%s value:%d\n", __func__, usb_temp_wait_us);

	return ret;
}

static int _custom_det_get_usbtemp_wait_us(char *val, const struct kernel_param *kp)
{
	return sprintf(val, "%d", usb_temp_wait_us);
}

MODULE_PARM_DESC(usb_temp_wait_us, "USB temperature wait us");
module_param_call(usb_temp_wait_us, _custom_det_set_usbtemp_wait_us, _custom_det_get_usbtemp_wait_us, &usb_temp_wait_us, 0644);

/******************************************************************************/
/* Interrupt handler                                                          */
/******************************************************************************/
static void _custom_det_usb_hot_handler_core(struct custom_det_platform_data *custom_pdata, int _usb_hot)
{
	struct status_info status;

	if (!CUSTOM_DET_FEATURE_CHK(custom_pdata, USB_HOT))
		return;

	CUSTOM_DET_LOG("%s:triggered\n", __func__);

	CUSTOM_DET_WAKE_LOCK(custom_pdata, CUSTOM_DET_WAKELOCK_USB_HOT);

	if (_usb_hot == CUSTOM_DET_CHATT) {
		_custom_det_status_chatt(custom_pdata, CUSTOM_DET_CHECK_USBHOT, &status);
	} else {
		status.usb_hot = _usb_hot;
	}
	_custom_det_usb_safety_notify_proc(custom_pdata, status.usb_hot, CUSTOM_DET_USB_SAFETY_USB_TEMP);

	CUSTOM_DET_WAKE_UNLOCK(custom_pdata, CUSTOM_DET_WAKELOCK_USB_HOT);
	return;
}

static irqreturn_t _custom_det_usb_hot_handler(int irq, void *pdata)
{
	struct custom_det_platform_data *custom_pdata = pdata;
	_custom_det_usb_hot_handler_core(custom_pdata, CUSTOM_DET_CHATT);
	return IRQ_HANDLED;
}

static void _custom_det_adp_safe_handler_core(struct custom_det_platform_data *custom_pdata, int _adp_safe)
{
	struct status_info status;

	if (!CUSTOM_DET_FEATURE_CHK(custom_pdata, ADP_SAFE))
		return;

	CUSTOM_DET_LOG("%s:triggered\n", __func__);

	CUSTOM_DET_WAKE_LOCK(custom_pdata, CUSTOM_DET_WAKELOCK_ADP_SAFE);

	if (_adp_safe == CUSTOM_DET_CHATT) {
		_custom_det_status_chatt(custom_pdata, CUSTOM_DET_CHECK_ADPSAFE, &status);
	} else {
		status.adp_safe = _adp_safe;
	}
	_custom_det_usb_safety_notify_proc(custom_pdata, status.adp_safe, CUSTOM_DET_USB_SAFETY_ADP_SAFE);

	CUSTOM_DET_WAKE_UNLOCK(custom_pdata, CUSTOM_DET_WAKELOCK_ADP_SAFE);
	return;
}

static irqreturn_t _custom_det_adp_safe_handler(int irq, void *pdata)
{
	struct custom_det_platform_data *custom_pdata = pdata;
	_custom_det_adp_safe_handler_core(custom_pdata, CUSTOM_DET_CHATT);
	return IRQ_HANDLED;
}

static void _custom_det_dcin_ov_handler_core(struct custom_det_platform_data *custom_pdata, CUSTOM_DET_OVP_STATUS _ovp_state)
{
	CUSTOM_DET_OVP_STATUS ovp_state;
	struct status_info status;

	if (!CUSTOM_DET_FEATURE_CHK(custom_pdata, VCDET))
		return;

	CUSTOM_DET_LOG("%s:triggered\n", __func__);

	CUSTOM_DET_WAKE_LOCK(custom_pdata, CUSTOM_DET_WAKELOCK_DCIN_OV);

	if (_ovp_state == CUSTOM_DET_OVP_INIT) {
		_custom_det_status_chatt(custom_pdata, CUSTOM_DET_CHECK_DCSTS, &status);
		ovp_state = status.ovp_state_dc;
	} else {
		ovp_state = _ovp_state;
	}

	_custom_det_ovp_notify_proc(custom_pdata, ovp_state, CUSTOM_DET_OVP_DC);

	CUSTOM_DET_WAKE_UNLOCK(custom_pdata, CUSTOM_DET_WAKELOCK_DCIN_OV);

	return;
}

void custom_det_dcin_ov_handler()
{
	struct custom_det_platform_data *custom_pdata = &custom_platform_data;
	_custom_det_dcin_ov_handler_core(custom_pdata, CUSTOM_DET_OVP_INIT);
}

static void _custom_det_usbin_ov_handler_core(struct custom_det_platform_data *custom_pdata, CUSTOM_DET_OVP_STATUS _ovp_state)
{
	CUSTOM_DET_OVP_STATUS ovp_state;
	struct status_info status;

	CUSTOM_DET_LOG("%s:triggered\n", __func__);

	CUSTOM_DET_WAKE_LOCK(custom_pdata, CUSTOM_DET_WAKELOCK_USBIN_OV);

	if (_ovp_state == CUSTOM_DET_OVP_INIT) {
		_custom_det_status_chatt(custom_pdata, CUSTOM_DET_CHECK_USBSTS, &status);
		ovp_state = status.ovp_state_usb;
	} else {
		ovp_state = _ovp_state;
	}
	_custom_det_ovp_notify_proc(custom_pdata, ovp_state, CUSTOM_DET_OVP_USB);

	CUSTOM_DET_WAKE_UNLOCK(custom_pdata, CUSTOM_DET_WAKELOCK_USBIN_OV);

	return;
}

void custom_det_usbin_ov_handler()
{
	struct custom_det_platform_data *custom_pdata = &custom_platform_data;
	_custom_det_usbin_ov_handler_core(custom_pdata, CUSTOM_DET_OVP_INIT);
}

static int _custom_det_set_power_role(const char *val, const struct kernel_param *kp)
{
	struct custom_det_platform_data *custom_pdata = &custom_platform_data;
	int i = 0;
	int ret;
	union power_supply_propval pval = {0, };

	for (i = 0; i < CUSTOM_DET_POWER_ROLE_TOTAL; i++)
	{
		if(strncmp(val, custom_det_power_role[i].power_role_name, strlen(custom_det_power_role[i].power_role_name)) == 0)
		{
			power_role = custom_det_power_role[i].power_role_val;
			break;
		}
	}

	if (i >= CUSTOM_DET_POWER_ROLE_TOTAL)
	{
		CUSTOM_DET_LOG("power role not supported: %s", val);
		return -EINVAL;
	}

	CUSTOM_DET_LOG("%s value:%s\n", __func__, val);

	pval.intval = power_role;
	ret = smblib_set_prop_typec_power_role(custom_pdata->chg, &pval);

	return ret;
}

static int _custom_det_get_power_role(char *val, const struct kernel_param *kp)
{
	struct custom_det_platform_data *custom_pdata = &custom_platform_data;
	int ret;
	int role = CUSTOM_DET_POWER_ROLE_NONE;
	union power_supply_propval pval = {0, };

	ret = smblib_get_prop_typec_power_role(custom_pdata->chg, &pval);

	if (ret) {
		return ret;
	}

	power_role = pval.intval;

	switch(power_role) {
	case POWER_SUPPLY_TYPEC_PR_NONE:
		role = CUSTOM_DET_POWER_ROLE_NONE;
		break;
	case POWER_SUPPLY_TYPEC_PR_DUAL:
		role = CUSTOM_DET_POWER_ROLE_DRP;
		break;
	case POWER_SUPPLY_TYPEC_PR_SINK:
		role = CUSTOM_DET_POWER_ROLE_UFP;
		break;
	case POWER_SUPPLY_TYPEC_PR_SOURCE:
		role = CUSTOM_DET_POWER_ROLE_DFP;
		break;
	default:
		role = CUSTOM_DET_POWER_ROLE_NONE;
		CUSTOM_DET_LOG("unknown power role:%d\n", power_role);
		break;
	}

	return sprintf(val, "%s", custom_det_power_role[role].power_role_name);
}

MODULE_PARM_DESC(power_role, "Type-C Power role");
module_param_call(power_role, _custom_det_set_power_role, _custom_det_get_power_role, &power_role, 0660);

static int _custom_det_set_default_power_role(const char *val, const struct kernel_param *kp)
{
	int i = 0;
	int ret = 0; 

	for (i = 0; i < CUSTOM_DET_POWER_ROLE_TOTAL; i++)
	{
		if(strncmp(val, custom_det_power_role[i].power_role_name, strlen(custom_det_power_role[i].power_role_name)) == 0)
		{
			default_power_role = custom_det_power_role[i].power_role_val;
			break;
		}
	}

	if (i >= CUSTOM_DET_POWER_ROLE_TOTAL)
	{
		CUSTOM_DET_LOG("power role not supported: %s", val);
		return -EINVAL;
	}

	CUSTOM_DET_LOG("%s value:%s\n", __func__, val);

	return ret;
}

static int _custom_det_get_default_power_role(char *val, const struct kernel_param *kp)
{
	int role = CUSTOM_DET_POWER_ROLE_NONE;

	switch(default_power_role) {
	case POWER_SUPPLY_TYPEC_PR_NONE:
		role = CUSTOM_DET_POWER_ROLE_NONE;
		break;
	case POWER_SUPPLY_TYPEC_PR_DUAL:
		role = CUSTOM_DET_POWER_ROLE_DRP;
		break;
	case POWER_SUPPLY_TYPEC_PR_SINK:
		role = CUSTOM_DET_POWER_ROLE_UFP;
		break;
	case POWER_SUPPLY_TYPEC_PR_SOURCE:
		role = CUSTOM_DET_POWER_ROLE_DFP;
		break;
	default:
		role = CUSTOM_DET_POWER_ROLE_NONE;
		CUSTOM_DET_LOG("unknown power role:%d\n", power_role);
		break;
	}

	return sprintf(val, "%s", custom_det_power_role[role].power_role_name);
}

MODULE_PARM_DESC(default_power_role, "Type-C Default Power role");
module_param_call(default_power_role, _custom_det_set_default_power_role, _custom_det_get_default_power_role, &default_power_role, 0660);

int custom_det_get_default_power_role(void)
{
	return default_power_role;
}

bool custom_det_otg_disable()
{
	struct custom_det_platform_data *custom_pdata = &custom_platform_data;
	bool ret = false;

#ifdef CONFIG_DEBUG_FS
	if (custom_pdata->debug_otg_disable)
		ret = true;
#endif

	CUSTOM_DET_MUTEX_LOCK(custom_pdata, CUSTOM_DET_MUTEX_SAFETY_STATE);
	if (custom_pdata->usb_safety_err_detect != 0)
	{
		ret = true;
	}

	CUSTOM_DET_MUTEX_UNLOCK(custom_pdata, CUSTOM_DET_MUTEX_SAFETY_STATE);

	return ret;
}

void custom_det_otg_en_set(bool enable)
{
	struct custom_det_platform_data *custom_pdata = &custom_platform_data;

	CUSTOM_DET_LOG("%s:%d\n", __func__, enable);

	if (enable) {
		_custom_det_gpio_control(custom_pdata, CUSTOM_DET_GPIO_OTG_EN, CUSTOM_DET_GPIO_OUT_H);
	} else {
		_custom_det_gpio_control(custom_pdata, CUSTOM_DET_GPIO_OTG_EN, CUSTOM_DET_GPIO_OUT_L);
	}
}

void custom_det_typec_change_notification(void)
{
	struct custom_det_platform_data *custom_pdata = &custom_platform_data;

	CUSTOM_DET_MUTEX_LOCK(custom_pdata, CUSTOM_DET_MUTEX_SAFETY_STATE);
	if ((custom_pdata->chg->custom_typec_present == 0)&& chg_notify_err_delay_flg) {
		CUSTOM_DET_LOG("CHG_NOTIFY_ERR_USB_HOT_NONE\n");
		oem_chg_notify_error(OEM_CHG_ERROR_USB_HOT_NONE);
		chg_notify_err_delay_flg = false;
	}
	CUSTOM_DET_MUTEX_UNLOCK(custom_pdata, CUSTOM_DET_MUTEX_SAFETY_STATE);

	return;
}

static void custom_det_charge_request_notification(void)
{
	if (chg_req_state)
		customdet_state = CUSTOM_DET_DEVICE_CHATTER_END;
}


void custom_det_charge_req(CUSTOM_DET_CHARGE_TYPE type, bool enable)
{
	switch (type) {
	case CUSTOM_DET_CHARGE_APSD:
		if (enable) {
			chg_req_state = true;
			oem_chg_apsd_vbus_draw(OEM_CHG_USB_CURRENT);
		} else {
			chg_req_state = false;
			oem_chg_apsd_vbus_draw(OEM_CHG_OFF_CURRENT);
		}
		break;
	}

	custom_det_charge_request_notification(); 
	CUSTOM_DET_LOG("%s value:%d\n", __func__, enable); 

	return;
}

int custom_det_request_usb_safety_irqs(struct device *dev)
{
	struct custom_det_platform_data *custom_pdata = &custom_platform_data;
	int rc = 0;
	int irq = 0;

	do {
		if (!CUSTOM_DET_FEATURE_CHK(custom_pdata, USB_HOT)) {
			CUSTOM_DET_LOG("USB_HOT disable\n");
			break;
		}

		irq = gpio_to_irq(CUSTOM_DET_GPIO(custom_pdata, CUSTOM_DET_GPIO_USB_HOT));

		/* request_irq usb_hot */
		rc = devm_request_threaded_irq(dev, irq,
						NULL, _custom_det_usb_hot_handler, /* Thread function */
						IRQF_TRIGGER_RISING | IRQF_TRIGGER_FALLING | IRQF_ONESHOT,
						custom_det_irq_name[CUSTOM_DET_IRQ_USB_HOT], custom_pdata); 
		if (rc < 0) {
			CUSTOM_DET_LOG("Unable to request usb_hot irq: %d\n", rc);
			_custom_det_gpio_control(custom_pdata, CUSTOM_DET_GPIO_USB_HOT, CUSTOM_DET_GPIO_FREE);
			break;
		}
		enable_irq_wake(irq);
		_custom_det_irq_control_init(custom_pdata, CUSTOM_DET_IRQ_USB_HOT, irq);

	} while (0);

	do {
		if (!CUSTOM_DET_FEATURE_CHK(custom_pdata, ADP_SAFE)) {
			CUSTOM_DET_LOG("ADP_SAFE disable\n");
			break;
		}

		irq = gpio_to_irq(CUSTOM_DET_GPIO(custom_pdata, CUSTOM_DET_GPIO_ADP_SAFE));

		/* request_irq adp_safe */
		rc = devm_request_threaded_irq(dev, irq,
						NULL, _custom_det_adp_safe_handler, /* Thread function */
						IRQF_TRIGGER_RISING | IRQF_TRIGGER_FALLING | IRQF_ONESHOT,
						custom_det_irq_name[CUSTOM_DET_IRQ_ADP_SAFE], custom_pdata); 
		if (rc < 0) {
			CUSTOM_DET_LOG("Unable to request adp_safe irq: %d\n", rc);
			_custom_det_gpio_control(custom_pdata, CUSTOM_DET_GPIO_ADP_SAFE, CUSTOM_DET_GPIO_FREE);
			break;
		}
		enable_irq_wake(irq);
		_custom_det_irq_control_init(custom_pdata, CUSTOM_DET_IRQ_ADP_SAFE, irq);

	} while (0);

	return rc;
}

int custom_det_determine_initial_status(void)
{
	struct custom_det_platform_data *custom_pdata = &custom_platform_data;
	struct status_info status;

	_custom_det_status_chatt(custom_pdata, CUSTOM_DET_CHECK_INIT, &status);

	_custom_det_usbin_ov_handler_core(custom_pdata, status.ovp_state_usb);
	_custom_det_dcin_ov_handler_core(custom_pdata, status.ovp_state_dc); 
	
	_custom_det_usb_hot_handler_core(custom_pdata, status.usb_hot);
	_custom_det_adp_safe_handler_core(custom_pdata, status.adp_safe);
	

	return 0;
}

#ifdef CONFIG_DEBUG_FS
static int custom_det_debug_otg_disable_read(struct seq_file *file, void *data)
{
	struct custom_det_platform_data *custom_pdata = &custom_platform_data;

	if (!custom_pdata)
		return -EINVAL;

	seq_printf(file, "%u\n", custom_det_otg_disable());

	return 0;
}

static ssize_t custom_det_debug_otg_disable_write(struct file *file,
				      const char __user *ubuf, size_t cnt,
				      loff_t *ppos)
{
	struct custom_det_platform_data *custom_pdata = &custom_platform_data;
	int value;
	int err;

	if (!custom_pdata)
		return -EINVAL;

	err = kstrtoint_from_user(ubuf, cnt, 0, &value);
	if (err) {
		CUSTOM_DET_LOG("failed to custom_det_debug_otg_disable_write\n");
		return err;
	}

	if (value) {
		custom_pdata->debug_otg_disable = true;
	} else {
		custom_pdata->debug_otg_disable = false;
	}

	return cnt;
}

static int custom_det_debug_otg_disable_open(struct inode *inode, struct file *file)
{
	return single_open(file, custom_det_debug_otg_disable_read, NULL);
}

static const struct file_operations custom_det_debug_otg_disable_fops = {
	.open		= custom_det_debug_otg_disable_open,
	.read		= seq_read,
	.write		= custom_det_debug_otg_disable_write,
};

static int custom_det_debug_charge_req_read(struct seq_file *file, void *data)
{
	struct custom_det_platform_data *custom_pdata = &custom_platform_data;

	if (!custom_pdata)
		return -EINVAL;

	seq_printf(file, "%u\n", chg_req_state);

	return 0;
}

static int custom_det_debug_charge_req_open(struct inode *inode, struct file *file)
{
	return single_open(file, custom_det_debug_charge_req_read, NULL);
}

static const struct file_operations custom_det_charge_req_fops = {
	.open		= custom_det_debug_charge_req_open,
	.read		= seq_read,
	.write		= NULL,
};

static int custom_det_debug_ov_status_read(struct seq_file *file, void *data)
{
	struct custom_det_platform_data *custom_pdata = &custom_platform_data;

	if (!custom_pdata)
		return -EINVAL;

	CUSTOM_DET_MUTEX_LOCK(custom_pdata, CUSTOM_DET_MUTEX_OVP_STATE);

	seq_printf(file, "dcin_ov_status : %s\n",
			(custom_pdata->ovp_detect & (1 << CUSTOM_DET_OVP_DC)) ? "ON":"OFF");
	seq_printf(file, "usbin_ov_status : %s\n",
			(custom_pdata->ovp_detect & (1 << CUSTOM_DET_OVP_USB)) ? "ON":"OFF");

	CUSTOM_DET_MUTEX_UNLOCK(custom_pdata, CUSTOM_DET_MUTEX_OVP_STATE);

	return 0;
}

static int custom_det_debug_ov_status_open(struct inode *inode, struct file *file)
{
	return single_open(file, custom_det_debug_ov_status_read, NULL);
}

static const struct file_operations custom_det_ov_status_fops = {
	.open		= custom_det_debug_ov_status_open,
	.read		= seq_read,
	.write		= NULL,
};


static int custom_det_debug_usb_safety_status_read(struct seq_file *file, void *data)
{
	struct custom_det_platform_data *custom_pdata = &custom_platform_data;

	if (!custom_pdata)
		return -EINVAL;

	CUSTOM_DET_MUTEX_LOCK(custom_pdata, CUSTOM_DET_MUTEX_SAFETY_STATE);

	seq_printf(file, "short_status : %s\n",
			(custom_pdata->usb_safety_err_detect & (1 << CUSTOM_DET_USB_SAFETY_ADP_SAFE)) ? "ON":"OFF");
	seq_printf(file, "usbhot_status : %s\n",
			(custom_pdata->usb_safety_err_detect & (1 << CUSTOM_DET_USB_SAFETY_USB_TEMP)) ? "ON":"OFF");

	CUSTOM_DET_MUTEX_UNLOCK(custom_pdata, CUSTOM_DET_MUTEX_SAFETY_STATE);

	return 0;
}

static int custom_det_debug_usb_safety_status_open(struct inode *inode, struct file *file)
{
	return single_open(file, custom_det_debug_usb_safety_status_read, NULL);
}

static const struct file_operations custom_det_usb_satety_status_fops = {
	.open		= custom_det_debug_usb_safety_status_open,
	.read		= seq_read,
	.write		= NULL,
};

static int custom_det_debug_wakelock_status_read(struct seq_file *file, void *data)
{
	struct custom_det_platform_data *custom_pdata = &custom_platform_data;
	CUSTOM_DET_WAKELOCK_TYPE wakelock_i;

	if (!custom_pdata)
		return -EINVAL;

	for (wakelock_i = 0; wakelock_i < CUSTOM_DET_WAKELOCK_TOTAL; wakelock_i++) {
		seq_printf(file, "%s: %s\n", custom_det_wakelock_setup[wakelock_i],
			CUSTOM_DET_WAKE_ACTIVE(custom_pdata, wakelock_i) ? "TRUE":"FALSE");
	}

	return 0;
}

static int custom_det_debug_wakelock_status_open(struct inode *inode, struct file *file)
{
	return single_open(file, custom_det_debug_wakelock_status_read, NULL);
}

static const struct file_operations custom_det_wakelock_status_fops = {
	.open		= custom_det_debug_wakelock_status_open,
	.read		= seq_read,
	.write		= NULL,
};

static int custom_det_debug_irq_spurious_read(struct seq_file *file, void *data)
{
	struct custom_det_platform_data *custom_pdata = &custom_platform_data;
	struct irq_desc *desc;
	CUSTOM_DET_IRQ_TYPE irq_i;

	if (!custom_pdata)
		return -EINVAL;

	smb5_debug_interrupt_statistics(file, data);

	for (irq_i = 0; irq_i < CUSTOM_DET_IRQ_TOTAL; irq_i++) {
		if (custom_pdata->irq_info[irq_i].ope == CUSTOM_DET_IRQ_ENABLE) {
			desc = irq_to_desc(custom_pdata->irq_info[irq_i].irq_no);
			seq_printf(file, "irq : %u\n" "name : %s\n" "count : %u\n"
					"unhandled : %u\n" "last_unhandled : %u ms\n\n",
					custom_pdata->irq_info[irq_i].irq_no, custom_det_irq_name[irq_i],
					desc->irq_count, desc->irqs_unhandled,
					jiffies_to_msecs(desc->last_unhandled));
		}
	}

	return 0;
}

static int custom_det_debug_irq_spurious_open(struct inode *inode, struct file *file)
{
	return single_open(file, custom_det_debug_irq_spurious_read, NULL);
}

static const struct file_operations custom_det_irq_spurious_fops = {
	.open		= custom_det_debug_irq_spurious_open,
	.read		= seq_read,
	.write		= NULL,
};


static void custom_det_debugfs_init(void)
{
	struct custom_det_platform_data *custom_pdata = &custom_platform_data;
	struct dentry *root;

	root = debugfs_create_dir("custom_det", NULL);

	if (!root)
		goto err_root;

	custom_pdata->customdet_debugfs_root = root;
	custom_pdata->debug_otg_disable = false;

	if (!debugfs_create_file("debug_otg_disable", S_IRUSR | S_IWUSR, root, NULL,
			&custom_det_debug_otg_disable_fops))
		goto err_node;

	if (!debugfs_create_file("charge_req_status", S_IRUSR, root, NULL,
			&custom_det_charge_req_fops))
		goto err_node;

	if (!debugfs_create_file("ov_status", S_IRUSR, root, NULL,
			&custom_det_ov_status_fops))
		goto err_node;

	if (!debugfs_create_file("usb_safety_status", S_IRUSR, root, NULL,
			&custom_det_usb_satety_status_fops))
		goto err_node;

	if (!debugfs_create_file("wakelock_status", S_IRUSR, root, NULL,
			&custom_det_wakelock_status_fops))
		goto err_node;

	if (!debugfs_create_file("irq_spurious", S_IRUSR, root, NULL,
			&custom_det_irq_spurious_fops))
		goto err_node;

	return;

err_node:
	debugfs_remove_recursive(root);
	custom_pdata->customdet_debugfs_root = NULL;
err_root:
	CUSTOM_DET_LOG("failed to debugfs_init\n");
}

static void custom_det_debugfs_remove(void)
{
	struct custom_det_platform_data *custom_pdata = &custom_platform_data;

	if (!custom_pdata->customdet_debugfs_root)
		return;

	debugfs_remove_recursive(custom_pdata->customdet_debugfs_root);
	custom_pdata->customdet_debugfs_root = NULL;
}

#else
static inline void custom_det_debugfs_init(void)
{
}

static inline void custom_det_debugfs_remove(void)
{
}
#endif

#define CUSTOM_DET_GPIO_SETUP(node, pdata, type, dt_name, gpio_name, ope)		\
do {																		\
	if (!_custom_det_gpio_control_init(node, pdata, type, dt_name, gpio_name))	\
		break;																\
	_custom_det_gpio_control(custom_pdata, type, ope);								\
} while (0)

int custom_det_init(struct smb_charger *chg)
{
	struct custom_det_platform_data *custom_pdata = &custom_platform_data;
	CUSTOM_DET_MUTEX_TYPE mutex_i;
	CUSTOM_DET_GPIO_TYPE gpio_i;
	CUSTOM_DET_WAKELOCK_TYPE wakelock_i;

	memset(custom_pdata, 0, sizeof(struct custom_det_platform_data));

	custom_pdata->chg = chg;

	for (mutex_i = 0; mutex_i < CUSTOM_DET_MUTEX_TOTAL; mutex_i++) {
		CUSTOM_DET_MUTEX_INIT(custom_pdata, mutex_i);
	}

	for (wakelock_i = 0; wakelock_i < CUSTOM_DET_WAKELOCK_TOTAL; wakelock_i++) {
		custom_pdata->wake_src[wakelock_i] = wakeup_source_register(chg->dev, custom_det_wakelock_setup[wakelock_i]);
	}

	for (gpio_i = 0; gpio_i < CUSTOM_DET_GPIO_TOTAL; gpio_i++) {
		CUSTOM_DET_GPIO_SETUP(chg->dev->of_node, custom_pdata, gpio_i,
							custom_det_gpio_setup[gpio_i].devtree_name,
							custom_det_gpio_setup[gpio_i].name,
							custom_det_gpio_setup[gpio_i].setting);
	}

	_custom_det_feature_check(custom_pdata);

	custom_det_debugfs_init();

	return 0;
}

void custom_det_probe_end_notification(int vbus_status)
{
	if (!vbus_status)
		customdet_state = CUSTOM_DET_DEVICE_PROBE_END;
}

void custom_det_shutdown(void)
{
	struct custom_det_platform_data *custom_pdata = &custom_platform_data;
	CUSTOM_DET_IRQ_TYPE irq_i;

	for (irq_i = 0; irq_i < CUSTOM_DET_IRQ_TOTAL; irq_i++) {
		_custom_det_irq_control(custom_pdata, irq_i, CUSTOM_DET_IRQ_DISABLE);
	}
}

int custom_det_remove(struct smb_charger *chg)
{
	struct custom_det_platform_data *custom_pdata = &custom_platform_data;
	CUSTOM_DET_WAKELOCK_TYPE wakelock_i;
	CUSTOM_DET_GPIO_TYPE gpio_i;
	CUSTOM_DET_IRQ_TYPE irq_i;

	custom_det_debugfs_remove();

	for (irq_i = 0; irq_i < CUSTOM_DET_IRQ_TOTAL; irq_i++) {
		_custom_det_irq_control(custom_pdata, irq_i, CUSTOM_DET_IRQ_FREE);
	}

	for (gpio_i = 0; gpio_i < CUSTOM_DET_GPIO_TOTAL; gpio_i++) {
		_custom_det_gpio_control(custom_pdata, gpio_i, CUSTOM_DET_GPIO_FREE);
	}

	for (wakelock_i = 0; wakelock_i < CUSTOM_DET_WAKELOCK_TOTAL; wakelock_i++) {
		 wakeup_source_unregister(custom_pdata->wake_src[wakelock_i]);
	}

	return 0;
}

#endif /* FEATURE_CUSTOM_DET_DRIVER */

MODULE_DESCRIPTION("OEM DET");
MODULE_LICENSE("GPL v2");
