/*----------------------------------------------------------------------------*/
// (C) 2021 FCNT LIMITED
/*----------------------------------------------------------------------------*/
// SPDX-License-Identifier: GPL-2.0+
#include <linux/device.h>

#include <linux/usb.h>
#include <linux/usb/hcd.h>
#include <linux/usb/cdc.h>
#include <linux/skbuff.h>
#include <linux/mii.h>
#include <linux/netdevice.h>
#include <linux/usb/usbnet.h>

#include <linux/usb/oem_kitting_usb.h>

#include <linux/nonvolatile_common.h>


#define APNV_KITTING_USB_FLAG_I			51001
#define APNV_KITTING_SPMODE_INFO_I  	51004

#define APNV_KITTING_USB_FLAG_SIZE		2
#define APNV_KITTING_SPMODE_INFO_SIZE	2

#define SPMODE_ENABLE					0x02

#define KITTING_USB						0x01
#define KITTING_USB_ETHERNET			0x02
#define KITTING_USB_HID_CLASS			0x04

static const struct usb_device_descriptor	use_dev[] = {
	{	/* xHCI Host Controller 1.1 root hub */
		.idVendor	= 0x1D6B,
		.idProduct	= 0x0001,
	},
	{	/* xHCI Host Controller 2.0 root hub  */
		.idVendor	= 0x1D6B,
		.idProduct	= 0x0002,
	},
	{	/* xHCI Host Controller 3.0 root hub  */
		.idVendor	= 0x1D6B,
		.idProduct	= 0x0003,
	},
};

static uint8_t kitting_data[APNV_KITTING_USB_FLAG_SIZE];
static uint8_t kitting_spmode_data[APNV_KITTING_SPMODE_INFO_SIZE];

extern int makercmd_mode;

static void _usb_kitting_init(void)
{
	int res;

	do {
		res = get_nonvolatile(	&kitting_spmode_data[0],
								APNV_KITTING_SPMODE_INFO_I,
								APNV_KITTING_SPMODE_INFO_SIZE);
		if (res < 0) {
			pr_err("%s spmode nonvolatile read error\n", __func__);
			break;
		}

		res = get_nonvolatile(	&kitting_data[0],
								APNV_KITTING_USB_FLAG_I,
								APNV_KITTING_USB_FLAG_SIZE);
		if (res < 0) {
			pr_err("%s nonvolatile read error\n", __func__);
			break;
		}

	} while (0);

	return;
}

static bool _usb_kitting_is_enable_nv(void)
{
	bool ret = false;

	do {
		if ((kitting_spmode_data[0] & SPMODE_ENABLE) != 0) {
			pr_info("%s free by spmode\n", __func__);
			break;
		}

		if ((kitting_data[0] & KITTING_USB) == 0)
			break;

		pr_info("%s restricted by kitting\n", __func__);
		ret = true;

	} while (0);

	return ret;
}

static bool _usb_kitting_is_enable_func(const struct device *dev, const struct usb_device_id *id)
{
	int	i;
	struct usb_interface *intf = to_usb_interface(dev);
	struct usb_device *udev = interface_to_usbdev(intf);
	struct usb_host_interface  *host_if;
	struct usb_driver *driver = to_usb_driver(dev->driver);
	struct driver_info *info;

	for (i = 0; i < (sizeof(use_dev) / sizeof(use_dev[0])) ; i++) {
		if ((le16_to_cpu(udev->descriptor.idVendor) == use_dev[i].idVendor) &&
			(le16_to_cpu(udev->descriptor.idProduct) == use_dev[i].idProduct)) {
			pr_info("%s Root HUB is free\n", __func__);
			return false;
		}
	}

	if (udev->descriptor.bDeviceClass == USB_CLASS_HUB) {
		pr_info("%s USB HUB Class is free\n", __func__);
		return false;
	}

	for (i = 0; i < intf->num_altsetting; i++) {
		host_if = &intf->altsetting[i];
		if ((host_if->desc.bInterfaceClass == USB_CLASS_COMM) &&
			(host_if->desc.bInterfaceSubClass == USB_CDC_SUBCLASS_ETHERNET)) {
			if ((kitting_data[0] & KITTING_USB_ETHERNET) == 0) {
				pr_info("%s descriptor: ETHERNET is free\n", __func__);
				return false;
			}
		}

		if (driver->probe == usbnet_probe) {
			info = (struct driver_info *)id->driver_info;
			if (!!info && (info->flags & FLAG_ETHER)) {
				if ((kitting_data[0] & KITTING_USB_ETHERNET) == 0) {
					pr_info("%s flag_ether: ETHERNET is free\n", __func__);
					return false;
				}
			}
		}

		if (!!driver->name && strncmp(driver->name, "r8152", 5) == 0) {
			if ((kitting_data[0] & KITTING_USB_ETHERNET) == 0) {
				pr_info("%s r8152: ETHERNET is free\n", __func__);
				return false;
			}
		}

		if (host_if->desc.bInterfaceClass == USB_CLASS_HID) {
			if ((kitting_data[0] & KITTING_USB_HID_CLASS) == 0) {
				pr_info("%s HID Class is free\n", __func__);
				return false;
			}
		}
	}

	pr_info("%s restricted by kitting\n", __func__);

	return true;
}

bool usb_kitting_is_enable_adb(void)
{
	bool ret = false;

	do {
		_usb_kitting_init();

		if (makercmd_mode)
			break;

		if (!_usb_kitting_is_enable_nv())
			break;

		ret = true;

	} while (0);

	return ret;
}

bool usb_kitting_is_enable_host(const struct device *dev, const struct usb_device_id *id)
{
	bool ret = false;

	do {
		if (!usb_kitting_is_enable_adb())
			break;

		if(!_usb_kitting_is_enable_func(dev, id))
			break;

		ret = true;

	} while (0);

	return ret;
}
