/**
* MxChat Image Generation JavaScript
*
* Handles frontend interactions for image generation functionality.
*
* @since      1.0.0
* @package    MxChat_Admin_Chat
*/

(function($) {
   'use strict';

   // Image Generation Module
   const MxChatImageGen = {
       // Properties
       isEnabled: false,
       imageGenToggle: null,
       currentImageUrl: null,
       currentPrompt: null,
       
       /**
        * Initialize the image generation module
        */
       init: function() {
           //console.log('MxChat Image Generation initializing...');
           
           // DOM elements
           this.imageGenToggle = $('.mxchat-image-gen-toggle');
           
           // Check if image generation is enabled (requires OpenAI API key)
           this.isEnabled = false; // Start disabled
           
           // Initialize event listeners
           this.initEventListeners();
           
           // Load preferences from localStorage
           this.loadPreferences();
           
           // Update localStorage with current settings from PHP if not already set
           this.initializeSettingsStorage();
           
           //console.log('MxChat Image Generation initialized');
       },
       
       /**
        * Initialize localStorage settings from PHP defaults if not already set
        */
       initializeSettingsStorage: function() {
           // Common settings
           if (!localStorage.getItem('mxchat_image_gen_model')) {
               localStorage.setItem('mxchat_image_gen_model', mxchatImageGen.default_model || 'openai');
           }
           
           // DALL-E settings
           if (!localStorage.getItem('mxchat_image_size')) {
               localStorage.setItem('mxchat_image_size', mxchatImageGen.default_size || '1024x1024');
           }
           if (!localStorage.getItem('mxchat_image_quality')) {
               localStorage.setItem('mxchat_image_quality', mxchatImageGen.default_quality || 'standard');
           }
           
           // GPT Image settings
           if (!localStorage.getItem('mxchat_gpt_image_size')) {
               localStorage.setItem('mxchat_gpt_image_size', mxchatImageGen.default_gpt_image_size || '1024x1024');
           }
           if (!localStorage.getItem('mxchat_gpt_image_quality')) {
               localStorage.setItem('mxchat_gpt_image_quality', mxchatImageGen.default_gpt_image_quality || 'medium');
           }
           if (!localStorage.getItem('mxchat_gpt_image_format')) {
               localStorage.setItem('mxchat_gpt_image_format', mxchatImageGen.default_gpt_image_format || 'png');
           }
           if (!localStorage.getItem('mxchat_gpt_image_background')) {
               localStorage.setItem('mxchat_gpt_image_background', mxchatImageGen.default_gpt_image_background || 'opaque');
           }
           if (!localStorage.getItem('mxchat_gpt_image_compression')) {
               localStorage.setItem('mxchat_gpt_image_compression', mxchatImageGen.default_gpt_image_compression || '75');
           }
           
           // xAI settings
           if (!localStorage.getItem('mxchat_image_count')) {
               localStorage.setItem('mxchat_image_count', mxchatImageGen.default_image_count || '1');
           }
           if (!localStorage.getItem('mxchat_image_response_format')) {
               localStorage.setItem('mxchat_image_response_format', mxchatImageGen.default_response_format || 'url');
           }
           
           // Gemini settings (minimal, just the model per docs)
           if (!localStorage.getItem('mxchat_gemini_model')) {
               localStorage.setItem('mxchat_gemini_model', mxchatImageGen.default_gemini_model || 'gemini-2.5-flash-image-preview');
           }
       },
       
       /**
        * Initialize event listeners
        */
       initEventListeners: function() {
           //console.log('Initializing image generation event listeners');
           const self = this;
           
           // Toggle button click
           this.imageGenToggle.on('click', function() {
               self.toggleImageGeneration();
           });
           
           // Alt+I keyboard shortcut for toggling image generation
           $(document).on('keydown', function(e) {
               if (e.altKey && e.key === 'i') {
                   self.toggleImageGeneration();
                   e.preventDefault();
               }
           });
           
           // Intercept message sending
           $(document).on('mxchat_admin_before_send', function(e, messageData, cancelCallback) {
               // Only intercept if image generation is enabled
               if (self.isEnabled) {
                   //console.log('Image generation intercepting message:', messageData);
                   
                   // Store the prompt
                   self.currentPrompt = messageData.message;
                   
                   // Cancel the normal message processing
                   cancelCallback();
                   
                   // Show user message in chat
                   if (window.MxChatAdmin) {
                       window.MxChatAdmin.addMessage('user', messageData.message);
                       window.MxChatAdmin.scrollToBottom();
                       
                       // Mark as having unsaved changes but DON'T save yet
                       window.MxChatAdmin.unsavedChanges = true;
                       
                       // Add thinking indicator
                       window.MxChatAdmin.addThinkingIndicator();
                   }
                   
                   // Clear the input field
                   $('#mxchat-message-input').val('');
                   $('#mxchat-message-input').css('height', 'auto');
                   
                   // Generate image
                   self.generateImage(messageData.message);
               }
           });
           
           // Handle image download clicks (delegated event)
           $(document).on('click', '.mxchat-image-download', function(e) {
               e.preventDefault();
               const imageUrl = $(this).data('url');
               const imageName = $(this).data('name') || 'generated-image.jpg';
               self.downloadImage(imageUrl, imageName);
           });
           
           //console.log('Image generation event listeners initialized');
       },
       
       /**
        * Load preferences from localStorage
        */
       loadPreferences: function() {
           const isEnabled = localStorage.getItem('mxchat_image_gen_enabled') === 'true';
           
           if (isEnabled) {
               this.enableImageGeneration(false); // Don't show notification on initial load
           }
       },
       
       /**
        * Toggle image generation on/off
        */
       toggleImageGeneration: function() {
           //console.log('Toggling image generation, current state:', this.isEnabled);
           
           if (this.isEnabled) {
               this.disableImageGeneration();
           } else {
               this.enableImageGeneration();
           }
       },
       
       /**
        * Enable image generation
        * 
        * @param {boolean} showNotification - Whether to show a notification
        */
       enableImageGeneration: function(showNotification = true) {
           //console.log('Enabling image generation');
           
           this.isEnabled = true;
           this.imageGenToggle.addClass('active');
           
           // Save preference
           localStorage.setItem('mxchat_image_gen_enabled', 'true');
           
           // Show notification
           if (showNotification) {
               this.showNotification(mxchatImageGen.i18n.image_gen_toggle_on);
           }
           
           // Update the message input placeholder to indicate image generation mode
           $('#mxchat-message-input').attr('placeholder', mxchatImageGen.i18n.image_gen_prompt);
       },
       
       /**
        * Disable image generation
        */
       disableImageGeneration: function() {
           //console.log('Disabling image generation');
           
           this.isEnabled = false;
           this.imageGenToggle.removeClass('active');
           
           // Save preference
           localStorage.setItem('mxchat_image_gen_enabled', 'false');
           
           // Show notification
           this.showNotification(mxchatImageGen.i18n.image_gen_toggle_off);
           
           // Restore original message input placeholder
           $('#mxchat-message-input').attr('placeholder', mxChatAdminChat.i18n.how_can_help);
       },
       
       /**
        * Generate an image based on the prompt
        * 
        * @param {string} prompt - The image description
        */
       generateImage: function(prompt) {
           //console.log('Generating image for prompt:', prompt);
           
           // Get settings from localStorage or defaults
           const model = localStorage.getItem('mxchat_image_gen_model') || mxchatImageGen.default_model || 'openai';
           
           // Create base request data
           let requestData = {
               action: 'mxchat_admin_generate_image',
               nonce: mxchatImageGen.nonce,
               prompt: prompt,
               model: model
           };
           
           // Add model-specific parameters
           if (model === 'openai') {
               const size = localStorage.getItem('mxchat_image_size') || mxchatImageGen.default_size || '1024x1024';
               const quality = localStorage.getItem('mxchat_image_quality') || mxchatImageGen.default_quality || 'standard';
               
               requestData.size = size;
               requestData.quality = quality;
               
               //console.log('OpenAI image generation settings:', { model, size, quality });
           } else if (model === 'gpt-image') {
               // Get GPT Image specific settings from localStorage or defaults
               const size = localStorage.getItem('mxchat_gpt_image_size') || '1024x1024';
               const quality = localStorage.getItem('mxchat_gpt_image_quality') || 'medium';
               const format = localStorage.getItem('mxchat_gpt_image_format') || 'png';
               const background = localStorage.getItem('mxchat_gpt_image_background') || 'opaque';
               
               // Add parameters to request
               requestData.gpt_image_size = size;
               requestData.gpt_image_quality = quality;
               requestData.gpt_image_format = format;
               requestData.gpt_image_background = background;
               
               // Add compression if format is jpeg or webp
               if (format === 'jpeg' || format === 'webp') {
                   const compression = localStorage.getItem('mxchat_gpt_image_compression') || '75';
                   requestData.gpt_image_compression = compression;
               }
               
               //console.log('GPT Image generation settings:', { model, size, quality, format, background });
           } else if (model === 'xai') {
               const imageCount = localStorage.getItem('mxchat_image_count') || mxchatImageGen.default_image_count || '1';
               const responseFormat = localStorage.getItem('mxchat_image_response_format') || mxchatImageGen.default_response_format || 'url';
               
               requestData.image_count = imageCount;
               requestData.response_format = responseFormat;
               
               //console.log('xAI image generation settings:', { model, imageCount, responseFormat });
           } else if (model === 'gemini') {
               const geminiModel = localStorage.getItem('mxchat_gemini_model') || mxchatImageGen.default_gemini_model || 'gemini-2.5-flash-image-preview';
               
               requestData.gemini_model = geminiModel; // Optional, but pass for consistency
               
               //console.log('Gemini image generation settings:', { model, geminiModel });
           }
           
           // Make AJAX request
           $.ajax({
               url: mxchatImageGen.ajaxurl,
               type: 'POST',
               data: requestData,
               dataType: 'json',
               success: (response) => {
                   //console.log('Image generation response:', response);
                   
                   // Remove thinking indicator
                   if (window.MxChatAdmin) {
                       window.MxChatAdmin.removeThinkingIndicator();
                   }
                   
                   if (response.success) {
                       // All models now return image_url
                       if (response.data.image_url) {
                           this.displayGeneratedImage(
                               response.data.image_url,
                               prompt,
                               response.data.revised_prompt || response.data.text_response || ''
                           );
                       } else if (response.data.data && Array.isArray(response.data.data)) {
                           // Handle multiple images (xAI)
                           this.displayMultipleImages(response.data.data, prompt);
                       } else {
                           this.displayErrorMessage('Invalid response format from server');
                       }
                   } else {
                       this.displayErrorMessage(
                           response.data && response.data.message
                               ? response.data.message
                               : mxchatImageGen.i18n.image_gen_error
                       );
                   }
               },
               error: (xhr, status, errorThrown) => {
                   // Remove thinking indicator on error
                   if (window.MxChatAdmin) {
                       window.MxChatAdmin.removeThinkingIndicator();
                   }
                   
                   let errMsg = mxchatImageGen.i18n.image_gen_error;
                   if (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                       errMsg = xhr.responseJSON.data.message;
                   }
                   console.error('Image gen error:', errMsg);
                   this.displayErrorMessage(errMsg);
               }
           });
       },
       
       /**
        * Display a generated image in the chat
        * 
        * @param {string} imageUrl - URL of the generated image
        * @param {string} originalPrompt - Original prompt used for generation
        * @param {string} revisedPrompt - Revised prompt from the AI (if available)
        */
       displayGeneratedImage: function(imageUrl, originalPrompt, revisedPrompt) {
           //console.log('Displaying generated image:', imageUrl);
           
           // Store the image URL
           this.currentImageUrl = imageUrl;
           
           // Create message content with the image URL
           let messageContent = `<p>${originalPrompt}</p>`;
           messageContent += `<img src="${imageUrl}" alt="${originalPrompt}" class="mxchat-message-image" />`;
           
           // Add revised prompt if available
           if (revisedPrompt) {
               messageContent += `<p class="mxchat-revised-prompt"><strong>Revised prompt:</strong> ${revisedPrompt}</p>`;
           }
           
           // Add to chat using MxChatAdmin's addMessage function
           if (window.MxChatAdmin) {
               // Just add the message visually, don't save to history yet
               window.MxChatAdmin.addMessage('assistant', messageContent);
               window.MxChatAdmin.scrollToBottom();
               
               // Mark as having unsaved changes
               window.MxChatAdmin.unsavedChanges = true;
               
               // DON'T call saveMessageToHistory here - let the auto-save handle it
               // This prevents multiple saves of the same message
           }
       },
       
       /**
        * Display multiple images in a single chat message
        * 
        * @param {Array} images - Array of image objects from xAI response
        * @param {string} originalPrompt - Original prompt used for generation
        */
       displayMultipleImages: function(images, originalPrompt) {
           //console.log('Displaying multiple images:', images.length);
           
           // Get response format from localStorage
           const responseFormat = localStorage.getItem('mxchat_image_response_format') || 'url';
           
           // Create message content with all images
           let messageContent = `<p>${originalPrompt}</p>`;
           messageContent += '<div class="mxchat-images-grid">';
           
           // Add each image to the grid
           images.forEach((image, index) => {
               const imageUrl = image.url || (responseFormat === 'url' ? 
                   image.url : 'data:image/jpeg;base64,' + image.b64_json);
               
               // Store the first image URL
               if (index === 0) {
                   this.currentImageUrl = imageUrl;
               }
               
               const imageName = `image-${index + 1}-${Date.now()}.jpg`;
               
               messageContent += `
               <div class="mxchat-image-container">
                   <div class="mxchat-image-counter">${index + 1}/${images.length}</div>
                   <div class="mxchat-image-wrapper">
                       <img src="${imageUrl}" alt="${originalPrompt}" class="mxchat-message-image" />
                   </div>
                   <div class="mxchat-image-actions">
                       <button type="button" class="mxchat-image-download" data-url="${imageUrl}" data-name="${imageName}" title="Download image">
                           <i class="fa-solid fa-download"></i>
                       </button>
                   </div>
               </div>`;
           });
           
           messageContent += '</div>';
           
           // Add revised prompt if available
           if (images[0] && images[0].revised_prompt) {
               messageContent += `<p class="mxchat-revised-prompt"><strong>Revised prompt:</strong> ${images[0].revised_prompt}</p>`;
           }
           
           // Add to chat
           if (window.MxChatAdmin) {
               // Just add the message visually, don't save to history yet
               window.MxChatAdmin.addMessage('assistant', messageContent);
               window.MxChatAdmin.scrollToBottom();
               
               // Mark as having unsaved changes
               window.MxChatAdmin.unsavedChanges = true;
               
               // DON'T call saveMessageToHistory here
           }
       },
       
       /**
        * Download an image locally - CORS-safe version
        * 
        * @param {string} imageUrl - URL of the image to download
        * @param {string} imageName - Filename for the downloaded image
        */
       downloadImage: function(imageUrl, imageName) {
           //console.log('Downloading image:', imageUrl, imageName);
           
           // For base64 encoded images (these work fine)
           if (imageUrl.startsWith('data:')) {
               const downloadLink = document.createElement('a');
               downloadLink.style.display = 'none';
               downloadLink.href = imageUrl;
               downloadLink.download = imageName;
               document.body.appendChild(downloadLink);
               downloadLink.click();
               document.body.removeChild(downloadLink);
               return;
           }
           
           // For remote URLs with potential CORS issues, use a server-side proxy
           const proxyUrl = mxchatImageGen.ajaxurl;
           
           // Show loading indicator
           this.showNotification('Downloading image...', 'info');
           
           // Send AJAX request to your server to download the image
           $.ajax({
               url: proxyUrl,
               type: 'POST',
               data: {
                   action: 'mxchat_admin_proxy_image',
                   nonce: mxchatImageGen.nonce,
                   image_url: imageUrl,
                   image_name: imageName
               },
               dataType: 'json',
               success: (response) => {
                   if (response.success && response.data && response.data.base64) {
                       // Create download link for the base64 data
                       const downloadLink = document.createElement('a');
                       downloadLink.style.display = 'none';
                       downloadLink.href = 'data:image/jpeg;base64,' + response.data.base64;
                       downloadLink.download = imageName;
                       document.body.appendChild(downloadLink);
                       downloadLink.click();
                       document.body.removeChild(downloadLink);
                       this.showNotification('Image downloaded successfully', 'success');
                   } else {
                       // Fallback method - open in new tab
                       this.showNotification('Please save the image manually from the new tab', 'info');
                       window.open(imageUrl, '_blank');
                   }
               },
               error: (xhr, status, error) => {
                   console.error('Error downloading image:', error);
                   // Fallback method - open in new tab
                   this.showNotification('Please save the image manually from the new tab', 'info');
                   window.open(imageUrl, '_blank');
               }
           });
       },
       
       /**
        * Display an error message in the chat
        * 
        * @param {string} errorMessage - The error message to display
        */
       displayErrorMessage: function(errorMessage) {
           if (window.MxChatAdmin) {
               window.MxChatAdmin.addErrorMessage(errorMessage);
               window.MxChatAdmin.scrollToBottom();
           } else {
               console.error('MxChatAdmin not available, cannot display error:', errorMessage);
               alert(errorMessage);
           }
       },
       
       /**
        * Show a notification
        * 
        * @param {string} message - The notification message
        * @param {string} type - The notification type (info, success, error, etc.)
        */
       showNotification: function(message, type = 'info') {
           //console.log('Showing notification:', message);
           
           // Use MxChatAdmin's notification system if available
           if (window.MxChatAdmin && typeof window.MxChatAdmin.showNotification === 'function') {
               //console.log('Using MxChatAdmin.showNotification');
               window.MxChatAdmin.showNotification(message, type);
           } else {
               //console.log('Falling back to alert notification');
               alert(message);
           }
       }
   };
   
   // Initialize on document ready
   $(document).ready(function() {
       //console.log('Document ready, initializing MxChatImageGen');
       MxChatImageGen.init();
   });
   
   // Expose to global scope
   window.MxChatImageGen = MxChatImageGen;
   //console.log('MxChatImageGen exposed to global scope');

})(jQuery);