<?php
/**
 * MxChat Admin Chat API Handler
 *
 * Handles all AJAX API interactions for the admin chat.
 *
 * @since      1.0.0
 * @package    MxChat_Admin_Chat
 * @subpackage MxChat_Admin_Chat/includes
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

/**
 * API handler class for Admin Chat
 *
 * This class manages all AJAX interactions between the frontend and
 * the backend, and also manages communication with the AI providers.
 *
 * @since      1.0.0
 * @package    MxChat_Admin_Chat
 * @subpackage MxChat_Admin_Chat/includes
 */
class MxChat_Admin_Chat_API {
    private $image_storage;

    /**
     * The loader that's responsible for maintaining and registering all hooks.
     *
     * @since    1.0.0
     * @access   protected
     * @var      MxChat_Admin_Chat_Loader    $loader    Maintains and registers all hooks.
     */
    protected $loader;

    /**
     * The core chat class instance.
     *
     * @since    1.0.0
     * @access   protected
     * @var      MxChat_Admin_Chat_Core    $chat    Core chat functionality.
     */
    protected $chat;

    /**
     * Initialize the class.
     *
     * @since    1.0.0
     * @param    MxChat_Admin_Chat_Loader    $loader    The loader for hooks.
     */
    public function __construct($loader) {
        $this->loader = $loader;

        // Create instance of the core chat class
        $this->chat = new MxChat_Admin_Chat_Core();

        // Register AJAX handlers
        $this->loader->add_action('wp_ajax_mxchat_admin_chat_send_message', $this, 'handle_send_message');
        $this->loader->add_action('wp_ajax_mxchat_admin_chat_clear_conversation', $this, 'handle_clear_conversation');
        $this->loader->add_action('wp_ajax_mxchat_admin_chat_get_conversation', $this, 'handle_get_conversation');
        $this->loader->add_action('wp_ajax_mxchat_admin_chat_update_model', $this, 'handle_update_model');
        $this->loader->add_action('wp_ajax_mxchat_admin_chat_get_models', $this, 'handle_get_models');
        $this->loader->add_action('wp_ajax_mxchat_admin_chat_save_message', $this, 'handle_save_message');

        // Register new AJAX handlers for chat history
        $this->loader->add_action('wp_ajax_mxchat_admin_chat_get_history', $this, 'handle_get_history');
        $this->loader->add_action('wp_ajax_mxchat_admin_chat_save_thread', $this, 'handle_save_thread');
        $this->loader->add_action('wp_ajax_mxchat_admin_chat_load_thread', $this, 'handle_load_thread');
        $this->loader->add_action('wp_ajax_mxchat_admin_chat_delete_thread', $this, 'handle_delete_thread');
        $this->loader->add_action('wp_ajax_mxchat_admin_generate_image', $this, 'handle_generate_image');
        $this->loader->add_action('wp_ajax_mxchat_admin_save_settings', $this, 'handle_save_settings');
        $this->loader->add_action('wp_ajax_mxchat_admin_proxy_image', $this, 'handle_proxy_image');
        $this->loader->add_action('wp_ajax_mxchat_admin_analyze_image', $this, 'handle_analyze_image');
        
        // **NEW** - Register image editing handler
        $this->loader->add_action('wp_ajax_mxchat_admin_edit_image', $this, 'handle_edit_image');

        $this->loader->add_action('wp_ajax_mxchat_admin_chat_quick_save', $this, 'handle_quick_save');
        $this->loader->add_action('wp_ajax_mxchat_admin_chat_check_updates', $this, 'handle_check_updates');
        
        // Load image storage handler with error checking
        $image_storage_file = MXCHAT_ADMIN_CHAT_PLUGIN_DIR . 'includes/class-mxchat-image-storage-handler.php';
        if (file_exists($image_storage_file)) {
            require_once $image_storage_file;
            $this->image_storage = new MxChat_Image_Storage_Handler();
        } else {
            //error_log('MxChat: Image storage handler file not found at ' . $image_storage_file);
        }
    }
    
    /**
     * Verify nonce for AJAX requests
     *
     * @since    1.0.0
     * @return   boolean    True if nonce is valid
     */
    private function verify_nonce() {
        $nonce = isset($_REQUEST['nonce']) ? sanitize_text_field($_REQUEST['nonce']) : '';
        return wp_verify_nonce($nonce, 'mxchat_admin_chat_nonce');
    }

    /**
     * Verify capability for admin operations
     *
     * @since    1.0.0
     * @return   boolean    True if user has capability
     */
    private function verify_capability() {
        return current_user_can('manage_options');
    }

/**
 * UPDATED handle_send_message function with timeout handling
 */
public function handle_send_message() {
    // Check permissions
    if (!$this->verify_nonce() || !$this->verify_capability()) {
        wp_send_json_error(array(
            'message' => __('Unauthorized access', 'mxchat-admin-chat')
        ), 403);
    }
    
    // Get the message - IMPORTANT: Don't use sanitize_textarea_field here
    // as it will strip HTML tags which will cause code examples to be improperly rendered
    $message = isset($_POST['message']) ? $_POST['message'] : '';
    
    // Get context messages if provided
    $context_messages_json = isset($_POST['context_messages']) ? $_POST['context_messages'] : '[]';
    $context_messages = json_decode(stripslashes($context_messages_json), true);
    if (!is_array($context_messages)) {
        $context_messages = array();
    }
    
    // Basic validation without stripping HTML
    if (empty($message)) {
        wp_send_json_error(array(
            'message' => __('Message cannot be empty', 'mxchat-admin-chat')
        ), 400);
    }
    
    // Rate limiting (optional)
    $user_id = get_current_user_id();
    $rate_limit_key = 'mxchat_admin_chat_rate_limit_' . $user_id;
    $last_request = get_transient($rate_limit_key);
    
    if (false !== $last_request) {
        wp_send_json_error(array(
            'message' => __('Please wait a moment before sending another message', 'mxchat-admin-chat')
        ), 429);
    }
    
    // Set rate limit (1 second)
    set_transient($rate_limit_key, time(), 1);
    
    // INCREASE PHP EXECUTION TIME FOR LONG RESPONSES
    if (function_exists('set_time_limit')) {
        set_time_limit(300); // 5 minutes
    }
    
    // Send the message to the AI using limited context
    try {
        // ALWAYS use send_message_with_context since that's the only method that exists
        $response = $this->chat->send_message_with_context($message, $context_messages);
        
        if (!$response['success']) {
            // Check if this is a context length error
            if (isset($response['error_type']) && $response['error_type'] === 'context_length') {
                wp_send_json_error(array(
                    'message' => $response['message'],
                    'error_type' => 'context_length'
                ), 500);
            } 
            // Check if this is a timeout error from the response
            elseif (isset($response['error_type']) && $response['error_type'] === 'timeout') {
                wp_send_json_error(array(
                    'message' => __('The AI is taking longer than usual to respond. Please try again with a shorter, more specific question.', 'mxchat-admin-chat'),
                    'error_type' => 'timeout'
                ), 500);
            }
            // Check if the response message contains timeout indicators
            elseif (isset($response['message']) && (
                strpos($response['message'], 'timed out') !== false || 
                strpos($response['message'], 'timeout') !== false ||
                strpos($response['message'], 'cURL error 28') !== false
            )) {
                wp_send_json_error(array(
                    'message' => __('The AI is taking longer than usual to respond. Please try again with a shorter, more specific question.', 'mxchat-admin-chat'),
                    'error_type' => 'timeout'
                ), 500);
            }
            else {
                wp_send_json_error(array(
                    'message' => $response['message']
                ), 500);
            }
        }
        
        // Return successful response
        wp_send_json_success(array(
            'response' => $response['response'],
            'model' => $response['model']
        ));
    } catch (Exception $e) {
        $error_message = $e->getMessage();
        
        // Check if this is a context length error
        if (strpos($error_message, 'This model\'s maximum context length') !== false || 
            strpos($error_message, 'token limit') !== false ||
            strpos($error_message, 'maximum tokens') !== false) {
            
            wp_send_json_error(array(
                'message' => __('The conversation has exceeded the maximum allowed length.', 'mxchat-admin-chat'),
                'error_type' => 'context_length'
            ), 500);
        } 
        // Check if this is a timeout error - THIS WAS MISSING IN YOUR ORIGINAL
        elseif (strpos($error_message, 'timed out') !== false || 
                strpos($error_message, 'timeout') !== false ||
                strpos($error_message, 'cURL error 28') !== false ||
                strpos($error_message, 'Operation timed out') !== false) {
            
            wp_send_json_error(array(
                'message' => __('The AI is taking longer than usual to respond. Please try again with a shorter, more specific question.', 'mxchat-admin-chat'),
                'error_type' => 'timeout'
            ), 500);
        } 
        else {
            wp_send_json_error(array(
                'message' => $error_message
            ), 500);
        }
    }
}


/**
 * Helper method to clean HTML content for API requests
 */
private function clean_html_content($content) {
    // Remove HTML tags but preserve line breaks and spacing
    $content = strip_tags($content, '<br><p>');
    
    // Replace <br> and <p> tags with newlines
    $content = str_replace(array('<br>', '<br/>', '<br />', '</p>'), "\n", $content);
    $content = str_replace('<p>', "\n\n", $content);
    
    // Decode HTML entities
    $content = html_entity_decode($content, ENT_QUOTES | ENT_HTML5, 'UTF-8');
    
    // Clean up extra whitespace
    $content = preg_replace('/\n\s+\n/', "\n\n", $content);
    $content = preg_replace('/\n{3,}/', "\n\n", $content);
    
    return trim($content);
}

/**
 * Get system message for the conversation
 */
private function get_system_message() {
    return 'message is long and strong'; // Override this with your preferred system message
}


    /**
     * Handle clearing the conversation
     *
     * @since    1.0.0
     */
    public function handle_clear_conversation() {
        // Check permissions
        if (!$this->verify_nonce() || !$this->verify_capability()) {
            wp_send_json_error(array(
                'message' => __('Unauthorized access', 'mxchat-admin-chat')
            ), 403);
        }
        
        // Clear the conversation
        $this->chat->clear_conversation();
        
        // Return successful response
        wp_send_json_success(array(
            'message' => __('Conversation cleared successfully', 'mxchat-admin-chat')
        ));
    }

    /**
     * Handle retrieving the conversation history
     *
     * @since    1.0.0
     */
    public function handle_get_conversation() {
        // Check permissions
        if (!$this->verify_nonce() || !$this->verify_capability()) {
            wp_send_json_error(array(
                'message' => __('Unauthorized access', 'mxchat-admin-chat')
            ), 403);
        }
        
        // Get the conversation
        $conversation = $this->chat->get_conversation_history();
        
        // Get current model
        $model = $this->chat->get_current_model();
        
        // Return successful response
        wp_send_json_success(array(
            'conversation' => $conversation,
            'model' => $model
        ));
    }

    /**
     * Handle updating the selected model
     *
     * @since    1.0.0
     */
    public function handle_update_model() {
        // Check permissions
        if (!$this->verify_nonce() || !$this->verify_capability()) {
            wp_send_json_error(array(
                'message' => __('Unauthorized access', 'mxchat-admin-chat')
            ), 403);
        }
        
        // Get the model
        $model_id = isset($_POST['model_id']) ? sanitize_text_field($_POST['model_id']) : '';
        
        if (empty($model_id)) {
            wp_send_json_error(array(
                'message' => __('Model ID cannot be empty', 'mxchat-admin-chat')
            ), 400);
        }
        
        // Update the model
        $success = $this->chat->update_selected_model($model_id);
        
        if (!$success) {
            wp_send_json_error(array(
                'message' => __('Invalid model ID', 'mxchat-admin-chat')
            ), 400);
        }
        
        // Get the updated model
        $model = $this->chat->get_current_model();
        
        // Return successful response
        wp_send_json_success(array(
            'model' => $model,
            'message' => __('Model updated successfully', 'mxchat-admin-chat')
        ));
    }

    /**
     * Handle retrieving the available models
     *
     * @since    1.0.0
     */
    public function handle_get_models() {
        // Check permissions
        if (!$this->verify_nonce() || !$this->verify_capability()) {
            wp_send_json_error(array(
                'message' => __('Unauthorized access', 'mxchat-admin-chat')
            ), 403);
        }
        
        // Get the models
        $models = $this->chat->get_available_models();
        
        // Get current model
        $current_model = $this->chat->get_current_model();
        
        // Return successful response
        wp_send_json_success(array(
            'models' => $models,
            'current_model' => $current_model
        ));
    }

/**
 * Handle saving a single message to conversation history with image optimization
 * 
 * @since    1.0.0
 */
public function handle_save_message() {
    // Verify nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'mxchat_admin_chat_nonce')) {
        wp_send_json_error(array('message' => 'Security check failed'));
        return;
    }
    
    // Initialize image storage handler if not already done
    if (!isset($this->image_storage)) {
        require_once MXCHAT_ADMIN_CHAT_PLUGIN_PATH . 'includes/class-mxchat-image-storage-handler.php';
        $this->image_storage = new MxChat_Image_Storage_Handler();
    }
    
    // Get message data
    $role = isset($_POST['role']) ? sanitize_text_field($_POST['role']) : '';
    
    // Get raw content to preserve code formatting
    $content = isset($_POST['content']) ? $_POST['content'] : '';
    
    $source = isset($_POST['source']) ? sanitize_text_field($_POST['source']) : '';
    
    if (empty($role) || empty($content)) {
        wp_send_json_error(array('message' => 'Missing required data'));
        return;
    }
    
    // IMPORTANT: Process content to replace base64 images with URLs
    if (strpos($content, 'data:image') !== false) {
        $content = $this->image_storage->process_message_content($content, 'message');
    }
    
    // Get current conversation
    $conversation = get_user_meta(get_current_user_id(), 'mxchat_admin_conversation', true);
    if (!is_array($conversation)) {
        $conversation = array();
    }
    
    // Add the processed message to conversation
    $conversation[] = array(
        'role' => $role,
        'content' => $content, // Now contains URLs instead of base64
        'timestamp' => time(),
        'source' => $source
    );
    
    // Save updated conversation
    update_user_meta(get_current_user_id(), 'mxchat_admin_conversation', $conversation);
    
    wp_send_json_success();
}


    /**
     * Handle getting the chat history
     * 
     * @since    1.0.0
     */
    public function handle_get_history() {
        // Check permissions
        if (!$this->verify_nonce() || !$this->verify_capability()) {
            wp_send_json_error(array(
                'message' => __('Unauthorized access', 'mxchat-admin-chat')
            ), 403);
        }
        
        // Get user ID
        $user_id = get_current_user_id();
        
        // Get chat threads from user meta
        $chat_threads = get_user_meta($user_id, 'mxchat_admin_chat_threads', true);
        
        // If no threads exist yet, initialize empty array
        if (!is_array($chat_threads)) {
            $chat_threads = array();
        }
        
        // Return successful response
        wp_send_json_success(array(
            'chat_threads' => $chat_threads
        ));
    }

/**
 * Handle saving a chat thread with image optimization
 * 
 * @since    1.0.0
 */
public function handle_save_thread() {
    // Check permissions
    if (!$this->verify_nonce() || !$this->verify_capability()) {
        wp_send_json_error(array(
            'message' => __('Unauthorized access', 'mxchat-admin-chat')
        ), 403);
    }
    
    // Initialize image storage handler if not already done
    if (!isset($this->image_storage)) {
        require_once MXCHAT_ADMIN_CHAT_PLUGIN_PATH . 'includes/class-mxchat-image-storage-handler.php';
        $this->image_storage = new MxChat_Image_Storage_Handler();
    }
    
    // Get parameters
    $chat_id = isset($_POST['chat_id']) ? sanitize_text_field($_POST['chat_id']) : null;
    $title = isset($_POST['title']) ? sanitize_text_field($_POST['title']) : '';
    $messages_json = isset($_POST['messages']) ? $_POST['messages'] : '[]';
    $model_id = isset($_POST['model_id']) ? sanitize_text_field($_POST['model_id']) : '';
    $model_name = isset($_POST['model_name']) ? sanitize_text_field($_POST['model_name']) : '';
    
    // Parse messages JSON
    $messages = json_decode(stripslashes($messages_json), true);
    if (!is_array($messages)) {
        $messages = array();
    }
    
    // IMPORTANT: Process each message to replace base64 images with URLs
    foreach ($messages as &$message) {
        if (isset($message['content']) && strpos($message['content'], 'data:image') !== false) {
            $message['content'] = $this->image_storage->process_message_content(
                $message['content'], 
                'thread'
            );
        }
    }
    
    // Get user ID
    $user_id = get_current_user_id();
    
    // Get existing chat threads
    $chat_threads = get_user_meta($user_id, 'mxchat_admin_chat_threads', true);
    if (!is_array($chat_threads)) {
        $chat_threads = array();
    }
    
    // Current timestamp
    $timestamp = time();
    
    // If chat_id is null or not found, create a new chat thread
    if (empty($chat_id) || !isset($chat_threads[$chat_id])) {
        $chat_id = 'chat_' . uniqid();
        
        // Create new chat thread with processed messages
        $chat_threads[$chat_id] = array(
            'id' => $chat_id,
            'title' => $title,
            'messages' => $messages, // Now contains URLs instead of base64
            'model_id' => $model_id,
            'model_name' => $model_name,
            'created_at' => $timestamp,
            'updated_at' => $timestamp
        );
    } else {
        // Update existing chat thread with processed messages
        $chat_threads[$chat_id]['title'] = $title;
        $chat_threads[$chat_id]['messages'] = $messages; // Now contains URLs instead of base64
        $chat_threads[$chat_id]['model_id'] = $model_id;
        $chat_threads[$chat_id]['model_name'] = $model_name;
        $chat_threads[$chat_id]['updated_at'] = $timestamp;
    }
    
    // Save updated chat threads
    update_user_meta($user_id, 'mxchat_admin_chat_threads', $chat_threads);
    
    // Return successful response
    wp_send_json_success(array(
        'chat_id' => $chat_id,
        'message' => __('Chat saved successfully', 'mxchat-admin-chat')
    ));
}

    /**
     * Handle loading a chat thread
     * 
     * @since    1.0.0
     */
    public function handle_load_thread() {
        // Check permissions
        if (!$this->verify_nonce() || !$this->verify_capability()) {
            wp_send_json_error(array(
                'message' => __('Unauthorized access', 'mxchat-admin-chat')
            ), 403);
        }
        
        // Get chat ID
        $chat_id = isset($_POST['chat_id']) ? sanitize_text_field($_POST['chat_id']) : '';
        
        if (empty($chat_id)) {
            wp_send_json_error(array(
                'message' => __('Chat ID cannot be empty', 'mxchat-admin-chat')
            ), 400);
        }
        
        // Get user ID
        $user_id = get_current_user_id();
        
        // Get chat threads
        $chat_threads = get_user_meta($user_id, 'mxchat_admin_chat_threads', true);
        
        if (!is_array($chat_threads) || !isset($chat_threads[$chat_id])) {
            wp_send_json_error(array(
                'message' => __('Chat thread not found', 'mxchat-admin-chat')
            ), 404);
        }
        
        // Get the chat thread
        $chat_thread = $chat_threads[$chat_id];
        
        // Also update the current conversation in the session
        if (method_exists($this->chat, 'set_conversation')) {
            $this->chat->set_conversation($chat_thread['messages']);
        } else {
            // Fallback if set_conversation method doesn't exist
            // Clear existing conversation
            $this->chat->clear_conversation();
            
            // Add each message to the conversation
            foreach ($chat_thread['messages'] as $message) {
                $this->chat->add_to_conversation($message['role'], $message['content']);
            }
        }
        
        // If the chat has a model, update it
        if (!empty($chat_thread['model_id'])) {
            $this->chat->update_selected_model($chat_thread['model_id']);
        }
        
        // Return successful response
        wp_send_json_success(array(
            'chat_id' => $chat_id,
            'title' => $chat_thread['title'],
            'messages' => $chat_thread['messages'],
            'model_id' => $chat_thread['model_id'],
            'model_name' => $chat_thread['model_name']
        ));
    }

    /**
     * Handle deleting a chat thread
     * 
     * @since    1.0.0
     */
    public function handle_delete_thread() {
        // Check permissions
        if (!$this->verify_nonce() || !$this->verify_capability()) {
            wp_send_json_error(array(
                'message' => __('Unauthorized access', 'mxchat-admin-chat')
            ), 403);
        }
        
        // Get chat ID
        $chat_id = isset($_POST['chat_id']) ? sanitize_text_field($_POST['chat_id']) : '';
        
        if (empty($chat_id)) {
            wp_send_json_error(array(
                'message' => __('Chat ID cannot be empty', 'mxchat-admin-chat')
            ), 400);
        }
        
        // Get user ID
        $user_id = get_current_user_id();
        
        // Get chat threads
        $chat_threads = get_user_meta($user_id, 'mxchat_admin_chat_threads', true);
        
        if (!is_array($chat_threads) || !isset($chat_threads[$chat_id])) {
            wp_send_json_error(array(
                'message' => __('Chat thread not found', 'mxchat-admin-chat')
            ), 404);
        }
        
        // Remove the chat thread
        unset($chat_threads[$chat_id]);
        
        // Save updated chat threads
        update_user_meta($user_id, 'mxchat_admin_chat_threads', $chat_threads);
        
        // Return successful response
        wp_send_json_success(array(
            'message' => __('Chat deleted successfully', 'mxchat-admin-chat')
        ));
    }
    
    
    
    /**
 * Handle quick save for session recovery
 * This is optimized for fast saves during page unload
 * 
 * @since    1.0.0
 */
public function handle_quick_save() {
    // Quick nonce check - simplified for speed
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'mxchat_admin_chat_nonce')) {
        wp_send_json_error(array('message' => 'Security check failed'), 403);
        return;
    }
    
    // Quick capability check
    if (!current_user_can('manage_options')) {
        wp_send_json_error(array('message' => 'Unauthorized'), 403);
        return;
    }
    
    // Get parameters - minimal validation for speed
    $chat_id = isset($_POST['chat_id']) ? sanitize_text_field($_POST['chat_id']) : null;
    $title = isset($_POST['title']) ? sanitize_text_field($_POST['title']) : '';
    $messages_json = isset($_POST['messages']) ? $_POST['messages'] : '[]';
    $model_id = isset($_POST['model_id']) ? sanitize_text_field($_POST['model_id']) : '';
    $model_name = isset($_POST['model_name']) ? sanitize_text_field($_POST['model_name']) : '';
    
    // Parse messages - minimal processing
    $messages = json_decode(stripslashes($messages_json), true);
    if (!is_array($messages)) {
        $messages = array();
    }
    
    // Don't save empty chats
    if (empty($messages)) {
        wp_send_json_success(array('saved' => false));
        return;
    }
    
    // Get user ID
    $user_id = get_current_user_id();
    
    // Get existing chat threads
    $chat_threads = get_user_meta($user_id, 'mxchat_admin_chat_threads', true);
    if (!is_array($chat_threads)) {
        $chat_threads = array();
    }
    
    // Current timestamp
    $timestamp = time();
    
    // If chat_id is null or not found, create a new chat thread
    if (empty($chat_id) || !isset($chat_threads[$chat_id])) {
        $chat_id = 'chat_' . uniqid();
        
        // Create new chat thread
        $chat_threads[$chat_id] = array(
            'id' => $chat_id,
            'title' => $title,
            'messages' => $messages,
            'model_id' => $model_id,
            'model_name' => $model_name,
            'created_at' => $timestamp,
            'updated_at' => $timestamp,
            'quick_save' => true // Mark as quick save
        );
    } else {
        // Update existing chat thread
        $chat_threads[$chat_id]['messages'] = $messages;
        $chat_threads[$chat_id]['updated_at'] = $timestamp;
        $chat_threads[$chat_id]['quick_save'] = true;
        
        // Update title if provided and different
        if (!empty($title) && $title !== $chat_threads[$chat_id]['title']) {
            $chat_threads[$chat_id]['title'] = $title;
        }
        
        // Update model if provided
        if (!empty($model_id)) {
            $chat_threads[$chat_id]['model_id'] = $model_id;
            $chat_threads[$chat_id]['model_name'] = $model_name;
        }
    }
    
    // Save updated chat threads - use update_user_meta which is faster than transients
    update_user_meta($user_id, 'mxchat_admin_chat_threads', $chat_threads);
    
    // Return minimal response for speed
    wp_send_json_success(array(
        'saved' => true,
        'chat_id' => $chat_id
    ));
}

/**
 * Handle checking for updates to a chat thread
 * Used when tab regains focus to check if chat was modified elsewhere
 * 
 * @since    1.0.0
 */
public function handle_check_updates() {
    // Check permissions
    if (!$this->verify_nonce() || !$this->verify_capability()) {
        wp_send_json_error(array(
            'message' => __('Unauthorized access', 'mxchat-admin-chat')
        ), 403);
    }
    
    // Get parameters
    $chat_id = isset($_POST['chat_id']) ? sanitize_text_field($_POST['chat_id']) : '';
    $last_update = isset($_POST['last_update']) ? intval($_POST['last_update']) : 0;
    
    if (empty($chat_id)) {
        wp_send_json_error(array(
            'message' => __('Chat ID is required', 'mxchat-admin-chat')
        ), 400);
    }
    
    // Get user ID
    $user_id = get_current_user_id();
    
    // Get chat threads
    $chat_threads = get_user_meta($user_id, 'mxchat_admin_chat_threads', true);
    
    if (!is_array($chat_threads) || !isset($chat_threads[$chat_id])) {
        wp_send_json_error(array(
            'message' => __('Chat thread not found', 'mxchat-admin-chat')
        ), 404);
    }
    
    // Get the chat thread
    $chat_thread = $chat_threads[$chat_id];
    
    // Check if chat has been updated since last_update timestamp
    $has_updates = false;
    if (isset($chat_thread['updated_at']) && $chat_thread['updated_at'] > $last_update) {
        $has_updates = true;
    }
    
    // Prepare response
    $response_data = array(
        'has_updates' => $has_updates,
        'current_timestamp' => isset($chat_thread['updated_at']) ? $chat_thread['updated_at'] : time()
    );
    
    // If there are updates, optionally include the updated data
    if ($has_updates) {
        $response_data['updated_data'] = array(
            'message_count' => count($chat_thread['messages']),
            'title' => $chat_thread['title'],
            'model_id' => isset($chat_thread['model_id']) ? $chat_thread['model_id'] : '',
            'model_name' => isset($chat_thread['model_name']) ? $chat_thread['model_name'] : '',
            // Optionally include last message preview
            'last_message' => !empty($chat_thread['messages']) 
                ? end($chat_thread['messages'])['role'] 
                : null
        );
    }
    
    wp_send_json_success($response_data);
}

    /**
     * Register the JavaScript for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_scripts() {
        wp_enqueue_script(
            'mxchat-admin-chat',
            MXCHAT_ADMIN_CHAT_PLUGIN_URL . 'admin/js/mxchat-admin-chat.js',
            array('jquery'),
            MXCHAT_ADMIN_CHAT_VERSION,
            true
        );
        
        // Localize the script with data for JavaScript
        wp_localize_script(
            'mxchat-admin-chat',
            'mxChatAdminChat',
            array(
                'ajaxurl' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('mxchat_admin_chat_nonce'),
                'i18n' => array(
                    'sending' => __('Sending...', 'mxchat-admin-chat'),
                    'error' => __('Error', 'mxchat-admin-chat'),
                    'clear_confirm' => __('Are you sure you want to clear the conversation?', 'mxchat-admin-chat'),
                    'clear_success' => __('Conversation cleared', 'mxchat-admin-chat'),
                    'update_success' => __('Model updated', 'mxchat-admin-chat'),
                    'loading' => __('Loading...', 'mxchat-admin-chat'),
                    'thinking' => __('Thinking...', 'mxchat-admin-chat'),
                    'you' => __('You', 'mxchat-admin-chat'),
                    'assistant' => __('Assistant', 'mxchat-admin-chat'),
                    'welcome_message' => __('Welcome! How can I assist you today?', 'mxchat-admin-chat'),
                    'rate_limit' => __('Please wait a moment before sending another message', 'mxchat-admin-chat'),
                    'type_message' => __('Type a message...', 'mxchat-admin-chat'),
                    'send' => __('Send', 'mxchat-admin-chat'),
                    'clear' => __('Clear conversation', 'mxchat-admin-chat'),
                    'new_chat' => __('New chat', 'mxchat-admin-chat'),
                    'chat_history' => __('Chat History', 'mxchat-admin-chat'),
                    'model_select' => __('Select model', 'mxchat-admin-chat'),
                    'select_model' => __('Select an AI model', 'mxchat-admin-chat'),
                    'copy' => __('Copy to clipboard', 'mxchat-admin-chat'),
                    'copied' => __('Copied!', 'mxchat-admin-chat'),
                    'copy_error' => __('Failed to copy', 'mxchat-admin-chat'),
                    'retry' => __('Retry', 'mxchat-admin-chat'),
                    'empty_models' => __('No AI models available. Please check your API keys in the MxChat settings.', 'mxchat-admin-chat'),
                    'save_chat_confirm' => __('Do you want to save the current chat before starting a new one?', 'mxchat-admin-chat'),
                    'delete_chat_confirm' => __('Are you sure you want to delete this chat? This cannot be undone.', 'mxchat-admin-chat'),
                    'switch_chat_confirm' => __('Save changes to the current chat before switching?', 'mxchat-admin-chat'),
                    'save_before_history' => __('Save current chat before viewing history?', 'mxchat-admin-chat'),
                    'new_chat_started' => __('New chat started', 'mxchat-admin-chat'),
                    'chat_saved' => __('Chat saved successfully', 'mxchat-admin-chat'),
                    'saving' => __('Saving chat...', 'mxchat-admin-chat'),
                    'chat_loaded' => __('Chat loaded successfully', 'mxchat-admin-chat'),
                    'chat_deleted' => __('Chat deleted successfully', 'mxchat-admin-chat'),
                    'chat_history' => __('Chat History', 'mxchat-admin-chat'),
                    'search_chats' => __('Search chats...', 'mxchat-admin-chat'),
                    'context_length_error_title' => __('Conversation Too Long', 'mxchat-admin-chat'),
                    'context_length_error_message' => __('The conversation has reached the maximum allowed length. To continue, you can:', 'mxchat-admin-chat'),
                    'context_length_error_option1' => __('Start a new chat (your current chat will be saved)', 'mxchat-admin-chat'),
                    'context_length_error_option2' => __('Save this chat and continue in a new thread', 'mxchat-admin-chat'),
                    'context_length_error_option3' => __('Try a shorter, more focused question', 'mxchat-admin-chat'),
                    'context_length_error_notification' => __('Conversation length limit reached', 'mxchat-admin-chat'),
                    'vision_model' => get_option('mxchat_image_analysis_model', 'grok-vision'),
                    'vision_model_name' => $this->get_vision_model_name(get_option('mxchat_image_analysis_model', 'grok-vision')),
                )
            )
        );
    }

    /**
     * Register the stylesheets for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_styles() {
        wp_enqueue_style(
            'mxchat-admin-chat',
            MXCHAT_ADMIN_CHAT_PLUGIN_URL . 'admin/css/mxchat-admin-chat.css',
            array(),
            MXCHAT_ADMIN_CHAT_VERSION,
            'all'
        );
    }
    
/**
 * Handle image generation request with efficient storage
 * 
 * @since    1.0.0
 */
public function handle_generate_image() {
    // Check permissions
    if (
        ! isset( $_POST['nonce'] ) ||
        ! wp_verify_nonce( $_POST['nonce'], 'mxchat_admin_image_gen_nonce' )
    ) {
        wp_send_json_error( array(
            'message' => __( 'Security check failed', 'mxchat-admin-chat' )
        ), 403 );
        return;
    }

    if ( ! current_user_can( 'manage_options' ) ) {
        wp_send_json_error( array(
            'message' => __( 'Unauthorized access', 'mxchat-admin-chat' )
        ), 403 );
        return;
    }

    // Initialize image storage handler if not already done
    if (!isset($this->image_storage)) {
        require_once MXCHAT_ADMIN_CHAT_PLUGIN_PATH . 'includes/class-mxchat-image-storage-handler.php';
        $this->image_storage = new MxChat_Image_Storage_Handler();
    }

    // Get parameters
    $prompt      = isset( $_POST['prompt'] ) ? sanitize_textarea_field( $_POST['prompt'] ) : '';
    $model_param = isset( $_POST['model'] )  ? sanitize_text_field( $_POST['model'] )   : '';

    // Determine which model to use
    $image_gen_model = ! empty( $model_param )
        ? $model_param
        : get_option( 'mxchat_image_gen_model', 'openai' );

    if ( empty( $prompt ) ) {
        wp_send_json_error( array(
            'message' => __( 'Image description cannot be empty', 'mxchat-admin-chat' )
        ), 400 );
        return;
    }

    // Get API keys (updated to include Gemini from main plugin options)
    $main_plugin_options = get_option( 'mxchat_options', array() );
    $api_key             = isset( $main_plugin_options['api_key'] )
        ? $main_plugin_options['api_key']
        : '';

    // Dispatch to the correct generator
    if ( $image_gen_model === 'openai' ) {
        if ( empty( $api_key ) ) {
            wp_send_json_error( array(
                'message' => __( 'OpenAI API key is not configured', 'mxchat-admin-chat' )
            ), 400 );
            return;
        }
        $size    = get_option( 'mxchat_image_size', '1024x1024' );
        $quality = get_option( 'mxchat_image_quality', 'standard' );
        $response = $this->generate_dalle_image( $api_key, $prompt, $size, $quality );
    }
    elseif ( $image_gen_model === 'gpt-image' ) {
        if ( empty( $api_key ) ) {
            wp_send_json_error( array(
                'message' => __( 'OpenAI API key is not configured', 'mxchat-admin-chat' )
            ), 400 );
            return;
        }
        $size        = get_option( 'mxchat_gpt_image_size', '1024x1024' );
        $quality     = get_option( 'mxchat_gpt_image_quality', 'medium' );
        $format      = get_option( 'mxchat_gpt_image_format', 'png' );
        $background  = get_option( 'mxchat_gpt_image_background', 'opaque' );
        $compression = null;
        if ( in_array( $format, array( 'jpeg', 'webp' ), true ) ) {
            $compression = intval( get_option( 'mxchat_gpt_image_compression', 75 ) );
        }
        $response = $this->generate_gpt_image(
            $api_key,
            $prompt,
            $size,
            $quality,
            $format,
            $background,
            $compression
        );
    }
    elseif ( $image_gen_model === 'xai' ) {
        $image_count     = intval( get_option( 'mxchat_image_count', 1 ) );
        $response_format = get_option( 'mxchat_image_response_format', 'url' );
        $xai_key = isset( $main_plugin_options['xai_api_key'] )
            ? $main_plugin_options['xai_api_key']
            : '';
        if ( empty( $xai_key ) && method_exists( $this->chat, 'get_api_key' ) ) {
            $xai_key = $this->chat->get_api_key( 'xai' );
        }
        if ( empty( $xai_key ) ) {
            wp_send_json_error( array(
                'message' => __( 'xAI API key is not configured', 'mxchat-admin-chat' )
            ), 400 );
            return;
        }
        $response = $this->generate_xai_image(
            $xai_key,
            $prompt,
            $image_count,
            $response_format
        );
    }
    elseif ( $image_gen_model === 'gemini' ) {
        $gemini_key = isset( $main_plugin_options['gemini_api_key'] )
            ? $main_plugin_options['gemini_api_key']
            : '';
        
        if ( empty( $gemini_key ) ) {
            if (method_exists($this->chat, 'get_api_key')) {
                $gemini_key = $this->chat->get_api_key('gemini');
            }
            
            if (empty($gemini_key)) {
                wp_send_json_error( array(
                    'message' => __( 'Gemini API key is not configured', 'mxchat-admin-chat' )
                ), 400 );
                return;
            }
        }
        
        $response = $this->generate_gemini_image( $gemini_key, $prompt );
    }
    else {
        wp_send_json_error( array(
            'message' => __( 'Unsupported image generation model', 'mxchat-admin-chat' )
        ), 400 );
        return;
    }

    // Handle any WP_Error
    if ( is_wp_error( $response ) ) {
        wp_send_json_error( array(
            'message' => $response->get_error_message()
        ), 500 );
        return;
    }

    $body = json_decode( wp_remote_retrieve_body( $response ), true );
    if ( isset( $body['error'] ) ) {
        wp_send_json_error( array(
            'message' => $body['error']['message']
        ), 500 );
        return;
    }

    // Return based on model with storage optimization
    if ( $image_gen_model === 'openai' ) {
        $url = $body['data'][0]['url'] ?? '';
        if ( ! $url ) {
            wp_send_json_error( array(
                'message' => __( 'Error generating image', 'mxchat-admin-chat' )
            ), 500 );
            return;
        }
        // OpenAI already provides URLs, no need to save
        wp_send_json_success( array(
            'image_url'      => $url,
            'revised_prompt' => $body['data'][0]['revised_prompt'] ?? '',
            'prompt'         => $prompt,
            'storage_type'   => 'url'
        ) );
    }
    elseif ( $image_gen_model === 'gpt-image' ) {
        $b64 = $body['data'][0]['b64_json'] ?? '';
        if ( ! $b64 ) {
            wp_send_json_error( array(
                'message' => __( 'Error generating image with GPT Image', 'mxchat-admin-chat' )
            ), 500 );
            return;
        }
        
        // IMPORTANT: Save base64 to file instead of sending directly
        $data_uri = 'data:image/' .
            ( $format === 'jpeg' ? 'jpeg' : ( $format === 'webp' ? 'webp' : 'png' ) ) .
            ';base64,' . $b64;
        
        $saved_image = $this->image_storage->save_image_from_base64($data_uri, 'generated');
        
        if (is_wp_error($saved_image)) {
            wp_send_json_error(array(
                'message' => $saved_image->get_error_message()
            ), 500);
            return;
        }
        
        wp_send_json_success( array(
            'image_url'      => $saved_image['url'],
            'prompt'         => $prompt,
            'format'         => $format,
            'storage_type'   => 'file'
        ) );
    }
    elseif ( $image_gen_model === 'xai' ) {
        if ( empty( $body['data'] ) ) {
            wp_send_json_error( array(
                'message' => __( 'Error generating image with xAI', 'mxchat-admin-chat' )
            ), 500 );
            return;
        }
        
        // Process xAI images if they're base64
        $response_format = get_option( 'mxchat_image_response_format', 'url' );
        if ($response_format === 'b64_json' && isset($body['data']) && is_array($body['data'])) {
            foreach ($body['data'] as &$image_item) {
                if (isset($image_item['b64_json'])) {
                    $data_uri = 'data:image/jpeg;base64,' . $image_item['b64_json'];
                    $saved_image = $this->image_storage->save_image_from_base64($data_uri, 'generated');
                    
                    if (!is_wp_error($saved_image)) {
                        $image_item['url'] = $saved_image['url'];
                        unset($image_item['b64_json']); // Remove base64 data
                    }
                }
            }
        }
        
        wp_send_json_success( $body );
    }
    elseif ( $image_gen_model === 'gemini' ) {
        // Check for API error in response
        if (isset($body['error'])) {
            $error_msg = isset($body['error']['message']) ? $body['error']['message'] : 'Unknown Gemini API error';
            wp_send_json_error( array(
                'message' => $error_msg
            ), 500 );
            return;
        }
        
        // Extract image data from response
        $image_data = '';
        $text_response = '';
        
        // Check if we have candidates
        if (!isset($body['candidates']) || empty($body['candidates'])) {
            wp_send_json_error( array(
                'message' => __( 'No image data received from Gemini', 'mxchat-admin-chat' )
            ), 500 );
            return;
        }
        
        // Get the first candidate
        $candidate = $body['candidates'][0];
        
        // Check if candidate has content and parts
        if (!isset($candidate['content']['parts']) || empty($candidate['content']['parts'])) {
            wp_send_json_error( array(
                'message' => __( 'Invalid response structure from Gemini', 'mxchat-admin-chat' )
            ), 500 );
            return;
        }
        
        // Process all parts to find image data
        foreach ( $candidate['content']['parts'] as $part ) {
            if ( isset( $part['inlineData'] ) && isset( $part['inlineData']['data'] ) ) {
                $mime_type = isset($part['inlineData']['mimeType']) ? $part['inlineData']['mimeType'] : 'image/png';
                $image_data = 'data:' . $mime_type . ';base64,' . $part['inlineData']['data'];
                break;
            }
            if ( isset( $part['text'] ) ) {
                $text_response = $part['text'];
            }
        }
        
        if ( ! $image_data ) {
            wp_send_json_error( array(
                'message' => __( 'No image data found in Gemini response', 'mxchat-admin-chat' )
            ), 500 );
            return;
        }
        
        // IMPORTANT: Save base64 to file
        $saved_image = $this->image_storage->save_image_from_base64($image_data, 'generated');
        
        if (is_wp_error($saved_image)) {
            wp_send_json_error(array(
                'message' => $saved_image->get_error_message()
            ), 500);
            return;
        }
        
        wp_send_json_success( array(
            'image_url'      => $saved_image['url'],
            'text_response'  => $text_response,
            'prompt'         => $prompt,
            'model'          => 'gemini-2.5-flash-image-preview',
            'storage_type'   => 'file'
        ) );
    }
}

/**
 * Generate image using Google's Gemini (Nano Banana)
 * 
 * Uses the Gemini API for text-to-image generation, based on the docs.
 * Model: gemini-2.5-flash-image-preview
 * Returns inline image data (base64) as per API response.
 * 
 * @param string $api_key Gemini API key
 * @param string $prompt Image description
 * @return WP_Error|array
 */
private function generate_gemini_image( $api_key, $prompt ) {
    // Correct Gemini API endpoint format from docs
    $url = 'https://generativelanguage.googleapis.com/v1beta/models/gemini-2.5-flash-image-preview:generateContent?key=' . $api_key;

    // Prepare the request body (simplified structure from docs)
    $body = array(
        'contents' => array(
            array(
                'parts' => array(
                    array( 'text' => $prompt )
                )
            )
        ),
        // Add generation config for better control
        'generationConfig' => array(
            'temperature' => 0.7,
            'maxOutputTokens' => 8192
        )
    );

    $args = array(
        'method'  => 'POST',
        'timeout' => 120, // Increased timeout for image generation
        'headers' => array(
            'Content-Type' => 'application/json',
        ),
        'body'    => wp_json_encode( $body ),
    );

    // Debug: Log the request details
    //error_log('Gemini API Request URL: ' . $url);
    //error_log('Gemini API Request Body: ' . wp_json_encode($body));
    //error_log('Gemini API Key (first 10 chars): ' . substr($api_key, 0, 10) . '...');

    // Make the request
    $response = wp_remote_post( $url, $args );

    // Check for WP_Error first
    if ( is_wp_error( $response ) ) {
        //error_log('Gemini API WP Error: ' . $response->get_error_message());
        //error_log('Gemini API Error Code: ' . $response->get_error_code());
        return $response;
    }

    $response_code = wp_remote_retrieve_response_code( $response );
    $response_body = wp_remote_retrieve_body( $response );
    $response_headers = wp_remote_retrieve_headers( $response );

    // Debug: Log detailed response information
    //error_log('Gemini API Response Code: ' . $response_code);
    //error_log('Gemini API Response Headers: ' . print_r($response_headers, true));
    //error_log('Gemini API Response Body Length: ' . strlen($response_body));
    //error_log('Gemini API Response Body: ' . $response_body);

    // Check if response body is empty
    if (empty($response_body)) {
        //error_log('Gemini API returned empty response body');
        return new WP_Error( 'gemini_empty_response', 'Empty response from Gemini API', array( 'status' => $response_code ) );
    }

    if ( $response_code !== 200 ) {
        $error_data = json_decode( $response_body, true );
        
        // Better error message extraction
        $error_message = 'Unknown error occurred while generating image with Gemini';
        
        if (isset($error_data['error']['message'])) {
            $error_message = $error_data['error']['message'];
        } elseif (isset($error_data['error'])) {
            $error_message = is_string($error_data['error']) ? $error_data['error'] : json_encode($error_data['error']);
        } elseif (isset($error_data['message'])) {
            $error_message = $error_data['message'];
        }
        
        //error_log('Gemini API Error: ' . $error_message);
        return new WP_Error( 'gemini_error', $error_message, array( 'status' => $response_code ) );
    }

    return $response;
}


/**
 * Generate image using xAI's Grok-2-Image
 * 
 * @param string $api_key xAI API key
 * @param string $prompt Image description
 * @param int $n Number of images to generate (1-10)
 * @param string $response_format Response format (url or b64_json)
 * @return mixed WP_Error or response array
 */
private function generate_xai_image($api_key, $prompt, $n = 1, $response_format = 'url') {
    // Validate number of images
    $n = max(1, min(10, intval($n)));
    
    // Debug logging to check the actual value being passed
    //error_log('xAI image generation - Number of images requested: ' . $n);
    
    // Validate response format
    if (!in_array($response_format, array('url', 'b64_json'))) {
        $response_format = 'url';
    }
    
    // Prepare the request
    $url = 'https://api.x.ai/v1/images/generations';
    
    $body = array(
        'model' => 'grok-2-image',
        'prompt' => $prompt,
        'n' => intval($n), // Ensure it's an integer
        'response_format' => $response_format
    );
    
    // Log the actual request body for debugging
    //error_log('xAI image generation request body: ' . json_encode($body));
    
    $args = array(
        'method' => 'POST',
        'timeout' => 60, // Longer timeout for image generation
        'redirection' => 5,
        'httpversion' => '1.0',
        'blocking' => true,
        'headers' => array(
            'Authorization' => 'Bearer ' . $api_key,
            'Content-Type' => 'application/json'
        ),
        'body' => json_encode($body),
        'cookies' => array()
    );
    
    // Make the request
    $response = wp_remote_post($url, $args);
    
    // Check for errors
    if (is_wp_error($response)) {
        return $response;
    }
    
    $response_code = wp_remote_retrieve_response_code($response);
    $response_body = wp_remote_retrieve_body($response);
    
    // Log the response for debugging
    //error_log('xAI image generation response code: ' . $response_code);
    //error_log('xAI image generation response: ' . $response_body);
    
    if ($response_code !== 200) {
        $error_message = $response_body;
        $error_data = json_decode($error_message, true);
        
        $error_text = isset($error_data['error']['message']) 
            ? $error_data['error']['message'] 
            : __('Unknown error occurred while generating image with xAI', 'mxchat-admin-chat');
            
        return new WP_Error('xai_error', $error_text, array('status' => $response_code));
    }
    
    return $response;
}


/**
 * Generate image using OpenAI's DALL-E 3
 * 
 * @param string $api_key OpenAI API key
 * @param string $prompt Image description
 * @param string $size Image size (1024x1024, 1792x1024, or 1024x1792)
 * @param string $quality Image quality (standard or hd)
 * @return mixed WP_Error or response array
 */
private function generate_dalle_image($api_key, $prompt, $size, $quality) {
    // Validate size
    $valid_sizes = array('1024x1024', '1792x1024', '1024x1792');
    if (!in_array($size, $valid_sizes)) {
        $size = '1024x1024'; // Default to square if invalid
    }
    
    // Validate quality
    $valid_qualities = array('standard', 'hd');
    if (!in_array($quality, $valid_qualities)) {
        $quality = 'standard'; // Default to standard if invalid
    }
    
    // Prepare the request
    $url = 'https://api.openai.com/v1/images/generations';
    
    $body = array(
        'model' => 'dall-e-3',
        'prompt' => $prompt,
        'n' => 1,
        'size' => $size,
        'quality' => $quality,
        'response_format' => 'url'
    );
    
    $args = array(
        'method' => 'POST',
        'timeout' => 45,
        'redirection' => 5,
        'httpversion' => '1.0',
        'blocking' => true,
        'headers' => array(
            'Authorization' => 'Bearer ' . $api_key,
            'Content-Type' => 'application/json'
        ),
        'body' => json_encode($body),
        'cookies' => array()
    );
    
    // Make the request
    $response = wp_remote_post($url, $args);
    
    // Check for errors
    if (is_wp_error($response)) {
        return $response;
    }
    
    $response_code = wp_remote_retrieve_response_code($response);
    
    if ($response_code !== 200) {
        $error_message = wp_remote_retrieve_body($response);
        $error_data = json_decode($error_message, true);
        
        $error_text = isset($error_data['error']['message']) 
            ? $error_data['error']['message'] 
            : __('Unknown error occurred while generating image', 'mxchat-admin-chat');
            
        return new WP_Error('openai_error', $error_text, array('status' => $response_code));
    }
    
    return $response;
}


/**
 * Generate image using OpenAI's GPT Image (gpt-image-1)
 * 
 * @param string      $api_key     OpenAI API key
 * @param string      $prompt      Image description
 * @param string      $size        Image size
 * @param string      $quality     Image quality
 * @param string      $format      Output format (png, jpeg, webp)
 * @param string      $background  Background type (opaque, transparent, auto)
 * @param int|null    $compression Compression level (0–100) for jpeg/webp
 * @return WP_Error|array
 */
private function generate_gpt_image(
    $api_key,
    $prompt,
    $size,
    $quality,
    $format = 'png',
    $background = 'opaque',
    $compression = null
) {
    // Validate inputs
    $valid_sizes       = array( '1024x1024', '1536x1024', '1024x1536', 'auto' );
    $valid_quals       = array( 'low', 'medium', 'high', 'auto' );
    $valid_formats     = array( 'png', 'jpeg', 'webp' );
    $valid_backgrounds = array( 'opaque', 'transparent', 'auto' );

    if ( ! in_array( $size, $valid_sizes, true ) ) {
        $size = '1024x1024';
    }
    if ( ! in_array( $quality, $valid_quals, true ) ) {
        $quality = 'medium';
    }
    if ( ! in_array( $format, $valid_formats, true ) ) {
        $format = 'png';
    }
    if ( ! in_array( $background, $valid_backgrounds, true ) ) {
        $background = 'opaque';
    }
    if ( $background === 'transparent' && $format === 'jpeg' ) {
        $format = 'png';
    }
    if ( $compression !== null ) {
        $compression = intval( $compression );
        if ( $compression < 0 || $compression > 100 ) {
            $compression = 75;
        }
    }

    // Build the request body
    $body = array(
        'model'   => 'gpt-image-1',
        'prompt'  => $prompt,
        'n'       => 1,
        'size'    => $size,
        'quality' => $quality,
    );

    // Background setting
    if ( $background !== 'opaque' ) {
        $body['background'] = $background;
    }

    // Output format & compression
    if ( $format !== 'png' ) {
        $body['output_format'] = $format;
        if ( in_array( $format, array( 'jpeg', 'webp' ), true ) && $compression !== null ) {
            $body['output_compression'] = $compression;
        }
    }

    $args = array(
        'method'  => 'POST',
        'timeout' => 60,
        'headers' => array(
            'Authorization' => 'Bearer ' . $api_key,
            'Content-Type'  => 'application/json',
        ),
        'body'    => wp_json_encode( $body ),
    );

    $url      = 'https://api.openai.com/v1/images/generations';
    $response = wp_remote_post( $url, $args );

    if ( is_wp_error( $response ) ) {
        return $response;
    }

    $code = wp_remote_retrieve_response_code( $response );
    if ( $code !== 200 ) {
        $data = json_decode( wp_remote_retrieve_body( $response ), true );
        $msg  = isset( $data['error']['message'] )
            ? $data['error']['message']
            : __( 'Unknown error occurred while generating image', 'mxchat-admin-chat' );
        return new WP_Error( 'openai_error', $msg, array( 'status' => $code ) );
    }

    return $response;
}



/**
 * Handle saving user settings
 * 
 * @since    1.0.0
 */
public function handle_save_settings() {
    // Check permissions
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'mxchat_admin_chat_nonce')) {
        wp_send_json_error(array(
            'message' => __('Security check failed', 'mxchat-admin-chat')
        ), 403);
        return;
    }
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error(array(
            'message' => __('Unauthorized access', 'mxchat-admin-chat')
        ), 403);
        return;
    }
    
    // Get common parameters
    $image_gen_model = isset($_POST['image_gen_model']) ? sanitize_text_field($_POST['image_gen_model']) : 'openai';
    $perplexity_model = isset($_POST['perplexity_model']) ? sanitize_text_field($_POST['perplexity_model']) : 'pplx-7b-online';
    $image_analysis_model = isset($_POST['image_analysis_model']) ? sanitize_text_field($_POST['image_analysis_model']) : 'grok-vision';

    // Get DALL-E 3 specific parameters
    $image_size = isset($_POST['image_size']) ? sanitize_text_field($_POST['image_size']) : '1024x1024';
    $image_quality = isset($_POST['image_quality']) ? sanitize_text_field($_POST['image_quality']) : 'standard';
    
    // Get xAI specific parameters
    $image_count = isset($_POST['image_count']) ? intval($_POST['image_count']) : 1;
    $image_response_format = isset($_POST['image_response_format']) ? sanitize_text_field($_POST['image_response_format']) : 'url';
    
    // Get GPT Image specific parameters
    $gpt_image_size = isset($_POST['gpt_image_size']) ? sanitize_text_field($_POST['gpt_image_size']) : '1024x1024';
    $gpt_image_quality = isset($_POST['gpt_image_quality']) ? sanitize_text_field($_POST['gpt_image_quality']) : 'medium';
    $gpt_image_format = isset($_POST['gpt_image_format']) ? sanitize_text_field($_POST['gpt_image_format']) : 'png';
    $gpt_image_background = isset($_POST['gpt_image_background']) ? sanitize_text_field($_POST['gpt_image_background']) : 'opaque';
    $gpt_image_compression = isset($_POST['gpt_image_compression']) ? intval($_POST['gpt_image_compression']) : 75;
    
    // Validate parameters
    $image_count = max(1, min(10, $image_count)); // 1-10 range for image count
    $gpt_image_compression = max(0, min(100, $gpt_image_compression)); // 0-100 range for compression
    
    // User meta settings (personal settings)
    $user_id = get_current_user_id();
    
    // Save common settings
    update_user_meta($user_id, 'mxchat_image_gen_model', $image_gen_model);
    update_user_meta($user_id, 'mxchat_perplexity_model', $perplexity_model);
    update_user_meta($user_id, 'mxchat_image_analysis_model', $image_analysis_model);

    // Save DALL-E specific settings
    update_user_meta($user_id, 'mxchat_image_size', $image_size);
    update_user_meta($user_id, 'mxchat_image_quality', $image_quality);
    
    // Save xAI specific settings
    update_user_meta($user_id, 'mxchat_image_count', $image_count);
    update_user_meta($user_id, 'mxchat_image_response_format', $image_response_format);
    
    // Save GPT Image specific settings
    update_user_meta($user_id, 'mxchat_gpt_image_size', $gpt_image_size);
    update_user_meta($user_id, 'mxchat_gpt_image_quality', $gpt_image_quality);
    update_user_meta($user_id, 'mxchat_gpt_image_format', $gpt_image_format);
    update_user_meta($user_id, 'mxchat_gpt_image_background', $gpt_image_background);
    update_user_meta($user_id, 'mxchat_gpt_image_compression', $gpt_image_compression);
    
    // Plugin options (site-wide settings)
    
    // Save common settings
    update_option('mxchat_image_gen_model', $image_gen_model);
    update_option('mxchat_perplexity_model', $perplexity_model);
    
    // Save DALL-E specific settings
    update_option('mxchat_image_size', $image_size);
    update_option('mxchat_image_quality', $image_quality);
    
    // Save xAI specific settings
    update_option('mxchat_image_count', $image_count);
    update_option('mxchat_image_response_format', $image_response_format);
    
    // Save GPT Image specific settings
    update_option('mxchat_gpt_image_size', $gpt_image_size);
    update_option('mxchat_gpt_image_quality', $gpt_image_quality);
    update_option('mxchat_gpt_image_format', $gpt_image_format);
    update_option('mxchat_gpt_image_background', $gpt_image_background);
    update_option('mxchat_gpt_image_compression', $gpt_image_compression);
    update_option('mxchat_image_analysis_model', $image_analysis_model);

    // Return successful response
    wp_send_json_success(array(
        'message' => __('Settings saved successfully', 'mxchat-admin-chat')
    ));
}

/**
 * Handle image proxy for downloads (CORS workaround)
 * 
 * @since    1.0.0
 */
public function handle_proxy_image() {
    // Check permissions
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'mxchat_admin_chat_nonce')) {
        wp_send_json_error(array(
            'message' => __('Security check failed', 'mxchat-admin-chat')
        ), 403);
        return;
    }
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error(array(
            'message' => __('Unauthorized access', 'mxchat-admin-chat')
        ), 403);
        return;
    }
    
    // Get parameters
    $image_url = isset($_POST['image_url']) ? esc_url_raw($_POST['image_url']) : '';
    $image_name = isset($_POST['image_name']) ? sanitize_file_name($_POST['image_name']) : 'image.jpg';
    
    if (empty($image_url)) {
        wp_send_json_error(array(
            'message' => __('Image URL is required', 'mxchat-admin-chat')
        ), 400);
        return;
    }
    
    // Download the image from the source
    $response = wp_remote_get($image_url, array(
        'timeout' => 30,
        'sslverify' => false,
    ));
    
    // Check for errors
    if (is_wp_error($response)) {
        wp_send_json_error(array(
            'message' => $response->get_error_message()
        ), 500);
        return;
    }
    
    // Get the response code
    $response_code = wp_remote_retrieve_response_code($response);
    
    if ($response_code !== 200) {
        wp_send_json_error(array(
            'message' => sprintf(__('Error downloading image: HTTP %s', 'mxchat-admin-chat'), $response_code)
        ), 500);
        return;
    }
    
    // Get the image data
    $image_data = wp_remote_retrieve_body($response);
    
    // Convert to base64
    $base64_data = base64_encode($image_data);
    
    // Return the base64 encoded image
    wp_send_json_success(array(
        'base64' => $base64_data,
        'name' => $image_name
    ));
}    






/**
 * Handle image analysis request with model selection
 * 
 * @since    1.0.0
 */
public function handle_analyze_image() {
    // Check permissions
    if (!$this->verify_nonce() || !$this->verify_capability()) {
        wp_send_json_error(array(
            'message' => __('Unauthorized access', 'mxchat-admin-chat')
        ), 403);
    }
    
    // Get parameters
    $message = isset($_POST['message']) ? $_POST['message'] : '';
    $image_data = isset($_POST['image_data']) ? $_POST['image_data'] : '';
    $context_messages_json = isset($_POST['context_messages']) ? $_POST['context_messages'] : '[]';
    $context_messages = json_decode(stripslashes($context_messages_json), true);
    
    if (!is_array($context_messages)) {
        $context_messages = array();
    }
    
    // Validate inputs
    if (empty($message) && empty($image_data)) {
        wp_send_json_error(array(
            'message' => __('Message or image is required', 'mxchat-admin-chat')
        ), 400);
    }
    
    if (empty($image_data)) {
        wp_send_json_error(array(
            'message' => __('No image provided for analysis', 'mxchat-admin-chat')
        ), 400);
    }
    
    // Get selected vision model
    $vision_model = get_option('mxchat_image_analysis_model', 'grok-vision');
    
    // Set execution time limit
    if (function_exists('set_time_limit')) {
        set_time_limit(300);
    }
    
    try {
        // Route to appropriate vision model
        switch ($vision_model) {
            case 'gpt-4-vision':
                $response = $this->analyze_image_with_openai($message, $image_data, $context_messages);
                break;
            case 'claude-vision':
                $response = $this->analyze_image_with_claude($message, $image_data, $context_messages);
                break;
            case 'gemini-vision':
                $response = $this->analyze_image_with_gemini($message, $image_data, $context_messages);
                break;
            case 'grok-vision':
            default:
                $response = $this->analyze_image_with_grok($message, $image_data, $context_messages);
                break;
        }
        
        if (!$response['success']) {
            wp_send_json_error(array(
                'message' => $response['message']
            ), 500);
        }
        
        wp_send_json_success(array(
            'response' => $response['response'],
            'model' => $response['model']
        ));
        
    } catch (Exception $e) {
        $error_message = $e->getMessage();
        
        // Check for specific error types
        if (strpos($error_message, 'timed out') !== false || 
            strpos($error_message, 'timeout') !== false ||
            strpos($error_message, 'cURL error 28') !== false) {
            
            wp_send_json_error(array(
                'message' => __('The AI is taking longer than usual to analyze the image. Please try again.', 'mxchat-admin-chat'),
                'error_type' => 'timeout'
            ), 500);
        } else {
            wp_send_json_error(array(
                'message' => $error_message
            ), 500);
        }
    }
}

/**
 * Analyze image using Grok Vision
 * 
 * @param string $message User message/prompt
 * @param string $image_data Base64 encoded image data
 * @param array $context_messages Previous conversation context
 * @return array Response array
 */
private function analyze_image_with_grok($message, $image_data, $context_messages = array()) {
    // Get xAI API key
    $main_plugin_options = get_option('mxchat_options', array());
    $xai_key = isset($main_plugin_options['xai_api_key']) 
        ? $main_plugin_options['xai_api_key'] 
        : '';
    
    if (empty($xai_key) && method_exists($this->chat, 'get_api_key')) {
        $xai_key = $this->chat->get_api_key('xai');
    }
    
    if (empty($xai_key)) {
        return array(
            'success' => false,
            'message' => __('xAI API key is not configured for image analysis', 'mxchat-admin-chat')
        );
    }
    
    // Your existing Grok vision code here...
    // (I'll include the full function, but it's the same as what you had before)
    
    // Prepare the messages array
    $messages = array();
    
    // Add context messages if any
    foreach ($context_messages as $context_msg) {
        $messages[] = array(
            'role' => $context_msg['role'],
            'content' => $context_msg['content']
        );
    }
    
    // Prepare the user message with image
    $user_content = array();
    
    // Add text if provided
    if (!empty($message)) {
        $user_content[] = array(
            'type' => 'text',
            'text' => $message
        );
    } else {
        // Default prompt if no text provided
        $user_content[] = array(
            'type' => 'text',
            'text' => 'Please analyze this image and describe what you see in detail.'
        );
    }
    
    // Add image
    $user_content[] = array(
        'type' => 'image_url',
        'image_url' => array(
            'url' => $image_data
        )
    );
    
    // Add the user message
    $messages[] = array(
        'role' => 'user',
        'content' => $user_content
    );
    
    // Prepare the API request
    $url = 'https://api.x.ai/v1/chat/completions';
    
    $body = array(
        'model' => 'grok-2-vision-1212',
        'messages' => $messages,
        'max_tokens' => 4000,
        'temperature' => 0.7
    );
    
    $args = array(
        'method' => 'POST',
        'timeout' => 60,
        'headers' => array(
            'Authorization' => 'Bearer ' . $xai_key,
            'Content-Type' => 'application/json'
        ),
        'body' => wp_json_encode($body)
    );
    
    // Make the request
    $response = wp_remote_post($url, $args);
    
    if (is_wp_error($response)) {
        return array(
            'success' => false,
            'message' => $response->get_error_message()
        );
    }
    
    $response_code = wp_remote_retrieve_response_code($response);
    $response_body = wp_remote_retrieve_body($response);
    
    if ($response_code !== 200) {
        $error_data = json_decode($response_body, true);
        $error_message = isset($error_data['error']['message']) 
            ? $error_data['error']['message'] 
            : 'Unknown error occurred during image analysis';
            
        return array(
            'success' => false,
            'message' => $error_message
        );
    }
    
    $data = json_decode($response_body, true);
    
    if (!isset($data['choices'][0]['message']['content'])) {
        return array(
            'success' => false,
            'message' => 'Invalid response from image analysis API'
        );
    }
    
    return array(
        'success' => true,
        'response' => $data['choices'][0]['message']['content'],
        'model' => array(
            'id' => 'grok-2-vision-1212',
            'name' => 'Grok 2 Vision'
        )
    );
}


/**
 * Analyze image using OpenAI GPT-4 Vision
 * 
 * @param string $message User message/prompt
 * @param string $image_data Base64 encoded image data
 * @param array $context_messages Previous conversation context
 * @return array Response array
 */
private function analyze_image_with_openai($message, $image_data, $context_messages = array()) {
    // Get OpenAI API key
    $main_plugin_options = get_option('mxchat_options', array());
    $api_key = isset($main_plugin_options['api_key']) 
        ? $main_plugin_options['api_key'] 
        : '';
    
    if (empty($api_key) && method_exists($this->chat, 'get_api_key')) {
        $api_key = $this->chat->get_api_key('openai');
    }
    
    if (empty($api_key)) {
        return array(
            'success' => false,
            'message' => __('OpenAI API key is not configured for image analysis', 'mxchat-admin-chat')
        );
    }
    
    // Prepare the messages array
    $messages = array();
    
    // Add context messages if any
    foreach ($context_messages as $context_msg) {
        $messages[] = array(
            'role' => $context_msg['role'],
            'content' => $context_msg['content']
        );
    }
    
    // Prepare the user message with image
    $user_content = array();
    
    // Add text if provided
    if (!empty($message)) {
        $user_content[] = array(
            'type' => 'text',
            'text' => $message
        );
    } else {
        // Default prompt if no text provided
        $user_content[] = array(
            'type' => 'text',
            'text' => 'Please analyze this image and describe what you see in detail.'
        );
    }
    
    // Add image
    $user_content[] = array(
        'type' => 'image_url',
        'image_url' => array(
            'url' => $image_data
        )
    );
    
    // Add the user message
    $messages[] = array(
        'role' => 'user',
        'content' => $user_content
    );
    
    // Prepare the API request
    $url = 'https://api.openai.com/v1/chat/completions';
    
    $body = array(
        'model' => 'gpt-4o',
        'messages' => $messages,
        'max_tokens' => 4000,
        'temperature' => 0.7
    );
    
    $args = array(
        'method' => 'POST',
        'timeout' => 60,
        'headers' => array(
            'Authorization' => 'Bearer ' . $api_key,
            'Content-Type' => 'application/json'
        ),
        'body' => wp_json_encode($body)
    );
    
    // Make the request
    $response = wp_remote_post($url, $args);
    
    if (is_wp_error($response)) {
        return array(
            'success' => false,
            'message' => $response->get_error_message()
        );
    }
    
    $response_code = wp_remote_retrieve_response_code($response);
    $response_body = wp_remote_retrieve_body($response);
    
    if ($response_code !== 200) {
        $error_data = json_decode($response_body, true);
        $error_message = isset($error_data['error']['message']) 
            ? $error_data['error']['message'] 
            : 'Unknown error occurred during image analysis';
            
        return array(
            'success' => false,
            'message' => $error_message
        );
    }
    
    $data = json_decode($response_body, true);
    
    if (!isset($data['choices'][0]['message']['content'])) {
        return array(
            'success' => false,
            'message' => 'Invalid response from image analysis API'
        );
    }
    
    return array(
        'success' => true,
        'response' => $data['choices'][0]['message']['content'],
        'model' => array(
            'id' => 'gpt-4o',
            'name' => 'GPT-4o Vision'
        )
    );
}

/**
 * Handle image editing request with efficient storage
 * 
 * @since    1.0.0
 */
public function handle_edit_image() {
    // Check permissions
    if (!$this->verify_nonce() || !$this->verify_capability()) {
        wp_send_json_error(array(
            'message' => __('Unauthorized access', 'mxchat-admin-chat')
        ), 403);
    }
    
    // Initialize image storage handler if not already done
    if (!isset($this->image_storage)) {
        require_once MXCHAT_ADMIN_CHAT_PLUGIN_PATH . 'includes/class-mxchat-image-storage-handler.php';
        $this->image_storage = new MxChat_Image_Storage_Handler();
    }
    
    // Get parameters
    $instructions = isset($_POST['instructions']) ? sanitize_textarea_field($_POST['instructions']) : '';
    $image_data = isset($_POST['image_data']) ? $_POST['image_data'] : '';
    $model = isset($_POST['model']) ? sanitize_text_field($_POST['model']) : 'gemini';
    
    // Validate inputs
    if (empty($instructions)) {
        wp_send_json_error(array(
            'message' => __('Edit instructions cannot be empty', 'mxchat-admin-chat')
        ), 400);
    }
    
    if (empty($image_data)) {
        wp_send_json_error(array(
            'message' => __('No image provided for editing', 'mxchat-admin-chat')
        ), 400);
    }
    
    // Only Gemini supports image editing currently
    if ($model !== 'gemini') {
        wp_send_json_error(array(
            'message' => __('Image editing is only supported with Gemini models', 'mxchat-admin-chat')
        ), 400);
    }
    
    // Get Gemini API key
    $main_plugin_options = get_option('mxchat_options', array());
    $gemini_key = isset($main_plugin_options['gemini_api_key']) 
        ? $main_plugin_options['gemini_api_key'] 
        : '';
    
    if (empty($gemini_key)) {
        wp_send_json_error(array(
            'message' => __('Gemini API key is not configured', 'mxchat-admin-chat')
        ), 400);
    }
    
    // INCREASE MEMORY LIMIT AND EXECUTION TIME
    ini_set('memory_limit', '1024M'); // Increase to 1GB
    if (function_exists('set_time_limit')) {
        set_time_limit(300);
    }
    
    try {
        $response = $this->edit_image_with_gemini($instructions, $image_data, $gemini_key);
        
        if (!$response['success']) {
            wp_send_json_error(array(
                'message' => $response['message']
            ), 500);
        }
        
        // IMPORTANT: Save edited image to file instead of sending base64
        if (isset($response['data']['image_data_uri'])) {
            $saved_image = $this->image_storage->save_image_from_base64(
                $response['data']['image_data_uri'], 
                'edited'
            );
            
            if (is_wp_error($saved_image)) {
                wp_send_json_error(array(
                    'message' => $saved_image->get_error_message()
                ), 500);
                return;
            }
            
            // Replace data URI with URL
            $response['data']['image_url'] = $saved_image['url'];
            unset($response['data']['image_data_uri']); // Remove the base64 data
            $response['data']['storage_type'] = 'file';
        }
        
        wp_send_json_success($response['data']);
        
    } catch (Exception $e) {
        $error_message = $e->getMessage();
        
        // Check for memory errors
        if (strpos($error_message, 'memory') !== false || 
            strpos($error_message, 'exhausted') !== false) {
            
            wp_send_json_error(array(
                'message' => __('Image is too large to process. Please try with a smaller image.', 'mxchat-admin-chat'),
                'error_type' => 'memory'
            ), 500);
        }
        elseif (strpos($error_message, 'timed out') !== false || 
            strpos($error_message, 'timeout') !== false ||
            strpos($error_message, 'cURL error 28') !== false) {
            
            wp_send_json_error(array(
                'message' => __('The AI is taking longer than usual to edit the image. Please try again.', 'mxchat-admin-chat'),
                'error_type' => 'timeout'
            ), 500);
        } else {
            wp_send_json_error(array(
                'message' => $error_message
            ), 500);
        }
    }
}


/**
 * Extract base64 data from data URI - MEMORY OPTIMIZED
 * 
 * @param string $data_uri Data URI string
 * @return string Base64 data
 */
private function extract_base64_from_data_uri($data_uri) {
    if (strpos($data_uri, 'data:') === 0) {
        $comma_pos = strpos($data_uri, ',');
        if ($comma_pos !== false) {
            return substr($data_uri, $comma_pos + 1);
        }
    }
    return $data_uri; // Already base64
}

/**
 * Edit image using Gemini API - MEMORY OPTIMIZED VERSION
 * 
 * @param string $instructions Edit instructions
 * @param string $image_data Base64 image data or data URI
 * @param string $api_key Gemini API key
 * @return array Response array
 */
private function edit_image_with_gemini($instructions, $image_data, $api_key) {
    // Extract base64 data from data URI if needed - avoid creating copies
    $base64_data = $this->extract_base64_from_data_uri($image_data);
    
    // Determine MIME type from data URI or default to jpeg
    $mime_type = 'image/jpeg';
    if (strpos($image_data, 'data:') === 0) {
        preg_match('/data:([^;]+)/', $image_data, $matches);
        if (isset($matches[1])) {
            $mime_type = $matches[1];
        }
    }
    
    // Check image size before processing to prevent memory issues
    $image_size_mb = (strlen($base64_data) * 3/4) / (1024 * 1024); // Approximate decoded size
    if ($image_size_mb > 20) { // 20MB limit
        return array(
            'success' => false,
            'message' => 'Image is too large for editing. Please use an image smaller than 20MB.'
        );
    }
    
    // Prepare the request body for image editing
    $body = array(
        'contents' => array(
            array(
                'parts' => array(
                    array('text' => $instructions),
                    array(
                        'inlineData' => array(
                            'mimeType' => $mime_type,
                            'data' => $base64_data
                        )
                    )
                )
            )
        ),
        'generationConfig' => array(
            'temperature' => 0.7,
            'maxOutputTokens' => 8192
        )
    );
    
    $url = 'https://generativelanguage.googleapis.com/v1beta/models/gemini-2.5-flash-image-preview:generateContent?key=' . $api_key;
    
    $args = array(
        'method'  => 'POST',
        'timeout' => 120,
        'headers' => array(
            'Content-Type' => 'application/json',
        ),
        'body'    => wp_json_encode($body),
    );
    
    // Reduced logging to prevent memory issues
    //error_log('Gemini Image Edit Request - Instructions: ' . substr($instructions, 0, 100) . '...');
    //error_log('Gemini Image Edit Request - Image size: ' . number_format($image_size_mb, 2) . 'MB');
    
    $response = wp_remote_post($url, $args);
    
    // Clear the body from memory as soon as possible
    unset($body);
    unset($base64_data);
    
    if (is_wp_error($response)) {
        //error_log('Gemini Image Edit WP Error: ' . $response->get_error_message());
        return array(
            'success' => false,
            'message' => $response->get_error_message()
        );
    }
    
    $response_code = wp_remote_retrieve_response_code($response);
    $response_body = wp_remote_retrieve_body($response);
    
    //error_log('Gemini Image Edit Response Code: ' . $response_code);
    
    if ($response_code !== 200) {
        //error_log('Gemini Image Edit Error Response: ' . substr($response_body, 0, 500) . '...');
        $error_data = json_decode($response_body, true);
        $error_message = isset($error_data['error']['message']) 
            ? $error_data['error']['message'] 
            : 'Unknown error occurred while editing image';
            
        return array(
            'success' => false,
            'message' => $error_message
        );
    }
    
    $data = json_decode($response_body, true);
    
    if (!isset($data['candidates'][0]['content']['parts'])) {
        return array(
            'success' => false,
            'message' => 'Invalid response from image editing API'
        );
    }
    
    // Extract image and text from response
    $image_data_result = '';
    $text_response = '';
    
    foreach ($data['candidates'][0]['content']['parts'] as $part) {
        if (isset($part['inlineData']) && isset($part['inlineData']['data'])) {
            $mime_type = isset($part['inlineData']['mimeType']) ? $part['inlineData']['mimeType'] : 'image/png';
            $image_data_result = 'data:' . $mime_type . ';base64,' . $part['inlineData']['data'];
        }
        if (isset($part['text'])) {
            $text_response = $part['text'];
        }
    }
    
    if (empty($image_data_result)) {
        return array(
            'success' => false,
            'message' => 'No edited image received from Gemini'
        );
    }
    
    return array(
        'success' => true,
        'data' => array(
            'image_data_uri' => $image_data_result,
            'text_response' => $text_response,
            'model' => 'gemini-2.5-flash-image-preview'
        )
    );
}

    
}