<?php
/**
 * Perplexity REST API Endpoint
 *
 * @package MxChat_Admin_Chat
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

/**
 * Class for handling Perplexity REST API endpoints
 */
class MxChat_Perplexity_Endpoint {

    /**
     * The namespace for the REST API
     *
     * @var string
     */
    private $namespace = 'mxchat/v1';

    /**
     * Reference to the Perplexity API handler
     *
     * @var MxChat_Admin_Chat_Perplexity_API
     */
    private $perplexity_api;

    /**
     * Initialize the class and set its properties.
     *
     * @param MxChat_Admin_Chat_Perplexity_API $perplexity_api The Perplexity API handler.
     */
    public function __construct($perplexity_api) {
        $this->perplexity_api = $perplexity_api;
        
        // Register REST API routes
        add_action('rest_api_init', array($this, 'register_routes'));
    }

    /**
     * Register the REST API routes
     */
    public function register_routes() {
        // Register the search endpoint
        register_rest_route(
            $this->namespace,
            '/perplexity/search',
            array(
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => array($this, 'handle_search_request'),
                'permission_callback' => array($this, 'check_admin_permission'),
                'args'                => array(
                    'query' => array(
                        'required'          => true,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'validate_callback' => function($param) {
                            return !empty($param) && is_string($param);
                        },
                    ),
                    'system_prompt' => array(
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_textarea_field',
                    ),
                    'model' => array(
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                    ),
                ),
            )
        );
        
        // Register the settings endpoint
        register_rest_route(
            $this->namespace,
            '/perplexity/settings',
            array(
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => array($this, 'get_settings'),
                'permission_callback' => array($this, 'check_admin_permission'),
            )
        );
    }

    /**
     * Check if the current user has admin permissions
     *
     * @return bool Whether the user has permission
     */
    public function check_admin_permission() {
        return current_user_can('manage_options');
    }

    /**
     * Handle search request
     *
     * @param WP_REST_Request $request The request object.
     * @return WP_REST_Response The response.
     */
    public function handle_search_request($request) {
        // Get parameters
        $query = $request->get_param('query');
        $system_prompt = $request->get_param('system_prompt');
        $model = $request->get_param('model');
        
        // Log the request (for debugging)
        do_action('mxchat_log', 'Perplexity search request: ' . $query);
        
        // Perform the search
        $result = $this->perplexity_api->perform_perplexity_search($query, $system_prompt);
        
        // Check for errors
        if (is_wp_error($result)) {
            return new WP_REST_Response(
                array(
                    'success' => false,
                    'message' => $result->get_error_message(),
                ),
                400
            );
        }
        
        // Process the response
        $response_data = $this->perplexity_api->process_perplexity_response($result, $query);
        
        // Add template HTML if available
        if (function_exists('mxchat_render_perplexity_response')) {
            $response_data['template_html'] = mxchat_render_perplexity_response($response_data, $query);
        }
        
        // Return the response
        return new WP_REST_Response(
            array(
                'success' => true,
                'data'    => $response_data,
            ),
            200
        );
    }

    /**
     * Get Perplexity settings
     *
     * @return WP_REST_Response The response.
     */
    public function get_settings() {
        // Get settings
        $settings = array(
            'api_key_configured' => !empty(get_option('mxchat_perplexity_api_key', '')),
            'response_mode'      => get_option('mxchat_perplexity_response_mode', 'conversational'),
            'model'              => get_option('mxchat_perplexity_model', 'sonar'),
            'available_models'   => array(
                'sonar'              => __('Sonar (Default)', 'mxchat-admin-chat'),
                'sonar-medium'       => __('Sonar Medium', 'mxchat-admin-chat'),
                'sonar-small'        => __('Sonar Small', 'mxchat-admin-chat'),
                'sonar-deep-research' => __('Sonar Deep Research', 'mxchat-admin-chat'),
                'mixtral-8x7b'       => __('Mixtral 8x7B', 'mxchat-admin-chat'),
                'codellama-70b'      => __('CodeLlama 70B', 'mxchat-admin-chat'),
                'llama-3-70b-instruct' => __('Llama 3 70B', 'mxchat-admin-chat'),
            ),
        );
        
        // Return the settings
        return new WP_REST_Response(
            array(
                'success' => true,
                'data'    => $settings,
            ),
            200
        );
    }
}
