<?php
/**
 * Template for rendering Perplexity responses in a conversational format
 *
 * @package MxChat_Admin_Chat
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Render a Perplexity response in conversational format
 *
 * @param array $response The Perplexity API response data
 * @param string $query The original search query
 * @return string HTML output
 */
function mxchat_render_perplexity_response($response, $query = '') {
    // Initialize output buffer
    ob_start();
    
    // Get response format
    $format = isset($response['format']) ? $response['format'] : 'conversational';
    
    // Get content
    $content = isset($response['text']) ? $response['text'] : '';
    $html = isset($response['html']) ? $response['html'] : '';
    
    // Get citations if available
    $citations = isset($response['citations']) ? $response['citations'] : [];
    
    // Begin template
    ?>
    <div class="mxchat-perplexity-response <?php echo esc_attr('mxchat-perplexity-' . $format); ?>">
        <?php if ($format === 'structured'): ?>
            <?php if (!empty($query)): ?>
                <div class="mxchat-perplexity-query">
                    <strong>Search:</strong> <?php echo esc_html($query); ?>
                </div>
            <?php endif; ?>
            
            <?php if (!empty($html)): ?>
                <?php echo $html; // HTML is already sanitized in the API class ?>
            <?php else: ?>
                <div class="mxchat-perplexity-content">
                    <?php echo wp_kses_post(nl2br($content)); ?>
                </div>
                
                <?php if (!empty($citations)): ?>
                    <div class="mxchat-perplexity-sources">
                        <h4><?php _e('Sources:', 'mxchat-admin-chat'); ?></h4>
                        <ul>
                            <?php foreach ($citations as $index => $citation): ?>
                                <li>
                                    <?php if (filter_var($citation, FILTER_VALIDATE_URL)): ?>
                                        <a href="<?php echo esc_url($citation); ?>" target="_blank" rel="noopener noreferrer">
                                            <?php echo esc_html($citation); ?>
                                        </a>
                                    <?php else: ?>
                                        <?php echo esc_html($citation); ?>
                                    <?php endif; ?>
                                </li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
            
        <?php else: /* Conversational format */ ?>
            <div class="mxchat-perplexity-content">
                <?php 
                // For conversational format, process some basic markdown
                $processed_content = $content;
                
                // Convert **bold** to <strong>
                $processed_content = preg_replace('/\*\*(.*?)\*\*/s', '<strong>$1</strong>', $processed_content);
                
                // Convert *italic* to <em>
                $processed_content = preg_replace('/\*(.*?)\*/s', '<em>$1</em>', $processed_content);
                
                // Convert URLs to links
                $processed_content = preg_replace('/(https?:\/\/[^\s<>"\']+)/i', '<a href="$1" target="_blank" rel="noopener noreferrer">$1</a>', $processed_content);
                
                // Output with line breaks
                echo wp_kses_post(nl2br($processed_content));
                ?>
            </div>
        <?php endif; ?>
        
        <div class="mxchat-perplexity-attribution">
            <span class="mxchat-perplexity-logo"></span>
            <?php _e('Powered by Perplexity', 'mxchat-admin-chat'); ?>
        </div>
    </div>
    <?php
    
    // Return the template output
    return ob_get_clean();
}

/**
 * Render a Perplexity error message
 *
 * @param string $error_message The error message
 * @return string HTML output
 */
function mxchat_render_perplexity_error($error_message) {
    ob_start();
    ?>
    <div class="mxchat-perplexity-response mxchat-perplexity-error">
        <div class="mxchat-perplexity-content">
            <div class="mxchat-error-icon">
                <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <circle cx="12" cy="12" r="10"></circle>
                    <line x1="12" y1="8" x2="12" y2="12"></line>
                    <line x1="12" y1="16" x2="12.01" y2="16"></line>
                </svg>
            </div>
            <div class="mxchat-error-message">
                <?php echo esc_html($error_message); ?>
            </div>
        </div>
    </div>
    <?php
    return ob_get_clean();
}

/**
 * Render a typing indicator for Perplexity search
 *
 * @return string HTML output
 */
function mxchat_render_perplexity_typing_indicator() {
    ob_start();
    ?>
    <div class="mxchat-perplexity-response mxchat-perplexity-typing">
        <div class="mxchat-typing-indicator">
            <div class="typing-dots">
                <span></span>
                <span></span>
                <span></span>
            </div>
            <div class="typing-text">
                <?php _e('Searching the web...', 'mxchat-admin-chat'); ?>
            </div>
        </div>
    </div>
    <?php
    return ob_get_clean();
}
?>
