<?php
/**
 * Plugin Name: MxChat Admin Assistant
 * Plugin URI: https://mxchat.ai/
 * Description: Admin-only AI chat interface for WordPress using your existing MxChat API keys
 * Version: 1.0.7
 * Author: MxChat
 * Author URI: https://mxchat.ai/
 * Text Domain: mxchat-admin-chat
 * Requires at least: 5.8
 * Requires PHP: 7.4
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

// Define plugin constants
define('MXCHAT_ADMIN_CHAT_VERSION', '1.0.7');
define('MXCHAT_ADMIN_CHAT_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('MXCHAT_ADMIN_CHAT_PLUGIN_URL', plugin_dir_url(__FILE__));
define('MXCHAT_ADMIN_CHAT_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * Manually load Perplexity API classes that don't follow the standard naming pattern
 */
function mxchat_admin_chat_load_perplexity_classes() {
    // Load Perplexity API class
    $perplexity_api_file = MXCHAT_ADMIN_CHAT_PLUGIN_DIR . 'includes/class-mxchat-perplexity-api.php';
    if (file_exists($perplexity_api_file)) {
        require_once $perplexity_api_file;
    }
    
    // Load Perplexity endpoint class
    $perplexity_endpoint_file = MXCHAT_ADMIN_CHAT_PLUGIN_DIR . 'includes/endpoints/class-mxchat-perplexity-endpoint.php';
    if (file_exists($perplexity_endpoint_file)) {
        require_once $perplexity_endpoint_file;
    }
    
    // Load Perplexity response template
    $perplexity_template_file = MXCHAT_ADMIN_CHAT_PLUGIN_DIR . 'includes/partials/perplexity-response-template.php';
    if (file_exists($perplexity_template_file)) {
        require_once $perplexity_template_file;
    }
}

// Load Perplexity classes
mxchat_admin_chat_load_perplexity_classes();

// Update checker for plugin updates
require_once dirname(__FILE__) . '/update-checker.php';
PluginUpdateCheckerAdminChat::init([
    'user_id' => 'wp_4_SGKQ2sGzkIGB',
    'plugin_slug' => 'mxchat-admin-assistant'
]);

// Autoloader for core classes
spl_autoload_register(function ($class) {
    $prefix = 'MxChat_Admin_Chat_';
    $base_dir = MXCHAT_ADMIN_CHAT_PLUGIN_DIR . 'includes/';

    $len = strlen($prefix);
    if (strncmp($prefix, $class, $len) !== 0) {
        return;
    }

    $relative_class = substr($class, $len);
    $file = $base_dir . 'class-mxchat-admin-chat-' . str_replace('_', '-', strtolower($relative_class)) . '.php';

    if (file_exists($file)) {
        require $file;
    }
});

/**
 * Check if MxChat core plugin is active
 * 
 * @return bool True if MxChat core is active
 */
function is_mxchat_admin_chat_requirements_met() {
    if (is_multisite()) {
        $active_sitewide_plugins = get_site_option('active_sitewide_plugins');
        $mxchat_active = isset($active_sitewide_plugins['mxchat-basic/mxchat-basic.php']);
    }
    
    // Check site-specific plugins if not found in network
    if (!isset($mxchat_active)) {
        $active_plugins = apply_filters('active_plugins', get_option('active_plugins'));
        $mxchat_active = in_array('mxchat-basic/mxchat-basic.php', $active_plugins);
    }
    
    return $mxchat_active;
}

/**
 * Main plugin class
 */
final class MxChat_Admin_Chat {
    /**
     * Single instance of the class
     * 
     * @var MxChat_Admin_Chat
     */
    private static $instance = null;

    /**
     * Loader to register hooks and filters
     *
     * @var MxChat_Admin_Chat_Loader
     */
    public $loader;

    /**
     * UI handler
     *
     * @var MxChat_Admin_Chat_UI
     */
    public $ui;

    /**
     * API handler
     *
     * @var MxChat_Admin_Chat_API
     */
    public $api;
    
    /**
     * Perplexity API handler
     *
     * @var MxChat_Admin_Chat_Perplexity_API
     */
    public $perplexity_api;
    
    /**
     * Perplexity endpoint
     *
     * @var MxChat_Perplexity_Endpoint
     */
    public $perplexity_endpoint;

    /**
     * Main plugin singleton instance
     * 
     * @return MxChat_Admin_Chat Main instance
     */
    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        $this->check_dependencies();
        $this->init_components();
        register_deactivation_hook(__FILE__, array($this, 'handle_deactivation'));
    }

    /**
     * Check if dependencies are available
     */
    private function check_dependencies() {
        if (!is_mxchat_admin_chat_requirements_met()) {
            add_action('admin_notices', function() {
                ?>
                <div class="notice notice-error is-dismissible">
                    <p><?php _e('<strong>MxChat Admin Chat</strong> requires the MxChat plugin to be installed and active.', 'mxchat-admin-chat'); ?></p>
                </div>
                <?php
            });
            return;
        }
    }

    /**
     * Initialize components
     */
    private function init_components() {
        require_once MXCHAT_ADMIN_CHAT_PLUGIN_DIR . 'includes/class-mxchat-admin-chat-loader.php';
        $this->loader = new MxChat_Admin_Chat_Loader();

        if ($this->is_license_active()) {
            // Load dependencies
            require_once MXCHAT_ADMIN_CHAT_PLUGIN_DIR . 'includes/class-mxchat-admin-chat-core.php';
            require_once MXCHAT_ADMIN_CHAT_PLUGIN_DIR . 'includes/class-mxchat-admin-chat-api.php';
            require_once MXCHAT_ADMIN_CHAT_PLUGIN_DIR . 'includes/class-mxchat-admin-chat-ui.php';
            
            // Initialize components
            $this->api = new MxChat_Admin_Chat_API($this->loader);
            $this->ui = new MxChat_Admin_Chat_UI($this->loader);
            
            // Initialize Perplexity API if class exists
            if (class_exists('MxChat_Admin_Chat_Perplexity_API')) {
                $this->perplexity_api = new MxChat_Admin_Chat_Perplexity_API($this->loader);
                
                // Initialize Perplexity endpoint if class exists
                if (class_exists('MxChat_Perplexity_Endpoint')) {
                    $this->perplexity_endpoint = new MxChat_Perplexity_Endpoint($this->perplexity_api);
                }
            }
            
            // Add admin menu
            add_action('admin_menu', array($this, 'add_admin_menu'), 100);
            
            // Register AJAX handlers - use the correct method names
            add_action('wp_ajax_mxchat_admin_chat_send_message', array($this->api, 'handle_send_message'));
            add_action('wp_ajax_mxchat_admin_chat_clear_conversation', array($this->api, 'handle_clear_conversation'));
            add_action('wp_ajax_mxchat_admin_chat_get_conversation', array($this->api, 'handle_get_conversation'));
            add_action('wp_ajax_mxchat_admin_chat_update_model', array($this->api, 'handle_update_model'));
            add_action('wp_ajax_mxchat_admin_chat_get_models', array($this->api, 'handle_get_models'));
            add_action('wp_ajax_mxchat_admin_chat_save_message', array($this->api, 'handle_save_message'));
            
            // Run the loader
            $this->loader->run();
        } else {
            add_action('admin_notices', array($this, 'show_pro_requirement_notice'));
        }
    }

    /**
     * Add admin menu item
     */
    public function add_admin_menu() {
        if (is_multisite() && is_network_admin()) {
            add_submenu_page(
                'settings.php',  // Network settings menu
                __('Admin AI Chat', 'mxchat-admin-chat'),
                __('Admin AI Chat', 'mxchat-admin-chat'),
                'manage_network_options',
                'mxchat-admin-chat',
                array($this->ui, 'display_chat_interface')
            );
        } else {
            add_submenu_page(
                'mxchat-max',  // Main MxChat menu slug
                __('Admin AI Chat', 'mxchat-admin-chat'),
                __('Admin AI Chat', 'mxchat-admin-chat'),
                'manage_options',
                'mxchat-admin-chat',
                array($this->ui, 'display_chat_interface')
            );
        }
    }

    /**
     * Check if license is active
     */
    private function is_license_active() {
        if (is_multisite()) {
            $network_license_status = get_site_option('mxchat_license_status');
            if ($network_license_status === 'active') {
                return true;
            }
        }
        $license_status = get_option('mxchat_license_status', 'inactive');
        return $license_status === 'active';
    }

    /**
     * Show notice that Pro is required
     */
    public function show_pro_requirement_notice() {
        $class = 'notice notice-warning';
        $message = sprintf(
            __('MxChat Admin Chat requires MxChat Pro to be activated. Please <a href="%s">activate your pro license</a> to use this addon.', 'mxchat-admin-chat'),
            admin_url('admin.php?page=mxchat-activation')
        );
        printf('<div class="%1$s"><p>%2$s</p></div>', esc_attr($class), wp_kses_post($message));
    }

    /**
     * Handle plugin deactivation
     */
    public function handle_deactivation() {
        // Clean up if needed
    }

    /**
     * Prevent cloning
     */
    public function __clone() {
        _doing_it_wrong(__FUNCTION__, __('Cheatin&#8217; huh?', 'mxchat-admin-chat'), '1.0.7');
    }

    /**
     * Prevent unserializing
     */
    public function __wakeup() {
        _doing_it_wrong(__FUNCTION__, __('Cheatin&#8217; huh?', 'mxchat-admin-chat'), '1.0.7');
    }
}

/**
 * Returns the main instance of MxChat_Admin_Chat
 * 
 * @return MxChat_Admin_Chat
 */
function MxChat_Admin_Chat() {
    return MxChat_Admin_Chat::instance();
}

// Initialize the plugin
add_action('plugins_loaded', 'MxChat_Admin_Chat', 20);

// Register activation hook
register_activation_hook(__FILE__, function() {
    // Activation tasks if needed
    // No need to include a separate file if it doesn't exist
});