/**
 * MxChat Image Analysis JavaScript
 *
 * Handles image upload, paste, and analysis functionality.
 *
 * @since      1.0.0
 * @package    MxChat_Admin_Chat
 */

(function($) {
    'use strict';

    // Image Analysis object
    const MxChatImageAnalysis = {
        isEnabled: false,
        currentImage: null,
        currentImageData: null,

        init: function() {
            this.bindEvents();
            this.setupPasteHandler();
        },

        bindEvents: function() {
            const self = this;

            // Toggle button click
            $('.mxchat-image-upload-toggle').on('click', function() {
                self.toggleImageAnalysis();
            });

            // File input change
            $('#mxchat-image-upload').on('change', function(e) {
                const file = e.target.files[0];
                if (file) {
                    self.handleImageFile(file);
                }
            });

            // Remove image button
            $('#mxchat-image-remove-btn').on('click', function() {
                self.removeImage();
            });

            // Keyboard shortcuts
            $(document).on('keydown', function(e) {
                // Alt+U for image upload
                if (e.altKey && e.keyCode === 85) {
                    e.preventDefault();
                    self.toggleImageAnalysis();
                }
            });

            // **UPDATED** - Intercept form submission with edit mode detection
            $(document).on('mxchat_admin_before_send', function(event, messageData, cancelCallback) {
                if (self.currentImageData) {
                    // **NEW** - Check if we're in edit mode and should use editing instead
                    if (window.MxChatImageEdit && window.MxChatImageEdit.isEditMode) {
                        //console.log('Redirecting to image editing instead of analysis');
                        // Let the edit handler take over
                        return;
                    }
                    
                    // Cancel normal processing for analysis
                    cancelCallback();
                    
                    // Handle as image analysis
                    self.sendImageAnalysis(messageData.message);
                    
                    // Prevent further event propagation
                    event.stopImmediatePropagation();
                }
            });
        },

        setupPasteHandler: function() {
            const self = this;

            // Handle paste events on the textarea
            $('#mxchat-message-input').on('paste', function(e) {
                const items = e.originalEvent.clipboardData.items;
                
                for (let i = 0; i < items.length; i++) {
                    const item = items[i];
                    
                    if (item.type.indexOf('image') !== -1) {
                        e.preventDefault();
                        const file = item.getAsFile();
                        self.handleImageFile(file);
                        break;
                    }
                }
            });
        },

        toggleImageAnalysis: function() {
            this.isEnabled = !this.isEnabled;
            const $toggle = $('.mxchat-image-upload-toggle');
            
            if (this.isEnabled) {
                $toggle.addClass('active');
                this.showImageUploadPrompt();
            } else {
                $toggle.removeClass('active');
                this.removeImage();
            }
            
            this.updatePlaceholder();
        },

        showImageUploadPrompt: function() {
            // Trigger file input click
            $('#mxchat-image-upload').click();
        },

        // Add this debugging to your handleImageFile function in mxchat-image-analysis.js

handleImageFile: function(file) {
    const self = this;
    //console.log('=== IMAGE ANALYSIS DEBUG ===');
    //console.log('File received:', file.name, file.type, file.size);

    // Validate file type
    if (!file.type.startsWith('image/')) {
        MxChatAdmin.showNotification('Please select a valid image file', 'error');
        return;
    }

    // Validate file size (max 10MB)
    if (file.size > 10 * 1024 * 1024) {
        MxChatAdmin.showNotification('Image file too large. Please select an image under 10MB', 'error');
        return;
    }

    const reader = new FileReader();
    
    reader.onload = function(e) {
        //console.log('FileReader loaded successfully');
        //console.log('Image data length:', e.target.result.length);
        
        self.currentImageData = e.target.result;
        self.displayImagePreview(e.target.result);
        self.isEnabled = true;
        $('.mxchat-image-upload-toggle').addClass('active');
        self.updatePlaceholder();
        
        //console.log('Triggering mxchat_image_uploaded event');
        //console.log('Edit module exists:', !!window.MxChatImageEdit);
        //console.log('Edit module initialized:', window.MxChatImageEdit ? window.MxChatImageEdit.isEditMode : 'N/A');
        
        // **NEW** - Trigger image uploaded event for other modules
        $(document).trigger('mxchat_image_uploaded', [e.target.result]);
        
        // Debug: Check if edit module received the event
        setTimeout(() => {
            if (window.MxChatImageEdit) {
                //console.log('Edit module state after upload:');
                //console.log('- Has image data:', !!window.MxChatImageEdit.currentImageData);
               //console.log('- Supports editing:', window.MxChatImageEdit.supportsImageEditing());
                //console.log('- Is edit mode:', window.MxChatImageEdit.isEditMode);
                //console.log('- Current model:', window.MxChatImageEdit.getCurrentImageModel());
                //console.log('- Has Gemini key:', window.MxChatImageEdit.hasGeminiApiKey());
            }
        }, 100);
    };
    
    reader.onerror = function() {
        console.error('FileReader error');
        MxChatAdmin.showNotification('Error reading image file', 'error');
    };
    
    reader.readAsDataURL(file);
},

        displayImagePreview: function(imageData) {
            $('#mxchat-image-preview').attr('src', imageData);
            $('#mxchat-image-preview-area').show();
        },

        removeImage: function() {
            this.currentImageData = null;
            this.currentImage = null;
            $('#mxchat-image-preview-area').hide();
            $('#mxchat-image-preview').attr('src', '');
            $('#mxchat-image-upload').val('');
            this.isEnabled = false;
            $('.mxchat-image-upload-toggle').removeClass('active');
            this.updatePlaceholder();
            
            // **NEW** - Trigger image removed event for other modules
            $(document).trigger('mxchat_image_removed');
        },

        updatePlaceholder: function() {
            const $input = $('#mxchat-message-input');
            
            // **NEW** - Check if we're in edit mode
            if (window.MxChatImageEdit && window.MxChatImageEdit.isEditMode && this.currentImageData) {
                $input.attr('placeholder', mxchatImageGen.i18n.image_edit_prompt || 'Describe the changes you want to make...');
            } else if (this.isEnabled && this.currentImageData) {
                $input.attr('placeholder', 'Describe what you want to know about this image...');
            } else if (this.isEnabled) {
                $input.attr('placeholder', 'Upload or paste an image to analyze...');
            } else {
                $input.attr('placeholder', mxChatAdminChat.i18n.how_can_help || 'How can MxChat help?');
            }
        },

        /**
         * **UPDATED** - Enhanced image analysis with edit mode detection
         */
        sendImageAnalysis: function(message) {
            const self = this;

            if (!this.currentImageData) {
                MxChatAdmin.showNotification('No image selected for analysis', 'error');
                return;
            }

            // **NEW** - Double-check if we should redirect to editing
            if (window.MxChatImageEdit && window.MxChatImageEdit.isEditMode) {
                //console.log('Redirecting analysis request to image editing');
                window.MxChatImageEdit.editImage(message);
                return;
            }

            //console.log('Processing image analysis request');

            // Store image data for the message
            const imageDataForMessage = this.currentImageData;

            // REMOVE IMAGE PREVIEW IMMEDIATELY after storing the data
            this.removeImage();

            // Update UI state
            MxChatAdmin.isWaitingForResponse = true;
            MxChatAdmin.messageInput.val('');
            MxChatAdmin.messageInput.css('height', 'auto');
            MxChatAdmin.messageInput.prop('disabled', true);
            MxChatAdmin.sendButton.prop('disabled', true);

            // Add user message to UI (with image preview) using stored data
            this.addImageAnalysisMessage('user', message, imageDataForMessage);
            MxChatAdmin.scrollToBottom();

            // Add thinking indicator
            MxChatAdmin.addThinkingIndicator();

            // Get context messages
            const contextMessages = MxChatAdmin.getContextMessages(10);

            // **NEW** - Trigger analysis request event for other modules to potentially intercept
            const analysisEvent = $.Event('mxchat_analyze_image_request');
            $(document).trigger(analysisEvent, [message, imageDataForMessage, contextMessages]);
            
            // If event was prevented (e.g., by edit mode), stop here
            if (analysisEvent.isDefaultPrevented()) {
                //console.log('Image analysis was intercepted by another handler');
                return;
            }

            // Send to backend using stored data
            $.ajax({
                url: mxChatAdminChat.ajaxurl,
                type: 'POST',
                data: {
                    action: 'mxchat_admin_analyze_image',
                    nonce: mxChatAdminChat.nonce,
                    message: message,
                    image_data: imageDataForMessage, // Use stored data
                    context_messages: JSON.stringify(contextMessages)
                },
                dataType: 'json',
                success: function(response) {
                    MxChatAdmin.removeThinkingIndicator();
                    
                    if (response.success) {
                        // Add assistant response
                        MxChatAdmin.addMessage('assistant', response.data.response, response.data.model);
                        
                        // Apply syntax highlighting
                        setTimeout(() => {
                            if (typeof Prism !== 'undefined') {
                                Prism.highlightAll();
                            }
                        }, 100);

                        // Save chat after successful analysis
                        MxChatAdmin.saveCurrentChatSilently();
                    } else {
                        MxChatAdmin.addErrorMessage(response.data.message);
                    }
                    
                    // UI cleanup (image already removed)
                    MxChatAdmin.scrollToBottom();
                    MxChatAdmin.isWaitingForResponse = false;
                    MxChatAdmin.messageInput.prop('disabled', false);
                    MxChatAdmin.sendButton.prop('disabled', false);
                    MxChatAdmin.messageInput.focus();
                },
                error: function(xhr, status, error) {
                    MxChatAdmin.removeThinkingIndicator();
                    MxChatAdmin.addErrorMessage('Error analyzing image: ' + error);
                    
                    // UI cleanup (image already removed)
                    MxChatAdmin.scrollToBottom();
                    MxChatAdmin.isWaitingForResponse = false;
                    MxChatAdmin.messageInput.prop('disabled', false);
                    MxChatAdmin.sendButton.prop('disabled', false);
                    MxChatAdmin.messageInput.focus();
                }
            });
        },

        addImageAnalysisMessage: function(role, content, imageData) {
            const $message = $('<div class="mxchat-message mxchat-' + role + '"></div>');
            const $contentContainer = $('<div class="mxchat-message-content"></div>');
            
            // Add sender info
            const senderLabel = role === 'user' ? mxChatAdminChat.i18n.you : mxChatAdminChat.i18n.assistant;
            const $header = $('<div class="mxchat-message-header"><span class="mxchat-message-sender">' + senderLabel + '</span></div>');
            $contentContainer.append($header);
            
            // Add image if this is a user message with image
            if (role === 'user' && imageData) {
                const $imageContainer = $('<div class="mxchat-message-image"></div>');
                const $image = $('<img src="' + imageData + '" alt="Analyzed image" class="mxchat-analyzed-image" />');
                $imageContainer.append($image);
                $contentContainer.append($imageContainer);
            }
            
            // Add text content
            const $textContainer = $('<div class="mxchat-message-text"></div>');
            if (content) {
                const processedContent = MxChatAdmin.processMessageContent(content, role);
                $textContainer.html(processedContent);
            } else {
                $textContainer.html('<em>Image for analysis</em>');
            }
            $contentContainer.append($textContainer);
            
            // Add actions for assistant messages
            if (role === 'assistant') {
                const $actions = $('<div class="mxchat-message-actions"></div>');
                const $copyBtn = $('<button class="mxchat-action-btn mxchat-copy-btn" title="' + mxChatAdminChat.i18n.copy + '">' +
                    '<i class="fa-regular fa-copy"></i></button>');
                $actions.append($copyBtn);
                $contentContainer.append($actions);
            }
            
            $message.append($contentContainer);
            MxChatAdmin.conversation.append($message);
        },

        /**
         * **NEW** - Method to check if editing is possible
         */
        canSwitchToEditMode: function() {
            return this.currentImageData && 
                   window.MxChatImageEdit && 
                   typeof window.MxChatImageEdit.supportsImageEditing === 'function' &&
                   window.MxChatImageEdit.supportsImageEditing();
        },

        /**
         * **NEW** - Method to update placeholders when edit mode changes
         */
        updatePlaceholderForEditMode: function(isEditMode) {
            if (isEditMode && this.currentImageData) {
                this.updatePlaceholder();
            }
        }
    };

    // Initialize when document is ready
    $(document).ready(function() {
        MxChatImageAnalysis.init();
    });

    // Make globally accessible
    window.MxChatImageAnalysis = MxChatImageAnalysis;

})(jQuery);