/**
 * MxChat Image Editing JavaScript
 *
 * Handles image editing functionality using Gemini API.
 *
 * @since      1.0.0
 * @package    MxChat_Admin_Chat
 */

(function($) {
    'use strict';

    // Image Editing Module
    const MxChatImageEdit = {
        // Properties
        isEditMode: false,
        currentImage: null,
        currentImageData: null,
        modeToggleBtn: null,
        previewTitle: null,
        previewInfo: null,
        
        /**
         * Initialize the image editing module
         */
        init: function() {
            //console.log('MxChat Image Editing initializing...');
            
            // DOM elements
            this.modeToggleBtn = $('#mxchat-image-mode-toggle');
            this.previewTitle = $('#mxchat-image-preview-title');
            this.previewInfo = $('#mxchat-image-preview-info');
            
            // Initialize event listeners
            this.initEventListeners();
            
            //console.log('MxChat Image Editing initialized');
        },
        
        /**
         * Initialize event listeners
         */
        initEventListeners: function() {
            const self = this;
            
            // Mode toggle button click
            this.modeToggleBtn.on('click', function() {
                self.toggleMode();
            });
            
            // Listen for image upload events
            $(document).on('mxchat_image_uploaded', function(e, imageData) {
                self.handleImageUpload(imageData);
            });
            
            // Listen for image removal events
            $(document).on('mxchat_image_removed', function() {
                self.resetMode();
            });
            
            // **UPDATED** - Higher priority event handler for edit mode
            $(document).on('mxchat_admin_before_send', function(e, messageData, cancelCallback) {
                if (self.isEditMode && self.currentImageData) {
                   //console.log('Image editing intercepting message:', messageData);
                    
                    // Cancel normal processing
                    cancelCallback();
                    
                    // Handle image editing
                    self.editImage(messageData.message);
                    
                    // Prevent further event propagation
                    e.stopImmediatePropagation();
                    return false;
                }
            });
            
            // **NEW** - Also override the image analysis trigger specifically
            $(document).on('mxchat_analyze_image_request', function(e, message, imageData, contextMessages) {
                if (self.isEditMode && self.currentImageData) {
                    //console.log('Redirecting image analysis to editing');
                    e.preventDefault();
                    e.stopImmediatePropagation();
                    self.editImage(message);
                    return false;
                }
            });
        },
        
        /**
         * Handle image upload for editing
         */
        handleImageUpload: function(imageData) {
            //console.log('Image edit module received image upload:', !!imageData);
            
            this.currentImageData = imageData;
            this.currentImage = $('#mxchat-image-preview')[0];
            
            // Check if current model supports editing
            if (this.supportsImageEditing()) {
                this.modeToggleBtn.show();
                
                // Auto-enable edit mode if Gemini is selected
                const currentModel = this.getCurrentImageModel();
                //console.log('Current image model:', currentModel);
                
                if (currentModel === 'gemini') {
                    //console.log('Auto-enabling edit mode for Gemini');
                    this.enableEditMode(false); // Don't show notification on auto-enable
                }
            } else {
                //console.log('Image editing not supported for current model');
                this.modeToggleBtn.hide();
                this.disableEditMode();
            }
        },
        
        /**
         * Toggle between analyze and edit modes
         */
        toggleMode: function() {
            if (this.isEditMode) {
                this.disableEditMode();
            } else {
                this.enableEditMode();
            }
        },
        
        /**
         * Enable edit mode
         */
        enableEditMode: function(showNotification = true) {
            if (!this.supportsImageEditing()) {
                if (showNotification) {
                    this.showNotification(mxchatImageGen.i18n.gemini_editing_only, 'warning');
                }
                return;
            }
            
            //console.log('Enabling edit mode');
            this.isEditMode = true;
            
            // Update UI elements
            this.modeToggleBtn.removeClass('analyze-mode').addClass('edit-mode');
            this.modeToggleBtn.html('<i class="fa-solid fa-edit"></i> ' + mxchatImageGen.i18n.image_edit_mode);
            
            this.previewTitle.text('Image for Editing');
            this.previewInfo.show();
            
            // Update input placeholder
            $('#mxchat-message-input').attr('placeholder', mxchatImageGen.i18n.image_edit_prompt);
            
            if (showNotification) {
                this.showNotification(mxchatImageGen.i18n.switch_to_edit);
            }
            
            //console.log('Edit mode enabled. isEditMode:', this.isEditMode);
        },
        
        /**
         * Disable edit mode
         */
        disableEditMode: function() {
            this.isEditMode = false;
            
            // Update UI elements
            this.modeToggleBtn.removeClass('edit-mode').addClass('analyze-mode');
            this.modeToggleBtn.html('<i class="fa-solid fa-eye"></i> ' + mxchatImageGen.i18n.image_analyze_mode);
            
            this.previewTitle.text('Image for Analysis');
            this.previewInfo.hide();
            
            // Restore input placeholder
            $('#mxchat-message-input').attr('placeholder', mxChatAdminChat.i18n.how_can_help);
            
            this.showNotification(mxchatImageGen.i18n.switch_to_analyze);
        },
        
        /**
         * Reset mode when image is removed
         */
        resetMode: function() {
            this.isEditMode = false;
            this.currentImage = null;
            this.currentImageData = null;
            this.modeToggleBtn.hide();
            this.previewInfo.hide();
        },
        
        /**
         * Check if current model supports image editing
         */
        supportsImageEditing: function() {
            const currentModel = this.getCurrentImageModel();
            const geminiApiKey = this.hasGeminiApiKey();
            
            //console.log('Checking edit support - Model:', currentModel, 'Gemini Key:', geminiApiKey);
            
            return currentModel === 'gemini' && geminiApiKey;
        },

        /**
         * Check if Gemini API key is available
         */
        hasGeminiApiKey: function() {
            return mxchatImageGen.editing_enabled || false;
        },
        
        /**
         * Get current image generation model
         */
        getCurrentImageModel: function() {
            return localStorage.getItem('mxchat_image_gen_model') || mxchatImageGen.default_model || 'openai';
        },
        
/**
 * Modified editImage function to handle both URLs and base64
 */
editImage: function(instructions) {
    if (!this.currentImageData) {
        this.showNotification('No image available for editing', 'error');
        return;
    }
    
    //console.log('Starting image edit with instructions:', instructions);
    //console.log('Current image data type:', this.currentImageData.substring(0, 30));
    
    // Function to proceed with editing after we have base64 data
    const proceedWithEdit = (base64Data) => {
        // Show user message in chat
        if (window.MxChatAdmin) {
            const userMessage = `Edit image: ${instructions}`;
            window.MxChatAdmin.addMessage('user', userMessage);
            window.MxChatAdmin.saveMessageToHistory('user', userMessage, 'image-editing');
            window.MxChatAdmin.scrollToBottom();
            window.MxChatAdmin.addThinkingIndicator();
        }
        
        // Clear input
        $('#mxchat-message-input').val('');
        $('#mxchat-message-input').css('height', 'auto');
        
        // Add loading overlay to the preview
        const $previewArea = $('#mxchat-image-preview-area');
        if (!$previewArea.find('.edit-loading-overlay').length) {
            $previewArea.append(`
                <div class="edit-loading-overlay" style="
                    position: absolute;
                    top: 0;
                    left: 0;
                    right: 0;
                    bottom: 0;
                    background: rgba(0,0,0,0.7);
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    color: white;
                    font-size: 14px;
                    z-index: 10;
                ">
                    <div>
                        <i class="fa fa-spinner fa-spin"></i>
                        <div style="margin-top: 8px;">Editing image...</div>
                    </div>
                </div>
            `);
        }
        
        // Make API request for image editing
        const requestData = {
            action: 'mxchat_admin_edit_image',
            nonce: mxChatAdminChat.nonce,
            instructions: instructions,
            image_data: base64Data,
            model: 'gemini'
        };
        
        $.ajax({
            url: mxChatAdminChat.ajaxurl,
            type: 'POST',
            data: requestData,
            dataType: 'json',
            timeout: 120000,
            success: (response) => {
               
               //console.log('Edit response received:', response);
                
                // Remove loading overlay
                $('.edit-loading-overlay').remove();
                
                if (window.MxChatAdmin) {
                    window.MxChatAdmin.removeThinkingIndicator();
                }
                
                if (response.success) {
                    this.displayEditedImage(response.data, instructions);
                } else {
                    this.displayErrorMessage(response.data.message || mxchatImageGen.i18n.image_edit_error);
                }
            },
            error: (xhr, status, error) => {
                console.error('Edit request failed:', xhr.responseText);
                
                // Remove loading overlay
                $('.edit-loading-overlay').remove();
                
                if (window.MxChatAdmin) {
                    window.MxChatAdmin.removeThinkingIndicator();
                }
                
                let errorMsg = mxchatImageGen.i18n.image_edit_error;
                
                if (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                    errorMsg = xhr.responseJSON.data.message;
                } else if (status === 'timeout') {
                    errorMsg = 'Image editing timed out. Please try again or use a smaller image.';
                }
                
                this.displayErrorMessage(errorMsg);
            }
        });
    };
    
    // Check if currentImageData is a URL or base64
    if (this.currentImageData.startsWith('http://') || this.currentImageData.startsWith('https://')) {
        // It's a URL, we need to fetch and convert it to base64
        //console.log('Converting URL to base64 for editing...');
        
        // Use fetch API to get the image
        fetch(this.currentImageData)
            .then(response => response.blob())
            .then(blob => {
                const reader = new FileReader();
                reader.onloadend = () => {
                    const base64Data = reader.result;
                    //console.log('Successfully converted URL to base64');
                    proceedWithEdit(base64Data);
                };
                reader.readAsDataURL(blob);
            })
            .catch(error => {
                console.error('Error converting image URL to base64:', error);
                
                // Try alternative method using canvas
                const img = new Image();
                img.crossOrigin = 'anonymous';
                
                img.onload = function() {
                    const canvas = document.createElement('canvas');
                    canvas.width = img.width;
                    canvas.height = img.height;
                    
                    const ctx = canvas.getContext('2d');
                    ctx.drawImage(img, 0, 0);
                    
                    const base64Data = canvas.toDataURL('image/png');
                    //console.log('Successfully converted URL to base64 using canvas');
                    proceedWithEdit(base64Data);
                };
                
                img.onerror = function() {
                    console.error('Failed to load image for conversion');
                    this.displayErrorMessage('Failed to load image for editing. Please try uploading the image again.');
                }.bind(this);
                
                img.src = this.currentImageData;
            });
    } else if (this.currentImageData.startsWith('data:')) {
        // It's already base64, proceed directly
        proceedWithEdit(this.currentImageData);
    } else {
        // Unknown format
        console.error('Unknown image data format');
        this.displayErrorMessage('Invalid image format. Please upload a new image.');
    }
},


/**
 * Modified displayEditedImage to store the URL for future edits
 */
displayEditedImage: function(responseData, instructions) {
    let messageContent = `<p>Edited image: ${instructions}</p>`;
    
    // Handle both URL and base64 responses
    const imageUrl = responseData.image_url || responseData.image_data_uri;
    
    if (imageUrl) {
        messageContent += `<img src="${imageUrl}" alt="Edited image" class="mxchat-message-image" />`;
        
        // Update current image data with the URL for future editing
        this.currentImageData = imageUrl;
        
        // Update the preview with the edited image
        $('#mxchat-image-preview').attr('src', imageUrl);
        
        // Keep preview visible
        $('#mxchat-image-preview-area').show();
        $('.mxchat-image-preview-controls').show();
        
        // Add success indicator
        const $previewArea = $('#mxchat-image-preview-area');
        $previewArea.append(`
            <div class="edit-success-indicator" style="
                position: absolute;
                top: 10px;
                right: 10px;
                background: #28a745;
                color: white;
                padding: 4px 8px;
                border-radius: 4px;
                font-size: 12px;
                z-index: 5;
                animation: fadeInOut 3s ease-in-out;
            ">
                ✓ Edited
            </div>
        `);
        
        setTimeout(() => {
            $('.edit-success-indicator').fadeOut(500, function() {
                $(this).remove();
            });
        }, 3000);
    }
    
    if (responseData.text_response) {
        messageContent += `<p class="mxchat-edit-description">${responseData.text_response}</p>`;
    }
    
    // Add to chat
    if (window.MxChatAdmin) {
        window.MxChatAdmin.addMessage('assistant', messageContent);
        window.MxChatAdmin.saveMessageToHistory('assistant', messageContent, 'image-editing');
        window.MxChatAdmin.scrollToBottom();
        window.MxChatAdmin.unsavedChanges = true;
        
        // Auto-save
        setTimeout(() => {
            window.MxChatAdmin.saveCurrentChatSilently();
        }, 500);
    }
},
/**
 * Add this utility function to handle image uploads
 * Converts uploaded images to URLs by saving server-side
 */
handleImageUploadOptimized: function(file, callback) {
    const reader = new FileReader();
    
    reader.onload = function(e) {
        const base64Data = e.target.result;
        
        // Send to server to save and get URL back
        $.ajax({
            url: mxChatAdminChat.ajaxurl,
            type: 'POST',
            data: {
                action: 'mxchat_admin_save_uploaded_image',
                nonce: mxChatAdminChat.nonce,
                image_data: base64Data,
                filename: file.name
            },
            success: function(response) {
                if (response.success && response.data.url) {
                    // Use the URL instead of base64
                    callback(response.data.url);
                } else {
                    // Fallback to base64 if save fails
                    callback(base64Data);
                }
            },
            error: function() {
                // Fallback to base64
                callback(base64Data);
            }
        });
    };
    
    reader.readAsDataURL(file);
},
        
        /**
         * Display error message
         */
        displayErrorMessage: function(errorMessage) {
            if (window.MxChatAdmin) {
                window.MxChatAdmin.addErrorMessage(errorMessage);
                window.MxChatAdmin.scrollToBottom();
            } else {
                alert(errorMessage);
            }
        },
        
        /**
         * Show notification
         */
        showNotification: function(message, type = 'info') {
            if (window.MxChatAdmin && typeof window.MxChatAdmin.showNotification === 'function') {
                window.MxChatAdmin.showNotification(message, type);
            } else if (window.mxchatShowNotification) {
                window.mxchatShowNotification(message, type);
            } else {
                alert(message);
            }
        }
    };
    
    // Initialize on document ready
    $(document).ready(function() {
        MxChatImageEdit.init();
    });
    
    // Expose to global scope
    window.MxChatImageEdit = MxChatImageEdit;

})(jQuery);