/**
 * MxChat Admin Chat - Perplexity Integration
 * 
 * Handles the Perplexity search toggle and API interactions in the admin panel.
 */
(function($) {
    'use strict';

    // Main Perplexity handler object
    const MxChatPerplexity = {
        // State variables
        isEnabled: false,
        isSearching: false,
        lastQuery: '',
        
        // DOM elements
        $toggleButton: null,
        $chatInput: null,
        $chatContainer: null,
        
/**
 * Initialize the Perplexity functionality
 */
init: function() {
    // Debug global objects
    //console.log('mxchatPerplexity object available:', typeof mxchatPerplexity !== 'undefined');
    //console.log('mxChatAdminChat object available:', typeof mxChatAdminChat !== 'undefined');
    
    if (typeof mxchatPerplexity !== 'undefined') {
        //console.log('mxchatPerplexity properties:', Object.keys(mxchatPerplexity));
        //console.log('mxchatPerplexity API key configured:', mxchatPerplexity.perplexity_enabled);
        
        if (!mxchatPerplexity.perplexity_enabled) {
            console.warn('Perplexity API key is not configured. Please add your API key in the MxChat Perplexity settings.');
        }
    }
    
    // Initialize DOM references with correct selectors
    this.$toggleButton = $('.mxchat-perplexity-toggle');
    this.$chatInput = $('#mxchat-message-input');
    this.$chatContainer = $('#mxchat-conversation');
    
    // Debug logs to verify elements found
    //console.log('Perplexity: Toggle button found:', this.$toggleButton.length > 0);
    //console.log('Perplexity: Chat input found:', this.$chatInput.length > 0);
    //console.log('Perplexity: Chat container found:', this.$chatContainer.length > 0);
    
    // Verify that toggle button exists
    if (!this.$toggleButton.length) {
        console.error('MxChat Perplexity: Toggle button not found');
        return;
    }
    
    if (!this.$chatContainer.length) {
        console.error('MxChat Perplexity: Chat container not found');
        return;
    }
    
    if (!this.$chatInput.length) {
        console.error('MxChat Perplexity: Chat input not found');
        return;
    }
    
    // Restore state from localStorage if available
    this.isEnabled = localStorage.getItem('mxchat_admin_perplexity_enabled') === 'true';
    this.updateToggleState();
    
    // Bind events
    this.bindEvents();
    
    //console.log('MxChat Perplexity initialized. Enabled:', this.isEnabled);
},
        
        /**
         * Bind event handlers
         */
        bindEvents: function() {
            // Toggle button click
            this.$toggleButton.on('click', this.handleToggleClick.bind(this));
            
            // Intercept message submission
            $(document).on('mxchat_admin_before_send', this.interceptMessage.bind(this));
            
            // Handle keyboard shortcuts (Alt+P to toggle)
            $(document).on('keydown', function(e) {
                if (e.altKey && e.key === 'p') {
                    e.preventDefault();
                    this.togglePerplexity();
                }
            }.bind(this));
        },
        
        /**
         * Handle toggle button click
         * @param {Event} e Click event
         */
        handleToggleClick: function(e) {
            e.preventDefault();
            this.togglePerplexity();
        },
        
        /**
         * Toggle Perplexity mode on/off
         */
        togglePerplexity: function() {
            this.isEnabled = !this.isEnabled;
            
            // Save state to localStorage
            localStorage.setItem('mxchat_admin_perplexity_enabled', this.isEnabled);
            
            // Update UI
            this.updateToggleState();
            
            // Show feedback
            const status = this.isEnabled ? 'enabled' : 'disabled';
            this.showFeedback(`Perplexity web search ${status}`);
        },
        
        /**
         * Update toggle button state
         */
        updateToggleState: function() {
            if (this.isEnabled) {
                this.$toggleButton.addClass('active');
                this.$toggleButton.attr('title', 'Perplexity web search enabled (Alt+P)');
            } else {
                this.$toggleButton.removeClass('active');
                this.$toggleButton.attr('title', 'Perplexity web search disabled (Alt+P)');
            }
        },
        
        /**
         * Show user feedback message
         * @param {string} message Feedback message
         */
        showFeedback: function(message) {
            // Create temporary feedback element
            const $feedback = $('<div class="mxchat-feedback"></div>')
                .text(message)
                .appendTo('body');
            
            // Animate and remove
            $feedback.fadeIn(200).delay(1500).fadeOut(200, function() {
                $(this).remove();
            });
        },
        
/**
 * Intercept message before sending to check for Perplexity mode
 * @param {Event} e Custom event
 * @param {Object} data Message data
 * @returns {boolean} False to cancel normal sending
 */
interceptMessage: function(e, data, cancelCallback) {
    // If Perplexity not enabled, continue normal flow
    if (!this.isEnabled) {
        return true;
    }
    
    // Don't intercept empty messages
    if (!data.message || data.message.trim() === '') {
        return true;
    }
    
    //console.log('Perplexity: Intercepting message for web search:', data.message);
    
    // Call the cancel callback to stop normal processing
    if (typeof cancelCallback === 'function') {
        cancelCallback();
    }
    
    // Store the query
    this.lastQuery = data.message;
    
    // Clear the input field immediately when intercepting
    this.$chatInput.val('');
    this.$chatInput.css('height', 'auto');
    
    // Show user message in chat
    this.addUserMessage(data.message);
    
    // Show typing indicator
    this.showTypingIndicator();
    
    // Perform Perplexity search
    this.performPerplexitySearch(data.message);
    
    // Return false to indicate we've taken over
    return false;
},
        
/**
 * Add user message to chat and save to conversation history
 * @param {string} message User message
 */
addUserMessage: function(message) {
    // Create message element with proper classes to match main chat
    const $message = $('<div class="mxchat-message mxchat-user"></div>');
    
    // Message content container
    const $contentContainer = $('<div class="mxchat-message-content"></div>');
    
    // Add sender info header to match main chat format
    const $header = $('<div class="mxchat-message-header"><span class="mxchat-message-sender">You</span></div>');
    $contentContainer.append($header);
    
    // Add message text
    const $textContainer = $('<div class="mxchat-message-text"></div>');
    $textContainer.text(message);
    $contentContainer.append($textContainer);
    
    // Add content to message
    $message.append($contentContainer);
    
    // Add to conversation
    this.$chatContainer.append($message);
    this.scrollToBottom();
    
    // Save to conversation history via AJAX if main chat has this capability
    if (typeof MxChatAdmin !== 'undefined' && MxChatAdmin.saveMessageToHistory) {
        MxChatAdmin.saveMessageToHistory('user', message);
    } else {
        // Fallback: Use AJAX to save message directly
        $.ajax({
            url: mxChatAdminChat.ajaxurl,
            type: 'POST',
            data: {
                action: 'mxchat_admin_chat_save_message',
                nonce: mxChatAdminChat.nonce,
                role: 'user',
                content: message,
                is_perplexity: true
            },
            dataType: 'json'
        });
    }
},
        
        /**
         * Show typing indicator
         */
        showTypingIndicator: function() {
            // Remove any existing indicator
            $('.mxchat-typing-indicator').remove();
            
            // Create and add new indicator
            const $indicator = $('<div class="mxchat-message mxchat-bot-message mxchat-typing-indicator"></div>');
            const $content = $('<div class="mxchat-message-content"><div class="typing-dots"><span></span><span></span><span></span></div></div>');
            
            $indicator.append($content);
            this.$chatContainer.append($indicator);
            this.scrollToBottom();
        },
        
        /**
         * Hide typing indicator
         */
        hideTypingIndicator: function() {
            $('.mxchat-typing-indicator').remove();
        },
        
        /**
         * Perform Perplexity search
         * @param {string} query Search query
         */
performPerplexitySearch: function(query) {
    // Set searching state
    this.isSearching = true;
    
    // Check if API key is configured
    if (typeof mxchatPerplexity !== 'undefined' && mxchatPerplexity.perplexity_enabled === false) {
        this.hideTypingIndicator();
        this.addErrorMessage('Error: Perplexity API key is not configured. Please add your API key in the MxChat Perplexity settings.');
        return;
    }
    
    // Use the correct AJAX URL
    const ajaxUrl = typeof mxchatPerplexity !== 'undefined' && mxchatPerplexity.ajaxurl 
        ? mxchatPerplexity.ajaxurl 
        : (typeof ajaxurl !== 'undefined' ? ajaxurl : '/wp-admin/admin-ajax.php');
    
    // Get the nonce from mxchatPerplexity
    const nonce = typeof mxchatPerplexity !== 'undefined' && mxchatPerplexity.nonce 
        ? mxchatPerplexity.nonce 
        : '';
    
    //console.log('Starting Perplexity search for:', query);
    //console.log('Using AJAX URL:', ajaxUrl);
    //console.log('Using nonce:', nonce);
    
    // Make AJAX request
    $.ajax({
        url: ajaxUrl,
        type: 'POST',
        data: {
            action: 'mxchat_admin_perplexity_search',
            nonce: nonce,
            query: query
        },
        success: function(response) {
            //console.log('Perplexity search response:', response);
            if (response.success) {
                this.handleSearchSuccess(response);
            } else {
                let errorMsg = response.data || 'Unknown error';
                if (response.data && response.data.includes('API key')) {
                    errorMsg = 'Perplexity API key is not configured or is invalid. Please check your settings.';
                }
                this.handleSearchError(null, 'Error', errorMsg);
            }
        }.bind(this),
        error: function(xhr, status, error) {
            console.error('Perplexity search error:', xhr);
            
            let errorMsg = error || 'Unknown error';
            
            // Try to get more detailed error message from response
            try {
                if (xhr.responseJSON && xhr.responseJSON.data) {
                    errorMsg = xhr.responseJSON.data;
                } else if (xhr.responseText) {
                    const jsonResp = JSON.parse(xhr.responseText);
                    if (jsonResp.data) errorMsg = jsonResp.data;
                }
                
                // Check if it's an API key issue
                if (errorMsg.toLowerCase().includes('api key') || 
                    errorMsg.toLowerCase().includes('authorization') ||
                    errorMsg.toLowerCase().includes('missing_api_key')) {
                    errorMsg = 'Perplexity API key is not configured or is invalid. Please check your settings.';
                }
            } catch (e) {
                console.error('Error parsing error response:', e);
            }
            
            this.handleSearchError(xhr, status, errorMsg);
        }.bind(this),
        complete: function() {
            this.isSearching = false;
        }.bind(this)
    });
},
        
        /**
         * Handle successful search
         * @param {Object} response AJAX response
         */
        handleSearchSuccess: function(response) {
            // Hide typing indicator
            this.hideTypingIndicator();
            
            if (!response.success) {
                this.handleSearchError(null, 'Error', response.data || 'Unknown error');
                return;
            }
            
            const data = response.data;
            
            // Add response to chat
            this.addBotMessage(data);
            
            // Clear input
            this.$chatInput.val('');
            
            // Trigger event for other components
            $(document).trigger('mxchat_perplexity_response', [data]);
        },
        
        /**
         * Handle search error
         * @param {Object} xhr XHR object
         * @param {string} status Error status
         * @param {string} error Error message
         */
handleSearchError: function(xhr, status, error) {
    // Hide typing indicator
    this.hideTypingIndicator();
    
    // Try to extract a more meaningful error message
    let errorMessage = 'Error searching the web: ' + (error || 'Unknown error');
    
    if (xhr && xhr.responseJSON && xhr.responseJSON.data) {
        errorMessage = 'Error searching the web: ' + xhr.responseJSON.data;
    }
    
    console.error('MxChat Perplexity error:', {
        status: status,
        error: error,
        xhr: xhr
    });
    
    // Add error message to chat
    this.addErrorMessage(errorMessage);
    
    // Enable the input field again
    $('#mxchat-message-input').prop('disabled', false);
    $('#mxchat-send-btn').prop('disabled', false);
},
        
/**
 * Add bot message to chat and save to conversation history
 * @param {Object} data Response data
 */
addBotMessage: function(data) {
    // Create message element with proper classes
    const $message = $('<div class="mxchat-message mxchat-assistant mxchat-perplexity-message"></div>');
    
    // Create content container
    const $contentContainer = $('<div class="mxchat-message-content"></div>');
    
    // Add sender info header
    const $header = $('<div class="mxchat-message-header"><span class="mxchat-message-sender">Assistant</span></div>');
    $contentContainer.append($header);
    
    // Create text container
    const $textContainer = $('<div class="mxchat-message-text"></div>');
    
    // Process content based on format
    let processedText = '';
    if (data.html && data.format === 'structured') {
        $textContainer.html(data.html);
        processedText = data.html;
    } else {
        // Simple markdown processing
        processedText = data.text;
        
        // Bold text
        let displayText = processedText.replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>');
        
        // Italic text
        displayText = displayText.replace(/\*(.*?)\*/g, '<em>$1</em>');
        
        // Line breaks
        displayText = displayText.replace(/\n/g, '<br>');
        
        $textContainer.html(displayText);
    }
    
    // Add text container to content
    $contentContainer.append($textContainer);
    
    // Add action buttons (like copy)
    const $actions = $('<div class="mxchat-message-actions"></div>');
    
    // Copy button
    const copyText = typeof mxChatAdminChat !== 'undefined' && mxChatAdminChat.i18n && mxChatAdminChat.i18n.copy 
        ? mxChatAdminChat.i18n.copy 
        : 'Copy';
    const $copyBtn = $('<button class="mxchat-action-btn mxchat-copy-btn" title="' + copyText + '">' +
        '<i class="fa-regular fa-copy"></i></button>');
    $actions.append($copyBtn);
    
    $contentContainer.append($actions);
    
    // Add content to message
    $message.append($contentContainer);
    
    // Add attribution
    const $attribution = $('<div class="mxchat-message-attribution"></div>')
        .html('<span class="mxchat-perplexity-logo"></span> Powered by Perplexity');
    
    $message.append($attribution);
    
    // Add to conversation
    this.$chatContainer.append($message);
    this.scrollToBottom();
    
    // Save to conversation history
    if (typeof MxChatAdmin !== 'undefined' && MxChatAdmin.saveMessageToHistory) {
        MxChatAdmin.saveMessageToHistory('assistant', processedText, 'perplexity');
    } else {
        // Fallback: Use AJAX to save message directly
        $.ajax({
            url: mxChatAdminChat.ajaxurl,
            type: 'POST',
            data: {
                action: 'mxchat_admin_chat_save_message',
                nonce: mxChatAdminChat.nonce,
                role: 'assistant',
                content: processedText,
                source: 'perplexity'
            },
            dataType: 'json'
        });
    }
},
        
/**
 * Add error message to chat
 * @param {string} message Error message
 */
addErrorMessage: function(message) {
    const $errorMsg = $('<div class="mxchat-message mxchat-assistant mxchat-error"></div>');
    
    // Add content
    const $content = $('<div class="mxchat-message-content"></div>');
    
    // Add header
    const $header = $('<div class="mxchat-message-header"><span class="mxchat-message-sender">Assistant</span></div>');
    $content.append($header);
    
    // Add error text
    const $errorContainer = $('<div class="mxchat-message-text"></div>');
    $errorContainer.html('<p><strong>Error:</strong> ' + message + '</p>');
    $content.append($errorContainer);
    
    $errorMsg.append($content);
    
    // Add to conversation
    this.$chatContainer.append($errorMsg);
    this.scrollToBottom();
    
    // Save to conversation history if available
    if (typeof MxChatAdmin !== 'undefined' && MxChatAdmin.saveMessageToHistory) {
        MxChatAdmin.saveMessageToHistory('assistant', 'Error: ' + message, 'perplexity_error');
    }
},
        
        /**
         * Scroll chat to bottom
         */
        scrollToBottom: function() {
            if (this.$chatContainer && this.$chatContainer.length) {
                const container = this.$chatContainer[0];
                if (container && typeof container.scrollHeight !== 'undefined') {
                    this.$chatContainer.scrollTop(container.scrollHeight);
                } else {
                    console.warn('Perplexity: Cannot scroll, container has no scrollHeight property');
                }
            } else {
                console.warn('Perplexity: Cannot scroll, chat container not found');
            }
        }
    };
    
    // Initialize on document ready
    $(document).ready(function() {
        MxChatPerplexity.init();
    });

})(jQuery);
