(function($) {
    'use strict';
    
    $(document).ready(function() {
        // Handle image generation model change
        $('#mxchat-image-gen-model').on('change', function() {
            const selectedModel = $(this).val();
            
            // Hide all model-specific settings first
            $('.openai-specific, .xai-specific, .gpt-image-specific, .gemini-specific').hide();
            
            // Show settings specific to the selected model
            if (selectedModel === 'openai') {
                $('.openai-specific').show();
            } else if (selectedModel === 'xai') {
                $('.xai-specific').show();
            } else if (selectedModel === 'gpt-image') {
                $('.gpt-image-specific').show();
                // Check if compression field should be visible
                toggleCompressionField();
            } else if (selectedModel === 'gemini') {
                $('.gemini-specific').show();
            }
        });
        
        // Handle showing/hiding compression settings for GPT Image
        $('#mxchat-gpt-image-format').on('change', function() {
            toggleCompressionField();
        });
        
        // Function to toggle compression field visibility based on format
        function toggleCompressionField() {
            const format = $('#mxchat-gpt-image-format').val();
            const isGptImage = $('#mxchat-image-gen-model').val() === 'gpt-image';
            
            if (isGptImage && (format === 'jpeg' || format === 'webp')) {
                $('.gpt-image-specific.compression-field').show();
            } else {
                $('.gpt-image-specific.compression-field').hide();
            }
        }
        
        // Handle background compatibility with format
        $('#mxchat-gpt-image-background, #mxchat-gpt-image-format').on('change', function() {
            const background = $('#mxchat-gpt-image-background').val();
            const format = $('#mxchat-gpt-image-format').val();
            
            // If transparent background is selected but format is not supported
            if (background === 'transparent' && format === 'jpeg') {
                // Show a warning or switch to a supported format
                alert('Transparent background is only supported with PNG and WebP formats. Switching to PNG format.');
                $('#mxchat-gpt-image-format').val('png');
                toggleCompressionField();
            }
        });
        
        // Initialize the visibility of model-specific settings on page load
        const initialModel = $('#mxchat-image-gen-model').val();
        if (initialModel === 'openai') {
            $('.openai-specific').show();
            $('.xai-specific, .gpt-image-specific, .gemini-specific').hide();
        } else if (initialModel === 'xai') {
            $('.xai-specific').show();
            $('.openai-specific, .gpt-image-specific, .gemini-specific').hide();
        } else if (initialModel === 'gpt-image') {
            $('.gpt-image-specific').show();
            $('.openai-specific, .xai-specific, .gemini-specific').hide();
            // Initialize compression field visibility
            toggleCompressionField();
        } else if (initialModel === 'gemini') {
            $('.gemini-specific').show();
            $('.openai-specific, .xai-specific, .gpt-image-specific').hide();
        }
        
        // **FIXED** - Complete form submission handling with all required fields
        $('#mxchat-settings-form').on('submit', function(e) {
            e.preventDefault();
            
            //console.log('Settings form submission started');
            
            // Get ALL the form values - **FIXED** - Added missing fields
            const image_gen_model = $('#mxchat-image-gen-model').val();
            const image_analysis_model = $('#mxchat-image-analysis-model').val() || 'grok-vision'; // Default fallback
            const perplexity_model = $('#mxchat-perplexity-model').val() || 'pplx-7b-online'; // Default fallback
            
            // Prepare AJAX data with ALL required fields
            const data = {
                action: 'mxchat_admin_save_settings',
                nonce: mxChatAdminChat.nonce,
                image_gen_model: image_gen_model,
                image_analysis_model: image_analysis_model, // **ADDED** - This was missing!
                perplexity_model: perplexity_model // **ADDED** - This was missing!
            };
            
            // Model-specific settings
            if (image_gen_model === 'openai') {
                data.image_size = $('#mxchat-image-size').val();
                data.image_quality = $('#mxchat-image-quality').val();
            } else if (image_gen_model === 'xai') {
                data.image_count = $('#mxchat-image-count').val();
                data.image_response_format = $('#mxchat-image-response-format').val();
            } else if (image_gen_model === 'gpt-image') {
                data.gpt_image_size = $('#mxchat-gpt-image-size').val();
                data.gpt_image_quality = $('#mxchat-gpt-image-quality').val();
                data.gpt_image_format = $('#mxchat-gpt-image-format').val();
                data.gpt_image_background = $('#mxchat-gpt-image-background').val();
                const fmt = $('#mxchat-gpt-image-format').val();
                if (fmt === 'jpeg' || fmt === 'webp') {
                    data.gpt_image_compression = $('#mxchat-gpt-image-compression').val();
                }
            } else if (image_gen_model === 'gemini') {
                // No additional parameters needed for Gemini
            }
            
            //console.log('Saving settings with data:', data);
            
            // Show "Saving" state
            $('#mxchat-save-settings-btn')
                .prop('disabled', true)
                .html('<i class="fa-solid fa-spinner fa-spin"></i> ' + mxChatAdminChat.i18n.saving);
            
            // **ENHANCED** - Better AJAX handling with debugging
            $.ajax({
                url: mxChatAdminChat.ajaxurl,
                type: 'POST',
                data: data,
                dataType: 'json', // **ADDED** - Specify expected response type
                timeout: 30000, // **ADDED** - 30 second timeout
                success: function(response) {
                    //console.log('Settings save response:', response);
                    
                    if (!response.success) {
                        console.error('Settings save failed:', response.data);
                        // Show error notification
                        if (window.mxchatShowNotification) {
                            window.mxchatShowNotification(response.data.message || mxChatAdminChat.i18n.error, 'error');
                        } else {
                            alert(response.data.message || mxChatAdminChat.i18n.error);
                        }
                        return;
                    }
        
                    // **ENHANCED** - Sync ALL settings to localStorage including missing ones
                    localStorage.setItem('mxchat_image_gen_model', image_gen_model);
                    localStorage.setItem('mxchat_image_analysis_model', image_analysis_model); // **ADDED**
                    localStorage.setItem('mxchat_perplexity_model', perplexity_model); // **ADDED**
                    
                    if (image_gen_model === 'openai') {
                        localStorage.setItem('mxchat_image_size', data.image_size);
                        localStorage.setItem('mxchat_image_quality', data.image_quality);
                    } else if (image_gen_model === 'xai') {
                        localStorage.setItem('mxchat_image_count', data.image_count);
                        localStorage.setItem('mxchat_image_response_format', data.image_response_format);
                    } else if (image_gen_model === 'gpt-image') {
                        localStorage.setItem('mxchat_gpt_image_size', data.gpt_image_size);
                        localStorage.setItem('mxchat_gpt_image_quality', data.gpt_image_quality);
                        localStorage.setItem('mxchat_gpt_image_format', data.gpt_image_format);
                        localStorage.setItem('mxchat_gpt_image_background', data.gpt_image_background);
                        if (data.gpt_image_compression !== undefined) {
                            localStorage.setItem('mxchat_gpt_image_compression', data.gpt_image_compression);
                        }
                    } else if (image_gen_model === 'gemini') {
                        localStorage.setItem('mxchat_gemini_model', 'gemini-2.5-flash-image-preview');
                    }
        
                    // **NEW** - Trigger event for other modules to update
                    $(document).trigger('mxchat_settings_saved', [data]);
                    
                    // **NEW** - If edit mode module exists and has image, update its capabilities
                    if (window.MxChatImageEdit && window.MxChatImageEdit.currentImageData) {
                        //console.log('Updating edit mode after settings save');
                        window.MxChatImageEdit.handleImageUpload(window.MxChatImageEdit.currentImageData);
                    }
                    
                    // Show success notification
                    if (window.mxchatShowNotification) {
                        window.mxchatShowNotification(mxChatAdminChat.i18n.settings_saved, 'success');
                    } else {
                        alert(mxChatAdminChat.i18n.settings_saved);
                    }
                    
                    // Hide modal after a short delay to allow user to see notification
                    setTimeout(function() {
                        $('#mxchat-settings-modal').hide();
                    }, 1000);
                },
                error: function(xhr, status, error) {
                    console.error('Settings save AJAX error:', {xhr, status, error});
                    console.error('Response text:', xhr.responseText);
                    
                    let errorMessage = mxChatAdminChat.i18n.error;
                    
                    // **ENHANCED** - Better error message extraction
                    if (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                        errorMessage = xhr.responseJSON.data.message;
                    } else if (xhr.responseText) {
                        try {
                            const response = JSON.parse(xhr.responseText);
                            if (response.data && response.data.message) {
                                errorMessage = response.data.message;
                            }
                        } catch (e) {
                            // If can't parse JSON, use status info
                            errorMessage = 'Network error: ' + status + ' (' + error + ')';
                        }
                    }
                    
                    // Show error notification
                    if (window.mxchatShowNotification) {
                        window.mxchatShowNotification(errorMessage, 'error');
                    } else {
                        alert(errorMessage);
                    }
                },
                complete: function() {
                    // Restore button state
                    $('#mxchat-save-settings-btn')
                        .prop('disabled', false)
                        .html('<i class="fa-solid fa-save"></i> ' + mxChatAdminChat.i18n.save);
                }
            });
        
            return false;
        });
    });
})(jQuery);

// Add this to mxchat-settings.js
(function($) {
    'use strict';
    
    // Create notification system
    function showNotification(message, type = 'success') {
        // Remove any existing notifications first
        $('.mxchat-notification').remove();
        
        // Create notification element
        const notification = $('<div class="mxchat-notification mxchat-notification-' + type + '"></div>');
        
        // Add icon based on notification type
        let icon = 'fa-check-circle';
        if (type === 'error') {
            icon = 'fa-exclamation-circle';
        } else if (type === 'warning') {
            icon = 'fa-exclamation-triangle';
        } else if (type === 'info') {
            icon = 'fa-info-circle';
        }
        
        // Add content to notification
        notification.html('<i class="fa-solid ' + icon + '"></i> ' + message);
        
        // Add close button
        const closeBtn = $('<button class="mxchat-notification-close"><i class="fa-solid fa-times"></i></button>');
        notification.append(closeBtn);
        
        // Add to body and animate in
        $('body').append(notification);
        setTimeout(() => {
            notification.addClass('show');
        }, 10);
        
        // Set auto-hide after 3 seconds
        const timer = setTimeout(() => {
            hideNotification(notification);
        }, 3000);
        
        // Handle close button click
        closeBtn.on('click', function() {
            clearTimeout(timer);
            hideNotification(notification);
        });
    }
    
    function hideNotification(notification) {
        notification.removeClass('show');
        setTimeout(() => {
            notification.remove();
        }, 300); // Match transition duration in CSS
    }
    
    // Make functions available globally
    window.mxchatShowNotification = showNotification;
    window.mxchatHideNotification = hideNotification;
    
})(jQuery);