<?php
/**
 * MxChat Admin Chat Core
 *
 * Core functionality for the admin chat interface.
 *
 * @since      1.0.0
 * @package    MxChat_Admin_Chat
 * @subpackage MxChat_Admin_Chat/includes
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

/**
 * The core plugin class.
 *
 * This is the main class that handles the core functionality of the admin chat.
 *
 * @since      1.0.0
 * @package    MxChat_Admin_Chat
 * @subpackage MxChat_Admin_Chat/includes
 */
class MxChat_Admin_Chat_Core {

    /**
     * Session helper for managing conversation history
     *
     * @since    1.0.0
     * @access   private
     * @var      MxChat_Admin_Chat_Session    $session    Manages conversation sessions
     */
    private $session;

    /**
     * Available AI models
     *
     * @since    1.0.0
     * @access   private
     * @var      array    $available_models    List of supported AI models
     */
    private $available_models;

    /**
     * Selected model for the current user
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $selected_model    Currently selected AI model
     */
    private $selected_model;

    /**
     * API keys from core plugin
     *
     * @since    1.0.0
     * @access   private
     * @var      array    $api_keys    API keys for different providers
     */
    private $api_keys;

    /**
     * Session key for storing conversation data
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $session_key    Session key based on user ID
     */
    private $session_key;

    /**
     * Constructor
     *
     * @since    1.0.0
     */
    public function __construct() {
        $this->init_api_keys();
        $this->init_models();
        $this->load_user_preferences();
        
        // Initialize session
        $user_id = get_current_user_id();
        $this->set_session_key($user_id);
    }

    /**
     * Initialize API keys from MxChat core plugin
     *
     * @since    1.0.0
     */
    private function init_api_keys() {
        // Get the main plugin options
        $main_plugin_options = get_option('mxchat_options', array());
        
        $this->api_keys = array(
            'openai' => isset($main_plugin_options['api_key']) ? $main_plugin_options['api_key'] : '',
            'claude' => isset($main_plugin_options['claude_api_key']) ? $main_plugin_options['claude_api_key'] : '',
            'gemini' => isset($main_plugin_options['gemini_api_key']) ? $main_plugin_options['gemini_api_key'] : '',
            'xai' => isset($main_plugin_options['xai_api_key']) ? $main_plugin_options['xai_api_key'] : '',
            'deepseek' => isset($main_plugin_options['deepseek_api_key']) ? $main_plugin_options['deepseek_api_key'] : '',
        );
    }


/**
 * Initialize available models
 *
 * @since    1.0.0
 */
private function init_models() {
    $this->available_models = array(
        // Google Gemini models
        'gemini-2.0-flash' => array(
            'provider' => 'gemini',
            'name' => 'Gemini 2.0 Flash',
            'description' => 'Next-Gen features, speed & multimodal generation',
            'max_tokens' => 32000,
            'available' => !empty($this->api_keys['gemini']),
        ),
        'gemini-2.0-flash-lite' => array(
            'provider' => 'gemini',
            'name' => 'Gemini 2.0 Flash-Lite',
            'description' => 'Cost-efficient with low latency',
            'max_tokens' => 32000,
            'available' => !empty($this->api_keys['gemini']),
        ),
        'gemini-1.5-pro' => array(
            'provider' => 'gemini',
            'name' => 'Gemini 1.5 Pro',
            'description' => 'Complex reasoning tasks requiring more intelligence',
            'max_tokens' => 32000,
            'available' => !empty($this->api_keys['gemini']),
        ),
        'gemini-1.5-flash' => array(
            'provider' => 'gemini',
            'name' => 'Gemini 1.5 Flash',
            'description' => 'Fast and versatile performance',
            'max_tokens' => 32000,
            'available' => !empty($this->api_keys['gemini']),
        ),
        
        // X.AI models - INCLUDING NEW GROK CODE FAST 1
        'grok-code-fast-1' => array(
            'provider' => 'xai',
            'name' => 'Grok Code Fast 1',
            'description' => 'Speedy and economical reasoning model that excels at agentic coding',
            'max_tokens' => 256000,
            'available' => !empty($this->api_keys['xai']),
        ),
        'grok-4-0709' => array(
            'provider' => 'xai',
            'name' => 'Grok 4',
            'description' => 'Latest flagship model - unparalleled performance in natural language, math and reasoning',
            'max_tokens' => 256000,
            'available' => !empty($this->api_keys['xai']),
        ),
        'grok-3-beta' => array(
            'provider' => 'xai',
            'name' => 'Grok-3',
            'description' => 'Powerful model with 131K context',
            'max_tokens' => 131000,
            'available' => !empty($this->api_keys['xai']),
        ),
        'grok-3-fast-beta' => array(
            'provider' => 'xai',
            'name' => 'Grok-3 Fast',
            'description' => 'High performance with faster responses',
            'max_tokens' => 131000,
            'available' => !empty($this->api_keys['xai']),
        ),
        'grok-3-mini-beta' => array(
            'provider' => 'xai',
            'name' => 'Grok-3 Mini',
            'description' => 'Affordable model with good performance',
            'max_tokens' => 131000,
            'available' => !empty($this->api_keys['xai']),
        ),
        'grok-3-mini-fast-beta' => array(
            'provider' => 'xai',
            'name' => 'Grok-3 Mini Fast',
            'description' => 'Quick and cost-effective',
            'max_tokens' => 131000,
            'available' => !empty($this->api_keys['xai']),
        ),
        'grok-2' => array(
            'provider' => 'xai',
            'name' => 'Grok 2',
            'description' => 'Latest X.AI model',
            'max_tokens' => 4096,
            'available' => !empty($this->api_keys['xai']),
        ),
        
        // DeepSeek models
        'deepseek-chat' => array(
            'provider' => 'deepseek',
            'name' => 'DeepSeek-V3',
            'description' => 'Advanced AI assistant',
            'max_tokens' => 4096,
            'available' => !empty($this->api_keys['deepseek']),
        ),
        
        // Claude models
        'claude-4-opus-20250514' => array(
            'provider' => 'claude',
            'name' => 'Claude 4 Opus',
            'description' => 'Most powerful and intelligent AI model',
            'max_tokens' => 300000,
            'available' => !empty($this->api_keys['claude']),
        ),
        'claude-4-sonnet-20250514' => array(
            'provider' => 'claude',
            'name' => 'Claude 4 Sonnet',
            'description' => 'Advanced intelligence with enhanced capabilities',
            'max_tokens' => 200000,
            'available' => !empty($this->api_keys['claude']),
        ),
        'claude-3-7-sonnet-20250219' => array(
            'provider' => 'claude',
            'name' => 'Claude 3.7 Sonnet',
            'description' => 'Most intelligent Claude 3 model',
            'max_tokens' => 64000,
            'available' => !empty($this->api_keys['claude']),
        ),
        'claude-3-5-sonnet-20241022' => array(
            'provider' => 'claude',
            'name' => 'Claude 3.5 Sonnet',
            'description' => 'Intelligent and balanced',
            'max_tokens' => 100000,
            'available' => !empty($this->api_keys['claude']),
        ),
        'claude-3-opus-20240229' => array(
            'provider' => 'claude',
            'name' => 'Claude 3 Opus',
            'description' => 'Highly complex tasks',
            'max_tokens' => 200000,
            'available' => !empty($this->api_keys['claude']),
        ),
        'claude-3-sonnet-20240229' => array(
            'provider' => 'claude',
            'name' => 'Claude 3 Sonnet',
            'description' => 'Balanced performance',
            'max_tokens' => 200000,
            'available' => !empty($this->api_keys['claude']),
        ),
        'claude-3-haiku-20240307' => array(
            'provider' => 'claude',
            'name' => 'Claude 3 Haiku',
            'description' => 'Fastest Claude model',
            'max_tokens' => 200000,
            'available' => !empty($this->api_keys['claude']),
        ),
        
        // OpenAI models - including GPT-5 series
        'gpt-5' => array(
            'provider' => 'openai',
            'name' => 'GPT-5',
            'description' => 'The best model for coding and agentic tasks across domains',
            'max_tokens' => 128000,
            'available' => !empty($this->api_keys['openai']),
        ),
        'gpt-5-mini' => array(
            'provider' => 'openai',
            'name' => 'GPT-5 Mini',
            'description' => 'A faster, cost-efficient version of GPT-5 for well-defined tasks',
            'max_tokens' => 128000,
            'available' => !empty($this->api_keys['openai']),
        ),
        'gpt-5-nano' => array(
            'provider' => 'openai',
            'name' => 'GPT-5 Nano',
            'description' => 'Fastest, most cost-efficient version of GPT-5',
            'max_tokens' => 128000,
            'available' => !empty($this->api_keys['openai']),
        ),
        'gpt-4.1-2025-04-14' => array(
            'provider' => 'openai',
            'name' => 'GPT-4.1',
            'description' => 'Flagship model for complex tasks',
            'max_tokens' => 128000,
            'available' => !empty($this->api_keys['openai']),
        ),
        'gpt-4o' => array(
            'provider' => 'openai',
            'name' => 'GPT-4o',
            'description' => 'Recommended for most use cases',
            'max_tokens' => 128000,
            'available' => !empty($this->api_keys['openai']),
        ),
        'gpt-4o-mini' => array(
            'provider' => 'openai',
            'name' => 'GPT-4o Mini',
            'description' => 'Fast and lightweight',
            'max_tokens' => 128000,
            'available' => !empty($this->api_keys['openai']),
        ),
        'gpt-4-turbo' => array(
            'provider' => 'openai',
            'name' => 'GPT-4 Turbo',
            'description' => 'High-performance model',
            'max_tokens' => 128000,
            'available' => !empty($this->api_keys['openai']),
        ),
        'gpt-4' => array(
            'provider' => 'openai',
            'name' => 'GPT-4',
            'description' => 'High intelligence model',
            'max_tokens' => 128000,
            'available' => !empty($this->api_keys['openai']),
        ),
        'gpt-3.5-turbo' => array(
            'provider' => 'openai',
            'name' => 'GPT-3.5 Turbo',
            'description' => 'Affordable and fast',
            'max_tokens' => 16385,
            'available' => !empty($this->api_keys['openai']),
        ),
    );
}

    /**
     * Get available models for the dropdown
     *
     * @since    1.0.0
     * @return   array    Available models with provider group
     */
    public function get_available_models() {
        $grouped_models = array();
        
        foreach ($this->available_models as $model_id => $model) {
            if ($model['available']) {
                $provider = ucfirst($model['provider']);
                if (!isset($grouped_models[$provider])) {
                    $grouped_models[$provider] = array();
                }
                
                $grouped_models[$provider][$model_id] = $model['name'];
            }
        }
        
        return $grouped_models;
    }

    /**
     * Load user preferences for model selection
     *
     * @since    1.0.0
     */
    private function load_user_preferences() {
        $user_id = get_current_user_id();
        $this->selected_model = get_user_meta($user_id, 'mxchat_admin_chat_model', true);
        
        // If no model selected or selected model is not available, pick first available model
        if (empty($this->selected_model) || !isset($this->available_models[$this->selected_model]) || !$this->available_models[$this->selected_model]['available']) {
            // Find first available model
            foreach ($this->available_models as $model_id => $model) {
                if ($model['available']) {
                    $this->selected_model = $model_id;
                    break;
                }
            }
            
            // Save preference
            update_user_meta($user_id, 'mxchat_admin_chat_model', $this->selected_model);
        }
    }

    /**
     * Update the selected model
     *
     * @since    1.0.0
     * @param    string    $model_id    The model ID to select
     * @return   boolean                Success or failure
     */
    public function update_selected_model($model_id) {
        if (!isset($this->available_models[$model_id]) || !$this->available_models[$model_id]['available']) {
            return false;
        }
        
        $user_id = get_current_user_id();
        $this->selected_model = $model_id;
        update_user_meta($user_id, 'mxchat_admin_chat_model', $model_id);
        
        return true;
    }

    /**
     * Get current model information
     *
     * @since    1.0.0
     * @return   array    Selected model information
     */
    public function get_current_model() {
        if (empty($this->selected_model) || !isset($this->available_models[$this->selected_model])) {
            return null;
        }
        
        return array(
            'id' => $this->selected_model,
            'name' => $this->available_models[$this->selected_model]['name'],
            'provider' => $this->available_models[$this->selected_model]['provider'],
            'description' => $this->available_models[$this->selected_model]['description'],
        );
    }

    /**
     * Set session key based on user ID
     *
     * @since    1.0.0
     * @param    int    $user_id    User ID
     */
    private function set_session_key($user_id) {
        $this->session_key = 'mxchat_admin_chat_' . $user_id;
    }

    /**
     * Get the conversation history
     *
     * @since    1.0.0
     * @return   array    Conversation history
     */
    public function get_conversation_history() {
        $history = get_transient($this->session_key);
        if (false === $history) {
            $history = array();
        }
        
        return $history;
    }

/**
 * Add a message to the conversation history - SECURE VERSION
 *
 * @since    1.0.0
 * @param    string    $role       The role (user/assistant)
 * @param    string    $content    The message content
 */
public function add_to_conversation($role, $content) {
    $history = $this->get_conversation_history();
    
    // Clean user input for security
    if ($role === 'user') {
        // For user messages, strip HTML and store as plain text
        $content = html_entity_decode(strip_tags($content), ENT_QUOTES | ENT_HTML5, 'UTF-8');
    }
    // Assistant messages are stored as-is since they come from trusted AI sources
    
    $history[] = array(
        'role' => $role,
        'content' => $content,
        'timestamp' => current_time('timestamp'),
    );
    
    // Limit history size (optional)
    if (count($history) > 100) {
        array_shift($history);
    }
    
    // Store for 24 hours
    set_transient($this->session_key, $history, 24 * HOUR_IN_SECONDS);
}

    /**
     * Clear the conversation history
     *
     * @since    1.0.0
     */
    public function clear_conversation() {
        delete_transient($this->session_key);
    }

    /**
     * Format conversation for API calls
     *
     * @since    1.0.0
     * @return   array    Formatted messages for API call
     */
    public function format_conversation_for_api() {
        $history = $this->get_conversation_history();
        $formatted = array();
        
        // Different formatting based on provider
        $provider = $this->available_models[$this->selected_model]['provider'];
        
        switch ($provider) {
            case 'claude':
                // Claude format
                foreach ($history as $message) {
                    $formatted[] = array(
                        'role' => $message['role'],
                        'content' => $message['content']
                    );
                }
                break;
                
            case 'gemini':
                // Gemini format: first add system message as user message
                $formatted[] = array(
                    'role' => 'user',
                    'parts' => array(
                        array('text' => "[System Instructions] You are a helpful AI assistant.")
                    )
                );
                
                // Add model acknowledgment
                $formatted[] = array(
                    'role' => 'model',
                    'parts' => array(
                        array('text' => "I understand and will follow these instructions.")
                    )
                );
                
                // Add conversation history
                foreach ($history as $message) {
                    $formatted[] = array(
                        'role' => ($message['role'] === 'assistant') ? 'model' : 'user',
                        'parts' => array(
                            array('text' => $message['content'])
                        )
                    );
                }
                break;
                
            case 'deepseek':
            case 'xai':
            case 'openai':
            default:
                // OpenAI-compatible format (also works for X.AI and DeepSeek)
                // Add system message
                $formatted[] = array(
                    'role' => 'system',
                    'content' => 'You are a helpful AI assistant.'
                );
                
                // Add conversation history
                foreach ($history as $message) {
                    $formatted[] = array(
                        'role' => $message['role'],
                        'content' => $message['content']
                    );
                }
                break;
        }
        
        return $formatted;
    }


/**
 * Send to OpenAI API
 *
 * @since    1.0.0
 * @param    array     $conversation   Formatted conversation
 * @param    string    $model_id       Model ID
 * @return   string|WP_Error           Response text or error
 */
private function send_to_openai($conversation, $model_id) {
    // Get the API key
    $api_key = $this->api_keys['openai'];
    
    // Validate API key
    if (empty($api_key)) {
        return new WP_Error('openai_api_key_missing', __('OpenAI API key is missing', 'mxchat-admin-chat'));
    }
    
    // Format conversation for OpenAI format
    $openai_messages = array();
    
    // First, add a system message if it exists
    $has_system = false;
    foreach ($conversation as $msg) {
        if ($msg['role'] === 'system') {
            $openai_messages[] = array(
                'role' => 'system',
                'content' => $msg['content']
            );
            $has_system = true;
            break;
        }
    }
    
    // If no system message, add a default one
    if (!$has_system) {
        $openai_messages[] = array(
            'role' => 'system',
            'content' => 'You are a helpful AI assistant.'
        );
    }
    
    // Add the rest of the messages
    foreach ($conversation as $msg) {
        if ($msg['role'] !== 'system') { // Skip system messages as we've already handled them
            $openai_messages[] = array(
                'role' => $msg['role'],
                'content' => $msg['content']
            );
        }
    }
    
    $url = 'https://api.openai.com/v1/chat/completions';
    
    $args = [
        'headers' => [
            'Authorization' => 'Bearer ' . $api_key,
            'Content-Type'  => 'application/json',
        ],
        'body'    => wp_json_encode([
            'model'       => $model_id,
            'messages'    => $openai_messages,
        ]),
        'timeout' => 60,
        'method'  => 'POST',
    ];
    
    // Add debug log
    //error_log('OpenAI Request - Model: ' . $model_id);
    //error_log('OpenAI Request - Messages Count: ' . count($openai_messages));

    $response = wp_remote_post($url, $args);

    if (is_wp_error($response)) {
        $this->log_api_error('openai', $response);
        return $response;
    }

    $status_code = wp_remote_retrieve_response_code($response);
    $body = wp_remote_retrieve_body($response);
    $decoded_body = json_decode($body, true);
    
    // Debug logging
    //error_log('OpenAI Response Code: ' . $status_code);
    if ($status_code != 200) {
        //error_log('OpenAI Error Response: ' . $body);
    }

    if ($status_code == 200 && isset($decoded_body['choices'][0]['message']['content'])) {
        return $decoded_body['choices'][0]['message']['content'];
    } else {
        $error_message = isset($decoded_body['error']['message']) 
            ? $decoded_body['error']['message'] 
            : __('Unexpected response from OpenAI API', 'mxchat-admin-chat');
            
        $error = new WP_Error('openai_error', $error_message);
        $this->log_api_error('openai', $error);
        return $error;
    }
}

/**
 * Send to Claude API with improved timeout handling
 *
 * @param array $conversation Formatted conversation messages
 * @param string $model_id The model ID to use
 * @return string|WP_Error Response text or error
 */
private function send_to_claude($conversation, $model_id) {
    $api_key = $this->api_keys['claude'];
    
    // Validate API key
    if (empty($api_key)) {
        return new WP_Error('claude_api_key_missing', __('Claude API key is missing', 'mxchat-admin-chat'));
    }
    
    // Set appropriate max_tokens based on model
    $max_tokens = 4096; // Default conservative value
    
    // Model-specific limits
    $model_limits = [
        'claude-3-7-sonnet-20250219' => 32000,
        'claude-3-5-sonnet-20241022' => 32000,
        'claude-3-opus-20240229' => 32000,
        'claude-3-sonnet-20240229' => 32000,
        'claude-3-haiku-20240307' => 32000,
    ];
    
    // If we have a specific limit for this model, use it
    if (isset($model_limits[$model_id])) {
        $max_tokens = $model_limits[$model_id];
    }
    
    // Create messages array and extract system message
    $messages = [];
    $system_message = '';
    
    foreach ($conversation as $message) {
        if ($message['role'] === 'system') {
            $system_message = $message['content'];
        } else {
            $messages[] = [
                'role' => $message['role'],
                'content' => $message['content']
            ];
        }
    }
    
    // If no system message found, use default
    if (empty($system_message)) {
        $system_message = 'You are a helpful assistant that provides clear and accurate information.';
    }
    
    $body = json_encode([
        'model' => $model_id,
        'max_tokens' => $max_tokens,
        'temperature' => 0.7,
        'messages' => $messages,
        'system' => $system_message
    ]);

    // Increased timeout from 120 to 180 seconds
    $args = [
        'body' => $body,
        'headers' => [
            'Content-Type' => 'application/json',
            'x-api-key' => $api_key,
            'anthropic-version' => '2023-06-01'
        ],
        'timeout' => 180, // Increased timeout for longer responses
        'redirection' => 5,
        'blocking' => true,
        'httpversion' => '1.0',
        'sslverify' => true,
    ];

    // Log that we're making a request
    //error_log('Claude API Request - Model: ' . $model_id . ' - Messages Count: ' . count($messages));
    
    $response = wp_remote_post('https://api.anthropic.com/v1/messages', $args);

    if (is_wp_error($response)) {
        $error_msg = $response->get_error_message();
        //error_log('Claude API Error (WP_Error): ' . $error_msg);
        
        // Special handling for timeout errors
        if (strpos($error_msg, 'timed out') !== false || strpos($error_msg, 'timeout') !== false) {
            return new WP_Error(
                'claude_timeout',
                __('The request to Claude API timed out. Please try again with a shorter message or try later.', 'mxchat-admin-chat')
            );
        }
        
        return $response;
    }

    $response_code = wp_remote_retrieve_response_code($response);
    $response_body = json_decode(wp_remote_retrieve_body($response), true);
    
    // Log response code
    //error_log('Claude API Response Code: ' . $response_code);
    
    // Log error response for debugging
    if ($response_code !== 200) {
        $error_message = isset($response_body['error']['message']) 
            ? $response_body['error']['message'] 
            : sprintf(__('API returned status code %d', 'mxchat-admin-chat'), $response_code);
            
        // Check if this is a context length error
        if (strpos($error_message, 'maximum context length') !== false || 
            strpos($error_message, 'token limit') !== false ||
            strpos($error_message, 'maximum tokens') !== false) {
            
            //error_log('Claude API Context Length Error: ' . $error_message);
            
            return new WP_Error(
                'claude_context_length',
                $error_message,
                ['error_type' => 'context_length']
            );
        }
        
        //error_log('Claude API Error: ' . print_r($response_body, true));
        return new WP_Error(
            'claude_error_' . $response_code,
            $error_message
        );
    }
    
    if (isset($response_body['content']) && 
        is_array($response_body['content']) && 
        !empty($response_body['content']) && 
        isset($response_body['content'][0]['text'])) {
        
        return trim($response_body['content'][0]['text']);
    }
    
    //error_log('Claude API Unexpected Response: ' . print_r($response_body, true));
    return new WP_Error(
        'claude_unexpected_response',
        __('Unexpected response format from Claude API', 'mxchat-admin-chat')
    );
}


/**
 * Send to Gemini API
 *
 * @since    1.0.0
 * @param    array     $conversation   Formatted conversation
 * @param    string    $model_id       Model ID
 * @return   string|WP_Error           Response text or error
 */
private function send_to_gemini($conversation, $model_id) {
    // Get the API key
    $api_key = $this->api_keys['gemini'];
    
    // Validate API key
    if (empty($api_key)) {
        return new WP_Error('gemini_api_key_missing', __('Gemini API key is missing', 'mxchat-admin-chat'));
    }
    
    // Gemini uses a different format for messages
    $gemini_messages = [];
    
    // Add system message as a user message
    $system_content = "You are a helpful AI assistant.";
    foreach ($conversation as $msg) {
        if ($msg['role'] == 'system') {
            $system_content = $msg['content'];
            break;
        }
    }
    
    // First message is system instructions as user message
    $gemini_messages[] = [
        'role' => 'user',
        'parts' => [
            ['text' => "[System Instructions] " . $system_content]
        ]
    ];
    
    // Add model acknowledgment
    $gemini_messages[] = [
        'role' => 'model',
        'parts' => [
            ['text' => "I understand and will follow these instructions."]
        ]
    ];
    
    // Add the rest of the conversation
    foreach ($conversation as $msg) {
        if ($msg['role'] == 'system') {
            continue; // Skip system messages as we already handled them
        }
        
        $gemini_messages[] = [
            'role' => ($msg['role'] === 'assistant') ? 'model' : 'user',
            'parts' => [
                ['text' => $msg['content']]
            ]
        ];
    }
    
    $body = json_encode([
        'contents' => $gemini_messages,
        'generationConfig' => [
            'temperature' => 0.7,
            'topP' => 0.95,
            'topK' => 40,
            'maxOutputTokens' => 1000,
        ],
    ]);
    
    // Debug logging
    //error_log('Gemini Request - Model: ' . $model_id);
    //error_log('Gemini Request - Messages Count: ' . count($gemini_messages));
    
    // Prepare the API endpoint
    $api_endpoint = 'https://generativelanguage.googleapis.com/v1/models/' . $model_id . ':generateContent?key=' . $api_key;
    
    $args = [
        'body'        => $body,
        'headers'     => [
            'Content-Type' => 'application/json',
        ],
        'timeout'     => 60,
    ];
    
    $response = wp_remote_post($api_endpoint, $args);
    
    if (is_wp_error($response)) {
        $this->log_api_error('gemini', $response);
        return $response;
    }
    
    $status_code = wp_remote_retrieve_response_code($response);
    $response_body = wp_remote_retrieve_body($response);
    $decoded_body = json_decode($response_body, true);
    
    // Debug logging
    //error_log('Gemini Response Code: ' . $status_code);
    if ($status_code != 200) {
        //error_log('Gemini Error Response: ' . $response_body);
    }
    
    if ($status_code == 200 && isset($decoded_body['candidates'][0]['content']['parts'][0]['text'])) {
        return trim($decoded_body['candidates'][0]['content']['parts'][0]['text']);
    }
    
    $error_message = isset($decoded_body['error']['message']) 
        ? $decoded_body['error']['message'] 
        : __('Unexpected response from Gemini API', 'mxchat-admin-chat');
        
    $error = new WP_Error('gemini_error', $error_message);
    $this->log_api_error('gemini', $error);
    return $error;
}

/**
 * Send to X.AI API - COMPLETE FUNCTION with 600s (10 minutes) timeout
 */
private function send_to_xai($conversation, $model_id) {
    // FORCE 600 second (10 minutes) timeout
    add_filter('http_request_timeout', function($timeout) {
        return 600;
    }, 999);
    
    // Get the API key
    $api_key = $this->api_keys['xai'];
    
    // Validate API key
    if (empty($api_key)) {
        return new WP_Error('xai_api_key_missing', __('X.AI API key is missing', 'mxchat-admin-chat'));
    }
    
    // X.AI uses OpenAI-compatible format
    $xai_messages = array();
    
    // First, add a system message if it exists
    $has_system = false;
    foreach ($conversation as $msg) {
        if ($msg['role'] === 'system') {
            $xai_messages[] = array(
                'role' => 'system',
                'content' => $msg['content']
            );
            $has_system = true;
            break;
        }
    }
    
    // If no system message, add a default one
    if (!$has_system) {
        $xai_messages[] = array(
            'role' => 'system',
            'content' => 'You are a helpful AI assistant.'
        );
    }
    
    // Add the rest of the messages
    foreach ($conversation as $msg) {
        if ($msg['role'] !== 'system') { // Skip system messages as we've already handled them
            $xai_messages[] = array(
                'role' => $msg['role'],
                'content' => $msg['content']
            );
        }
    }
    
    $body = json_encode([
        'model' => $model_id,
        'messages' => $xai_messages,
        'temperature' => 0.7,
        'stream' => false
        // Removed max_tokens to allow longer responses
    ]);
    
    // Debug logging
    //error_log('X.AI Request - Model: ' . $model_id);
    //error_log('X.AI Request - Messages Count: ' . count($xai_messages));

    $args = [
        'body' => $body,
        'headers' => [
            'Content-Type' => 'application/json',
            'Authorization' => 'Bearer ' . $api_key,
        ],
        'timeout' => 600,  // 10 minutes
        'redirection' => 5,
        'httpversion' => '1.0',
        'blocking' => true,
        'sslverify' => true,
    ];

    $response = wp_remote_post('https://api.x.ai/v1/chat/completions', $args);

    if (is_wp_error($response)) {
        $error_message = $response->get_error_message();
        
        // Special handling for timeout errors
        if (strpos($error_message, 'timed out') !== false || 
            strpos($error_message, 'timeout') !== false ||
            strpos($error_message, 'cURL error 28') !== false) {
            
            return new WP_Error(
                'xai_timeout',
                __('The AI took longer than 10 minutes to respond. This usually means it\'s generating a very detailed response. Please try again or break your question into smaller parts.', 'mxchat-admin-chat')
            );
        }
        
        $this->log_api_error('xai', $response);
        return $response;
    }

    $status_code = wp_remote_retrieve_response_code($response);
    $response_body = wp_remote_retrieve_body($response);
    $decoded_body = json_decode($response_body, true);
    
    // Debug logging
    //error_log('X.AI Response Code: ' . $status_code);
    if ($status_code != 200) {
        //error_log('X.AI Error Response: ' . $response_body);
    }
    
    if ($status_code == 200 && isset($decoded_body['choices'][0]['message']['content'])) {
        return trim($decoded_body['choices'][0]['message']['content']);
    }
    
    $error_message = isset($decoded_body['error']['message']) 
        ? $decoded_body['error']['message'] 
        : __('Unexpected response from X.AI API', 'mxchat-admin-chat');
        
    $error = new WP_Error('xai_error', $error_message);
    $this->log_api_error('xai', $error);
    return $error;
}
/**
 * Send to DeepSeek API
 *
 * @since    1.0.0
 * @param    array     $conversation   Formatted conversation
 * @param    string    $model_id       Model ID
 * @return   string|WP_Error           Response text or error
 */
private function send_to_deepseek($conversation, $model_id) {
    // Get the API key
    $api_key = $this->api_keys['deepseek'];
    
    // Validate API key
    if (empty($api_key)) {
        return new WP_Error('deepseek_api_key_missing', __('DeepSeek API key is missing', 'mxchat-admin-chat'));
    }
    
    // DeepSeek uses OpenAI-compatible format
    $deepseek_messages = array();
    
    // First, add a system message if it exists
    $has_system = false;
    foreach ($conversation as $msg) {
        if ($msg['role'] === 'system') {
            $deepseek_messages[] = array(
                'role' => 'system',
                'content' => $msg['content']
            );
            $has_system = true;
            break;
        }
    }
    
    // If no system message, add a default one
    if (!$has_system) {
        $deepseek_messages[] = array(
            'role' => 'system',
            'content' => 'You are a helpful AI assistant.'
        );
    }
    
    // Add the rest of the messages
    foreach ($conversation as $msg) {
        if ($msg['role'] !== 'system') { // Skip system messages as we've already handled them
            $deepseek_messages[] = array(
                'role' => $msg['role'],
                'content' => $msg['content']
            );
        }
    }
    
    $body = json_encode([
        'model' => $model_id,
        'messages' => $deepseek_messages,
        'temperature' => 0.7,
        'stream' => false
    ]);
    
    // Debug logging
    //error_log('DeepSeek Request - Model: ' . $model_id);
    //error_log('DeepSeek Request - Messages Count: ' . count($deepseek_messages));

    $args = [
        'body' => $body,
        'headers' => [
            'Content-Type' => 'application/json',
            'Authorization' => 'Bearer ' . $api_key,
        ],
        'timeout' => 60,
    ];

    $response = wp_remote_post('https://api.deepseek.com/v1/chat/completions', $args);

    if (is_wp_error($response)) {
        $this->log_api_error('deepseek', $response);
        return $response;
    }

    $status_code = wp_remote_retrieve_response_code($response);
    $response_body = wp_remote_retrieve_body($response);
    $decoded_body = json_decode($response_body, true);
    
    // Debug logging
    //error_log('DeepSeek Response Code: ' . $status_code);
    if ($status_code != 200) {
        //error_log('DeepSeek Error Response: ' . $response_body);
    }
    
    if ($status_code == 200 && isset($decoded_body['choices'][0]['message']['content'])) {
        return trim($decoded_body['choices'][0]['message']['content']);
    }
    
    $error_message = isset($decoded_body['error']['message']) 
        ? $decoded_body['error']['message'] 
        : __('Unexpected response from DeepSeek API', 'mxchat-admin-chat');
        
    $error = new WP_Error('deepseek_error', $error_message);
    $this->log_api_error('deepseek', $error);
    return $error;
}
    
    /**
 * Log API errors to WordPress error log
 *
 * @since    1.0.0
 * @param    string    $provider    Provider name
 * @param    mixed     $error       Error data
 */
private function log_api_error($provider, $error) {
    if (is_wp_error($error)) {
        $error_message = $error->get_error_message();
    } elseif (is_array($error) || is_object($error)) {
        $error_message = print_r($error, true);
    } else {
        $error_message = (string) $error;
    }
    
    //error_log(sprintf('[MxChat Admin Chat] %s API Error: %s', ucfirst($provider), $error_message));
}


/**
 * Send message with limited context
 *
 * @since    1.0.0
 * @param    string    $message           The user message
 * @param    array     $context_messages  Previous context messages
 * @return   array                        Response array
 */
public function send_message_with_context($message, $context_messages) {
    // Get the current model
    $model = $this->get_current_model();
    
    if (!$model) {
        return array(
            'success' => false,
            'message' => __('No AI model selected', 'mxchat-admin-chat')
        );
    }
    
    // Create a new conversation with just the context messages
    $conversation = array();
    
    // Add system message if needed
    $system_message = $this->get_system_message();
    if (!empty($system_message)) {
        $conversation[] = array(
            'role' => 'system',
            'content' => $system_message
        );
    }
    
    // Add context messages to the conversation
    foreach ($context_messages as $msg) {
        $conversation[] = array(
            'role' => $msg['role'],
            'content' => $this->clean_html_content($msg['content'])
        );
    }
    
    // Add the current user message
    $conversation[] = array(
        'role' => 'user',
        'content' => $message
    );
    
    // Make the API request using the provider-specific method
    try {
        $provider = $model['provider'];
        $model_id = $model['id'];
        
        $method_name = "send_to_{$provider}";
        if (method_exists($this, $method_name)) {
            $response = $this->$method_name($conversation, $model_id);
            
            // Add the response to the full conversation history
            $this->add_to_conversation('assistant', $response);
            
            return array(
                'success' => true,
                'response' => $response,
                'model' => $model_id
            );
        } else {
            return array(
                'success' => false,
                'message' => sprintf(__('Provider %s is not supported', 'mxchat-admin-chat'), $provider)
            );
        }
    } catch (Exception $e) {
        // Check if this is a context length error
        if (strpos($e->getMessage(), 'This model\'s maximum context length') !== false || 
            strpos($e->getMessage(), 'token limit') !== false ||
            strpos($e->getMessage(), 'maximum tokens') !== false) {
            
            return array(
                'success' => false,
                'message' => __('The conversation has exceeded the maximum allowed length.', 'mxchat-admin-chat'),
                'error_type' => 'context_length'
            );
        }
        
        return array(
            'success' => false,
            'message' => $e->getMessage()
        );
    }
}

/**
 * Helper method to clean HTML content for API requests
 *
 * @since    1.0.0
 * @param    string    $content    HTML content to clean
 * @return   string                Cleaned content
 */
private function clean_html_content($content) {
    // Remove HTML tags but preserve line breaks and spacing
    $content = strip_tags($content, '<br><p>');
    
    // Replace <br> and <p> tags with newlines
    $content = str_replace(array('<br>', '<br/>', '<br />', '</p>'), "\n", $content);
    $content = str_replace('<p>', "\n\n", $content);
    
    // Decode HTML entities
    $content = html_entity_decode($content, ENT_QUOTES | ENT_HTML5, 'UTF-8');
    
    // Clean up extra whitespace
    $content = preg_replace('/\n\s+\n/', "\n\n", $content);
    $content = preg_replace('/\n{3,}/', "\n\n", $content);
    
    return trim($content);
}

/**
 * Get system message for the conversation
 *
 * @since    1.0.0
 * @return   string    System message
 */
private function get_system_message() {
    return apply_filters('mxchat_system_message', '');
}

}