<?php
/**
 * MxChat Image Storage Handler
 *
 * Handles efficient storage of generated/edited images
 * by saving them as files instead of base64 in database
 *
 * @since      1.0.0
 * @package    MxChat_Admin_Chat
 */

class MxChat_Image_Storage_Handler {
    
    /**
     * Upload directory info
     */
    private $upload_dir;
    private $upload_url;
    
    /**
     * Constructor
     */
    public function __construct() {
        $upload_info = wp_upload_dir();
        $this->upload_dir = $upload_info['basedir'] . '/mxchat-images';
        $this->upload_url = $upload_info['baseurl'] . '/mxchat-images';
        
        // Create directory if it doesn't exist
        if (!file_exists($this->upload_dir)) {
            wp_mkdir_p($this->upload_dir);
            
            // Add .htaccess for security
            $htaccess_content = "Options -Indexes\n";
            file_put_contents($this->upload_dir . '/.htaccess', $htaccess_content);
        }
    }
    
    /**
     * Save base64 image to file and return URL
     * 
     * @param string $base64_data Base64 encoded image or data URI
     * @param string $type Type of image (generated/edited/analyzed)
     * @return array|WP_Error Array with URL and file info or error
     */
    public function save_image_from_base64($base64_data, $type = 'generated') {
        // Extract base64 data from data URI if needed
        if (strpos($base64_data, 'data:') === 0) {
            $parts = explode(',', $base64_data);
            if (count($parts) !== 2) {
                return new WP_Error('invalid_data_uri', 'Invalid data URI format');
            }
            
            // Get MIME type
            preg_match('/data:([^;]+)/', $parts[0], $matches);
            $mime_type = isset($matches[1]) ? $matches[1] : 'image/jpeg';
            
            // Get extension from MIME type
            $extension = $this->get_extension_from_mime($mime_type);
            
            $base64_data = $parts[1];
        } else {
            $extension = 'jpg'; // Default extension
            $mime_type = 'image/jpeg';
        }
        
        // Decode base64
        $image_data = base64_decode($base64_data);
        if ($image_data === false) {
            return new WP_Error('decode_failed', 'Failed to decode base64 image');
        }
        
        // Generate unique filename
        $user_id = get_current_user_id();
        $timestamp = time();
        $random = wp_generate_password(8, false);
        $filename = sprintf('%s_%d_%d_%s.%s', $type, $user_id, $timestamp, $random, $extension);
        
        // Full path
        $file_path = $this->upload_dir . '/' . $filename;
        
        // Save file
        $result = file_put_contents($file_path, $image_data);
        if ($result === false) {
            return new WP_Error('save_failed', 'Failed to save image file');
        }
        
        // Get file URL
        $file_url = $this->upload_url . '/' . $filename;
        
        // Store metadata in database
        $image_id = $this->store_image_metadata($file_url, $file_path, $type, $mime_type);
        
        return array(
            'url' => $file_url,
            'path' => $file_path,
            'filename' => $filename,
            'size' => $result,
            'mime_type' => $mime_type,
            'image_id' => $image_id
        );
    }
    
    /**
     * Store image metadata in database
     */
    private function store_image_metadata($url, $path, $type, $mime_type) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'mxchat_images';
        
        // Create table if it doesn't exist
        $this->create_images_table();
        
        $wpdb->insert(
            $table_name,
            array(
                'user_id' => get_current_user_id(),
                'image_url' => $url,
                'image_path' => $path,
                'image_type' => $type,
                'mime_type' => $mime_type,
                'created_at' => current_time('mysql')
            ),
            array('%d', '%s', '%s', '%s', '%s', '%s')
        );
        
        return $wpdb->insert_id;
    }
    
    /**
     * Create images table if it doesn't exist
     */
    private function create_images_table() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'mxchat_images';
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            user_id bigint(20) NOT NULL,
            image_url text NOT NULL,
            image_path text NOT NULL,
            image_type varchar(50) DEFAULT 'generated',
            mime_type varchar(50) DEFAULT 'image/jpeg',
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY user_id (user_id),
            KEY created_at (created_at)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    /**
     * Get extension from MIME type
     */
    private function get_extension_from_mime($mime_type) {
        $mime_to_ext = array(
            'image/jpeg' => 'jpg',
            'image/jpg' => 'jpg',
            'image/png' => 'png',
            'image/gif' => 'gif',
            'image/webp' => 'webp',
            'image/bmp' => 'bmp'
        );
        
        return isset($mime_to_ext[$mime_type]) ? $mime_to_ext[$mime_type] : 'jpg';
    }
    
    /**
     * Clean up old images (run via cron)
     * 
     * @param int $days_old Delete images older than this many days
     */
    public function cleanup_old_images($days_old = 30) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'mxchat_images';
        $cutoff_date = date('Y-m-d H:i:s', strtotime("-{$days_old} days"));
        
        // Get old images
        $old_images = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM $table_name WHERE created_at < %s",
                $cutoff_date
            )
        );
        
        // Delete files and records
        foreach ($old_images as $image) {
            if (file_exists($image->image_path)) {
                unlink($image->image_path);
            }
            
            $wpdb->delete(
                $table_name,
                array('id' => $image->id),
                array('%d')
            );
        }
        
        return count($old_images);
    }
    
    /**
     * Process message content to replace base64 images with URLs
     * 
     * @param string $content Message content
     * @param string $type Type of image
     * @return string Processed content
     */
    public function process_message_content($content, $type = 'generated') {
        // Pattern to match base64 images in src attributes
        $pattern = '/src="data:image\/[^;]+;base64,([^"]+)"/i';
        
        // Process each match
        $content = preg_replace_callback($pattern, function($matches) use ($type) {
            $full_data_uri = 'data:image/jpeg;base64,' . $matches[1];
            
            // Save image and get URL
            $result = $this->save_image_from_base64($full_data_uri, $type);
            
            if (is_wp_error($result)) {
                // If save fails, keep original (but this should be rare)
                return $matches[0];
            }
            
            // Replace with actual URL
            return 'src="' . esc_url($result['url']) . '"';
        }, $content);
        
        return $content;
    }
    
    /**
     * Get user's images
     * 
     * @param int $user_id User ID (0 for current user)
     * @param int $limit Number of images to retrieve
     * @return array Array of image records
     */
    public function get_user_images($user_id = 0, $limit = 50) {
        global $wpdb;
        
        if ($user_id === 0) {
            $user_id = get_current_user_id();
        }
        
        $table_name = $wpdb->prefix . 'mxchat_images';
        
        return $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM $table_name 
                WHERE user_id = %d 
                ORDER BY created_at DESC 
                LIMIT %d",
                $user_id,
                $limit
            )
        );
    }
}

// Initialize and hook into save processes
add_action('init', function() {
    if (class_exists('MxChat_Image_Storage_Handler')) {
        $GLOBALS['mxchat_image_storage'] = new MxChat_Image_Storage_Handler();
    }
});

// Schedule cleanup cron
add_action('wp', function() {
    if (!wp_next_scheduled('mxchat_cleanup_old_images')) {
        wp_schedule_event(time(), 'daily', 'mxchat_cleanup_old_images');
    }
});

add_action('mxchat_cleanup_old_images', function() {
    if (isset($GLOBALS['mxchat_image_storage'])) {
        $GLOBALS['mxchat_image_storage']->cleanup_old_images(30);
    }
});