/**
 * MxChat Admin Chat JavaScript
 *
 * Handles all frontend interactions for the admin chat interface.
 *
 * @since      1.0.0
 * @package    MxChat_Admin_Chat
 */

(function($) {
    'use strict';

    // Main MxChat Admin Chat object
    const MxChatAdmin = {
init: function() {
    // Main DOM elements
    this.pageWrapper = $('.mxchat-page-wrapper');
    this.conversation = $('#mxchat-conversation');
    this.messageForm = $('#mxchat-message-form');
    this.messageInput = $('#mxchat-message-input');
    this.sendButton = $('#mxchat-send-btn');
    this.newChatButton = $('.mxchat-new-chat-btn');
    this.historyButton = $('.mxchat-history-btn');
    this.modelSelector = $('#mxchat-model-selector');
    this.toolButtons = $('.mxchat-tool-btn');
    
    // State variables
    this.isWaitingForResponse = false;
    this.currentChatId = null;
    this.unsavedChanges = false;
    this.isInitializing = true;
    this.isLoading = false;
    this.autoSaveTimer = null;
    
    // Initialize components
    this.initAutoResize();
    this.initEventListeners();
    this.initModelSelectorModal();
    this.initChatHistoryModal();
    this.scrollToBottom();
    
    // Initialize syntax highlighting
    this.initPrismHighlighting();
    
    // Set up session recovery
    this.initSessionRecovery();
    
    // Check if we should load a previous chat
    const shouldLoadPrevious = this.shouldLoadPreviousChat();
    
    if (shouldLoadPrevious) {
        this.setLoadingState(true);
        this.loadChatWithFallback();
    } else {
        this.showWelcomeMessage();
        this.isInitializing = false;
        // Focus input after initialization
        setTimeout(() => {
            this.messageInput.focus();
        }, 100);
    }
},

initSessionRecovery: function() {
    const self = this;
    
    // Save before page unload
    window.addEventListener('beforeunload', function(e) {
        // Cancel any pending auto-save
        self.cancelAutoSave();
        
        // Check if we have unsaved changes
        if (self.unsavedChanges && self.currentChatId) {
            // Try to do a quick synchronous save
            self.quickSaveBeforeUnload();
            
            // In some browsers, we might want to show a warning
            // Uncomment if you want browser warning:
            // e.preventDefault();
            // e.returnValue = 'You have unsaved changes. Are you sure you want to leave?';
        }
        
        // Save current scroll position
        if (self.conversation.length) {
            sessionStorage.setItem('mxchat_scroll_position', self.conversation.scrollTop());
        }
    });
    
    // Restore scroll position when returning
    window.addEventListener('load', function() {
        const savedScrollPosition = sessionStorage.getItem('mxchat_scroll_position');
        if (savedScrollPosition && self.conversation.length) {
            setTimeout(() => {
                self.conversation.scrollTop(parseInt(savedScrollPosition));
            }, 100);
        }
    });
    
    // Handle visibility change (tab switching)
    document.addEventListener('visibilitychange', function() {
        if (document.hidden) {
            // Page is hidden, save if needed
            if (self.unsavedChanges && self.currentChatId) {
                self.scheduleAutoSave();
            }
        } else {
            // Page is visible again, check for updates
            if (self.currentChatId && !self.isLoading) {
                // Verify current chat is still in sync
                self.checkForUpdates();
            }
        }
    });
},

quickSaveBeforeUnload: function() {
    // Collect current messages
    const messages = [];
    $('.mxchat-message').each(function() {
        const $msg = $(this);
        const role = $msg.hasClass('mxchat-user') ? 'user' : 'assistant';
        const content = $msg.find('.mxchat-message-text').html();
        
        messages.push({
            role: role,
            content: content
        });
    });
    
    // Get the first user message as title
    let title = '';
    const firstUserMessage = $('.mxchat-user .mxchat-message-text').first().text();
    if (firstUserMessage) {
        title = firstUserMessage.substring(0, 50) + (firstUserMessage.length > 50 ? '...' : '');
    } else {
        title = 'Chat ' + new Date().toLocaleDateString();
    }
    
    // Get current model
    const modelId = mxChatAdminChat.current_model ? mxChatAdminChat.current_model.id : '';
    const modelName = mxChatAdminChat.current_model ? mxChatAdminChat.current_model.name : '';
    
    // Use sendBeacon for reliable transmission even as page unloads
    const formData = new FormData();
    formData.append('action', 'mxchat_admin_chat_quick_save');
    formData.append('nonce', mxChatAdminChat.nonce);
    formData.append('chat_id', this.currentChatId);
    formData.append('title', title);
    formData.append('messages', JSON.stringify(messages));
    formData.append('model_id', modelId);
    formData.append('model_name', modelName);
    
    // sendBeacon returns true if queued successfully
    const queued = navigator.sendBeacon(mxChatAdminChat.ajaxurl, formData);
    
    if (!queued) {
        // Fallback to synchronous XHR if sendBeacon fails
        const xhr = new XMLHttpRequest();
        xhr.open('POST', mxChatAdminChat.ajaxurl, false); // false makes it synchronous
        xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
        
        const params = new URLSearchParams({
            action: 'mxchat_admin_chat_quick_save',
            nonce: mxChatAdminChat.nonce,
            chat_id: this.currentChatId,
            title: title,
            messages: JSON.stringify(messages),
            model_id: modelId,
            model_name: modelName
        });
        
        try {
            xhr.send(params);
        } catch (e) {
            console.error('Failed to save chat before unload:', e);
        }
    }
},

checkForUpdates: function() {
    const self = this;
    
    if (!this.currentChatId) {
        return;
    }
    
    // Don't check if we're already loading or have unsaved changes
    if (this.isLoading || this.unsavedChanges) {
        return;
    }
    
    $.ajax({
        url: mxChatAdminChat.ajaxurl,
        type: 'POST',
        data: {
            action: 'mxchat_admin_chat_check_updates',
            nonce: mxChatAdminChat.nonce,
            chat_id: this.currentChatId,
            last_update: this.lastUpdateTimestamp || 0
        },
        dataType: 'json',
        timeout: 5000,
        success: function(response) {
            if (response.success && response.data.has_updates) {
                // Show a subtle notification that updates are available
                self.showUpdateNotification();
            }
        },
        error: function() {
            // Silent fail - this is just a background check
        }
    });
},

showUpdateNotification: function() {
    if ($('.mxchat-update-notification').length) {
        return; // Already showing
    }
    
    const self = this;
    const $notification = $('<div class="mxchat-update-notification">' +
        '<i class="fa-solid fa-sync"></i> ' +
        '<span>This chat has been updated elsewhere.</span> ' +
        '<a href="#" class="mxchat-refresh-link">Refresh</a>' +
        '</div>');
    
    $notification.find('.mxchat-refresh-link').on('click', function(e) {
        e.preventDefault();
        $notification.remove();
        
        // Reload the current chat
        if (self.currentChatId) {
            self.loadChatWithTransition(self.currentChatId);
        }
    });
    
    $('.mxchat-header').append($notification);
    
    // Auto-hide after 10 seconds
    setTimeout(function() {
        $notification.fadeOut(300, function() {
            $(this).remove();
        });
    }, 10000);
},

shouldLoadPreviousChat: function() {
    // Check URL params for explicit "new chat" flag
    const urlParams = new URLSearchParams(window.location.search);
    if (urlParams.get('new_chat') === 'true') {
        // Clear the URL param after reading it
        window.history.replaceState({}, document.title, window.location.pathname);
        return false;
    }
    
    // Check if user explicitly started new chat in this session
    if (sessionStorage.getItem('mxchat_new_chat_session') === 'true') {
        return false;
    }
    
    // Check if there's actually a cached chat to load
    const cachedChatId = localStorage.getItem('mxchat_last_chat_id');
    if (!cachedChatId) {
        return false;
    }
    
    // Check if the cached chat was recently deleted
    const deletedChats = JSON.parse(localStorage.getItem('mxchat_deleted_chats') || '[]');
    if (deletedChats.includes(cachedChatId)) {
        // Clean up
        localStorage.removeItem('mxchat_last_chat_id');
        return false;
    }
    
    return true;
},

/**
 * Show welcome message when no chat is loaded
 */
showWelcomeMessage: function() {
    // Clear conversation and show welcome
    this.conversation.empty();
    this.addMessage('assistant', mxChatAdminChat.i18n.welcome_message || 'Welcome! How can I assist you today?');
    this.scrollToBottom();
    
    // Reset state
    this.currentChatId = null;
    this.unsavedChanges = false;
    this.isInitializing = false; 
},

cleanup: function() {
    if (this.codeHighlightObserver) {
        this.codeHighlightObserver.disconnect();
        this.codeHighlightObserver = null;
    }
    if (this.thinkingAnimationInterval) {
        clearInterval(this.thinkingAnimationInterval);
    }
},

/**
 * Save the current chat thread silently without showing notifications
 * 
 * @param {Function} callback Function to call after saving
 */
saveCurrentChatSilently: function(callback) {
    // Set a flag to suppress notifications
    this._suppressSaveNotification = true;
    
    // Call the regular save function
    this.saveCurrentChat(callback);
},

/**
 * Manage loading state across the interface
 * 
 * @param {boolean} isLoading Whether the interface should be in loading state
 */
setLoadingState: function(isLoading) {
    if (isLoading) {
        this.messageInput.prop('disabled', true);
        this.sendButton.prop('disabled', true);
        this.showLoadingIndicator();
    } else {
        this.messageInput.prop('disabled', false);
        this.sendButton.prop('disabled', false);
        this.hideLoadingIndicator();
        
        // Focus input after loading completes
        setTimeout(() => {
            this.messageInput.focus();
        }, 100);
    }
},

/**
 * Show a proper loading indicator
 */
showLoadingIndicator: function() {
    this.conversation.html(`
        <div class="mxchat-page-loading">
            <div class="mxchat-loading-content">
                <div class="mxchat-loading-spinner">
                    <div class="spinner"></div>
                </div>
                <h3>Loading your conversation...</h3>
                <p>Restoring your last chat session</p>
            </div>
        </div>
    `);
},

/**
 * Hide loading indicator with smooth transition
 */
hideLoadingIndicator: function() {
    $('.mxchat-page-loading').fadeOut(200, function() {
        $(this).remove();
    });
},

/**
 * Load chat with local cache fallback and enhanced UX
 */
loadChatWithFallback: function() {
    const self = this;
    
    // Check if we're already loading
    if (this.isLoading) {
        return;
    }
    
    this.isLoading = true;
    
    // Try to get cached chat ID first
    const cachedChatId = localStorage.getItem('mxchat_last_chat_id');
    
    // Try cached data first for instant UI
    const cachedMessages = localStorage.getItem('mxchat_cached_messages_' + cachedChatId);
    
    if (cachedMessages && cachedChatId) {
        // Show cached content immediately for better UX
        this.displayCachedContent(JSON.parse(cachedMessages), cachedChatId);
        
        // Show subtle indicator that we're verifying with server
        this.showMinimalLoadingIndicator();
        
        // Then verify with server in background
        this.verifyAndUpdateFromServer(cachedChatId);
    } else {
        // No cache available, show full loading indicator
        this.showLoadingIndicator();
        
        // Always fetch from server for accuracy
        $.ajax({
            url: mxChatAdminChat.ajaxurl,
            type: 'POST',
            data: {
                action: 'mxchat_admin_chat_get_history',
                nonce: mxChatAdminChat.nonce
            },
            dataType: 'json',
            timeout: 5000, // 5 second timeout
            success: function(response) {
                if (response.success && response.data.chat_threads) {
                    const threadsArray = Object.values(response.data.chat_threads);
                    
                    if (threadsArray.length > 0) {
                        threadsArray.sort((a, b) => b.updated_at - a.updated_at);
                        const lastChat = threadsArray[0];
                        
                        // Cache the chat ID
                        localStorage.setItem('mxchat_last_chat_id', lastChat.id);
                        
                        // Cache the messages for next time
                        if (lastChat.messages && lastChat.messages.length > 0) {
                            localStorage.setItem('mxchat_cached_messages_' + lastChat.id, JSON.stringify(lastChat.messages));
                        }
                        
                        // Load with smooth transition
                        self.loadChatWithTransition(lastChat.id);
                    } else {
                        self.showWelcomeWithTransition();
                    }
                } else {
                    self.showWelcomeWithTransition();
                }
                
                self.isLoading = false;
            },
            error: function() {
                // If server fails and we have cache, try to show something
                if (cachedChatId) {
                    self.showOfflineMessage();
                } else {
                    self.showWelcomeWithTransition();
                }
                
                self.isLoading = false;
            }
        });
    }
},

displayCachedContent: function(messages, chatId) {
    // Immediately display cached messages without loading indicator
    this.conversation.empty();
    
    // Update state
    this.currentChatId = chatId;
    this.unsavedChanges = false;
    this.isInitializing = false;
    
    if (messages.length === 0) {
        this.addMessage('assistant', mxChatAdminChat.i18n.welcome_message);
    } else {
        messages.forEach(message => {
            this.addMessage(message.role, message.content);
        });
        
        // Apply syntax highlighting
        setTimeout(() => {
            if (typeof Prism !== 'undefined') {
                Prism.highlightAll();
            }
        }, 50);
    }
    
    this.scrollToBottom();
    
    // Enable input immediately
    this.messageInput.prop('disabled', false);
    this.sendButton.prop('disabled', false);
    this.messageInput.focus();
},

showMinimalLoadingIndicator: function() {
    // Add a subtle loading indicator that doesn't block the UI
    if (!$('.mxchat-sync-indicator').length) {
        const syncIndicator = $('<div class="mxchat-sync-indicator">' +
            '<i class="fa-solid fa-sync fa-spin"></i> ' +
            '<span>Syncing...</span>' +
            '</div>');
        
        $('.mxchat-header').append(syncIndicator);
    }
},

hideMinimalLoadingIndicator: function() {
    $('.mxchat-sync-indicator').fadeOut(200, function() {
        $(this).remove();
    });
},

verifyAndUpdateFromServer: function(chatId) {
    const self = this;
    
    $.ajax({
        url: mxChatAdminChat.ajaxurl,
        type: 'POST',
        data: {
            action: 'mxchat_admin_chat_load_thread',
            nonce: mxChatAdminChat.nonce,
            chat_id: chatId
        },
        dataType: 'json',
        timeout: 10000, // 10 second timeout for background verify
        success: function(response) {
            if (response.success) {
                // Check if messages differ from cache
                const cachedMessages = JSON.parse(localStorage.getItem('mxchat_cached_messages_' + chatId) || '[]');
                const serverMessages = response.data.messages || [];
                
                // Simple comparison - if lengths differ or last message differs
                if (cachedMessages.length !== serverMessages.length || 
                    (serverMessages.length > 0 && cachedMessages.length > 0 &&
                     JSON.stringify(serverMessages[serverMessages.length - 1]) !== 
                     JSON.stringify(cachedMessages[cachedMessages.length - 1]))) {
                    
                    // Update cache
                    localStorage.setItem('mxchat_cached_messages_' + chatId, JSON.stringify(serverMessages));
                    
                    // Update UI smoothly
                    self.updateMessagesSmooth(serverMessages);
                }
                
                // Update model if needed
                if (response.data.model_id) {
                    $('.mxchat-model-selector-btn span').text(response.data.model_name || response.data.model_id);
                }
            }
            
            self.hideMinimalLoadingIndicator();
            self.isLoading = false;
        },
        error: function() {
            // Silent fail for background sync - user can still work with cached data
            self.hideMinimalLoadingIndicator();
            self.isLoading = false;
        }
    });
},

updateMessagesSmooth: function(newMessages) {
    const self = this;
    const currentMessageCount = $('.mxchat-message').length;
    
    // If completely different, replace all
    if (newMessages.length < currentMessageCount) {
        this.conversation.fadeOut(200, function() {
            self.conversation.empty();
            newMessages.forEach(message => {
                self.addMessage(message.role, message.content);
            });
            self.conversation.fadeIn(200);
            self.scrollToBottom();
        });
    } else {
        // Just add new messages
        for (let i = currentMessageCount; i < newMessages.length; i++) {
            self.addMessage(newMessages[i].role, newMessages[i].content);
        }
        self.scrollToBottom();
    }
    
    // Reapply syntax highlighting
    setTimeout(() => {
        if (typeof Prism !== 'undefined') {
            Prism.highlightAll();
        }
    }, 100);
},
loadChatWithTransition: function(chatId) {
    const self = this;
    
    // Update the cached chat ID since we're loading this chat
    localStorage.setItem('mxchat_last_chat_id', chatId);
    
    // Clear the new chat session flag since we're loading an existing chat
    sessionStorage.removeItem('mxchat_new_chat_session');
    
    $.ajax({
        url: mxChatAdminChat.ajaxurl,
        type: 'POST',
        data: {
            action: 'mxchat_admin_chat_load_thread',
            nonce: mxChatAdminChat.nonce,
            chat_id: chatId
        },
        dataType: 'json',
        success: function(response) {
            if (response.success) {
                // Update state
                self.currentChatId = chatId;
                self.unsavedChanges = false;
                self.isInitializing = false;
                
                // Fade out loading, fade in content
                self.hideLoadingIndicator();
                
                setTimeout(() => {
                    self.conversation.hide().empty();
                    
                    const messages = response.data.messages || [];
                    if (messages.length === 0) {
                        self.addMessage('assistant', mxChatAdminChat.i18n.welcome_message);
                    } else {
                        messages.forEach(message => {
                            self.addMessage(message.role, message.content);
                        });
                        
                        // Apply syntax highlighting
                        setTimeout(() => {
                            if (typeof Prism !== 'undefined') {
                                Prism.highlightAll();
                            }
                        }, 100);
                    }
                    
                    // Smooth reveal
                    self.conversation.fadeIn(300);
                    self.scrollToBottom();
                    
                    // Update model if needed
                    if (response.data.model_id) {
                        $('.mxchat-model-selector-btn span').text(response.data.model_name || response.data.model_id);
                    }
                    
                    // Complete loading
                    self.setLoadingState(false);
                }, 200);
            } else {
                self.showWelcomeWithTransition();
            }
        },
        error: function() {
            self.showWelcomeWithTransition();
        }
    });
},

cleanupLocalStorage: function() {
    // Clean up old deleted chats list if it gets too large
    const deletedChats = JSON.parse(localStorage.getItem('mxchat_deleted_chats') || '[]');
    if (deletedChats.length > 50) {
        localStorage.setItem('mxchat_deleted_chats', JSON.stringify(deletedChats.slice(-50)));
    }
    
    // Remove any orphaned cached messages for deleted chats
    const keysToRemove = [];
    for (let i = 0; i < localStorage.length; i++) {
        const key = localStorage.key(i);
        if (key && key.startsWith('mxchat_cached_messages_')) {
            const chatId = key.replace('mxchat_cached_messages_', '');
            if (deletedChats.includes(chatId)) {
                keysToRemove.push(key);
            }
        }
    }
    
    keysToRemove.forEach(key => localStorage.removeItem(key));
},

/**
 * Show welcome message with smooth transition
 */
showWelcomeWithTransition: function() {
    const self = this;
    
    this.hideLoadingIndicator();
    
    setTimeout(() => {
        this.conversation.hide().empty();
        this.addMessage('assistant', mxChatAdminChat.i18n.welcome_message || 'Welcome! How can I assist you today?');
        this.conversation.fadeIn(300);
        this.scrollToBottom();
        
        // Reset state
        this.currentChatId = null;
        this.unsavedChanges = false;
        this.isInitializing = false;
        
        // Complete loading
        this.setLoadingState(false);
    }, 200);
},

/**
 * Show offline message if server is unavailable
 */
showOfflineMessage: function() {
    this.hideLoadingIndicator();
    
    this.conversation.html(`
        <div class="mxchat-offline-message">
            <h3>⚠️ Connection Issue</h3>
            <p>Unable to load your previous chat. Please check your connection and try refreshing the page.</p>
            <button class="button" onclick="location.reload()">Refresh Page</button>
        </div>
    `);
    
    this.setLoadingState(false);
    this.isInitializing = false;
},

/**
 * Initialize Prism syntax highlighting
 */
initPrismHighlighting: function() {
    //console.log('Initializing Prism highlighting');
    
    // Check if Prism is already defined
    if (typeof Prism !== 'undefined') {
        //console.log('Prism is available, attempting to highlight all code blocks');
        try {
            Prism.highlightAll();
            //console.log('Initial Prism highlighting completed');
        } catch (e) {
            console.error('Error in Prism.highlightAll():', e);
        }
    } else {
        console.warn('Prism.js not loaded - code highlighting disabled');
    }
    
    // Set up a MutationObserver to detect when new messages are added
    if (window.MutationObserver && this.conversation.length) {
        //console.log('Setting up MutationObserver for code highlighting');
        
        // Create an observer instance
        this.codeHighlightObserver = new MutationObserver((mutations) => {
            let shouldHighlight = false;
            
            // Check if any of the mutations involve adding code blocks
            mutations.forEach((mutation) => {
                if (mutation.type === 'childList' && mutation.addedNodes.length) {
                    for (let i = 0; i < mutation.addedNodes.length; i++) {
                        const node = mutation.addedNodes[i];
                        if (node.nodeType === 1 && // Element node
                            (node.querySelector('pre code') || 
                             node.tagName === 'PRE' || 
                             node.tagName === 'CODE')) {
                            shouldHighlight = true;
                            break;
                        }
                    }
                }
            });
            
            // Only call highlightAll if we found code elements
            if (shouldHighlight && typeof Prism !== 'undefined') {
                //console.log('New code blocks detected, re-highlighting');
                setTimeout(() => {
                    try {
                        Prism.highlightAll();
                        //console.log('Re-highlighting completed');
                    } catch (e) {
                        console.error('Error in Prism.highlightAll() from observer:', e);
                    }
                }, 50);
            }
        });
        
        // Start observing the conversation for changes
        this.codeHighlightObserver.observe(this.conversation[0], {
            childList: true,
            subtree: true
        });
    }
},

        /**
         * Initialize textarea auto-resize
         */
        initAutoResize: function() {
            this.messageInput.on('input', function() {
                this.style.height = 'auto';
                const newHeight = Math.min(this.scrollHeight, 200);
                this.style.height = newHeight + 'px';
            });
        },

/**
 * Initialize event listeners
 */
initEventListeners: function() {
    const self = this;

    // Form submission
    this.messageForm.on('submit', function(e) {
        e.preventDefault();
        self.sendMessage();
    });

    // New chat button (starts a new chat, saving current if needed)
    this.newChatButton.on('click', function() {
        if (self.unsavedChanges) {
            // Make sure we have the proper message text in the confirmation dialog
            if (confirm(mxChatAdminChat.i18n.save_chat_confirm || 'Do you want to save the current chat before starting a new one?')) {
                self.saveCurrentChat(function() {
                    self.startNewChat();
                });
            } else {
                self.startNewChat();
            }
        } else {
            self.startNewChat();
        }
    });
    
    // History button (shows chat history modal)
    this.historyButton.on('click', function() {
        self.showChatHistoryModal();
    });
    
    // Settings button click handler
        $('#mxchat-settings-btn').on('click', function() {
            //console.log('Settings button clicked');
            $('#mxchat-settings-modal').show();
        });
        
        // Settings modal close button
        $(document).on('click', '#mxchat-settings-close', function() {
            //console.log('Settings close button clicked');
            $('#mxchat-settings-modal').hide();
        });
        
$(document).on('submit', '#mxchat-settings-form', function(e) {
    e.preventDefault();
    //console.log('Settings form submitted');

    // Store 'self' reference for use in callbacks
    const self = MxChatAdmin; 
    
    // Get ALL form values
    const imageGenModel = $('#mxchat-image-gen-model').val();
    const imageSize = $('#mxchat-image-size').val();
    const imageQuality = $('#mxchat-image-quality').val();
    const imageCount = $('#mxchat-image-count').val();
    const imageResponseFormat = $('#mxchat-image-response-format').val();
    const perplexityModel = $('#mxchat-perplexity-model').val();

    // Log values for debugging
    //console.log('Saving image count:', imageCount);
    
    // Save to localStorage
    localStorage.setItem('mxchat_image_gen_model', imageGenModel);
    localStorage.setItem('mxchat_image_size', imageSize);
    localStorage.setItem('mxchat_image_quality', imageQuality);
    localStorage.setItem('mxchat_image_count', imageCount);
    localStorage.setItem('mxchat_image_response_format', imageResponseFormat);
    localStorage.setItem('mxchat_perplexity_model', perplexityModel);

    // Save to server
    $.ajax({
        url: mxChatAdminChat.ajaxurl,
        type: 'POST',
        data: {
            action: 'mxchat_admin_save_settings',
            nonce: mxChatAdminChat.nonce,
            image_gen_model: imageGenModel,
            image_size: imageSize,
            image_quality: imageQuality,
            image_count: imageCount,
            image_response_format: imageResponseFormat,
            perplexity_model: perplexityModel
        },
        dataType: 'json',
        success: function(response) {
            // Close the settings modal first
            $('#mxchat-settings-modal').hide();
            
            // Then show the notification
            if (response.success) {
                self.showNotification(mxChatAdminChat.i18n.settings_saved || 'Settings saved successfully');
            } else {
                self.showNotification(mxChatAdminChat.i18n.error + ': ' + 
                    (response.data && response.data.message ? response.data.message : 'Unknown error'), 'error');
            }
        },
        error: function(xhr, status, error) {
            // Close the modal
            $('#mxchat-settings-modal').hide();
            
            // Show error notification
            self.showNotification(mxChatAdminChat.i18n.error + ': ' + error, 'error');
        }
    });
});

    // Tool buttons (PDF, Word, Search, etc.)
    this.toolButtons.on('click', function() {
        const tool = $(this).attr('title');
        self.showNotification(tool + ' feature coming soon');
    });

    // Copy to clipboard buttons (using event delegation for dynamic content)
    $(document).on('click', '.mxchat-copy-btn', function() {
        const messageContent = $(this).closest('.mxchat-message-content').find('.mxchat-message-text').text();
        self.copyToClipboard(messageContent, $(this));
    });

    // Regenerate response button
    $(document).on('click', '.mxchat-regenerate-btn', function() {
        self.showNotification('Regenerate response feature coming soon');
    });

    // Enter key to send, Shift+Enter for new line
    this.messageInput.on('keydown', function(e) {
        if (e.key === 'Enter' && !e.shiftKey) {
            e.preventDefault();
            self.sendMessage();
        }
    });
    
    // Listen for chat clicks in history modal
    $(document).on('click', '.mxchat-history-item', function() {
        const chatId = $(this).data('chat-id');
        self.loadChatThread(chatId);
    });
    
    // Listen for delete button clicks in history modal
    $(document).on('click', '.mxchat-history-delete-btn', function(e) {
        e.stopPropagation(); // Prevent triggering parent click
        e.preventDefault(); // Prevent any default button behavior
        
        const chatId = $(this).data('chat-id');
        //console.log('Delete button clicked for chat ID:', chatId); // Debug
        
        if (chatId && confirm(mxChatAdminChat.i18n.delete_chat_confirm || 'Are you sure you want to delete this chat?')) {
            self.deleteChatThread(chatId);
        } else {
            //console.log('Chat ID missing or delete canceled'); // Debug
        }
    });
},


/**
 * Show a notification
 * 
 * @param {string} message - The notification message
 * @param {string} type - The notification type (success, error)
 */
showNotification: function(message, type = 'success') {
    //console.log('Showing notification:', message, type);
    
    const notificationId = 'mxchat-notification-' + Date.now();
    const notificationClass = type === 'error' ? 'mxchat-notification-error' : 'mxchat-notification-success';

    // Create notification element
    const $notification = $(`
        <div id="${notificationId}" class="mxchat-notification ${notificationClass}">
            <span>${message}</span>
            <button class="mxchat-notification-close">&times;</button>
        </div>
    `);

    // Add to page
    if ($('.mxchat-notifications-container').length === 0) {
        $('body').append('<div class="mxchat-notifications-container"></div>');
    }

    $('.mxchat-notifications-container').append($notification);
    
    // Make sure the notification is visible
    $notification.css({
        'display': 'flex',
        'opacity': '1'
    });

    // Add close button handler
    $notification.find('.mxchat-notification-close').on('click', function() {
        $notification.fadeOut(300, function() {
            $(this).remove();
        });
    });

    // Auto-hide after 5 seconds
    setTimeout(function() {
        $notification.fadeOut(300, function() {
            $(this).remove();
        });
    }, 5000);
},

/**
 * Initialize the chat history modal
 */
initChatHistoryModal: function() {
    // Create the modal HTML structure
    const chatHistoryModal = `
        <div id="mxchat-history-modal" class="mxchat-modal">
            <div class="mxchat-modal-content">
                <div class="mxchat-modal-header">
                    <h3>${mxChatAdminChat.i18n.chat_history || 'Chat History'}</h3>
                    <span class="mxchat-modal-close">&times;</span>
                </div>
                <div class="mxchat-modal-body">
                    <div class="mxchat-history-search">
                        <input type="text" id="mxchat-history-search-input" placeholder="${mxChatAdminChat.i18n.search_chats || 'Search chats...'}" />
                    </div>
                    <div class="mxchat-history-list" id="mxchat-history-list">
                        <!-- Chat history items will be loaded here -->
                    </div>
                </div>
            </div>
        </div>
    `;
    
    // Append modal to the body
    $('body').append(chatHistoryModal);
    
    // Set up event handlers
    const $modal = $('#mxchat-history-modal');
    
    // Close modal when clicking the close button
    $('.mxchat-modal-close', $modal).on('click', function() {
        $modal.hide();
    });
    
    // Close modal when clicking outside the modal content
    $(window).on('click', function(event) {
        if ($(event.target).is($modal)) {
            $modal.hide();
        }
    });
    
    // Search functionality
    $('#mxchat-history-search-input').on('input', function() {
        const searchTerm = $(this).val().toLowerCase();
        $('.mxchat-history-item').each(function() {
            const title = $(this).find('.mxchat-history-title').text().toLowerCase();
            const preview = $(this).find('.mxchat-history-preview').text().toLowerCase();
            
            if (title.includes(searchTerm) || preview.includes(searchTerm)) {
                $(this).show();
            } else {
                $(this).hide();
            }
        });
    });
},

/**
 * Show the chat history modal and load chat threads
 */
showChatHistoryModal: function() {
    // If there are unsaved changes, prompt to save first
    if (this.unsavedChanges) {
        if (confirm(mxChatAdminChat.i18n.save_before_history || 'Save current chat before viewing history?')) {
            this.saveCurrentChat(() => {
                this._openHistoryModal();
            });
        } else {
            this._openHistoryModal();
        }
    } else {
        this._openHistoryModal();
    }
},

/**
 * Internal method to open and populate the history modal
 * @private
 */
_openHistoryModal: function() {
    // Show the modal
    $('#mxchat-history-modal').show();
    
    // Load the chat history list
    this.loadChatHistory();
    
    // Clear the search input
    $('#mxchat-history-search-input').val('');
},

/**
 * Load the chat history list from the server
 */
loadChatHistory: function() {
    const self = this;
    const $historyList = $('#mxchat-history-list');
    
    // Show loading indicator
    $historyList.html('<div class="mxchat-history-loading"><i class="fa-solid fa-spinner fa-spin"></i> Loading chat history...</div>');
    
    // Make AJAX request to get chat history
    $.ajax({
        url: mxChatAdminChat.ajaxurl,
        type: 'POST',
        data: {
            action: 'mxchat_admin_chat_get_history',
            nonce: mxChatAdminChat.nonce
        },
        dataType: 'json',
        success: function(response) {
            if (response.success) {
                self.renderChatHistory(response.data.chat_threads);
            } else {
                $historyList.html('<div class="mxchat-history-error">Error loading chat history.</div>');
            }
        },
        error: function() {
            $historyList.html('<div class="mxchat-history-error">Error loading chat history.</div>');
        }
    });
},
/**
 * Render the chat history list
 * 
 * @param {Object} chatThreads Object of chat threads
 */
renderChatHistory: function(chatThreads) {
    const $historyList = $('#mxchat-history-list');
    $historyList.empty();
    
    if (!chatThreads || Object.keys(chatThreads).length === 0) {
        $historyList.html('<div class="mxchat-history-empty">No chat history found.</div>');
        return;
    }
    
    // Convert object to array
    const threadsArray = Object.values(chatThreads);
    
    // Sort chats by time (newest first)
    threadsArray.sort((a, b) => b.updated_at - a.updated_at);
    
    // Add each chat to the list
    threadsArray.forEach(chat => {
        const date = new Date(chat.updated_at * 1000);
        const formattedDate = date.toLocaleDateString() + ' ' + date.toLocaleTimeString();
        
        // Create title from first message or use default
        let title = chat.title || 'Chat ' + formattedDate;
        
        // Get preview text from the first few messages
        let preview = '';
        if (chat.messages && chat.messages.length > 0) {
            // Extract preview from user messages
            const userMessages = chat.messages.filter(msg => msg.role === 'user');
            if (userMessages.length > 0) {
                // Strip HTML for preview text
                const tempDiv = document.createElement('div');
                tempDiv.innerHTML = userMessages[0].content;
                const textContent = tempDiv.textContent || tempDiv.innerText || '';
                preview = textContent.substring(0, 100) + (textContent.length > 100 ? '...' : '');
            }
        }
        
        const $chatItem = $(`
            <div class="mxchat-history-item" data-chat-id="${chat.id}">
                <div class="mxchat-history-item-content">
                    <h4 class="mxchat-history-title">${title}</h4>
                    <p class="mxchat-history-preview">${preview}</p>
                    <div class="mxchat-history-meta">
                        <span class="mxchat-history-date">${formattedDate}</span>
                        <span class="mxchat-history-messages">${chat.messages ? chat.messages.length : 0} messages</span>
                    </div>
                </div>
                <div class="mxchat-history-actions">
                    <button type="button" class="mxchat-history-delete-btn" data-chat-id="${chat.id}" title="Delete chat">
                        <i class="fa-solid fa-trash"></i>
                    </button>
                </div>
            </div>
        `);
        
        $historyList.append($chatItem);
    });
},

/**
 * Start a new chat
 */
startNewChat: function() {
    const self = this;
    
    // Clear any cached chat references
    localStorage.removeItem('mxchat_last_chat_id');
    sessionStorage.setItem('mxchat_new_chat_session', 'true');
    
    // First check if there are any messages to save
    const hasMessages = $('.mxchat-message').length > 1; // More than just welcome message
    
    if (hasMessages) {
        // Save the current chat first
        this.saveCurrentChat(function() {
            // After saving, continue with starting a new chat
            self._clearAndStartNewChat();
        });
    } else {
        // No messages to save, just start a new chat
        self._clearAndStartNewChat();
    }
},

_clearAndStartNewChat: function() {
    // Clear the conversation display
    this.conversation.empty();
    
    // Add welcome message
    this.addMessage('assistant', mxChatAdminChat.i18n.welcome_message);
    
    // Reset variables
    this.currentChatId = null;
    this.unsavedChanges = false;
    
    // Clear session storage flag after a delay (so it doesn't affect immediate reloads)
    setTimeout(() => {
        sessionStorage.removeItem('mxchat_new_chat_session');
    }, 1000);
    
    // Focus on input
    this.messageInput.focus();
    
    // Show notification
    this.showNotification(mxChatAdminChat.i18n.new_chat_started || 'New chat started');
    
    // Clear the conversation history in the session
    $.ajax({
        url: mxChatAdminChat.ajaxurl,
        type: 'POST',
        data: {
            action: 'mxchat_admin_chat_clear_conversation',
            nonce: mxChatAdminChat.nonce
        },
        dataType: 'json'
    });
},


/**
 * Load a specific chat thread
 * 
 * @param {string} chatId The chat ID to load
 */
loadChatThread: function(chatId) {
    const self = this;
    
    // If there are unsaved changes in the current chat, save them first
    if (this.unsavedChanges && this.currentChatId !== chatId) {
        if (confirm(mxChatAdminChat.i18n.switch_chat_confirm || 'Save changes to the current chat before switching?')) {
            // Save current chat first, then load the new one
            this.saveCurrentChat(function() {
                self._loadChatThreadInternal(chatId);
            });
        } else {
            // User chose not to save, just load the new chat
            this._loadChatThreadInternal(chatId);
        }
    } else {
        // No unsaved changes, just load the chat
        this._loadChatThreadInternal(chatId);
    }
},

/**
 * Internal method to handle the actual chat thread loading
 * 
 * @param {string} chatId The chat ID to load
 * @private
 */
_loadChatThreadInternal: function(chatId) {
    const self = this;
    
    // Show loading indicator
    this.conversation.html('<div class="mxchat-loading"><i class="fa-solid fa-spinner fa-spin"></i> Loading chat...</div>');
    
    // Make AJAX request to get chat thread
    $.ajax({
        url: mxChatAdminChat.ajaxurl,
        type: 'POST',
        data: {
            action: 'mxchat_admin_chat_load_thread',
            nonce: mxChatAdminChat.nonce,
            chat_id: chatId
        },
        dataType: 'json',
        success: function(response) {
            if (response.success) {
                // Update state
                self.currentChatId = chatId;
                self.unsavedChanges = false;
                
                // Clear conversation display
                self.conversation.empty();
                
                // Render messages
                const messages = response.data.messages || [];
                if (messages.length === 0) {
                    // If no messages, add a welcome message
                    self.addMessage('assistant', mxChatAdminChat.i18n.welcome_message || 'Welcome! How can I assist you today?');
                } else {
                    // Add each message to the conversation
                    messages.forEach(message => {
                        self.addMessage(message.role, message.content);
                    });
                    
                    // Apply syntax highlighting
                    setTimeout(() => {
                        if (typeof Prism !== 'undefined') {
                            Prism.highlightAll();
                        }
                    }, 100);
                }
                
                // Close the modal
                $('#mxchat-history-modal').hide();
                
                // Scroll to bottom
                self.scrollToBottom();
                
                // Update model if needed
                if (response.data.model_id) {
                    // Update model display only, don't change backend
                    $('.mxchat-model-selector-btn span').text(response.data.model_name || response.data.model_id);
                }
                
                // Show notification
                self.showNotification(mxChatAdminChat.i18n.chat_loaded || 'Chat loaded successfully');
            } else {
                self.conversation.html('<div class="mxchat-error">Error loading chat: ' + (response.data && response.data.message ? response.data.message : 'Unknown error') + '</div>');
            }
        },
        error: function(xhr, status, error) {
            self.conversation.html('<div class="mxchat-error">Error loading chat: ' + error + '</div>');
        }
    });
},

/**
 * Save the current chat thread
 * 
 * @param {Function} callback Function to call after saving
 * @param {boolean} forceSave Whether to force saving even if there are no changes
 */
saveCurrentChat: function(callback, forceSave = false) {
    const self = this;
    
    // Get all messages from the UI
    const messages = [];
    $('.mxchat-message').each(function() {
        const $msg = $(this);
        const role = $msg.hasClass('mxchat-user') ? 'user' : 'assistant';
        const content = $msg.find('.mxchat-message-text').html();
        
        messages.push({
            role: role,
            content: content
        });
    });
    
    // If no messages and not forcing save, don't save
    if (messages.length === 0 && !forceSave) {
        if (typeof callback === 'function') {
            callback();
        }
        return;
    }
    
    // If no changes and not forcing save, don't save
    if (!this.unsavedChanges && !forceSave) {
        //console.log('No unsaved changes, skipping save operation');
        if (typeof callback === 'function') {
            callback();
        }
        return;
    }
    
    // Get the first user message as title (or part of it)
    let title = '';
    const firstUserMessage = $('.mxchat-user .mxchat-message-text').first().text();
    if (firstUserMessage) {
        title = firstUserMessage.substring(0, 50) + (firstUserMessage.length > 50 ? '...' : '');
    } else {
        title = 'Chat ' + new Date().toLocaleDateString();
    }
    
    // Get current model
    const modelId = mxChatAdminChat.current_model ? mxChatAdminChat.current_model.id : '';
    const modelName = mxChatAdminChat.current_model ? mxChatAdminChat.current_model.name : '';
    
    // Show saving indicator only if not suppressed
    if (!this._suppressSaveNotification) {
        self.showNotification(mxChatAdminChat.i18n.saving || 'Saving chat...');
    }
    
    //console.log('Saving chat thread with ID:', this.currentChatId, 'Messages:', messages.length);
    
    // Make AJAX request to save chat thread
    $.ajax({
        url: mxChatAdminChat.ajaxurl,
        type: 'POST',
        data: {
            action: 'mxchat_admin_chat_save_thread',
            nonce: mxChatAdminChat.nonce,
            chat_id: this.currentChatId, // Will be null for new chats
            title: title,
            messages: JSON.stringify(messages),
            model_id: modelId,
            model_name: modelName
        },
        dataType: 'json',
        success: function(response) {
            if (response.success) {
                // Update current chat ID with the one returned from server
                self.currentChatId = response.data.chat_id;
                self.unsavedChanges = false;
                
                // Show notification only if not suppressed
                if (!self._suppressSaveNotification) {
                    self.showNotification(mxChatAdminChat.i18n.chat_saved || 'Chat saved successfully');
                }
                
                //console.log('Chat saved successfully. New ID:', response.data.chat_id);
                
                // Reset the suppress flag
                self._suppressSaveNotification = false;
                
                // Call callback if provided
                if (typeof callback === 'function') {
                    callback();
                }
            } else {
                // Reset the suppress flag
                self._suppressSaveNotification = false;
                
                self.showNotification(mxChatAdminChat.i18n.error + ': ' + (response.data && response.data.message ? response.data.message : 'Error saving chat'));
                console.error('Error saving chat:', response);
                
                // Still call callback if provided, even on error
                if (typeof callback === 'function') {
                    callback();
                }
            }
        },
        error: function(xhr, status, error) {
            // Reset the suppress flag
            self._suppressSaveNotification = false;
            
            self.showNotification(mxChatAdminChat.i18n.error + ': Error saving chat - ' + error);
            console.error('AJAX error saving chat:', status, error, xhr.responseText);
            
            // Still call callback if provided, even on error
            if (typeof callback === 'function') {
                callback();
            }
        }
    });
},

/**
 * Delete a chat thread
 * 
 * @param {string} chatId The chat ID to delete
 */
deleteChatThread: function(chatId) {
    const self = this;
    
    if (!chatId) {
        this.showNotification(mxChatAdminChat.i18n.error + ': Cannot delete chat - ID is missing');
        console.error('Attempted to delete chat with empty ID');
        return;
    }
    
    //console.log('Deleting chat with ID:', chatId); // Debug
    
    // Check if this is the cached chat
    const cachedChatId = localStorage.getItem('mxchat_last_chat_id');
    if (cachedChatId === chatId) {
        localStorage.removeItem('mxchat_last_chat_id');
    }
    
    // Track deleted chats to prevent trying to load them
    let deletedChats = JSON.parse(localStorage.getItem('mxchat_deleted_chats') || '[]');
    if (!deletedChats.includes(chatId)) {
        deletedChats.push(chatId);
        // Keep only last 50 deleted chat IDs to prevent localStorage bloat
        if (deletedChats.length > 50) {
            deletedChats = deletedChats.slice(-50);
        }
        localStorage.setItem('mxchat_deleted_chats', JSON.stringify(deletedChats));
    }
    
    $.ajax({
        url: mxChatAdminChat.ajaxurl,
        type: 'POST',
        data: {
            action: 'mxchat_admin_chat_delete_thread',
            nonce: mxChatAdminChat.nonce,
            chat_id: chatId
        },
        dataType: 'json',
        success: function(response) {
            //console.log('Delete response:', response); // Debug
            
            if (response.success) {
                // If the deleted chat was the current one, start a new chat
                if (self.currentChatId === chatId) {
                    self.startNewChat();
                }
                
                // Reload the history list
                self.loadChatHistory();
                
                // Show notification
                self.showNotification(mxChatAdminChat.i18n.chat_deleted || 'Chat deleted successfully');
            } else {
                self.showNotification(mxChatAdminChat.i18n.error + ': ' + (response.data && response.data.message ? response.data.message : 'Error deleting chat'));
            }
        },
        error: function(xhr, status, error) {
            console.error('Delete AJAX error:', status, error); // Debug
            self.showNotification(mxChatAdminChat.i18n.error + ': ' + 'Error deleting chat - ' + error);
        }
    });
},


/**
 * Send message to the AI with a limited context window
 */
sendMessage: function() {
    //console.log('Raw message before sending:', this.messageInput.val().trim());
    this.unsavedChanges = true;

    const message = this.messageInput.val().trim();
    
    if (!message || this.isWaitingForResponse) {
        return;
    }

    // Package message data
    const messageData = {
        message: message
    };

    // Trigger the mxchat_admin_before_send event
    // This allows Perplexity to intercept the message if needed
    let continueProcessing = true;
    $(document).trigger('mxchat_admin_before_send', [messageData, function() {
        continueProcessing = false;
    }]);

    // If the event handler (Perplexity) cancelled the process, exit
    if (!continueProcessing) {
        //console.log('Message handling intercepted by plugin (likely Perplexity)');
        return;
    }

    // Continue with normal processing
    
    // IMPORTANT: Collect context messages BEFORE adding the new message to UI
    // This prevents the current message from being included in the context
    const contextSize = 10; // Adjust this number as needed
    const contextMessages = this.getContextMessages(contextSize);
    
    // Update UI state
    this.isWaitingForResponse = true;
    this.messageInput.val('');
    this.messageInput.css('height', 'auto');
    this.messageInput.prop('disabled', true);
    this.sendButton.prop('disabled', true);
    
    // Add user message to UI (AFTER collecting context)
    this.addMessage('user', message);
    this.scrollToBottom();
    
    // Add thinking indicator
    this.addThinkingIndicator();
    
    // Send to backend
    $.ajax({
        url: mxChatAdminChat.ajaxurl,
        type: 'POST',
        data: {
            action: 'mxchat_admin_chat_send_message',
            nonce: mxChatAdminChat.nonce,
            message: message,
            chat_id: this.currentChatId, // Include current chat ID if any
            context_messages: JSON.stringify(contextMessages) // Send limited context window
        },
        dataType: 'json',
        success: (response) => {
            // Remove thinking indicator
            this.removeThinkingIndicator();
            
            if (response.success) {
                // Add assistant message
                this.addMessage('assistant', response.data.response, response.data.model);
                
                // Ensure syntax highlighting is applied with a slightly longer delay
                setTimeout(() => {
                    if (typeof Prism !== 'undefined') {
                        Prism.highlightAll();
                    }
                }, 100);
                
                // Schedule auto-save instead of immediate save
                this.scheduleAutoSave();
                
            } else {
                // Handle specific error types
                if (response.data && response.data.error_type === 'context_length') {
                    // Handle context length error specifically
                    this.handleContextLengthError();
                } else {
                    // Add generic error message
                    this.addErrorMessage(response.data && response.data.message ? response.data.message : 'Unknown error');
                }
            }
            
            this.scrollToBottom();
            this.isWaitingForResponse = false;
            this.messageInput.prop('disabled', false);
            this.sendButton.prop('disabled', false);
            this.messageInput.focus();
        },
        error: (xhr, status, error) => {
            // Remove thinking indicator
            this.removeThinkingIndicator();
            
            // Add error message
            this.addErrorMessage(mxChatAdminChat.i18n.error + ': ' + error);
            
            this.scrollToBottom();
            this.isWaitingForResponse = false;
            this.messageInput.prop('disabled', false);
            this.sendButton.prop('disabled', false);
            this.messageInput.focus();
        }
    });
},
scheduleAutoSave: function() {
    // Clear any existing auto-save timer
    if (this.autoSaveTimer) {
        clearTimeout(this.autoSaveTimer);
    }
    
    // Schedule a new auto-save
    this.autoSaveTimer = setTimeout(() => {
        // Only save if there are actual unsaved changes
        if (this.unsavedChanges) {
           //console.log('Auto-saving chat after inactivity...');
            this.saveCurrentChatSilently(() => {
                //console.log('Auto-save completed');
            }, true);
        }
        this.autoSaveTimer = null;
    }, 2000); // Save 2 seconds after last activity
},

cancelAutoSave: function() {
    if (this.autoSaveTimer) {
        clearTimeout(this.autoSaveTimer);
        this.autoSaveTimer = null;
    }
},

/**
 * Get context messages for the AI request - FIXED VERSION
 * 
 * @param {number} contextSize Number of message pairs to include
 * @return {Array} Array of message objects to send to the AI
 */
getContextMessages: function(contextSize) {
    const contextMessages = [];
    const messageElements = $('.mxchat-message').toArray();
    
    // Simply take the last contextSize * 2 messages (whatever is in the DOM)
    const startIndex = Math.max(0, messageElements.length - (contextSize * 2));
    
    // Include ALL messages from startIndex to end
    for (let i = startIndex; i < messageElements.length; i++) {
        const $msg = $(messageElements[i]);
        const role = $msg.hasClass('mxchat-user') ? 'user' : 'assistant';
        const content = $msg.find('.mxchat-message-text').html();
        
        if (content && content.trim()) {
            contextMessages.push({
                role: role,
                content: content
            });
        }
    }
    
    return contextMessages;
},

/**
 * Handle context length error
 */
handleContextLengthError: function() {
    // Add a specialized error message
    const errorHtml = `
        <div class="mxchat-message mxchat-assistant mxchat-error">
            <div class="mxchat-message-content">
                <div class="mxchat-message-header">
                    <span class="mxchat-message-sender">${mxChatAdminChat.i18n.assistant || 'Assistant'}</span>
                </div>
                <div class="mxchat-message-text">
                    <p><strong>${mxChatAdminChat.i18n.context_length_error_title || 'Conversation Too Long'}</strong></p>
                    <p>${mxChatAdminChat.i18n.context_length_error_message || 'The conversation has reached the maximum allowed length. To continue, you can:'}</p>
                    <ul>
                        <li>${mxChatAdminChat.i18n.context_length_error_option1 || 'Start a new chat (your current chat will be saved)'}</li>
                        <li>${mxChatAdminChat.i18n.context_length_error_option2 || 'Save this chat and continue in a new thread'}</li>
                        <li>${mxChatAdminChat.i18n.context_length_error_option3 || 'Try a shorter, more focused question'}</li>
                    </ul>
                </div>
            </div>
        </div>
    `;
    
    // Add to conversation
    this.conversation.append(errorHtml);
    
    // Show a notification too
    this.showNotification(mxChatAdminChat.i18n.context_length_error_notification || 'Conversation length limit reached');
},

/**
 * Update the selected model
 * 
 * @param {string} modelId   The model ID
 * @param {string} modelName The model name for display
 */
updateModel: function(modelId, modelName) {
    const self = this;
    
    if (!modelId) {
        console.error('No model ID provided for updateModel');
        this.showNotification(mxChatAdminChat.i18n.error + ': Invalid model selection');
        return;
    }
    
    // Show loading in the button
    $('.mxchat-model-selector-btn span').text(mxChatAdminChat.i18n.loading + '...');
    
    
    $.ajax({
        url: mxChatAdminChat.ajaxurl,
        type: 'POST',
        data: {
            action: 'mxchat_admin_chat_update_model',
            nonce: mxChatAdminChat.nonce,
            model_id: modelId
        },
        dataType: 'json',
        success: function(response) {
            //console.log('Update model response:', response);
            
            if (response.success) {
                // Show notification
                self.showNotification(mxChatAdminChat.i18n.update_success);
                
                // Update global current model
                if (response.data && response.data.model) {
                    mxChatAdminChat.current_model = response.data.model;
                } else {
                    // Fallback if no model data returned
                    mxChatAdminChat.current_model = {
                        id: modelId,
                        name: modelName
                    };
                }
                
                // Update UI - update button text
                $('.mxchat-model-selector-btn span').text(modelName);
            } else {
                // Show error notification with details from server
                const errorMsg = response.data && response.data.message 
                    ? response.data.message 
                    : 'Unknown error updating model';
                    
                self.showNotification(mxChatAdminChat.i18n.error + ': ' + errorMsg);
                
                // Restore original text on error
                $('.mxchat-model-selector-btn span').text(
                    mxChatAdminChat.current_model ? mxChatAdminChat.current_model.name : mxChatAdminChat.i18n.select_model
                );
            }
        },
        error: function(xhr, status, error) {
            console.error('Update model error:', {
                status: status,
                error: error,
                response: xhr.responseText
            });
            
            let errorMsg = error;
            
            // Try to extract error message from response if possible
            try {
                const errorResponse = JSON.parse(xhr.responseText);
                if (errorResponse.data && errorResponse.data.message) {
                    errorMsg = errorResponse.data.message;
                }
            } catch(e) {
                // Use the original error
            }
            
            // Show error notification
            self.showNotification(mxChatAdminChat.i18n.error + ': ' + errorMsg);
            
            // Restore original text on error
            $('.mxchat-model-selector-btn span').text(
                mxChatAdminChat.current_model ? mxChatAdminChat.current_model.name : mxChatAdminChat.i18n.select_model
            );
        }
    });
},

/**
 * Save a message to conversation history
 * 
 * @param {string} role       The message role (user/assistant)
 * @param {string} content    The message content
 * @param {string} source     Optional source of the message (e.g., 'perplexity')
 */
saveMessageToHistory: function(role, content, source = '') {
    // Make an AJAX call to save this to the server-side conversation history
    $.ajax({
        url: mxChatAdminChat.ajaxurl,
        type: 'POST',
        data: {
            action: 'mxchat_admin_chat_save_message',
            nonce: mxChatAdminChat.nonce,
            role: role,
            content: content,
            source: source
        },
        dataType: 'json'
    });
},

        /**
         * Clear the conversation
         */
        clearConversation: function() {
    $.ajax({
        url: mxChatAdminChat.ajaxurl,
        type: 'POST',
        data: {
            action: 'mxchat_admin_chat_clear_conversation',
            nonce: mxChatAdminChat.nonce
        },
        dataType: 'json',
        success: (response) => {
            if (response.success) {
                // Clear all messages including Perplexity ones
                this.conversation.empty();
                
                // Disable Perplexity mode if it was enabled
                if (typeof MxChatPerplexity !== 'undefined' && MxChatPerplexity.isEnabled) {
                    MxChatPerplexity.togglePerplexity();
                }
                
                // Add welcome message
                this.addMessage('assistant', mxChatAdminChat.i18n.welcome_message);
                this.scrollToBottom();
                
                // Show notification
                this.showNotification(mxChatAdminChat.i18n.clear_success);
            }
        }
    });
},

/**
 * Add a message to the chat
 * 
 * @param {string} role      The message role (user/assistant)
 * @param {string|object} content   The message content
 * @param {string} modelName The model name (optional, for assistant messages)
 */
addMessage: function(role, content, modelName) {
    // Make sure content is a string
    if (content === null || content === undefined) {
        content = '';
    } else if (typeof content !== 'string') {
        // If content is an object or something else, convert it to string
        try {
            if (content instanceof Error) {
                content = content.message;
            } else if (typeof content === 'object') {
                content = JSON.stringify(content, null, 2);
            } else {
                content = String(content);
            }
        } catch (e) {
            content = 'Error converting message content to string';
        }
    }
    
    //console.log(`Adding ${role} message with content length: ${content.length}`);
    const hasCodeBlock = content.includes('```') || content.includes('<pre><code');
    if (hasCodeBlock) {
        //console.log('Message contains code blocks');
    }
    
    // Create the message element
    const $message = $('<div class="mxchat-message mxchat-' + role + '"></div>');
    
    // Message content container
    const $contentContainer = $('<div class="mxchat-message-content"></div>');
    
    // Add sender info
    const senderLabel = role === 'user' ? mxChatAdminChat.i18n.you : mxChatAdminChat.i18n.assistant;
    const $header = $('<div class="mxchat-message-header"><span class="mxchat-message-sender">' + senderLabel + '</span></div>');
    $contentContainer.append($header);
    
    // Process and add message text
    const $textContainer = $('<div class="mxchat-message-text"></div>');
    
    // Pass the role to processMessageContent
    const processedContent = this.processMessageContent(content, role);
    $textContainer.html(processedContent);

    $contentContainer.append($textContainer);
    
    // Add actions for assistant messages
    if (role === 'assistant') {
        const $actions = $('<div class="mxchat-message-actions"></div>');
        
        // Copy button
        const $copyBtn = $('<button class="mxchat-action-btn mxchat-copy-btn" title="' + mxChatAdminChat.i18n.copy + '">' +
            '<i class="fa-regular fa-copy"></i></button>');
        $actions.append($copyBtn);
        
        $contentContainer.append($actions);
    }
    
    // Add content container to message
    $message.append($contentContainer);
    
    // Add to conversation
    this.conversation.append($message);
    
    // Apply syntax highlighting to code blocks
    if (hasCodeBlock) {
        //console.log('Attempting to highlight code blocks in added message');
        
        // Force preservation of whitespace with inline styles
        const codeBlocks = $message.find('pre code');
        if (codeBlocks.length) {
            //console.log(`Found ${codeBlocks.length} code blocks to format`);
            codeBlocks.each(function() {
                $(this).css({
                    'white-space': 'pre',
                    'tab-size': '4',
                    'display': 'block'
                });
                
                $(this).parent('pre').css({
                    'white-space': 'pre',
                    'overflow-x': 'auto',
                    'tab-size': '4',
                    'display': 'block'
                });
            });
        }
        
        // Apply syntax highlighting with Prism.js
        if (typeof Prism !== 'undefined') {
            // Immediate highlighting
            try {
                Prism.highlightAllUnder($message[0]);
                //console.log('Immediate highlighting attempted');
            } catch (e) {
                console.error('Error in immediate highlighting:', e);
            }
            
            // Short delay highlighting
            setTimeout(() => {
                try {
                    Prism.highlightAllUnder($message[0]);
                    //console.log('Short delay highlighting attempted');
                } catch (e) {
                    console.error('Error in short delay highlighting:', e);
                }
            }, 50);
            
            // Longer delay highlighting
            setTimeout(() => {
                try {
                    Prism.highlightAll();
                    //console.log('Full highlighting attempted after delay');
                } catch (e) {
                    console.error('Error in full delayed highlighting:', e);
                }
            }, 200);
        } else {
            console.warn('Prism not available for message highlighting');
        }
    }
},


/**
 * Process message content based on role (user or assistant) - ROBUST VERSION
 * 
 * @param {string} content The message content
 * @param {string} role The message role ('user' or 'assistant')
 * @return {string} Processed HTML content
 */
processMessageContent: function(content, role = 'assistant') {
    // For assistant messages, use existing logic
    if (role === 'assistant') {
        // If content appears to be already HTML, return it as is
        if (content.trim().charAt(0) === '<') {
            return content;
        }
        
        // Otherwise, use marked.js to convert markdown to HTML
        return marked.parse(content);
    }
    
    // For USER messages - be very careful about HTML rendering
    
    // Step 1: If this looks like it might be already processed HTML from storage,
    // we need to be more careful
    if (content.includes('<') && content.includes('>')) {
        // This might be stored HTML, let's check if it needs to be treated as text
        
        // If it contains code block HTML tags, it's likely properly formatted
        if (content.includes('<pre><code class="language-') && content.includes('</code></pre>')) {
            return content; // Already properly formatted
        }
        
        // If it contains <br> tags but no code blocks, it's likely formatted text
        if (content.includes('<br>') && !content.includes('<pre>')) {
            return content; // Already formatted
        }
        
        // Otherwise, treat it as raw text that needs to be escaped
        content = this.unescapeHtml(content);
    }
    
    // Step 2: Now we have raw text content, process it properly
    
    // Check for code blocks first
    if (content.includes('```')) {
        return this.processUserCodeBlocks(content);
    }
    
    // For regular user messages, escape HTML and preserve line breaks
    const escapedContent = this.escapeHtml(content);
    
    // Convert line breaks to <br> tags
    return escapedContent.replace(/\n/g, '<br>');
},

/**
 * Process code blocks in user input - IMPROVED VERSION
 * 
 * @param {string} content The user message content
 * @return {string} HTML with properly formatted code blocks
 */
processUserCodeBlocks: function(content) {
    // First, ensure content is unescaped
    content = this.unescapeHtml(content);
    
    // Split content by code blocks
    const codeBlockRegex = /```([a-zA-Z0-9_+-]*)\n([\s\S]*?)```/g;
    let result = '';
    let lastIndex = 0;
    let match;
    
    while ((match = codeBlockRegex.exec(content)) !== null) {
        // Add text before code block (escaped)
        const textBefore = content.substring(lastIndex, match.index);
        if (textBefore) {
            result += this.escapeHtml(textBefore).replace(/\n/g, '<br>');
        }
        
        // Add code block
        const language = match[1] || 'plaintext';
        const code = match[2];
        
        result += '<pre><code class="language-' + this.escapeHtml(language) + '">';
        result += this.escapeHtml(code);
        result += '</code></pre>';
        
        lastIndex = match.index + match[0].length;
    }
    
    // Add remaining text after last code block
    const textAfter = content.substring(lastIndex);
    if (textAfter) {
        result += this.escapeHtml(textAfter).replace(/\n/g, '<br>');
    }
    
    return result;
},

/**
 * Helper function to escape HTML - IMPROVED
 * 
 * @param {string} text Text to escape
 * @return {string} Escaped HTML
 */
escapeHtml: function(text) {
    if (typeof text !== 'string') {
        text = String(text);
    }
    
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
},

/**
 * Helper function to unescape HTML
 * 
 * @param {string} text Text to unescape
 * @return {string} Unescaped text
 */
unescapeHtml: function(text) {
    if (typeof text !== 'string') {
        text = String(text);
    }
    
    const div = document.createElement('div');
    div.innerHTML = text;
    return div.textContent || div.innerText || '';
},

/**
 * Check if content is already formatted HTML
 * 
 * @param {string} content Content to check
 * @return {boolean} True if content appears to be formatted HTML
 */
isFormattedHtml: function(content) {
    // Check for common HTML patterns that indicate formatted content
    const htmlPatterns = [
        /<pre><code class="language-/,
        /<\/code><\/pre>/,
        /<br>/,
        /<p>/,
        /<div>/
    ];
    
    return htmlPatterns.some(pattern => pattern.test(content));
},


/**
 * Add thinking indicator while waiting for response
 */
addThinkingIndicator: function() {
    const $thinking = $('<div class="mxchat-message mxchat-assistant mxchat-thinking"></div>');
    
    // Add content
    const $content = $('<div class="mxchat-message-content"></div>');
    
    // Add header
    const $header = $('<div class="mxchat-message-header"><span class="mxchat-message-sender">' + 
        mxChatAdminChat.i18n.assistant + '</span></div>');
    $content.append($header);
    
    // Add enhanced thinking animation
    const $thinkingContainer = $('<div class="mxchat-message-text"></div>');
    
    // Improved thinking animation with a pulsing effect
    const $thinkingIndicator = $('<div class="mxchat-thinking-indicator"></div>');
    
    // Add multiple dots for a better visual effect
    for (let i = 0; i < 3; i++) {
        $thinkingIndicator.append('<div class="mxchat-thinking-dot"></div>');
    }
    
    $thinkingContainer.append($thinkingIndicator);
    $content.append($thinkingContainer);
    
    $thinking.append($content);
    
    // Add to conversation
    this.conversation.append($thinking);
    
    // Start animation
    this.animateThinking();
    
    // Scroll to make sure the thinking indicator is visible
    this.scrollToBottom();
},

/**
 * Animate the thinking dots in sequence
 */
animateThinking: function() {
    let dotIndex = 0;
    const dots = $('.mxchat-thinking-dot');
    
    // Clear any existing animation interval
    if (this.thinkingAnimationInterval) {
        clearInterval(this.thinkingAnimationInterval);
    }
    
    // Create animation interval
    this.thinkingAnimationInterval = setInterval(() => {
        dots.removeClass('active');
        
        // Activate current dot
        dots.eq(dotIndex).addClass('active');
        
        // Move to next dot
        dotIndex = (dotIndex + 1) % dots.length;
    }, 400);
},

/**
 * Stop thinking animation
 */
stopThinkingAnimation: function() {
    if (this.thinkingAnimationInterval) {
        clearInterval(this.thinkingAnimationInterval);
        this.thinkingAnimationInterval = null;
    }
},
        /**
         * Remove thinking indicator
         */
        removeThinkingIndicator: function() {
            $('.mxchat-thinking').remove();
        },

        /**
         * Add error message
         * 
         * @param {string} errorMessage The error message
         */
        addErrorMessage: function(errorMessage) {
            const $errorMsg = $('<div class="mxchat-message mxchat-assistant mxchat-error"></div>');
            
            // Add content
            const $content = $('<div class="mxchat-message-content"></div>');
            
            // Add header
            const $header = $('<div class="mxchat-message-header"><span class="mxchat-message-sender">' + 
                mxChatAdminChat.i18n.assistant + '</span></div>');
            $content.append($header);
            
            // Add error text
            const $errorContainer = $('<div class="mxchat-message-text"></div>');
            $errorContainer.html('<p><strong>' + mxChatAdminChat.i18n.error + ':</strong> ' + errorMessage + '</p>');
            $content.append($errorContainer);
            
            $errorMsg.append($content);
            
            // Add to conversation
            this.conversation.append($errorMsg);
        },

        /**
         * Copy text to clipboard
         * 
         * @param {string} text The text to copy
         * @param {jQuery} $button The button element
         */
        copyToClipboard: function(text, $button) {
            const self = this;
            
            // Use modern clipboard API if available
            if (navigator.clipboard && window.isSecureContext) {
                navigator.clipboard.writeText(text)
                    .then(() => {
                        self.showCopiedFeedback($button);
                    })
                    .catch(err => {
                        this.showNotification(mxChatAdminChat.i18n.copy_error);
                    });
            } else {
                // Fallback for older browsers
                const $temp = $('<textarea>');
                $('body').append($temp);
                $temp.val(text).select();
                
                try {
                    // Execute copy command
                    document.execCommand('copy');
                    self.showCopiedFeedback($button);
                } catch (err) {
                    this.showNotification(mxChatAdminChat.i18n.copy_error);
                }
                
                // Remove temporary textarea
                $temp.remove();
            }
        },

        /**
         * Show copied feedback on button
         * 
         * @param {jQuery} $button The button element
         */
        showCopiedFeedback: function($button) {
            // Show copied indicator
            const $icon = $button.find('i');
            const originalClass = $icon.attr('class');
            
            $icon.attr('class', 'fa-solid fa-check');
            
            setTimeout(function() {
                $icon.attr('class', originalClass);
            }, 1500);
            
            // Show notification
            this.showNotification(mxChatAdminChat.i18n.copied);
        },
        
/**
 * Initialize model selector modal
 */
initModelSelectorModal: function() {
    // Create the modal HTML structure
    const modelSelectorModal = `
        <div id="mxchat-model-selector-modal" class="mxchat-model-selector-modal">
            <div class="mxchat-model-selector-modal-content">
                <div class="mxchat-model-selector-modal-header">
                    <h3>${mxChatAdminChat.i18n.select_model}</h3>
                    <span class="mxchat-model-selector-modal-close">&times;</span>
                </div>
                <div class="mxchat-model-selector-modal-body">
                    <div class="mxchat-model-selector-search-container">
                        <input type="text" id="mxchat-model-search-input" class="mxchat-model-search-input" placeholder="Search models...">
                    </div>
                    <div class="mxchat-model-selector-categories">
                        <button class="mxchat-model-category-btn active" data-category="all">All</button>
                        <button class="mxchat-model-category-btn" data-category="gemini">Google Gemini</button>
                        <button class="mxchat-model-category-btn" data-category="openai">OpenAI</button>
                        <button class="mxchat-model-category-btn" data-category="claude">Claude</button>
                        <button class="mxchat-model-category-btn" data-category="xai">X.AI</button>
                        <button class="mxchat-model-category-btn" data-category="deepseek">DeepSeek</button>
                    </div>
                    <div class="mxchat-model-selector-grid" id="mxchat-models-grid"></div>
                </div>
                <div class="mxchat-model-selector-modal-footer">
                    <button id="mxchat-cancel-model-selection" class="button mxchat-model-cancel-btn">${mxChatAdminChat.i18n.cancel}</button>
                </div>
            </div>
        </div>
    `;
    
    // Append modal to the body
    $('body').append(modelSelectorModal);
    
    // Define our complete model data based on the PHP function
    this.models = {
        gemini: [
            { value: 'gemini-2.0-flash', label: 'Gemini 2.0 Flash', description: 'Next-Gen Features' },
            { value: 'gemini-2.0-flash-lite', label: 'Gemini 2.0 Flash-Lite', description: 'Cost-Efficient' },
            { value: 'gemini-1.5-pro', label: 'Gemini 1.5 Pro', description: 'Complex Reasoning' },
            { value: 'gemini-1.5-flash', label: 'Gemini 1.5 Flash', description: 'Fast & Versatile' },
        ],
        xai: [
            { value: 'grok-code-fast-1', label: 'Grok Code Fast 1', description: 'Speedy and economical reasoning model that excels at agentic coding' },
            { value: 'grok-4-0709', label: 'Grok 4', description: 'Latest flagship model - unparalleled performance' },
            { value: 'grok-3-beta', label: 'Grok-3', description: 'Powerful model with 131K context' },
            { value: 'grok-3-fast-beta', label: 'Grok-3 Fast', description: 'High performance with faster responses' },
            { value: 'grok-3-mini-beta', label: 'Grok-3 Mini', description: 'Affordable model with good performance' },
            { value: 'grok-3-mini-fast-beta', label: 'Grok-3 Mini Fast', description: 'Quick and cost-effective' },
            { value: 'grok-2', label: 'Grok 2', description: 'Latest X.AI model' },
        ],
        deepseek: [
            { value: 'deepseek-chat', label: 'DeepSeek-V3', description: 'Advanced AI assistant' },
        ],
        claude: [
            { value: 'claude-4-opus-20250514', label: 'Claude 4 Opus', description: 'Most Powerful & Intelligent' },
            { value: 'claude-4-sonnet-20250514', label: 'Claude 4 Sonnet', description: 'Advanced Intelligence' },
            { value: 'claude-3-7-sonnet-20250219', label: 'Claude 3.7 Sonnet', description: 'Most Intelligent (v3)' },
            { value: 'claude-3-5-sonnet-20241022', label: 'Claude 3.5 Sonnet', description: 'Intelligent' },
            { value: 'claude-3-opus-20240229', label: 'Claude 3 Opus', description: 'Highly Complex Tasks' },
            { value: 'claude-3-sonnet-20240229', label: 'Claude 3 Sonnet', description: 'Balanced' },
            { value: 'claude-3-haiku-20240307', label: 'Claude 3 Haiku', description: 'Fastest' },
        ],
        openai: [
            { value: 'gpt-5', label: 'GPT-5', description: 'The best model for coding and agentic tasks across domains' },
            { value: 'gpt-5-mini', label: 'GPT-5 Mini', description: 'A faster, cost-efficient version of GPT-5 for well-defined tasks' },
            { value: 'gpt-5-nano', label: 'GPT-5 Nano', description: 'Fastest, most cost-efficient version of GPT-5' },
            { value: 'gpt-4.1-2025-04-14', label: 'GPT-4.1', description: 'Flagship model for complex tasks' },
            { value: 'gpt-4o', label: 'GPT-4o', description: 'Recommended for most use cases' },
            { value: 'gpt-4o-mini', label: 'GPT-4o Mini', description: 'Fast and lightweight' },
            { value: 'gpt-4-turbo', label: 'GPT-4 Turbo', description: 'High-performance model' },
            { value: 'gpt-4', label: 'GPT-4', description: 'High intelligence model' },
            { value: 'gpt-3.5-turbo', label: 'GPT-3.5 Turbo', description: 'Affordable and fast' },
        ],
    };
    
    // Load server models information for debugging
    const serverModels = mxChatAdminChat.models;
    //console.log('Server models:', serverModels);
    //console.log('Our predefined models:', this.models);
    
    // Set up event handlers
    this.setupModelSelectorEvents();
},

/**
 * Set up event handlers for model selector modal
 */
setupModelSelectorEvents: function() {
    const self = this;
    const $modal = $('#mxchat-model-selector-modal');
    const $modelSelectorBtn = $('.mxchat-model-selector-btn');
    
    // Open modal when clicking the model selector button
    $modelSelectorBtn.on('click', function() {
        $modal.show();
        self.populateModelsGrid('', 'all');
    });
    
    // Close modal when clicking the close button or cancel button
    $('.mxchat-model-selector-modal-close, #mxchat-cancel-model-selection').on('click', function() {
        $modal.hide();
    });
    
    // Close modal when clicking outside the modal content
    $(window).on('click', function(event) {
        if ($(event.target).is($modal)) {
            $modal.hide();
        }
    });
    
    // Category filtering
    $(document).on('click', '.mxchat-model-category-btn', function() {
        $('.mxchat-model-category-btn').removeClass('active');
        $(this).addClass('active');
        const category = $(this).data('category');
        const searchTerm = $('#mxchat-model-search-input').val();
        self.populateModelsGrid(searchTerm, category);
    });
    
    // Search functionality
    $(document).on('input', '#mxchat-model-search-input', function() {
        const searchTerm = $(this).val();
        const activeCategory = $('.mxchat-model-category-btn.active').data('category');
        self.populateModelsGrid(searchTerm, activeCategory);
    });
    
    // Model selection
    $(document).on('click', '.mxchat-model-selector-card', function() {
        const $card = $(this);
        const modelId = $card.data('value');
        const modelName = $card.find('.mxchat-model-selector-title').text();
        
        
        // Highlight selected card
        $('.mxchat-model-selector-card').removeClass('mxchat-model-selected');
        $card.addClass('mxchat-model-selected');
        
        // Remove existing checkmarks
        $('.mxchat-model-selector-checkmark').remove();
        
        // Add checkmark to selected card
        $card.append('<div class="mxchat-model-selector-checkmark">✓</div>');
        
        // Update the model
        self.updateModel(modelId, modelName);
        
        // Close the modal
        $modal.hide();
    });
},

/**
 * Populate the models grid based on filter and category
 * 
 * @param {string} filter   Search filter term
 * @param {string} category Category filter
 */
populateModelsGrid: function(filter = '', category = 'all') {
    const $grid = $('#mxchat-models-grid');
    $grid.empty();
    
    // Check if we have models
    if (!this.models || Object.keys(this.models).length === 0) {
        $grid.html('<div class="mxchat-no-models-message">' + mxChatAdminChat.i18n.empty_models + '</div>');
        return;
    }
    
    let allModels = [];
    
    // Collect models based on category
    Object.keys(this.models).forEach(key => {
        if (category === 'all' || category === key) {
            allModels = allModels.concat(this.models[key].map(model => {
                // Add category to each model object for better filtering
                return {...model, category: key};
            }));
        }
    });
    
    // Filter by search term if present
    if (filter) {
        const lowerFilter = filter.toLowerCase();
        allModels = allModels.filter(model => 
            model.label.toLowerCase().includes(lowerFilter) || 
            (model.description && model.description.toLowerCase().includes(lowerFilter)) ||
            (model.category && model.category.toLowerCase().includes(lowerFilter))
        );
    }
    
    // Show message if no models match the filter
    if (allModels.length === 0) {
        $grid.html('<div class="mxchat-no-models-message">No models match your search criteria</div>');
        return;
    }
    
    // Get current model ID
    const currentModelId = mxChatAdminChat.current_model ? mxChatAdminChat.current_model.id : '';
    
    // Create model cards
    allModels.forEach(model => {
        const isSelected = currentModelId === model.value;
        const providerLabel = this.getProviderLabel(model.category);
        
        const $modelCard = $(`
            <div class="mxchat-model-selector-card ${isSelected ? 'mxchat-model-selected' : ''}" data-value="${model.value}">
                <div class="mxchat-model-selector-icon">${this.getModelIcon(model.value)}</div>
                <div class="mxchat-model-selector-info">
                    <h4 class="mxchat-model-selector-title">${model.label}</h4>
                    <p class="mxchat-model-selector-description">
                        <span class="mxchat-model-provider">${providerLabel}</span>
                        ${model.description ? ' - ' + model.description : ''}
                    </p>
                </div>
                ${isSelected ? '<div class="mxchat-model-selector-checkmark">✓</div>' : ''}
            </div>
        `);
        $grid.append($modelCard);
    });
},

/**
 * Get formatted provider label
 * 
 * @param {string} category Category key
 * @return {string} Formatted provider label
 */
getProviderLabel: function(category) {
    const providers = {
        'gemini': 'Google Gemini',
        'openai': 'OpenAI',
        'claude': 'Claude',
        'xai': 'X.AI',
        'deepseek': 'DeepSeek'
    };
    
    return providers[category] || category;
},

/**
 * Get appropriate icon for model type
 * 
 * @param {string} modelValue Model ID
 * @return {string} HTML for the icon
 */
getModelIcon: function(modelValue) {
     if (modelValue.startsWith('gemini-')) return '<svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" viewBox="0 0 48 48" class="mxchat-model-icon-gemini"><defs><path id="a" d="M44.5 20H24v8.5h11.8C34.7 33.9 30.1 37 24 37c-7.2 0-13-5.8-13-13s5.8-13 13-13c3.1 0 5.9 1.1 8.1 2.9l6.4-6.4C34.6 4.1 29.6 2 24 2 11.8 2 2 11.8 2 24s9.8 22 22 22c11 0 21-8 21-22 0-1.3-.2-2.7-.5-4z"></path></defs><clipPath id="b"><use xlink:href="#a" overflow="visible"></use></clipPath><path clip-path="url(#b)" fill="#FBBC05" d="M0 37V11l17 13z"></path><path clip-path="url(#b)" fill="#EA4335" d="M0 11l17 13 7-6.1L48 14V0H0z"></path><path clip-path="url(#b)" fill="#34A853" d="M0 37l30-23 7.9 1L48 0v48H0z"></path><path clip-path="url(#b)" fill="#4285F4" d="M48 48L17 24l-4-3 35-10z"></path></svg>';
    if (modelValue.startsWith('gpt-')) return '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 320 320" class="mxchat-model-icon-openai"><path fill="currentColor" d="M297 131a80.6 80.6 0 0 0-93.7-104.2 80.6 80.6 0 0 0-137 29A80.6 80.6 0 0 0 23 189a80.6 80.6 0 0 0 93.7 104.2 80.6 80.6 0 0 0 137-29A80.7 80.7 0 0 0 297.1 131zM176.9 299c-14 .1-27.6-4.8-38.4-13.8l1.9-1 63.7-36.9c3.3-1.8 5.3-5.3 5.2-9v-89.9l27 15.6c.3.1.4.4.5.7v74.4a60 60 0 0 1-60 60zM47.9 244a59.7 59.7 0 0 1-7.1-40.1l1.9 1.1 63.7 36.8c3.2 1.9 7.2 1.9 10.5 0l77.8-45V228c0 .3-.2.6-.4.8L129.9 266a60 60 0 0 1-82-22zM31.2 105c7-12.2 18-21.5 31.2-26.3v75.8c0 3.7 2 7.2 5.2 9l77.8 45-27 15.5a1 1 0 0 1-.9 0L53.1 187a60 60 0 0 1-22-82zm221.2 51.5-77.8-45 27-15.5a1 1 0 0 1 .9 0l64.4 37.1a60 60 0 0 1-9.3 108.2v-75.8c0-3.7-2-7.2-5.2-9zm26.8-40.4-1.9-1.1-63.7-36.8a10.4 10.4 0 0 0-10.5 0L125.4 123V92c0-.3 0-.6.3-.8L190.1 54a60 60 0 0 1 89.1 62.1zm-168.5 55.4-27-15.5a1 1 0 0 1-.4-.7V80.9a60 60 0 0 1 98.3-46.1l-1.9 1L116 72.8a10.3 10.3 0 0 0-5.2 9v89.8zm14.6-31.5 34.7-20 34.6 20v40L160 200l-34.7-20z"></path></svg>';
    if (modelValue.startsWith('claude-')) return '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 256 176" fill="none" class="mxchat-model-icon-claude"><path fill="currentColor" d="m147.487 0l70.081 175.78H256L185.919 0zM66.183 106.221l23.98-61.774l23.98 61.774zM70.07 0L0 175.78h39.18l14.33-36.914h73.308l14.328 36.914h39.179L110.255 0z"></path></svg>';
    if (modelValue.startsWith('grok-')) return '<svg fill="currentColor" fill-rule="evenodd" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" class="mxchat-model-icon-xai"><path d="M6.469 8.776L16.512 23h-4.464L2.005 8.776H6.47zm-.004 7.9l2.233 3.164L6.467 23H2l4.465-6.324zM22 2.582V23h-3.659V7.764L22 2.582zM22 1l-9.952 14.095-2.233-3.163L17.533 1H22z"></path></svg>';
    if (modelValue.startsWith('deepseek-')) return '<svg height="1em" viewBox="0 0 24 24" width="1em" xmlns="http://www.w3.org/2000/svg" class="mxchat-model-icon-deepseek"><path d="M23.748 4.482c-.254-.124-.364.113-.512.234-.051.039-.094.09-.137.136-.372.397-.806.657-1.373.626-.829-.046-1.537.214-2.163.848-.133-.782-.575-1.248-1.247-1.548-.352-.156-.708-.311-.955-.65-.172-.241-.219-.51-.305-.774-.055-.16-.11-.323-.293-.35-.2-.031-.278.136-.356.276-.313.572-.434 1.202-.422 1.84.027 1.436.633 2.58 1.838 3.393.137.093.172.187.129.323-.082.28-.18.552-.266.833-.055.179-.137.217-.329.14a5.526 5.526 0 01-1.736-1.18c-.857-.828-1.631-1.742-2.597-2.458a11.365 11.365 0 00-.689-.471c-.985-.957.13-1.743.388-1.836.27-.098.093-.432-.779-.428-.872.004-1.67.295-2.687.684a3.055 3.055 0 01-.465.137 9.597 9.597 0 00-2.883-.102c-1.885.21-3.39 1.102-4.497 2.623C.082 8.606-.231 10.684.152 12.85c.403 2.284 1.569 4.175 3.36 5.653 1.858 1.533 3.997 2.284 6.438 2.14 1.482-.085 3.133-.284 4.994-1.86.47.234.962.327 1.78.397.63.059 1.236-.03 1.705-.128.735-.156.684-.837.419-.961-2.155-1.004-1.682-.595-2.113-.926 1.096-1.296 2.746-2.642 3.392-7.003.05-.347.007-.565 0-.845-.004-.17.035-.237.23-.256a4.173 4.173 0 001.545-.475c1.396-.763 1.96-2.015 2.093-3.517.02-.23-.004-.467-.247-.588zM11.581 18c-2.089-1.642-3.102-2.183-3.52-2.16-.392.024-.321.471-.235.763.09.288.207.486.371.739.114.167.192.416-.113.603-.673.416-1.842-.14-1.897-.167-1.361-.802-2.5-1.86-3.301-3.307-.774-1.393-1.224-2.887-1.298-4.482-.02-.386.093-.522.477-.592a4.696 4.696 0 011.529-.039c2.132.312 3.946 1.265 5.468 2.774.868.86 1.525 1.887 2.202 2.891.72 1.066 1.494 2.082 2.48 2.914.348.292.625.514.891.677-.802.09-2.14.11-3.054-.614zm1-6.44a.306.306 0 01.415-.287.302.302 0 01.2.288.306.306 0 01-.31.307.303.303 0 01-.304-.308zm3.11 1.596c-.2.081-.399.151-.59.16a1.245 1.245 0 01-.798-.254c-.274-.23-.47-.358-.552-.758a1.73 1.73 0 01.016-.588c.07-.327-.008-.537-.239-.727-.187-.156-.426-.199-.688-.199a.559.559 0 01-.254-.078c-.11-.054-.2-.19-.114-.358.028-.054.16-.186.192-.21.356-.202.767-.136 1.146.016.352.144.618.408 1.001.782.391.451.462.576.685.914.176.265.336.537.445.848.067.195-.019.354-.25.452z" fill="currentColor"></path></svg>';
    return '<span class="dashicons dashicons-admin-generic mxchat-model-icon-generic"></span>';
},

        /**
         * Scroll conversation to bottom
         */
        scrollToBottom: function() {
            if (this.conversation.length) {
                this.conversation.scrollTop(this.conversation[0].scrollHeight);
            }
        }
    };

    // Initialize when document is ready
    $(document).ready(function() {
        MxChatAdmin.init();
    });
    
        // Make MxChatAdmin globally accessible
    window.MxChatAdmin = MxChatAdmin;

})(jQuery);


jQuery(document).ready(function($) {
  // Mobile menu toggle
  $('.mxchat-menu-toggle').on('click', function() {
    $('.mxchat-mobile-menu').toggleClass('active');
  });
  
  // Map mobile buttons to desktop button functionality
  $('#mxchat-settings-btn-mobile').on('click', function() {
    $('#mxchat-settings-btn').trigger('click');
  });
  
  $('.mxchat-history-btn-mobile').on('click', function() {
    $('.mxchat-history-btn').trigger('click');
  });
  
  $('.mxchat-new-chat-btn-mobile').on('click', function() {
    $('.mxchat-new-chat-btn').trigger('click');
  });
  
  // Close menu when clicking outside
  $(document).on('click', function(e) {
    if (!$(e.target).closest('.mxchat-menu-toggle, .mxchat-mobile-menu').length) {
      $('.mxchat-mobile-menu').removeClass('active');
    }
  });
});