<?php
/**
 * MxChat Admin Chat UI Handler
 *
 * Manages the user interface for the admin chat.
 *
 * @since      1.0.0
 * @package    MxChat_Admin_Chat
 * @subpackage MxChat_Admin_Chat/includes
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

/**
 * UI handler class for Admin Chat
 *
 * This class manages the admin interface, rendering the chat panel,
 * and coordinating with the API handlers.
 *
 * @since      1.0.0
 * @package    MxChat_Admin_Chat
 * @subpackage MxChat_Admin_Chat/includes
 */
class MxChat_Admin_Chat_UI {

    /**
     * Perplexity API handler instance.
     *
     * @since    1.0.0
     * @access   protected
     * @var      MxChat_Admin_Chat_Perplexity_API    $perplexity_api    Perplexity API handler.
     */
    protected $perplexity_api;

/**
 * Perplexity Endpoint handler instance.
 *
 * @since    1.0.0
 * @access   protected
 * @var      MxChat_Perplexity_Endpoint    $perplexity_endpoint    Perplexity Endpoint handler.
 */
protected $perplexity_endpoint;

    /**
     * The loader that's responsible for maintaining and registering all hooks.
     *
     * @since    1.0.0
     * @access   protected
     * @var      MxChat_Admin_Chat_Loader    $loader    Maintains and registers all hooks.
     */
    protected $loader;

    /**
     * The core chat class instance.
     *
     * @since    1.0.0
     * @access   protected
     * @var      MxChat_Admin_Chat_Core    $chat    Core chat functionality.
     */
    protected $chat;

    /**
     * API handler instance.
     *
     * @since    1.0.0
     * @access   protected
     * @var      MxChat_Admin_Chat_API    $api    API handler.
     */
    protected $api;

/**
 * Initialize the class.
 *
 * @since    1.0.0
 * @param    MxChat_Admin_Chat_Loader    $loader    The loader for hooks.
 */
public function __construct($loader) {
    $this->loader = $loader;
    $this->chat = new MxChat_Admin_Chat_Core();

    // Register admin scripts and styles
    $this->loader->add_action('admin_enqueue_scripts', $this, 'enqueue_assets');

    // Create API handler instance
    $this->api = new MxChat_Admin_Chat_API($loader);

    // Initialize Perplexity API handler
    if (class_exists('MxChat_Admin_Chat_Perplexity_API')) {
        $this->perplexity_api = new MxChat_Admin_Chat_Perplexity_API($loader);

        // Initialize Perplexity endpoint
        if (class_exists('MxChat_Perplexity_Endpoint')) {
            $this->perplexity_endpoint = new MxChat_Perplexity_Endpoint($this->perplexity_api);
        }
    }
}
public function enqueue_assets($hook_suffix) {
    // Only load assets on our plugin page
    if (strpos($hook_suffix, 'mxchat-admin-chat') === false) {
        return;
    }

    // Enqueue Font Awesome
    wp_enqueue_style(
        'font-awesome',
        'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css',
        array(),
        '6.5.1'
    );

    // Enqueue Prism.js for syntax highlighting
    wp_enqueue_style(
        'prism-css',
        'https://cdnjs.cloudflare.com/ajax/libs/prism/1.29.0/themes/prism.min.css',
        array(),
        '1.29.0'
    );
    wp_enqueue_script(
        'prism-js',
        'https://cdnjs.cloudflare.com/ajax/libs/prism/1.29.0/prism.min.js',
        array('jquery'),
        '1.29.0',
        true
    );
    wp_enqueue_script(
        'prism-autoloader',
        'https://cdnjs.cloudflare.com/ajax/libs/prism/1.29.0/plugins/autoloader/prism-autoloader.min.js',
        array('prism-js'),
        '1.29.0',
        true
    );
    wp_enqueue_style(
        'prism-toolbar',
        'https://cdnjs.cloudflare.com/ajax/libs/prism/1.29.0/plugins/toolbar/prism-toolbar.min.css',
        array('prism-css'),
        '1.29.0'
    );
    wp_enqueue_script(
        'prism-toolbar',
        'https://cdnjs.cloudflare.com/ajax/libs/prism/1.29.0/plugins/toolbar/prism-toolbar.min.js',
        array('prism-js'),
        '1.29.0',
        true
    );
    wp_enqueue_script(
        'prism-copy',
        'https://cdnjs.cloudflare.com/ajax/libs/prism/1.29.0/plugins/copy-to-clipboard/prism-copy-to-clipboard.min.js',
        array('prism-toolbar'),
        '1.29.0',
        true
    );

    // Enqueue marked.js for markdown parsing
    wp_enqueue_script(
        'marked-js',
        'https://cdn.jsdelivr.net/npm/marked/marked.min.js',
        array(),
        '4.3.0',
        true
    );

    // Enqueue plugin styles
    wp_enqueue_style(
        'mxchat-admin-chat',
        MXCHAT_ADMIN_CHAT_PLUGIN_URL . 'admin/css/mxchat-admin-chat.css',
        array(),
        MXCHAT_ADMIN_CHAT_VERSION
    );

    // Add admin-specific styles
    wp_enqueue_style('wp-components');

    // Enqueue plugin scripts and add marked.js as a dependency
    wp_enqueue_script(
        'mxchat-admin-chat',
        MXCHAT_ADMIN_CHAT_PLUGIN_URL . 'admin/js/mxchat-admin-chat.js',
        array('jquery', 'wp-element', 'wp-components', 'wp-i18n', 'wp-api-fetch', 'marked-js'),
        MXCHAT_ADMIN_CHAT_VERSION,
        true
    );

    // Enqueue Perplexity styles and scripts
    wp_enqueue_style(
        'mxchat-perplexity',
        MXCHAT_ADMIN_CHAT_PLUGIN_URL . 'admin/css/mxchat-perplexity.css',
        array('mxchat-admin-chat'),
        MXCHAT_ADMIN_CHAT_VERSION
    );

    wp_enqueue_script(
        'mxchat-perplexity',
        MXCHAT_ADMIN_CHAT_PLUGIN_URL . 'admin/js/mxchat-perplexity.js',
        array('jquery', 'mxchat-admin-chat'),
        MXCHAT_ADMIN_CHAT_VERSION,
        true
    );

    // Add Perplexity data to script localization
    wp_localize_script(
        'mxchat-perplexity',
        'mxchatPerplexity',
        array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('mxchat_admin_perplexity_nonce'),
            'perplexity_enabled' => !empty(get_option('mxchat_perplexity_api_key', '')),
            'i18n' => array(
                'perplexity_searching' => __('Searching the web...', 'mxchat-admin-chat'),
                'perplexity_error' => __('Error searching the web', 'mxchat-admin-chat'),
                'perplexity_toggle_on' => __('Web search enabled', 'mxchat-admin-chat'),
                'perplexity_toggle_off' => __('Web search disabled', 'mxchat-admin-chat')
            )
        )
    );

    // Enqueue Image Generation styles and scripts
    wp_enqueue_style(
        'mxchat-image-gen',
        MXCHAT_ADMIN_CHAT_PLUGIN_URL . 'admin/css/mxchat-image-gen.css',
        array('mxchat-admin-chat'),
        MXCHAT_ADMIN_CHAT_VERSION
    );

    wp_enqueue_script(
        'mxchat-image-gen',
        MXCHAT_ADMIN_CHAT_PLUGIN_URL . 'admin/js/mxchat-image-gen.js',
        array('jquery', 'mxchat-admin-chat'),
        MXCHAT_ADMIN_CHAT_VERSION,
        true
    );

    wp_enqueue_script(
        'mxchat-settings',
        MXCHAT_ADMIN_CHAT_PLUGIN_URL . 'admin/js/mxchat-settings.js',
        array('jquery'),
        MXCHAT_ADMIN_CHAT_VERSION,
        true
    );

    wp_enqueue_script(
        'mxchat-image-analysis',
        MXCHAT_ADMIN_CHAT_PLUGIN_URL . 'admin/js/mxchat-image-analysis.js',
        array('jquery', 'mxchat-admin-chat'),
        MXCHAT_ADMIN_CHAT_VERSION,
        true
    );

    // **NEW** - Enqueue image editing functionality
    wp_enqueue_script(
        'mxchat-image-editing',
        MXCHAT_ADMIN_CHAT_PLUGIN_URL . 'admin/js/mxchat-image-editing.js',
        array('jquery', 'mxchat-admin-chat', 'mxchat-image-gen'),
        MXCHAT_ADMIN_CHAT_VERSION,
        true
    );

    // **UPDATED** - Enhanced localization for image generation with editing support
    $main_plugin_options = get_option('mxchat_options', array());
    wp_localize_script(
        'mxchat-image-gen',
        'mxchatImageGen',
        array(
            // Core settings
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('mxchat_admin_image_gen_nonce'),
            'image_gen_enabled' => !empty($main_plugin_options['api_key']) || !empty($main_plugin_options['gemini_api_key']),

            // Common settings
            'default_model' => get_option('mxchat_image_gen_model', 'openai'),

            // DALL-E 3 specific settings
            'default_size' => get_option('mxchat_image_size', '1024x1024'),
            'default_quality' => get_option('mxchat_image_quality', 'standard'),

            // xAI Grok specific settings
            'default_image_count' => get_option('mxchat_image_count', '1'),
            'default_response_format' => get_option('mxchat_image_response_format', 'url'),

            // GPT Image specific settings
            'default_gpt_image_size' => get_option('mxchat_gpt_image_size', '1024x1024'),
            'default_gpt_image_quality' => get_option('mxchat_gpt_image_quality', 'medium'),
            'default_gpt_image_format' => get_option('mxchat_gpt_image_format', 'png'),
            'default_gpt_image_background' => get_option('mxchat_gpt_image_background', 'opaque'),
            'default_gpt_image_compression' => get_option('mxchat_gpt_image_compression', '75'),

            // Gemini-specific settings
            'default_gemini_model' => 'gemini-2.5-flash-image-preview',

            // **NEW** - Image editing capabilities
            'supports_image_editing' => array('gemini'), // Models that support image editing
            'editing_enabled' => !empty($main_plugin_options['gemini_api_key']), // Gemini supports editing

            // **UPDATED** - Enhanced translation strings
            'i18n' => array(
                'generating_image' => __('Generating image...', 'mxchat-admin-chat'),
                'editing_image' => __('Editing image...', 'mxchat-admin-chat'),
                'image_gen_error' => __('Error generating image', 'mxchat-admin-chat'),
                'image_edit_error' => __('Error editing image', 'mxchat-admin-chat'),
                'image_gen_prompt' => __('Enter a description for the image...', 'mxchat-admin-chat'),
                'image_edit_prompt' => __('Describe the changes you want to make...', 'mxchat-admin-chat'),
                'image_gen_toggle_on' => __('Image generation enabled', 'mxchat-admin-chat'),
                'image_gen_toggle_off' => __('Image generation disabled', 'mxchat-admin-chat'),
                'image_edit_mode' => __('Edit Mode', 'mxchat-admin-chat'),
                'image_analyze_mode' => __('Analyze Mode', 'mxchat-admin-chat'),
                'switch_to_edit' => __('Switch to edit mode', 'mxchat-admin-chat'),
                'switch_to_analyze' => __('Switch to analyze mode', 'mxchat-admin-chat'),
                'upload_image_to_edit' => __('Upload an image to edit with Gemini AI', 'mxchat-admin-chat'),
                'gemini_editing_only' => __('Image editing is only available with Gemini models', 'mxchat-admin-chat')
            )
        )
    );

    // Rest of the existing localize_script calls...
    wp_localize_script(
        'mxchat-admin-chat',
        'mxChatAdminChat',
        array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('mxchat_admin_chat_nonce'),
            'models' => $this->chat->get_available_models(),
            'current_model' => $this->chat->get_current_model(),
            'i18n' => array(
                'sending' => __('Sending...', 'mxchat-admin-chat'),
                'error' => __('Error', 'mxchat-admin-chat'),
                'clear_confirm' => __('Are you sure you want to clear the conversation?', 'mxchat-admin-chat'),
                'clear_success' => __('Conversation cleared', 'mxchat-admin-chat'),
                'update_success' => __('Model updated', 'mxchat-admin-chat'),
                'loading' => __('Loading...', 'mxchat-admin-chat'),
                'thinking' => __('Thinking...', 'mxchat-admin-chat'),
                'you' => __('You', 'mxchat-admin-chat'),
                'assistant' => __('Assistant', 'mxchat-admin-chat'),
                'welcome_message' => __('Welcome! How can I assist you today?', 'mxchat-admin-chat'),
                'rate_limit' => __('Please wait a moment before sending another message', 'mxchat-admin-chat'),
                'type_message' => __('Type a message...', 'mxchat-admin-chat'),
                'send' => __('Send', 'mxchat-admin-chat'),
                'clear' => __('Clear conversation', 'mxchat-admin-chat'),
                'new_chat' => __('New chat', 'mxchat-admin-chat'),
                'model_select' => __('Select model', 'mxchat-admin-chat'),
                'select_model' => __('Select an AI model', 'mxchat-admin-chat'),
                'copy' => __('Copy to clipboard', 'mxchat-admin-chat'),
                'copied' => __('Copied!', 'mxchat-admin-chat'),
                'copy_error' => __('Failed to copy', 'mxchat-admin-chat'),
                'retry' => __('Retry', 'mxchat-admin-chat'),
                'upload_pdf' => __('Upload PDF', 'mxchat-admin-chat'),
                'upload_word' => __('Upload Word Document', 'mxchat-admin-chat'),
                'search_web' => __('Search Web', 'mxchat-admin-chat'),
                'brave_search' => __('Brave Search', 'mxchat-admin-chat'),
                'image_generation' => __('Generate Image', 'mxchat-admin-chat'),
                'settings' => __('Settings', 'mxchat-admin-chat'),
                'settings_title' => __('MxChat Settings', 'mxchat-admin-chat'),
                'settings_saved' => __('Settings saved successfully', 'mxchat-admin-chat'),
                'saving' => __('Saving...', 'mxchat-admin-chat'),
                'image_models' => __('Image Generation Models', 'mxchat-admin-chat'),
                'openai_image' => __('OpenAI DALL-E', 'mxchat-admin-chat'),
                'grok_image' => __('Grok (Coming Soon)', 'mxchat-admin-chat'),
                'empty_models' => __('No AI models available. Please check your API keys in the MxChat settings.', 'mxchat-admin-chat'),
                'title' => __('Admin AI Chat', 'mxchat-admin-chat'),
                'subtitle' => __('Chat with AI using your configured API keys', 'mxchat-admin-chat'),
                'no_models' => __('No AI models available', 'mxchat-admin-chat'),
                'no_models_desc' => __('Please configure API keys in the MxChat settings to enable AI models.', 'mxchat-admin-chat'),
                'go_to_settings' => __('Go to MxChat Settings', 'mxchat-admin-chat'),
                'loading_models' => __('Loading available models...', 'mxchat-admin-chat'),
                'loading_conversation' => __('Loading conversation...', 'mxchat-admin-chat'),
                'error_loading' => __('Error loading data', 'mxchat-admin-chat'),
                'try_again' => __('Try Again', 'mxchat-admin-chat'),
                'using_model' => __('Using model', 'mxchat-admin-chat'),
                'change_model' => __('Change model', 'mxchat-admin-chat'),
                'save' => __('Save', 'mxchat-admin-chat'),
                'cancel' => __('Cancel', 'mxchat-admin-chat'),
                'how_can_help' => __('How can MxChat help?', 'mxchat-admin-chat'),
            )
        )
    );
}

/**
 * Format message content - ROBUST VERSION
 * 
 * @since    1.0.0
 * @param    mixed     $content    Message content to format
 * @param    string    $role       Message role (user/assistant)
 * @return   string                Formatted content
 */
public function format_message_content($content, $role = 'assistant') {
    // Check if content is a WP_Error object
    if (is_wp_error($content)) {
        $error_message = $content->get_error_message();
        return '<div class="mxchat-error-message">' . esc_html($error_message) . '</div>';
    }
    
    // Ensure content is a string
    if (!is_string($content)) {
        if (is_array($content) || is_object($content)) {
            $content = print_r($content, true);
        } else {
            $content = (string)$content;
        }
    }

    // For assistant messages, use existing logic
    if ($role === 'assistant') {
        // If content appears to be already HTML, return it as is
        if (strpos(trim($content), '<') === 0) {
            return $content;
        }
        
        // Process markdown-style code blocks
        $content = preg_replace_callback(
            '/```([a-zA-Z0-9_+-]+)?\s*\n(.*?)\n```/s',
            function($matches) {
                $language = !empty($matches[1]) ? $matches[1] : 'plaintext';
                $code = $matches[2];
                return '<pre><code class="language-' . esc_attr($language) . '">' . esc_html($code) . '</code></pre>';
            },
            $content
        );
        
        // Process inline code
        $content = preg_replace_callback(
            '/`([^`]+)`/',
            function($matches) {
                return '<code class="language-none">' . esc_html($matches[1]) . '</code>';
            },
            $content
        );
        
        return $content;
    }
    
    // For USER messages - be very strict about preventing HTML injection
    
    // Step 1: Check if this is already properly formatted HTML from storage
    if ($this->is_formatted_user_html($content)) {
        return $content; // Already safely formatted
    }
    
    // Step 2: Treat as raw user input that needs to be safely processed
    
    // If content contains HTML entities or tags, decode/strip them first
    if (strpos($content, '&') !== false || strpos($content, '<') !== false) {
        // Decode HTML entities first
        $content = html_entity_decode($content, ENT_QUOTES | ENT_HTML5, 'UTF-8');
        // Strip any remaining HTML tags (security measure)
        $content = strip_tags($content);
    }
    
    // Process code blocks if present
    if (strpos($content, '```') !== false) {
        return $this->process_user_code_blocks_safe($content);
    }
    
    // For regular text, escape and add line breaks
    $escaped = esc_html($content);
    return nl2br($escaped);
}

/**
 * Check if content is already formatted HTML for user messages
 * 
 * @param string $content Content to check
 * @return bool True if content is already safely formatted
 */
private function is_formatted_user_html($content) {
    // Check for patterns that indicate this is already formatted HTML
    $safe_patterns = [
        '/<pre><code class="language-[^"]*">/',
        '/<\/code><\/pre>/',
    ];
    
    foreach ($safe_patterns as $pattern) {
        if (preg_match($pattern, $content)) {
            // Additional safety check - make sure no dangerous HTML
            if (!preg_match('/<script|<iframe|<object|<embed|javascript:/i', $content)) {
                return true;
            }
        }
    }
    
    // Check for simple formatted text (just br tags)
    if (strpos($content, '<br>') !== false && 
        !preg_match('/<(?!br\s*\/?>)[^>]+>/i', $content)) {
        return true;
    }
    
    return false;
}

/**
 * Safely process code blocks in user input
 * 
 * @param string $content User input with code blocks
 * @return string Safely formatted HTML
 */
private function process_user_code_blocks_safe($content) {
    // Split by code blocks
    $parts = preg_split('/(```[a-zA-Z0-9_+-]*\n[\s\S]*?\n```)/s', $content, -1, PREG_SPLIT_DELIM_CAPTURE);
    
    $result = '';
    
    foreach ($parts as $part) {
        if (preg_match('/```([a-zA-Z0-9_+-]*)\n([\s\S]*?)\n```/s', $part, $matches)) {
            // This is a code block
            $language = !empty($matches[1]) ? $matches[1] : 'plaintext';
            $code = $matches[2];
            
            // Safely format the code block
            $result .= '<pre><code class="language-' . esc_attr($language) . '">';
            $result .= esc_html($code);
            $result .= '</code></pre>';
        } else {
            // This is regular text - escape and preserve line breaks
            if (trim($part) !== '') {
                $result .= nl2br(esc_html($part));
            }
        }
    }
    
    return $result;
}

/**
 * Clean and save message content to prevent HTML injection
 * 
 * @param string $role Message role
 * @param string $content Message content  
 * @return string Cleaned content safe for storage
 */
public function clean_message_for_storage($role, $content) {
    if ($role === 'user') {
        // For user messages, always store as plain text
        // Strip any HTML tags and decode entities
        $content = html_entity_decode(strip_tags($content), ENT_QUOTES | ENT_HTML5, 'UTF-8');
    }
    // For assistant messages, store as-is since they come from trusted sources
    
    return $content;
}


/**
 * Display the admin chat interface
 *
 * @since    1.0.0
 */
public function display_chat_interface() {
    // Ensure user has permission
    if (!current_user_can('manage_options')) {
        wp_die(__('You do not have sufficient permissions to access this page.', 'mxchat-admin-chat'));
    }
    
    $current_model = $this->chat->get_current_model();
    $available_models = $this->chat->get_available_models();
    $has_models = !empty($available_models);
    $conversation = $this->chat->get_conversation_history();
    
    // Output the chat interface
    echo '<div class="mxchat-page-wrapper">';
    // Header with logo and action buttons
    echo '<div class="mxchat-header">';
    // Logo section
    echo '<div class="mxchat-logo">';
    echo '<img src="' . esc_url(MXCHAT_ADMIN_CHAT_PLUGIN_URL . 'assets/images/icon-128x128.png') . '" alt="MxChat Logo" />';
    echo '<h2>' . esc_html__('MxChat', 'mxchat-admin-chat') . '</h2>';
    echo '</div>';

    // Desktop action buttons - keep original IDs for functionality
    echo '<div class="mxchat-actions">';
    // Settings button
    echo '<button class="mxchat-action-btn settings" id="mxchat-settings-btn">';
    echo '<i class="fa-solid fa-gear"></i> ' . esc_html__('Settings', 'mxchat-admin-chat');
    echo '</button>';
    // History button
    echo '<button class="mxchat-action-btn history mxchat-history-btn">';
    echo '<i class="fa-solid fa-clock-rotate-left"></i> ' . esc_html__('History', 'mxchat-admin-chat');
    echo '</button>';
    // New chat button
    echo '<button class="mxchat-action-btn new mxchat-new-chat-btn">';
    echo '<i class="fa-solid fa-plus"></i> ' . esc_html__('New', 'mxchat-admin-chat');
    echo '</button>';
    echo '</div>';

    // Mobile menu toggle
    echo '<button class="mxchat-menu-toggle">';
    echo '<i class="fa-solid fa-ellipsis-vertical"></i>';
    echo '</button>';

    // Mobile dropdown menu (hidden by default)
    echo '<div class="mxchat-mobile-menu">';
    echo '<button class="mxchat-action-btn settings" id="mxchat-settings-btn-mobile">';
    echo '<i class="fa-solid fa-gear"></i> ' . esc_html__('Settings', 'mxchat-admin-chat');
    echo '</button>';
    echo '<button class="mxchat-action-btn history mxchat-history-btn-mobile">';
    echo '<i class="fa-solid fa-clock-rotate-left"></i> ' . esc_html__('History', 'mxchat-admin-chat');
    echo '</button>';
    echo '<button class="mxchat-action-btn new mxchat-new-chat-btn-mobile">';
    echo '<i class="fa-solid fa-plus"></i> ' . esc_html__('New', 'mxchat-admin-chat');
    echo '</button>';
    echo '</div>';
    echo '</div>';
    
    if (!$has_models) {
        echo $this->generate_no_models_message();
    } else {
        echo '<div class="mxchat-content-area">';
        
        // Main chat area
        echo '<div class="mxchat-conversation" id="mxchat-conversation">';
        echo $this->generate_conversation_html($conversation);
        echo '</div>';
        
        echo '<div class="mxchat-input-area">';
        echo '<div class="mxchat-input-container">';

        // **UPDATED** - Enhanced image preview area with editing capabilities
        echo '<div class="mxchat-image-preview-area" id="mxchat-image-preview-area" style="display: none;">';
        echo '<div class="mxchat-image-preview-content">';
        echo '<div class="mxchat-image-preview-header">';
        echo '<span class="mxchat-image-preview-title" id="mxchat-image-preview-title">' . esc_html__('Image for Analysis', 'mxchat-admin-chat') . '</span>';
        echo '<div class="mxchat-image-preview-controls">';
        echo '<button type="button" class="mxchat-image-mode-btn" id="mxchat-image-mode-toggle" title="' . esc_attr__('Switch between analyze and edit modes', 'mxchat-admin-chat') . '">';
        echo '<i class="fa-solid fa-eye"></i> ' . esc_html__('Analyze', 'mxchat-admin-chat');
        echo '</button>';
        echo '<button type="button" class="mxchat-image-remove-btn" id="mxchat-image-remove-btn">';
        echo '<i class="fa-solid fa-times"></i>';
        echo '</button>';
        echo '</div>';
        echo '</div>';
        echo '<div class="mxchat-image-preview-wrapper">';
        echo '<img id="mxchat-image-preview" class="mxchat-image-preview" src="" alt="Image preview" />';
        echo '</div>';
        echo '<div class="mxchat-image-preview-info" id="mxchat-image-preview-info" style="display: none;">';
        echo '<p class="mxchat-image-mode-description">' . esc_html__('Describe the changes you want to make to this image...', 'mxchat-admin-chat') . '</p>';
        echo '</div>';
        echo '</div>';
        echo '</div>';

        // The main input form with textfield, tools and send button
        echo '<form id="mxchat-message-form">';

        // Top row: Text input with send button
        echo '<div class="mxchat-input-row">';

        // Text input wrapper
        echo '<div class="mxchat-input-field-wrapper">';
        // Actual textarea with placeholder attribute
        echo '<textarea id="mxchat-message-input" class="mxchat-message-input" name="message" placeholder="' . esc_attr__('How can MxChat help?', 'mxchat-admin-chat') . '" aria-label="Ask MxChat anything" rows="1"></textarea>';
        echo '</div>';

        // Send button
        echo '<div class="mxchat-send-btn-wrapper">';
        echo '<button type="submit" class="mxchat-send-btn" id="mxchat-send-btn" aria-label="' . esc_attr__('Submit', 'mxchat-admin-chat') . '">';
        echo '<i class="fa-solid fa-paper-plane"></i>';
        echo '</button>';
        echo '</div>';

        echo '</div>'; // End input row

        // Bottom row: Tools on left, model selector on right
        echo '<div class="mxchat-bottom-row">';

        // Tool buttons on left
        echo '<div class="mxchat-tools">';

        // **UPDATED** - Enhanced image upload button with mode awareness
        echo '<div class="mxchat-tool-button mxchat-image-upload-toggle" title="' . esc_attr__('Upload or paste image for analysis/editing (Alt+U)', 'mxchat-admin-chat') . '">';
        echo '<i class="fa-solid fa-camera"></i>';
        echo '<input type="file" id="mxchat-image-upload" class="mxchat-image-upload-input" accept="image/*" style="display: none;" />';
        echo '</div>';

        echo '<div class="mxchat-tool-button mxchat-perplexity-toggle" title="' . esc_attr__('Toggle web search (Alt+P)', 'mxchat-admin-chat') . '">';
        echo '<svg fill="currentColor" height="1em" viewBox="0 0 24 24" width="1em" xmlns="http://www.w3.org/2000/svg">';
        echo '<path d="M19.785 0v7.272H22.5V17.62h-2.935V24l-7.037-6.194v6.145h-1.091v-6.152L4.392 24v-6.465H1.5V7.188h2.884V0l7.053 6.494V.19h1.09v6.49L19.786 0zm-7.257 9.044v7.319l5.946 5.234V14.44l-5.946-5.397zm-1.099-.08l-5.946 5.398v7.235l5.946-5.234V8.965zm8.136 7.58h1.844V8.349H13.46l6.105 5.54v2.655zm-8.982-8.28H2.59v8.195h1.8v-2.576l6.192-5.62zM5.475 2.476v4.71h5.115l-5.115-4.71zm13.219 0l-5.115 4.71h5.115v-4.71z"></path>';
        echo '</svg>';
        echo '</div>';

        // Add image generation toggle button (similar to Perplexity toggle)
        echo '<div class="mxchat-tool-button mxchat-image-gen-toggle" title="' . esc_attr__('Toggle image generation (Alt+I)', 'mxchat-admin-chat') . '">';
        echo '<i class="fa-solid fa-image"></i>';
        echo '</div>';

        echo '</div>'; // End tools
        
        // Model selector button (right side)
        echo '<div class="mxchat-model-selector" id="mxchat-model-selector">';
        $model_name = $current_model ? $current_model['name'] : __('Select Model', 'mxchat-admin-chat');
        echo '<button type="button" class="mxchat-model-selector-btn">';
        echo '<span>' . esc_html($model_name) . '</span>';
        echo '<i class="fa-solid fa-chevron-down"></i>';
        echo '</button>';
        echo '</div>'; // End model selector
        
        echo '</div>'; // End bottom row
        
        echo '</form>';
        echo '</div>'; // End input container
        echo '</div>'; // End input area
        
        echo '</div>'; // End content area
        
        // Add settings modal (hidden by default)
        echo $this->generate_settings_modal();
        
    }
    
    echo '</div>'; // End page wrapper
}


/**
 * Generate the settings modal HTML
 *
 * @since    1.0.0
 * @return   string    HTML for settings modal
 */
  private function generate_settings_modal() {
        // Get saved options with defaults
        $image_gen_model = get_option('mxchat_image_gen_model', 'openai');
        $image_size = get_option('mxchat_image_size', '1024x1024');
        $image_quality = get_option('mxchat_image_quality', 'standard');
        $image_count = get_option('mxchat_image_count', '1');
        $image_response_format = get_option('mxchat_image_response_format', 'url');
        $perplexity_model = get_option('mxchat_perplexity_model', 'pplx-7b-online');

        $html = '<div id="mxchat-settings-modal" class="mxchat-modal-settings" style="display: none;">';
        $html .= '<div class="mxchat-modal-content-settings">';
        
        // Modal header
        $html .= '<div class="mxchat-modal-header">';
        $html .= '<h2>' . esc_html__('MxChat Settings', 'mxchat-admin-chat') . '</h2>';
        $html .= '<button type="button" class="mxchat-modal-close" id="mxchat-settings-close">&times;</button>';
        $html .= '</div>';
        
        // Modal body
        $html .= '<div class="mxchat-modal-body">';
        
        // Form for settings
        $html .= '<form id="mxchat-settings-form">';
        
        // Image Analysis section (unchanged)
        $html .= '<div class="mxchat-settings-section">';
        $html .= '<h3>' . esc_html__('Image Analysis', 'mxchat-admin-chat') . '</h3>';

        // Get current image analysis model setting
        $image_analysis_model = get_option('mxchat_image_analysis_model', 'grok-vision');

        // Image analysis model selection
        $html .= '<div class="mxchat-settings-field">';
        $html .= '<label for="mxchat-image-analysis-model">' . esc_html__('Vision Model for Image Analysis', 'mxchat-admin-chat') . '</label>';
        $html .= '<select id="mxchat-image-analysis-model" name="image_analysis_model">';
        $html .= '<option value="grok-vision"' . selected($image_analysis_model, 'grok-vision', false) . '>' . esc_html__('Grok 2 Vision (xAI)', 'mxchat-admin-chat') . '</option>';
        $html .= '<option value="gpt-4-vision"' . selected($image_analysis_model, 'gpt-4-vision', false) . '>' . esc_html__('GPT-4 Vision (OpenAI)', 'mxchat-admin-chat') . '</option>';
        $html .= '</select>';
        $html .= '</div>';

        // Add helpful note about vision models
        $html .= '<div class="mxchat-settings-field">';
        $html .= '<p class="mxchat-settings-note">' . esc_html__('Note: Make sure you have the appropriate API key configured for your selected vision model.', 'mxchat-admin-chat') . '</p>';
        $html .= '</div>';

        $html .= '</div>'; // End image analysis section

        // **UPDATED** - Image Generation section: Added Gemini as a new option.
        // Added Gemini-specific note (no extra fields, as Gemini uses text prompts only per docs).
        $html .= '<div class="mxchat-settings-section">';
        $html .= '<h3>' . esc_html__('Image Generation', 'mxchat-admin-chat') . '</h3>';
        
        // Image generation model selection - **UPDATED** with Gemini option
        $html .= '<div class="mxchat-settings-field">';
        $html .= '<label for="mxchat-image-gen-model">' . esc_html__('Image Model', 'mxchat-admin-chat') . '</label>';
        $html .= '<select id="mxchat-image-gen-model" name="image_gen_model">';
        $html .= '<option value="openai"' . selected($image_gen_model, 'openai', false) . '>' . esc_html__('OpenAI DALL-E 3', 'mxchat-admin-chat') . '</option>';
        $html .= '<option value="gpt-image"' . selected($image_gen_model, 'gpt-image', false) . '>' . esc_html__('OpenAI GPT Image', 'mxchat-admin-chat') . '</option>';
        $html .= '<option value="xai"' . selected($image_gen_model, 'xai', false) . '>' . esc_html__('xAI Grok-2-Image', 'mxchat-admin-chat') . '</option>';
        $html .= '<option value="gemini"' . selected($image_gen_model, 'gemini', false) . '>' . esc_html__('Google Gemini Nano Banana', 'mxchat-admin-chat') . '</option>'; // **NEW** Gemini option
        $html .= '</select>';
        $html .= '</div>';
        
        // OpenAI DALL-E specific settings (unchanged)
        $html .= '<div class="mxchat-settings-field openai-specific"' . ($image_gen_model !== 'openai' ? 'style="display:none;"' : '') . '>';
        $html .= '<label for="mxchat-image-size">' . esc_html__('Image Size', 'mxchat-admin-chat') . '</label>';
        $html .= '<select id="mxchat-image-size" name="image_size">';
        $html .= '<option value="1024x1024"' . selected($image_size, '1024x1024', false) . '>' . esc_html__('Square (1024x1024)', 'mxchat-admin-chat') . '</option>';
        $html .= '<option value="1792x1024"' . selected($image_size, '1792x1024', false) . '>' . esc_html__('Landscape (1792x1024)', 'mxchat-admin-chat') . '</option>';
        $html .= '<option value="1024x1792"' . selected($image_size, '1024x1792', false) . '>' . esc_html__('Portrait (1024x1792)', 'mxchat-admin-chat') . '</option>';
        $html .= '</select>';
        $html .= '</div>';
        
        // OpenAI DALL-E specific settings - quality (unchanged)
        $html .= '<div class="mxchat-settings-field openai-specific"' . ($image_gen_model !== 'openai' ? 'style="display:none;"' : '') . '>';
        $html .= '<label for="mxchat-image-quality">' . esc_html__('Image Quality', 'mxchat-admin-chat') . '</label>';
        $html .= '<select id="mxchat-image-quality" name="image_quality">';
        $html .= '<option value="standard"' . selected($image_quality, 'standard', false) . '>' . esc_html__('Standard', 'mxchat-admin-chat') . '</option>';
        $html .= '<option value="hd"' . selected($image_quality, 'hd', false) . '>' . esc_html__('HD', 'mxchat-admin-chat') . '</option>';
        $html .= '</select>';
        $html .= '</div>';
        
        // xAI Grok specific settings - number of images (unchanged)
        $html .= '<div class="mxchat-settings-field xai-specific"' . ($image_gen_model !== 'xai' ? 'style="display:none;"' : '') . '>';
        $html .= '<label for="mxchat-image-count">' . esc_html__('Number of Images (1-10)', 'mxchat-admin-chat') . '</label>';
        $html .= '<input type="number" id="mxchat-image-count" name="image_count" min="1" max="10" value="' . esc_attr($image_count) . '">';
        $html .= '</div>';
        
        // xAI Grok specific settings - response format (unchanged)
        $html .= '<div class="mxchat-settings-field xai-specific"' . ($image_gen_model !== 'xai' ? 'style="display:none;"' : '') . '>';
        $html .= '<label for="mxchat-image-response-format">' . esc_html__('Response Format', 'mxchat-admin-chat') . '</label>';
        $html .= '<select id="mxchat-image-response-format" name="image_response_format">';
        $html .= '<option value="url"' . selected($image_response_format, 'url', false) . '>' . esc_html__('URL', 'mxchat-admin-chat') . '</option>';
        $html .= '<option value="b64_json"' . selected($image_response_format, 'b64_json', false) . '>' . esc_html__('Base64 JSON', 'mxchat-admin-chat') . '</option>';
        $html .= '</select>';
        $html .= '</div>';
        
        // xAI note about limitations (unchanged)
        $html .= '<div class="mxchat-settings-field xai-specific"' . ($image_gen_model !== 'xai' ? 'style="display:none;"' : '') . '>';
        $html .= '<p class="mxchat-settings-note">' . esc_html__('Note: xAI Grok does not support quality, size, or style parameters.', 'mxchat-admin-chat') . '</p>';
        $html .= '</div>';
        
        // GPT Image specific settings (unchanged)
        $html .= '<div class="mxchat-settings-field gpt-image-specific"' . ($image_gen_model !== 'gpt-image' ? 'style="display:none;"' : '') . '>';
        $html .= '<label for="mxchat-gpt-image-size">' . esc_html__('Image Size', 'mxchat-admin-chat') . '</label>';
        $html .= '<select id="mxchat-gpt-image-size" name="gpt_image_size">';
        $html .= '<option value="1024x1024"' . selected(get_option('mxchat_gpt_image_size', '1024x1024'), '1024x1024', false) . '>' . esc_html__('Square (1024x1024)', 'mxchat-admin-chat') . '</option>';
        $html .= '<option value="1024x1536"' . selected(get_option('mxchat_gpt_image_size', '1024x1024'), '1024x1536', false) . '>' . esc_html__('Portrait (1024x1536)', 'mxchat-admin-chat') . '</option>';
        $html .= '<option value="1536x1024"' . selected(get_option('mxchat_gpt_image_size', '1024x1024'), '1536x1024', false) . '>' . esc_html__('Landscape (1536x1024)', 'mxchat-admin-chat') . '</option>';
        $html .= '<option value="auto"' . selected(get_option('mxchat_gpt_image_size', '1024x1024'), 'auto', false) . '>' . esc_html__('Auto (Model decides)', 'mxchat-admin-chat') . '</option>';
        $html .= '</select>';
        $html .= '</div>';

        // GPT Image specific settings - quality (unchanged)
        $html .= '<div class="mxchat-settings-field gpt-image-specific"' . ($image_gen_model !== 'gpt-image' ? 'style="display:none;"' : '') . '>';
        $html .= '<label for="mxchat-gpt-image-quality">' . esc_html__('Image Quality', 'mxchat-admin-chat') . '</label>';
        $html .= '<select id="mxchat-gpt-image-quality" name="gpt_image_quality">';
        $html .= '<option value="low"' . selected(get_option('mxchat_gpt_image_quality', 'medium'), 'low', false) . '>' . esc_html__('Low', 'mxchat-admin-chat') . '</option>';
        $html .= '<option value="medium"' . selected(get_option('mxchat_gpt_image_quality', 'medium'), 'medium', false) . '>' . esc_html__('Medium', 'mxchat-admin-chat') . '</option>';
        $html .= '<option value="high"' . selected(get_option('mxchat_gpt_image_quality', 'medium'), 'high', false) . '>' . esc_html__('High', 'mxchat-admin-chat') . '</option>';
        $html .= '<option value="auto"' . selected(get_option('mxchat_gpt_image_quality', 'medium'), 'auto', false) . '>' . esc_html__('Auto (Model decides)', 'mxchat-admin-chat') . '</option>';
        $html .= '</select>';
        $html .= '</div>';

        // GPT Image specific settings - format (unchanged)
        $html .= '<div class="mxchat-settings-field gpt-image-specific"' . ($image_gen_model !== 'gpt-image' ? 'style="display:none;"' : '') . '>';
        $html .= '<label for="mxchat-gpt-image-format">' . esc_html__('Output Format', 'mxchat-admin-chat') . '</label>';
        $html .= '<select id="mxchat-gpt-image-format" name="gpt_image_format">';
        $html .= '<option value="png"' . selected(get_option('mxchat_gpt_image_format', 'png'), 'png', false) . '>' . esc_html__('PNG', 'mxchat-admin-chat') . '</option>';
        $html .= '<option value="jpeg"' . selected(get_option('mxchat_gpt_image_format', 'png'), 'jpeg', false) . '>' . esc_html__('JPEG', 'mxchat-admin-chat') . '</option>';
        $html .= '<option value="webp"' . selected(get_option('mxchat_gpt_image_format', 'png'), 'webp', false) . '>' . esc_html__('WebP', 'mxchat-admin-chat') . '</option>';
        $html .= '</select>';
        $html .= '</div>';

        // GPT Image transparent background option (unchanged)
        $html .= '<div class="mxchat-settings-field gpt-image-specific"' . ($image_gen_model !== 'gpt-image' ? 'style="display:none;"' : '') . '>';
        $html .= '<label for="mxchat-gpt-image-background">' . esc_html__('Background', 'mxchat-admin-chat') . '</label>';
        $html .= '<select id="mxchat-gpt-image-background" name="gpt_image_background">';
        $html .= '<option value="opaque"' . selected(get_option('mxchat_gpt_image_background', 'opaque'), 'opaque', false) . '>' . esc_html__('Opaque', 'mxchat-admin-chat') . '</option>';
        $html .= '<option value="transparent"' . selected(get_option('mxchat_gpt_image_background', 'opaque'), 'transparent', false) . '>' . esc_html__('Transparent (PNG/WebP only)', 'mxchat-admin-chat') . '</option>';
        $html .= '<option value="auto"' . selected(get_option('mxchat_gpt_image_background', 'opaque'), 'auto', false) . '>' . esc_html__('Auto (Model decides)', 'mxchat-admin-chat') . '</option>';
        $html .= '</select>';
        $html .= '</div>';

        // GPT Image compression for JPEG/WebP (unchanged)
        $html .= '<div class="mxchat-settings-field gpt-image-specific compression-field"' . (($image_gen_model !== 'gpt-image' || get_option('mxchat_gpt_image_format', 'png') === 'png') ? 'style="display:none;"' : '') . '>';
        $html .= '<label for="mxchat-gpt-image-compression">' . esc_html__('Compression Level (0-100%)', 'mxchat-admin-chat') . '</label>';
        $html .= '<input type="number" id="mxchat-gpt-image-compression" name="gpt_image_compression" min="0" max="100" value="' . esc_attr(get_option('mxchat_gpt_image_compression', '75')) . '">';
        $html .= '</div>';

        // GPT Image note about features (unchanged)
        $html .= '<div class="mxchat-settings-field gpt-image-specific"' . ($image_gen_model !== 'gpt-image' ? 'style="display:none;"' : '') . '>';
        $html .= '<p class="mxchat-settings-note">' . esc_html__('Note: GPT Image provides superior instruction following, text rendering, detailed editing, and real-world knowledge.', 'mxchat-admin-chat') . '</p>';
        $html .= '</div>';

        // **NEW** - Gemini-specific note (no extra fields, as Gemini is prompt-based per docs)
        $html .= '<div class="mxchat-settings-field gemini-specific"' . ($image_gen_model !== 'gemini' ? 'style="display:none;"' : '') . '>';
        $html .= '<p class="mxchat-settings-note">' . esc_html__('Note: Gemini Nano Banana uses text prompts for generation and includes SynthID watermark. Ensure your Google API key is configured.', 'mxchat-admin-chat') . '</p>';
        $html .= '</div>';
        
        $html .= '</div>'; // End image generation section
        
        // API Keys Help Section (unchanged)
        $html .= '<div class="mxchat-settings-section mxchat-api-help-section">';
        $html .= '<h3>' . esc_html__('API Key Setup Help', 'mxchat-admin-chat') . '</h3>';

        $html .= '<div class="mxchat-api-help-content">';
        $html .= '<p class="mxchat-api-help-intro">' . esc_html__('To use different AI models in the Admin Chat, you need to configure their API keys in the appropriate locations:', 'mxchat-admin-chat') . '</p>';

        // API Keys info cards
        $html .= '<div class="mxchat-api-info-cards">';

        // OpenAI Card (unchanged)
        $html .= '<div class="mxchat-api-card">';
        $html .= '<div class="mxchat-api-card-header">';
        $html .= '<i class="fa-solid fa-key"></i>';
        $html .= '<h4>' . esc_html__('OpenAI API Key', 'mxchat-admin-chat') . '</h4>';
        $html .= '</div>';
        $html .= '<div class="mxchat-api-card-content">';
        $html .= '<p>' . esc_html__('Used for: Chat (GPT models) and image generation', 'mxchat-admin-chat') . '</p>';
        $html .= '<ol>';
        $html .= '<li>' . esc_html__('Go to MxChat → Chatbot → Chat Model Selection', 'mxchat-admin-chat') . '</li>';
        $html .= '<li>' . esc_html__('Select "OpenAI" model', 'mxchat-admin-chat') . '</li>';
        $html .= '<li>' . esc_html__('Enter your OpenAI API key', 'mxchat-admin-chat') . '</li>';
        $html .= '<li>' . esc_html__('Save settings', 'mxchat-admin-chat') . '</li>';
        $html .= '</ol>';
        $html .= '<div class="mxchat-api-note">' . esc_html__('Note: You can switch back to your preferred chat model after saving the API key', 'mxchat-admin-chat') . '</div>';
        $html .= '</div>';
        $html .= '</div>';

        // Anthropic Claude Card (unchanged)
        $html .= '<div class="mxchat-api-card">';
        $html .= '<div class="mxchat-api-card-header">';
        $html .= '<i class="fa-solid fa-key"></i>';
        $html .= '<h4>' . esc_html__('Anthropic API Key', 'mxchat-admin-chat') . '</h4>';
        $html .= '</div>';
        $html .= '<div class="mxchat-api-card-content">';
        $html .= '<p>' . esc_html__('Used for: Claude models (Opus, Sonnet, Haiku)', 'mxchat-admin-chat') . '</p>';
        $html .= '<ol>';
        $html .= '<li>' . esc_html__('Go to MxChat → Chatbot → Chat Model Selection', 'mxchat-admin-chat') . '</li>';
        $html .= '<li>' . esc_html__('Select any "Claude" model', 'mxchat-admin-chat') . '</li>';
        $html .= '<li>' . esc_html__('Enter your Anthropic API key', 'mxchat-admin-chat') . '</li>';
        $html .= '<li>' . esc_html__('Save settings', 'mxchat-admin-chat') . '</li>';
        $html .= '</ol>';
        $html .= '<div class="mxchat-api-note">' . esc_html__('Note: You can switch back to your preferred chat model after saving the API key', 'mxchat-admin-chat') . '</div>';
        $html .= '</div>';
        $html .= '</div>';

        // Gemini Card - **UPDATED** to include image generation
        $html .= '<div class="mxchat-api-card">';
        $html .= '<div class="mxchat-api-card-header">';
        $html .= '<i class="fa-solid fa-key"></i>';
        $html .= '<h4>' . esc_html__('Google Gemini API Key', 'mxchat-admin-chat') . '</h4>';
        $html .= '</div>';
        $html .= '<div class="mxchat-api-card-content">';
        $html .= '<p>' . esc_html__('Used for: Gemini Pro and Gemini Flash models, and Nano Banana image generation', 'mxchat-admin-chat') . '</p>'; // **UPDATED** to mention image gen
        $html .= '<ol>';
        $html .= '<li>' . esc_html__('Go to MxChat → Chatbot → Chat Model Selection', 'mxchat-admin-chat') . '</li>';
        $html .= '<li>' . esc_html__('Select any "Gemini" model', 'mxchat-admin-chat') . '</li>';
        $html .= '<li>' . esc_html__('Enter your Google API key', 'mxchat-admin-chat') . '</li>';
        $html .= '<li>' . esc_html__('Save settings', 'mxchat-admin-chat') . '</li>';
        $html .= '</ol>';
        $html .= '<div class="mxchat-api-note">' . esc_html__('Note: You can switch back to your preferred chat model after saving the API key', 'mxchat-admin-chat') . '</div>';
        $html .= '</div>';
        $html .= '</div>';

        // xAI Card (unchanged)
        $html .= '<div class="mxchat-api-card">';
        $html .= '<div class="mxchat-api-card-header">';
        $html .= '<i class="fa-solid fa-key"></i>';
        $html .= '<h4>' . esc_html__('xAI API Key', 'mxchat-admin-chat') . '</h4>';
        $html .= '</div>';
        $html .= '<div class="mxchat-api-card-content">';
        $html .= '<p>' . esc_html__('Used for: Grok models and image generation', 'mxchat-admin-chat') . '</p>';
        $html .= '<ol>';
        $html .= '<li>' . esc_html__('Go to MxChat → Chatbot → Chat Model Selection', 'mxchat-admin-chat') . '</li>';
        $html .= '<li>' . esc_html__('Select "xAI" model', 'mxchat-admin-chat') . '</li>';
        $html .= '<li>' . esc_html__('Enter your xAI API key', 'mxchat-admin-chat') . '</li>';
        $html .= '<li>' . esc_html__('Save settings', 'mxchat-admin-chat') . '</li>';
        $html .= '</ol>';
        $html .= '<div class="mxchat-api-note">' . esc_html__('Note: You can switch back to your preferred chat model after saving the API key', 'mxchat-admin-chat') . '</div>';
        $html .= '</div>';
        $html .= '</div>';

        // Perplexity Card (unchanged)
        $html .= '<div class="mxchat-api-card">';
        $html .= '<div class="mxchat-api-card-header">';
        $html .= '<i class="fa-solid fa-key"></i>';
        $html .= '<h4>' . esc_html__('Perplexity API Key', 'mxchat-admin-chat') . '</h4>';
        $html .= '</div>';
        $html .= '<div class="mxchat-api-card-content">';
        $html .= '<p>' . esc_html__('Used for: Web search capability', 'mxchat-admin-chat') . '</p>';
        $html .= '<ol>';
        $html .= '<li>' . esc_html__('Install the Perplexity add-on', 'mxchat-admin-chat') . '</li>';
        $html .= '<li>' . esc_html__('Go to MxChat → Perplexity', 'mxchat-admin-chat') . '</li>';
        $html .= '<li>' . esc_html__('Enter your Perplexity API key', 'mxchat-admin-chat') . '</li>';
        $html .= '<li>' . esc_html__('Save settings', 'mxchat-admin-chat') . '</li>';
        $html .= '</ol>';
        $html .= '<div class="mxchat-api-note">' . esc_html__('Note: Web search requires the Perplexity add-on to be installed', 'mxchat-admin-chat') . '</div>';
        $html .= '</div>';
        $html .= '</div>';

        $html .= '</div>'; // End api-info-cards

        // Add a tip/summary (unchanged)
        $html .= '<div class="mxchat-api-help-tip">';
        $html .= '<i class="fa-solid fa-lightbulb"></i>';
        $html .= '<p>' . esc_html__('API keys are stored securely once saved and can be used by both the main MxChat plugin and the Admin Chat. You can use different models in each place while sharing the same API keys.', 'mxchat-admin-chat') . '</p>';
        $html .= '</div>';

        $html .= '</div>'; // End api-help-content
        $html .= '</div>'; // End api-help-section
        
        // Submit button (unchanged)
        $html .= '<div class="mxchat-settings-actions">';
        $html .= '<button type="submit" class="mxchat-save-settings-btn" id="mxchat-save-settings-btn">';
        $html .= '<i class="fa-solid fa-save"></i> ' . esc_html__('Save Settings', 'mxchat-admin-chat');
        $html .= '</button>';
        $html .= '</div>';
        
        $html .= '</form>';
        $html .= '</div>'; // End modal body
        $html .= '</div>'; // End modal content
        $html .= '</div>'; // End modal
        
        return $html;
    }



/**
 * Generate conversation history HTML
 *
 * @since    1.0.0
 * @param    array     $conversation  Conversation history
 * @return   string                   HTML for conversation
 */
private function generate_conversation_html($conversation) {
    $html = '';
    
    if (empty($conversation)) {
        // Welcome message
        $html .= '<div class="mxchat-message mxchat-assistant">';
        $html .= '<div class="mxchat-message-content">';
        $html .= '<div class="mxchat-message-header">';
        $html .= '<span class="mxchat-message-sender">' . esc_html__('Assistant', 'mxchat-admin-chat') . '</span>';
        $html .= '</div>';
        $html .= '<div class="mxchat-message-text">';
        $html .= '<p>' . esc_html__('Hi there! How can I assist you today?', 'mxchat-admin-chat') . '</p>';
        $html .= '</div>';
        $html .= '<div class="mxchat-message-actions">';
        $html .= '<button class="mxchat-action-btn mxchat-copy-btn" title="' . esc_attr__('Copy', 'mxchat-admin-chat') . '">';
        $html .= '<i class="fa-regular fa-copy"></i>';
        $html .= '</button>';
        $html .= '</div>';
        $html .= '</div>'; // End message content
        $html .= '</div>'; // End message
    } else {
        foreach ($conversation as $message) {
            $role = $message['role'];
            $content = $message['content'];
            
            $html .= '<div class="mxchat-message mxchat-' . esc_attr($role) . '">';
            $html .= '<div class="mxchat-message-content">';
            
            // Header with sender name
            $html .= '<div class="mxchat-message-header">';
            $html .= '<span class="mxchat-message-sender">';
            $html .= $role === 'user' ? esc_html__('You', 'mxchat-admin-chat') : esc_html__('Assistant', 'mxchat-admin-chat');
            $html .= '</span>';
            $html .= '</div>';
            
            // Message text - process for code blocks
            $html .= '<div class="mxchat-message-text">';
            
            // Format the content based on role
            $formatted_content = $this->format_message_content($content, $role);
            
            // For assistant messages, apply wpautop to format paragraphs
            // For user messages, the formatting is already handled in format_message_content
            if ($role === 'assistant') {
                // Only apply wpautop if content doesn't already contain HTML block elements
                if (strpos($formatted_content, '<p>') === false && 
                    strpos($formatted_content, '<pre>') === false && 
                    strpos($formatted_content, '<div>') === false) {
                    $html .= wpautop($formatted_content);
                } else {
                    $html .= $formatted_content;
                }
            } else {
                // User messages - use the formatted content as-is
                $html .= $formatted_content;
            }
            
            $html .= '</div>';
            
            // Action buttons
            if ($role === 'assistant') {
                $html .= '<div class="mxchat-message-actions">';
                $html .= '<button class="mxchat-action-btn mxchat-copy-btn" title="' . esc_attr__('Copy', 'mxchat-admin-chat') . '">';
                $html .= '<i class="fa-regular fa-copy"></i>';
                $html .= '</button>';
                $html .= '</div>'; // End actions
            }
            
            $html .= '</div>'; // End message content
            $html .= '</div>'; // End message
        }
    }
    
    return $html;
}

    /**
     * Generate no models available message
     *
     * @since    1.0.0
     * @return   string    HTML for no models message
     */
    private function generate_no_models_message() {
        $html = '<div class="mxchat-no-models">';
        $html .= '<div class="mxchat-no-models-content">';
        
        $html .= '<div class="mxchat-no-models-icon">';
        $html .= '<i class="fa-solid fa-triangle-exclamation"></i>';
        $html .= '</div>';
        
        $html .= '<h2>' . esc_html__('No AI Models Available', 'mxchat-admin-chat') . '</h2>';
        
        $html .= '<p>' . esc_html__('To use MxChat, you need to configure at least one API key in the settings.', 'mxchat-admin-chat') . '</p>';
        
        $html .= '<div class="mxchat-providers-list">';
        $html .= '<p>' . esc_html__('Supported AI providers:', 'mxchat-admin-chat') . '</p>';
        
        $html .= '<ul>';
        $html .= '<li><i class="fa-solid fa-check"></i> ' . esc_html__('OpenAI (GPT-4, GPT-3.5)', 'mxchat-admin-chat') . '</li>';
        $html .= '<li><i class="fa-solid fa-check"></i> ' . esc_html__('Anthropic (Claude)', 'mxchat-admin-chat') . '</li>';
        $html .= '<li><i class="fa-solid fa-check"></i> ' . esc_html__('Google (Gemini)', 'mxchat-admin-chat') . '</li>';
        $html .= '<li><i class="fa-solid fa-check"></i> ' . esc_html__('X.AI (Grok)', 'mxchat-admin-chat') . '</li>';
        $html .= '<li><i class="fa-solid fa-check"></i> ' . esc_html__('DeepSeek', 'mxchat-admin-chat') . '</li>';
        $html .= '</ul>';
        $html .= '</div>';
        
        $settings_url = admin_url('admin.php?page=mxchat-settings');
        $html .= '<a href="' . esc_url($settings_url) . '" class="mxchat-settings-btn">';
        $html .= '<i class="fa-solid fa-gear"></i> ' . esc_html__('Go to MxChat Settings', 'mxchat-admin-chat');
        $html .= '</a>';
        
        $html .= '</div>'; // End content
        $html .= '</div>'; // End no models
        
        return $html;
    }
    
    
    
    
    
    
    /**
 * Get vision model display name
 * 
 * @param string $model_id Model identifier
 * @return string Display name
 */
private function get_vision_model_name($model_id) {
    $models = array(
        'grok-vision' => 'Grok 2 Vision',
        'gpt-4-vision' => 'GPT-4o Vision',
        'claude-vision' => 'Claude 3.5 Sonnet',
        'gemini-vision' => 'Gemini Pro Vision'
    );
    
    return isset($models[$model_id]) ? $models[$model_id] : 'Grok 2 Vision';
}
    
    
    
    
    
    
    
}