<?php
/**
 * Perplexity API Integration for MxChat Admin Chat
 *
 * @package MxChat_Admin_Chat
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

/**
 * Class for handling Perplexity API integration in admin panel
 */
class MxChat_Admin_Chat_Perplexity_API {

    /**
     * The loader that's responsible for maintaining and registering all hooks.
     *
     * @var MxChat_Admin_Chat_Loader
     */
    private $loader;

    /**
     * Perplexity API key
     *
     * @var string
     */
    private $api_key;

    /**
     * Response mode (conversational or structured)
     *
     * @var string
     */
    private $response_mode;

/**
 * Initialize the class and set its properties.
 *
 * @param MxChat_Admin_Chat_Loader $loader The loader for registering hooks.
 */
public function __construct($loader) {
    $this->loader = $loader;
    
    // Get API key from the Perplexity add-on plugin instead of a separate option
    $this->api_key = get_option('mxchat_perplexity_api_key', '');
    $this->response_mode = get_option('mxchat_perplexity_response_mode', 'conversational');
    
    // Register the AJAX handlers
    $this->register_ajax_handlers();
    
    // Log the API key status for debugging
    //error_log('Perplexity API key status: ' . (!empty($this->api_key) ? 'configured' : 'missing'));
}

    /**
     * Register AJAX handler for Perplexity search in admin
     */
    public function register_ajax_handlers() {
        add_action('wp_ajax_mxchat_admin_perplexity_search', array($this, 'ajax_perplexity_search'));
    }


/**
 * AJAX handler for Perplexity search
 */
public function ajax_perplexity_search() {
    // Debug logging
    //error_log('Perplexity search request received');
    //error_log('POST data: ' . print_r($_POST, true));
    
    // Verify nonce with better error reporting
    if (!isset($_POST['nonce'])) {
        //error_log('Perplexity nonce missing');
        wp_send_json_error('Security check failed: Nonce is missing');
        return;
    }
    
    // THIS LINE IS THE KEY ISSUE - make sure the nonce name matches exactly
    if (!wp_verify_nonce($_POST['nonce'], 'mxchat_admin_perplexity_nonce')) {
        //error_log('Perplexity nonce verification failed');
        //error_log('Received nonce: ' . $_POST['nonce']);
        wp_send_json_error('Security check failed: Invalid nonce');
        return;
    }
    
    // Get the query
    $query = isset($_POST['query']) ? sanitize_text_field($_POST['query']) : '';
    
    if (empty($query)) {
        wp_send_json_error('Empty query');
    }
    
    // Perform the search
    $result = $this->perform_perplexity_search($query);
    
    if (is_wp_error($result)) {
        wp_send_json_error($result->get_error_message());
    }
    
    // Process and return the result
    $response = $this->process_perplexity_response($result, $query);
    wp_send_json_success($response);
}

    /**
     * Perform search using Perplexity API
     *
     * @param string $query Search query
     * @param string $system_prompt Optional system prompt
     * @return array|WP_Error API response or error
     */
// In your perform_perplexity_search method
public function perform_perplexity_search($query, $system_prompt = null) {
    // Check if API key is available
    if (empty($this->api_key)) {
        //error_log('Perplexity Error: API key is missing');
        return new WP_Error('missing_api_key', 'Perplexity API key is not configured');
    }

    // Clean the query
    $query = $this->clean_search_query($query);
    
    if (empty($query) || strlen($query) < 3) {
        //error_log('Perplexity Error: Query too short: ' . $query);
        return new WP_Error('invalid_query', 'Search query is too short or empty');
    }

    // Log the request details for debugging
    //error_log('Perplexity Request: Query: ' . $query);
    //error_log('Perplexity Request: API Key (first 5 chars): ' . substr($this->api_key, 0, 5) . '...');
    
    // Set default system prompt if not provided
    if (empty($system_prompt)) {
        $system_prompt = ($this->response_mode === 'conversational') 
            ? 'You are a helpful research assistant in a WordPress admin panel. Search the web for accurate and up-to-date information to answer the user\'s query. Provide your response in a natural, conversational way.'
            : 'You are a helpful research assistant in a WordPress admin panel. Search the web for accurate and up-to-date information to answer the user\'s query. Include relevant facts, data, and cite sources.';
    }

    // Get the model name
    $model = get_option('mxchat_perplexity_model', 'sonar');
    //error_log('Perplexity Request: Using model: ' . $model);

    // Build the API request
    $request_data = array(
        'model' => $model,
        'messages' => array(
            array(
                'role' => 'system',
                'content' => $system_prompt
            ),
            array(
                'role' => 'user',
                'content' => $query
            )
        ),
        'options' => array(
            'web_search' => true
        )
    );

    // Log the full request for debugging
    //error_log('Perplexity Request Data: ' . json_encode($request_data));

    // Make the API request
    $response = wp_remote_post('https://api.perplexity.ai/chat/completions', array(
        'headers' => array(
            'Content-Type' => 'application/json',
            'Authorization' => 'Bearer ' . $this->api_key
        ),
        'body' => json_encode($request_data),
        'timeout' => 180
    ));

    // Check for errors
    if (is_wp_error($response)) {
        //error_log('Perplexity WP Error: ' . $response->get_error_message());
        return $response;
    }

    $body = wp_remote_retrieve_body($response);
    $status_code = wp_remote_retrieve_response_code($response);

    if ($status_code !== 200) {
        //error_log('Perplexity API Error: Status ' . $status_code . ', Response: ' . $body);
        
        // Try to extract more detailed error message
        $json_body = json_decode($body, true);
        if ($json_body && isset($json_body['error']) && isset($json_body['error']['message'])) {
            $error_message = $json_body['error']['message'];
            return new WP_Error('api_error', 'Perplexity API error: ' . $error_message);
        }
        
        return new WP_Error('api_error', 'Perplexity API error: ' . $status_code);
    }

    $result = json_decode($body, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        //error_log('Perplexity JSON Error: Invalid JSON response');
        return new WP_Error('json_error', 'Invalid JSON response from Perplexity API');
    }

    // Log success
    //error_log('Perplexity Success: Got valid response');
    
    return $result;
}
    /**
     * Process the Perplexity API response
     *
     * @param array $result API response
     * @param string $query Original query
     * @return array Processed response
     */
    public function process_perplexity_response($result, $query) {
        // Extract content from the response
        $content = '';
        if (isset($result['choices']) && !empty($result['choices'])) {
            $content = $result['choices'][0]['message']['content'] ?? '';
        }

        // Extract citations
        // This is more likely the correct path
        $citations = [];
        if (isset($result['choices'][0]['message']['web_search_citations'])) {
            $citations = $result['choices'][0]['message']['web_search_citations'];
        }
        // Get the model name
        $model = get_option('mxchat_perplexity_model', 'sonar');

        // Remove Chain of Thought for deep research and reasoning models
        if ($model === 'sonar-deep-research' || strpos($model, 'reasoning') !== false) {
            $content = $this->remove_chain_of_thought($content);
        }

        // Format response based on mode
        if ($this->response_mode === 'conversational') {
            // Convert any markdown headings to bold text
            $content = preg_replace('/^(#{1,6})\s+(.+)$/m', '**$2**', $content);
            
            return [
                'text' => $content,
                'html' => '',
                'format' => 'conversational',
                'query' => $query
            ];
        } else {
            // For structured mode with sources
            $html_content = $this->convert_markdown_to_html($content);
            
            // Format citations
            $citations_html = '';
            if (!empty($citations)) {
                $citations_html = '<div class="mxchat-perplexity-sources">';
                $citations_html .= '<h4>Sources:</h4><ul>';
                foreach ($citations as $index => $citation) {
                    $citations_html .= '<li>';
                    if (filter_var($citation, FILTER_VALIDATE_URL)) {
                        $citations_html .= '<a href="' . esc_url($citation) . '" target="_blank">' . esc_html($citation) . '</a>';
                    } else {
                        $citations_html .= esc_html($citation);
                    }
                    $citations_html .= '</li>';
                }
                $citations_html .= '</ul></div>';
            }
            
            // Build HTML response
            $html_response = '<div class="mxchat-perplexity-response">';
            $html_response .= '<div class="mxchat-perplexity-content">' . $html_content . '</div>';
            $html_response .= $citations_html;
            $html_response .= '</div>';
            
            return [
                'text' => $content,
                'html' => $html_response,
                'format' => 'structured',
                'query' => $query,
                'citations' => $citations
            ];
        }
    }

    /**
     * Clean up a search query to remove duplicate text
     * 
     * @param string $query The original query
     * @return string The cleaned query
     */
    private function clean_search_query($query) {
        // Safety check - return original if empty or too short
        if (empty($query) || strlen(trim($query)) < 2) {
            return $query;
        }
        
        // Store original for comparison
        $original_query = $query;
        
        // Remove any obvious escape slashes from the query
        $cleaned_query = stripslashes($query);
        
        // Basic sanitization - remove any HTML tags
        $cleaned_query = strip_tags($cleaned_query);
        
        // Trim whitespace
        $cleaned_query = trim($cleaned_query);
        
        // SAFETY CHECK: If cleaning resulted in an extremely short query, use original
        if (strlen($cleaned_query) < 3) {
            return $original_query;
        }
        
        return $cleaned_query;
    }

    /**
     * Clean response from reasoning models to remove Chain of Thought sections
     * 
     * @param string $content The response content from Perplexity
     * @return string Cleaned content without CoT sections
     */
    private function remove_chain_of_thought($content) {
        // Remove content between <thinking> tags
        $content = preg_replace('/<thinking>.*?<\/thinking>/s', '', $content);
        
        // Also check for incomplete think tags (if the closing tag is missing)
        if (strpos($content, '<thinking>') !== false) {
            $parts = explode('<thinking>', $content, 2);
            $content = $parts[0]; // Keep only the content before the <thinking> tag
        }
        
        // Clean up any leftover artifacts
        $content = preg_replace('/\n{3,}/', "\n\n", $content); // Remove excessive newlines
        $content = preg_replace('/^[\s\n]+|[\s\n]+$/', '', $content); // Trim whitespace and newlines
        
        return $content;
    }

    /**
     * Convert markdown text to HTML
     * 
     * @param string $markdown Markdown formatted text
     * @return string HTML formatted text
     */
    private function convert_markdown_to_html($markdown) {
        // Convert heading markers to bold text
        $markdown = preg_replace('/^#{1,6}\s+(.*?)$/m', '<strong>$1</strong>', $markdown);
        
        // Process code blocks
        $markdown = preg_replace('/```(.*?)```/s', '<pre><code>$1</code></pre>', $markdown);
        
        // Process bold
        $markdown = preg_replace('/\*\*(.*?)\*\*/s', '<strong>$1</strong>', $markdown);
        
        // Process numbered lists
        $markdown = preg_replace_callback('/^(\d+)\.\s+(.*)$/m', function($matches) {
            return "<li>{$matches[2]}</li>";
        }, $markdown);
        
        // Process bullet lists
        $markdown = preg_replace('/^- (.*?)$/m', '<li>$1</li>', $markdown);
        
        // Wrap consecutive list items in ul tags
        $markdown = preg_replace('/((?:<li>.*?<\/li>\s*)+)/', '<ul>$1</ul>', $markdown);
        
        // Process italics
        $markdown = preg_replace('/\*(.*?)\*/s', '<em>$1</em>', $markdown);
        
        // Process links
        $markdown = preg_replace('/\[(.*?)\]\((.*?)\)/', '<a href="$2" target="_blank">$1</a>', $markdown);
        
        // Process paragraphs
        $markdown = preg_replace('/\n\n+/', '</p><p>', $markdown);
        
        // Process line breaks
        $markdown = str_replace("\n", "<br>", $markdown);
        
        // Wrap in a paragraph if not already
        if (strpos($markdown, '<p>') === false) {
            $markdown = '<p>' . $markdown . '</p>';
        }
        
        return $markdown;
    }
}
