/**
 * MxChat Assistant API Admin JavaScript
 * Handles admin interface interactions
 */

(function($) {
    'use strict';

    /**
     * Admin object to handle all functionality
     */
    const MxChatAssistantAdmin = {
        
        /**
         * Initialize admin functionality
         */
        init: function() {
            this.bindEvents();
            this.initializeTooltips();
        },

        /**
         * Bind event handlers
         */
        bindEvents: function() {

            // Form submission with validation
            $('.mxchat-assistant-api-form').on('submit', this.validateForm.bind(this));
            
            // Real-time Assistant ID validation
            $('input[name="mxchat_assistant_api_options[assistant_id]"]').on('input', this.validateAssistantId.bind(this));
            
            // Enable/disable toggle handler
            $('input[name="mxchat_assistant_api_options[enabled]"]').on('change', this.handleEnableToggle.bind(this));
            
            // Show confirmation on critical actions
            $('.mxchat-reset-config').on('click', this.confirmReset.bind(this));
        },

        /**
         * Initialize tooltips and help text
         */
        initializeTooltips: function() {
            // Add helpful tooltips using WordPress admin styles
            if (typeof window.wp !== 'undefined' && window.wp.hooks) {
                // Use WordPress built-in tooltip system if available
                $('.mxchat-tooltip').each(function() {
                    $(this).attr('title', $(this).data('tooltip'));
                });
            }
        },


        /**
         * Update connection status indicator
         */
        updateConnectionStatus: function(success, message) {
            // Remove existing status indicators
            $('.mxchat-connection-status').remove();
            
            // Create new status indicator
            const statusClass = success ? 'mxchat-status-success' : 'mxchat-status-error';
            const statusIcon = success ? '✓' : '✗';
            const statusHtml = `
                <div class="mxchat-connection-status ${statusClass}" style="margin-top: 10px;">
                    ${statusIcon} ${message}
                </div>
            `;
            
            // Add after Assistant ID field
            $('input[name="mxchat_assistant_api_options[assistant_id]"]').closest('td').append(statusHtml);
            
            // Auto-hide after 10 seconds
            setTimeout(function() {
                $('.mxchat-connection-status').fadeOut(500, function() {
                    $(this).remove();
                });
            }, 10000);
        },

        /**
         * Validate form before submission
         */
        validateForm: function(e) {
            const enabled = $('input[name="mxchat_assistant_api_options[enabled]"]').is(':checked');
            const assistantId = $('input[name="mxchat_assistant_api_options[assistant_id]"]').val();
            
            if (enabled && (!assistantId || !assistantId.startsWith('asst_'))) {
                e.preventDefault();
                this.showNotice('error', 'Please enter a valid Assistant ID before enabling the override.');
                $('input[name="mxchat_assistant_api_options[assistant_id]"]').focus();
                return false;
            }

            // Show confirmation for enabling
            if (enabled) {
                const confirmMessage = 'Are you sure you want to enable Assistant API override? This will bypass all MxChat knowledge base information.';
                if (!confirm(confirmMessage)) {
                    e.preventDefault();
                    return false;
                }
            }

            // Show saving notification
            this.showNotice('info', 'Saving configuration...', false);
            
            return true;
        },

        /**
         * Real-time Assistant ID validation
         */
        validateAssistantId: function(e) {
            const $input = $(e.currentTarget);
            const value = $input.val().trim();
            
            // Remove existing validation messages
            $input.closest('td').find('.mxchat-validation-message').remove();
            
            if (value && !value.startsWith('asst_')) {
                const errorHtml = '<div class="mxchat-validation-message mxchat-status-error" style="margin-top: 5px;">Assistant ID must start with "asst_"</div>';
                $input.closest('td').append(errorHtml);
            } else if (value && value.startsWith('asst_') && value.length > 20) {
                const successHtml = '<div class="mxchat-validation-message mxchat-status-success" style="margin-top: 5px;">✓ Valid Assistant ID format</div>';
                $input.closest('td').append(successHtml);
            }
        },

        /**
         * Handle enable/disable toggle
         */
        handleEnableToggle: function(e) {
            const $checkbox = $(e.currentTarget);
            const isEnabled = $checkbox.is(':checked');
            const assistantId = $('input[name="mxchat_assistant_api_options[assistant_id]"]').val();
            
            if (isEnabled && (!assistantId || !assistantId.startsWith('asst_'))) {
                // Prevent enabling without valid Assistant ID
                $checkbox.prop('checked', false);
                this.showNotice('warning', 'Please enter a valid Assistant ID before enabling the override.');
                $('input[name="mxchat_assistant_api_options[assistant_id]"]').focus();
                return;
            }

            // Update UI based on state
            this.updateUIState(isEnabled);
        },

        /**
         * Update UI based on enabled state
         */
        updateUIState: function(enabled) {
            const $testButton = $('.mxchat-test-assistant');
            const $warningNotice = $('.mxchat-notice-warning');
            
            if (enabled) {
                $testButton.show();
                if ($warningNotice.length === 0) {
                    // Show warning if not already visible
                    const warningHtml = `
                        <div class="mxchat-notice mxchat-notice-warning" style="margin-bottom: 20px;">
                            <div class="mxchat-notice-icon">⚠️</div>
                            <div class="mxchat-notice-content">
                                <strong>Assistant Override will be enabled after saving</strong><br>
                                This will bypass all MxChat knowledge base and intent features.
                            </div>
                        </div>
                    `;
                    $('.mxchat-assistant-api-form').before(warningHtml);
                }
            } else {
                $testButton.hide();
                $('.mxchat-notice-warning').fadeOut(300, function() {
                    $(this).remove();
                });
            }
        },

        /**
         * Confirm reset action
         */
        confirmReset: function(e) {
            if (!confirm(mxchatAssistantAPI.i18n.confirmReset)) {
                e.preventDefault();
                return false;
            }
        },

        /**
         * Show admin notice
         */
        showNotice: function(type, message, dismissible = true) {
            // Remove existing notices of the same type
            $(`.mxchat-admin-notice.mxchat-notice-${type}`).remove();
            
            const dismissibleClass = dismissible ? 'is-dismissible' : '';
            const noticeHtml = `
                <div class="notice notice-${type} mxchat-admin-notice mxchat-notice-${type} ${dismissibleClass}" style="margin: 15px 0;">
                    <p>${message}</p>
                    ${dismissible ? '<button type="button" class="notice-dismiss"><span class="screen-reader-text">Dismiss this notice.</span></button>' : ''}
                </div>
            `;
            
            // Add notice at top of content
            $('.mxchat-assistant-api-content').prepend(noticeHtml);
            
            // Handle dismiss button
            if (dismissible) {
                $('.notice-dismiss').on('click', function() {
                    $(this).closest('.notice').fadeOut(300, function() {
                        $(this).remove();
                    });
                });
                
                // Auto-dismiss success/info notices after 5 seconds
                if (type === 'success' || type === 'info') {
                    setTimeout(function() {
                        $(`.mxchat-notice-${type}.is-dismissible`).fadeOut(300, function() {
                            $(this).remove();
                        });
                    }, 5000);
                }
            }
        },

        /**
         * Add loading animation styles
         */
        addLoadingStyles: function() {
            if ($('#mxchat-assistant-loading-styles').length === 0) {
                const styles = `
                    <style id="mxchat-assistant-loading-styles">
                        .mxchat-loading {
                            position: relative;
                            pointer-events: none;
                        }
                        .mxchat-spinner {
                            display: inline-block;
                            width: 12px;
                            height: 12px;
                            border: 2px solid #ffffff;
                            border-radius: 50%;
                            border-top-color: transparent;
                            animation: mxchat-spin 1s ease-in-out infinite;
                            margin-right: 8px;
                        }
                        @keyframes mxchat-spin {
                            to { transform: rotate(360deg); }
                        }
                        .mxchat-validation-message {
                            font-size: 12px;
                            padding: 4px 8px;
                            border-radius: 4px;
                            display: inline-block;
                        }
                        .mxchat-connection-status {
                            font-size: 12px;
                            padding: 6px 10px;
                            border-radius: 4px;
                            display: inline-block;
                            animation: mxchat-fade-in 0.3s ease-in-out;
                        }
                        @keyframes mxchat-fade-in {
                            from { opacity: 0; transform: translateY(-10px); }
                            to { opacity: 1; transform: translateY(0); }
                        }
                    </style>
                `;
                $('head').append(styles);
            }
        }
    };

    /**
     * AJAX handler registration
     */
    function registerAjaxHandlers() {
        // Test Assistant AJAX handler
        $(document).ajaxSuccess(function(event, xhr, settings) {
            if (settings.data && settings.data.includes('action=mxchat_test_assistant')) {
                // Handle any post-test actions if needed
            }
        });

        // Form save success handler
        $(document).ajaxComplete(function(event, xhr, settings) {
            if (settings.data && settings.data.includes('option_page=mxchat_assistant_api')) {
                // Check if form was saved successfully
                if (xhr.status === 200) {
                    MxChatAssistantAdmin.showNotice('success', mxchatAssistantAPI.i18n.saved);
                }
            }
        });
    }

    /**
     * Utility functions
     */
    const Utils = {
        /**
         * Debounce function calls
         */
        debounce: function(func, wait, immediate) {
            let timeout;
            return function() {
                const context = this, args = arguments;
                const later = function() {
                    timeout = null;
                    if (!immediate) func.apply(context, args);
                };
                const callNow = immediate && !timeout;
                clearTimeout(timeout);
                timeout = setTimeout(later, wait);
                if (callNow) func.apply(context, args);
            };
        },

        /**
         * Sanitize HTML content
         */
        sanitizeHtml: function(html) {
            const temp = document.createElement('div');
            temp.textContent = html;
            return temp.innerHTML;
        }
    };

    /**
     * Initialize when document is ready
     */
    $(document).ready(function() {
        // Add loading styles
        MxChatAssistantAdmin.addLoadingStyles();
        
        // Initialize admin functionality
        MxChatAssistantAdmin.init();
        
        // Register AJAX handlers
        registerAjaxHandlers();
        
        // Initialize UI state based on current settings
        const isEnabled = $('input[name="mxchat_assistant_api_options[enabled]"]').is(':checked');
        MxChatAssistantAdmin.updateUIState(isEnabled);
        
        // Add smooth scroll to anchors
        $('a[href^="#"]').on('click', function(e) {
            e.preventDefault();
            const target = $(this.hash);
            if (target.length) {
                $('html, body').animate({
                    scrollTop: target.offset().top - 50
                }, 300);
            }
        });
        
        console.log('MxChat Assistant API Admin initialized');
    });

})(jQuery);