<?php
/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://mxchat.ai/
 * @since      1.0.0
 * @package    MxChat_Assistant_API
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class MxChat_Assistant_API_Admin
 *
 * Handles all admin functionality including:
 * - Menu creation
 * - Asset management
 * - Admin page rendering
 * - Settings management
 *
 * @since      1.0.0
 * @package    MxChat_Assistant_API
 */
class MxChat_Assistant_API_Admin {

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     */
    public function __construct() {
        $this->setup_hooks();
    }

    /**
     * Register all hooks for the admin area.
     *
     * @since    1.0.0
     * @access   private
     */
    private function setup_hooks() {
        // Add menu items
        add_action('admin_menu', array($this, 'add_menu_pages'), 20);

        // Register scripts and styles
        add_action('admin_enqueue_scripts', array($this, 'enqueue_assets'));

        // Register settings
        add_action('admin_init', array($this, 'register_settings'));
    }

    /**
     * Register the admin menu pages.
     *
     * @since    1.0.0
     * @access   public
     */
    public function add_menu_pages() {
        if (is_multisite() && is_network_admin()) {
            add_submenu_page(
                'settings.php',  // Network settings menu
                esc_html__('Assistant API', 'mxchat-assistant-api'),
                esc_html__('Assistant API', 'mxchat-assistant-api'),
                'manage_network_options',
                'mxchat-assistant-api',
                array($this, 'render_admin_page')
            );
        } else {
            add_submenu_page(
                'mxchat-max',
                esc_html__('Assistant API', 'mxchat-assistant-api'),
                esc_html__('Assistant API', 'mxchat-assistant-api'),
                'manage_options',
                'mxchat-assistant-api',
                array($this, 'render_admin_page')
            );
        }
    }

    /**
     * Enqueue admin-specific scripts and styles.
     *
     * @since    1.0.0
     * @access   public
     * @param    string    $hook    The current admin page.
     */
    public function enqueue_assets($hook) {
        // Only load on our plugin's page
        if ('mxchat_page_mxchat-assistant-api' !== $hook) {
            return;
        }

        // Enqueue CSS
        wp_enqueue_style(
            'mxchat-assistant-api-admin',
            MXCHAT_ASSISTANT_API_PLUGIN_URL . 'admin/css/admin.css',
            array(),
            MXCHAT_ASSISTANT_API_VERSION
        );

        // Enqueue JavaScript (if needed in future)
        wp_enqueue_script(
            'mxchat-assistant-api-admin',
            MXCHAT_ASSISTANT_API_PLUGIN_URL . 'admin/js/admin.js',
            array('jquery'),
            MXCHAT_ASSISTANT_API_VERSION,
            true
        );

        // Localize script
        wp_localize_script('mxchat-assistant-api-admin', 'mxchatAssistantAPI', array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce'   => wp_create_nonce('mxchat_assistant_api_nonce'),
            'i18n'    => array(
                'confirmReset' => __('Are you sure you want to reset the configuration?', 'mxchat-assistant-api'),
                'saved'        => __('Settings saved successfully!', 'mxchat-assistant-api'),
                'error'        => __('An error occurred. Please try again.', 'mxchat-assistant-api')
            )
        ));
    }

    /**
     * Register settings for the plugin.
     *
     * @since    1.0.0
     * @access   public
     */
    public function register_settings() {
        register_setting('mxchat_assistant_api', 'mxchat_assistant_api_options');
        
        add_settings_section(
            'mxchat_assistant_api_main',
            'OpenAI Assistant Configuration',
            array($this, 'render_section_description'),
            'mxchat_assistant_api'
        );
        
        add_settings_field(
            'enabled',
            'Enable Assistant API',
            array($this, 'render_enabled_field'),
            'mxchat_assistant_api',
            'mxchat_assistant_api_main'
        );
        
        add_settings_field(
            'assistant_id',
            'Assistant ID',
            array($this, 'render_assistant_id_field'),
            'mxchat_assistant_api',
            'mxchat_assistant_api_main'
        );
    }

    /**
     * Render the section description.
     *
     * @since    1.0.0
     * @access   public
     */
    public function render_section_description() {

        // Show current OpenAI API key status
        $main_plugin_options = get_option('mxchat_options', array());
        $api_key = $main_plugin_options['api_key'] ?? '';
        
        if (!empty($api_key)) {
            $masked_key = substr($api_key, 0, 7) . '...' . substr($api_key, -4);
            echo '<p><strong>OpenAI API Key:</strong> <span style="color: green;">Configured (' . esc_html($masked_key) . ')</span></p>';
        } else {
            echo '<p><strong>OpenAI API Key:</strong> <span style="color: red;">Not configured - please add your OpenAI API key in MxChat settings first</span></p>';
        }
    }

    /**
     * Render the enabled field.
     *
     * @since    1.0.0
     * @access   public
     */
    public function render_enabled_field() {
        $options = get_option('mxchat_assistant_api_options', array());
        $main_plugin_options = get_option('mxchat_options', array());
        $api_key = $main_plugin_options['api_key'] ?? '';
        
        $value = $options['enabled'] ?? '';
        $checked = ($value === 'on') ? 'checked' : '';
        $disabled = empty($api_key) ? 'disabled' : '';
        
        echo "<input type='checkbox' name='mxchat_assistant_api_options[enabled]' value='on' {$checked} {$disabled} />";
        echo '<label for="enabled"><strong>Enable complete override</strong> (This will disable all MxChat features except basic chat interface)</label>';
        
        if (empty($api_key)) {
            echo '<p style="color: red;"><em>Please configure your OpenAI API key in MxChat settings first.</em></p>';
        }
    }

    /**
     * Render the assistant ID field.
     *
     * @since    1.0.0
     * @access   public
     */
    public function render_assistant_id_field() {
        $options = get_option('mxchat_assistant_api_options', array());
        $value = $options['assistant_id'] ?? '';
        
        echo "<input type='text' name='mxchat_assistant_api_options[assistant_id]' value='" . esc_attr($value) . "' class='regular-text' placeholder='asst_...' />";
        echo '<p class="description">The ID of your OpenAI Assistant (starts with "asst_").</p>';
        echo '<p class="description">You can find this in your <a href="https://platform.openai.com/assistants" target="_blank">OpenAI dashboard under Assistants</a>.</p>';
    }

    /**
     * Render the main admin page.
     *
     * @since    1.0.0
     * @access   public
     */
    public function render_admin_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        // Include the admin page template
        require_once MXCHAT_ASSISTANT_API_PLUGIN_DIR . 'admin/partials/admin-page.php';
    }
}