<?php
/**
 * OpenAI Assistant API handler
 *
 * @link       https://mxchat.ai/
 * @since      1.0.0
 * @package    MxChat_Assistant_API
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class MxChat_Assistant_API_OpenAI
 *
 * Handles all OpenAI Assistant API interactions including:
 * - Thread management
 * - Message handling
 * - Run execution
 * - Response retrieval
 *
 * @since      1.0.0
 * @package    MxChat_Assistant_API
 */
class MxChat_Assistant_API_OpenAI {

    /**
     * OpenAI API base URL
     */
    const API_BASE_URL = 'https://api.openai.com/v1';

    /**
     * API timeout in seconds
     */
    const API_TIMEOUT = 30;

    /**
     * Maximum attempts for polling run status
     */
    const MAX_POLLING_ATTEMPTS = 60;

    /**
     * Initialize the class
     *
     * @since    1.0.0
     */
    public function __construct() {
        // Nothing to initialize for now
    }

    /**
     * Get response from OpenAI Assistant
     *
     * @since    1.0.0
     * @param    string    $message        User message
     * @param    string    $session_id     Chat session ID
     * @param    string    $api_key        OpenAI API key
     * @param    string    $assistant_id   OpenAI Assistant ID
     * @return   string                   Assistant response
     * @throws   Exception               On API errors
     */
    public function get_assistant_response($message, $session_id, $api_key, $assistant_id) {
        try {
            // Get or create thread for this session
            $thread_id = $this->get_or_create_thread($session_id, $api_key);
            
            // Add user message to thread
            $this->add_message_to_thread($api_key, $thread_id, $message);
            
            // Create and run the assistant
            $run_id = $this->create_run($api_key, $thread_id, $assistant_id);
            
            // Wait for completion and get response
            return $this->wait_for_completion($api_key, $thread_id, $run_id);
            
        } catch (Exception $e) {
            error_log('MxChat Assistant API Error: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Get or create thread for session
     *
     * @since    1.0.0
     * @param    string    $session_id    Chat session ID
     * @param    string    $api_key       OpenAI API key
     * @return   string                  Thread ID
     * @throws   Exception              On API errors
     */
    private function get_or_create_thread($session_id, $api_key) {
        $thread_id = get_transient('mxchat_assistant_thread_' . $session_id);
        
        if (!$thread_id) {
            $thread_id = $this->create_thread($api_key);
            
            if ($thread_id) {
                // Store thread ID for 2 hours
                set_transient('mxchat_assistant_thread_' . $session_id, $thread_id, 24 * HOUR_IN_SECONDS);
            }
        }
        
        return $thread_id;
    }

    /**
     * Create a new thread
     *
     * @since    1.0.0
     * @param    string    $api_key    OpenAI API key
     * @return   string               Thread ID
     * @throws   Exception           On API errors
     */
    private function create_thread($api_key) {
        $response = wp_remote_post(self::API_BASE_URL . '/threads', array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $api_key,
                'Content-Type' => 'application/json',
                'OpenAI-Beta' => 'assistants=v2'
            ),
            'body' => json_encode(array()),
            'timeout' => self::API_TIMEOUT
        ));
        
        if (is_wp_error($response)) {
            throw new Exception('Failed to create thread: ' . $response->get_error_message());
        }
        
        $http_code = wp_remote_retrieve_response_code($response);
        if ($http_code !== 200) {
            throw new Exception('Failed to create thread. HTTP code: ' . $http_code);
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (!isset($body['id'])) {
            throw new Exception('Invalid response from OpenAI when creating thread');
        }
        
        return $body['id'];
    }

    /**
     * Add message to thread
     *
     * @since    1.0.0
     * @param    string    $api_key     OpenAI API key
     * @param    string    $thread_id   Thread ID
     * @param    string    $message     User message
     * @return   bool                  Success status
     * @throws   Exception            On API errors
     */
    private function add_message_to_thread($api_key, $thread_id, $message) {
        $response = wp_remote_post(self::API_BASE_URL . "/threads/{$thread_id}/messages", array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $api_key,
                'Content-Type' => 'application/json',
                'OpenAI-Beta' => 'assistants=v2'
            ),
            'body' => json_encode(array(
                'role' => 'user',
                'content' => $message
            )),
            'timeout' => self::API_TIMEOUT
        ));
        
        if (is_wp_error($response)) {
            throw new Exception('Failed to add message to thread: ' . $response->get_error_message());
        }
        
        $http_code = wp_remote_retrieve_response_code($response);
        if ($http_code !== 200) {
            throw new Exception('Failed to add message to thread. HTTP code: ' . $http_code);
        }
        
        return true;
    }

    /**
     * Create a run
     *
     * @since    1.0.0
     * @param    string    $api_key        OpenAI API key
     * @param    string    $thread_id      Thread ID
     * @param    string    $assistant_id   Assistant ID
     * @return   string                   Run ID
     * @throws   Exception               On API errors
     */
    private function create_run($api_key, $thread_id, $assistant_id) {
        $response = wp_remote_post(self::API_BASE_URL . "/threads/{$thread_id}/runs", array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $api_key,
                'Content-Type' => 'application/json',
                'OpenAI-Beta' => 'assistants=v2'
            ),
            'body' => json_encode(array(
                'assistant_id' => $assistant_id
            )),
            'timeout' => self::API_TIMEOUT
        ));
        
        if (is_wp_error($response)) {
            throw new Exception('Failed to create run: ' . $response->get_error_message());
        }
        
        $http_code = wp_remote_retrieve_response_code($response);
        if ($http_code !== 200) {
            throw new Exception('Failed to create run. HTTP code: ' . $http_code);
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (!isset($body['id'])) {
            throw new Exception('Invalid response from OpenAI when creating run');
        }
        
        return $body['id'];
    }

    /**
     * Wait for run completion and get response
     *
     * @since    1.0.0
     * @param    string    $api_key     OpenAI API key
     * @param    string    $thread_id   Thread ID
     * @param    string    $run_id      Run ID
     * @return   string                Assistant response
     * @throws   Exception            On API errors or timeout
     */
    private function wait_for_completion($api_key, $thread_id, $run_id) {
        $attempt = 0;
        
        while ($attempt < self::MAX_POLLING_ATTEMPTS) {
            $response = wp_remote_get(self::API_BASE_URL . "/threads/{$thread_id}/runs/{$run_id}", array(
                'headers' => array(
                    'Authorization' => 'Bearer ' . $api_key,
                    'OpenAI-Beta' => 'assistants=v2'
                ),
                'timeout' => 10
            ));
            
            if (is_wp_error($response)) {
                throw new Exception('Failed to check run status: ' . $response->get_error_message());
            }
            
            $http_code = wp_remote_retrieve_response_code($response);
            if ($http_code !== 200) {
                throw new Exception('Failed to check run status. HTTP code: ' . $http_code);
            }
            
            $body = json_decode(wp_remote_retrieve_body($response), true);
            $status = $body['status'] ?? '';
            
            switch ($status) {
                case 'completed':
                    return $this->get_latest_assistant_message($api_key, $thread_id);
                    
                case 'failed':
                    $error_message = $body['last_error']['message'] ?? 'Unknown error';
                    throw new Exception('Assistant run failed: ' . $error_message);
                    
                case 'cancelled':
                    throw new Exception('Assistant run was cancelled');
                    
                case 'expired':
                    throw new Exception('Assistant run expired');
                    
                case 'requires_action':
                    // Handle function calls if needed in the future
                    throw new Exception('Assistant requires action (function calls not supported yet)');
                    
                case 'queued':
                case 'in_progress':
                case 'cancelling':
                    // Continue polling
                    break;
                    
                default:
                    throw new Exception('Unknown run status: ' . $status);
            }
            
            // Wait 1 second before next poll
            sleep(1);
            $attempt++;
        }
        
        throw new Exception('Assistant response timeout after ' . self::MAX_POLLING_ATTEMPTS . ' seconds');
    }

    /**
     * Get the latest assistant message from thread
     *
     * @since    1.0.0
     * @param    string    $api_key     OpenAI API key
     * @param    string    $thread_id   Thread ID
     * @return   string                Assistant message content
     * @throws   Exception            On API errors
     */
    private function get_latest_assistant_message($api_key, $thread_id) {
        $response = wp_remote_get(self::API_BASE_URL . "/threads/{$thread_id}/messages", array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $api_key,
                'OpenAI-Beta' => 'assistants=v2'
            ),
            'timeout' => 10
        ));
        
        if (is_wp_error($response)) {
            throw new Exception('Failed to retrieve messages: ' . $response->get_error_message());
        }
        
        $http_code = wp_remote_retrieve_response_code($response);
        if ($http_code !== 200) {
            throw new Exception('Failed to retrieve messages. HTTP code: ' . $http_code);
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        $messages = $body['data'] ?? array();
        
        // Find the latest assistant message
        foreach ($messages as $message) {
            if ($message['role'] === 'assistant') {
                // Handle different content types
                $content = $message['content'][0] ?? null;
                
                if (!$content) {
                    continue;
                }
                
                switch ($content['type']) {
                    case 'text':
                        return $content['text']['value'] ?? '';
                        
                    case 'image_file':
                        return 'Assistant sent an image (not supported in this integration)';
                        
                    case 'image_url':
                        return 'Assistant sent an image URL (not supported in this integration)';
                        
                    default:
                        return 'Assistant sent unsupported content type: ' . $content['type'];
                }
            }
        }
        
        throw new Exception('No assistant response found in thread');
    }

    /**
     * Test connection to OpenAI Assistant
     *
     * @since    1.0.0
     * @param    string    $api_key        OpenAI API key
     * @param    string    $assistant_id   Assistant ID
     * @return   array                    Test result
     */
    public function test_assistant_connection($api_key, $assistant_id) {
        try {
            // Try to retrieve the assistant to verify it exists
            $response = wp_remote_get(self::API_BASE_URL . "/assistants/{$assistant_id}", array(
                'headers' => array(
                    'Authorization' => 'Bearer ' . $api_key,
                    'OpenAI-Beta' => 'assistants=v2'
                ),
                'timeout' => 10
            ));
            
            if (is_wp_error($response)) {
                return array(
                    'success' => false,
                    'message' => 'Connection error: ' . $response->get_error_message()
                );
            }
            
            $http_code = wp_remote_retrieve_response_code($response);
            
            if ($http_code === 200) {
                $body = json_decode(wp_remote_retrieve_body($response), true);
                $assistant_name = $body['name'] ?? 'Unknown';
                $assistant_model = $body['model'] ?? 'Unknown';
                
                return array(
                    'success' => true,
                    'message' => "Successfully connected to Assistant: {$assistant_name} (Model: {$assistant_model})"
                );
            } elseif ($http_code === 404) {
                return array(
                    'success' => false,
                    'message' => 'Assistant not found. Please check your Assistant ID.'
                );
            } elseif ($http_code === 401) {
                return array(
                    'success' => false,
                    'message' => 'Invalid API key. Please check your OpenAI API key.'
                );
            } else {
                return array(
                    'success' => false,
                    'message' => "API error (HTTP {$http_code}). Please try again."
                );
            }
            
        } catch (Exception $e) {
            return array(
                'success' => false,
                'message' => 'Test failed: ' . $e->getMessage()
            );
        }
    }

    /**
     * Clear thread for session (useful for testing or resetting)
     *
     * @since    1.0.0
     * @param    string    $session_id    Chat session ID
     * @return   bool                    Success status
     */
    public function clear_session_thread($session_id) {
        return delete_transient('mxchat_assistant_thread_' . $session_id);
    }

    /**
     * Get thread information for debugging
     *
     * @since    1.0.0
     * @param    string    $session_id    Chat session ID
     * @return   array                   Thread info
     */
    public function get_thread_info($session_id) {
        $thread_id = get_transient('mxchat_assistant_thread_' . $session_id);
        $expiry = get_option('_transient_timeout_mxchat_assistant_thread_' . $session_id);
        
        return array(
            'thread_id' => $thread_id,
            'exists' => !empty($thread_id),
            'expires_at' => $expiry ? date('Y-m-d H:i:s', $expiry) : null,
            'time_remaining' => $expiry ? max(0, $expiry - time()) : 0
        );
    }
}