<?php
/**
 * Plugin Name: MxChat Assistant API
 * Plugin URI: https://mxchat.ai/
 * Description: OpenAI Assistants API integration for MxChat - bypasses knowledge base and uses your custom OpenAI Assistant
 * Version: 1.0.0
 * Author: MxChat
 * Author URI: https://mxchat.ai/
 * Text Domain: mxchat-assistant-api
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

function is_mxchat_active_assistant_api() {
    if (is_multisite()) {
        $active_sitewide_plugins = get_site_option('active_sitewide_plugins');
        if (isset($active_sitewide_plugins['mxchat-basic/mxchat-basic.php'])) {
            return true;
        }
    }
    return in_array(
        'mxchat-basic/mxchat-basic.php',
        apply_filters('active_plugins', get_option('active_plugins'))
    );
}

// Check if MxChat is installed and active
if (!is_mxchat_active_assistant_api()) {
    add_action('admin_notices', function() {
        ?>
        <div class="notice notice-error is-dismissible">
            <p><?php _e('<strong>MxChat Assistant API</strong> requires the MxChat plugin to be installed and active.', 'mxchat-assistant-api'); ?></p>
        </div>
        <?php
    });
    return;
}

// Define plugin constants
define('MXCHAT_ASSISTANT_API_VERSION', '1.0.0');
define('MXCHAT_ASSISTANT_API_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('MXCHAT_ASSISTANT_API_PLUGIN_URL', plugin_dir_url(__FILE__));
define('MXCHAT_ASSISTANT_API_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * Load the required files
 */
function load_mxchat_assistant_files() {
    $files = array(
        MXCHAT_ASSISTANT_API_PLUGIN_DIR . 'includes/class-mxchat-assistant-api-openai.php',
        MXCHAT_ASSISTANT_API_PLUGIN_DIR . 'includes/class-mxchat-assistant-api-admin.php'
    );
    
    foreach ($files as $file) {
        if (file_exists($file)) {
            require_once $file;
        }
    }
}

// Load files
load_mxchat_assistant_files();

require_once dirname(__FILE__) . '/update-checker.php';
MxChatAssistantsAPIUpdateChecker::init([
    'user_id' => 'wp_4_SGKQ2sGzkIGB',
    'plugin_slug' => 'mxchat-assistants-api'
]);

class MxChat_Assistant_API {
    private static $instance = null;
    private $options;
    
    public $admin;
    public $openai;

    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        $this->options = get_option('mxchat_options', array());
        
        // Add admin notice for pro requirement
        add_action('admin_notices', array($this, 'show_pro_requirement_notice'));
        
        // Only load features if pro is active
        if ($this->is_license_active()) {
            $this->init_components();
        }
    }

    /**
     * Check if MxChat Pro is activated
     */
    private function is_license_active() {
        if (is_multisite()) {
            $network_license_status = get_site_option('mxchat_license_status');
            if ($network_license_status === 'active') {
                return true;
            }
        }
        $license_status = get_option('mxchat_license_status', 'inactive');
        return $license_status === 'active';
    }

    /**
     * Admin notice for pro requirement
     */
    public function show_pro_requirement_notice() {
        if (!$this->is_license_active()) {
            $class = 'notice notice-warning';
            $message = sprintf(
                __('MxChat Assistant API requires MxChat Pro to be activated. Please <a href="%s">activate your pro license</a> to use this addon.', 'mxchat-assistant-api'),
                admin_url('admin.php?page=mxchat-activation')
            );
            printf('<div class="%1$s"><p>%2$s</p></div>', esc_attr($class), wp_kses_post($message));
        }
    }

    private function init_components() {
        // Initialize components
        if (class_exists('MxChat_Assistant_API_OpenAI')) {
            $this->openai = new MxChat_Assistant_API_OpenAI();
        }
        
        if (class_exists('MxChat_Assistant_API_Admin')) {
            $this->admin = new MxChat_Assistant_API_Admin();
        }

        // Hook into MxChat's message processing
        add_filter('mxchat_pre_process_message', array($this, 'maybe_override_with_assistant'), 5, 3);
        
        // Add AJAX handlers
        add_action('wp_ajax_mxchat_test_assistant', array($this, 'ajax_test_assistant'));
        add_action('wp_ajax_mxchat_clear_assistant_session', array($this, 'ajax_clear_session'));
    }

/**
 * Intercept the message processing if assistant override is enabled
 */
public function maybe_override_with_assistant($message, $user_id, $session_id) {
    $assistant_options = get_option('mxchat_assistant_api_options', array());
    
    // Only override if the add-on is enabled
    if (($assistant_options['enabled'] ?? '') !== 'on') {
        return $message; // Let original processing continue
    }
    
    // Check rate limits first (use core plugin's method if available)
    if (method_exists('MxChat_Integrator', 'check_rate_limit')) {
        // Get the core plugin instance
        global $wp_filter;
        if (isset($wp_filter['wp_ajax_mxchat_handle_chat_request'])) {
            foreach ($wp_filter['wp_ajax_mxchat_handle_chat_request']->callbacks as $priority => $callbacks) {
                foreach ($callbacks as $callback) {
                    if (is_array($callback['function']) && 
                        is_object($callback['function'][0]) && 
                        get_class($callback['function'][0]) === 'MxChat_Integrator') {
                        
                        $integrator = $callback['function'][0];
                        $rate_limit_result = $integrator->check_rate_limit();
                        
                        if ($rate_limit_result !== true) {
                            return array(
                                'text' => $rate_limit_result['message'],
                                'html' => '',
                                'status' => 'rate_limit_exceeded'
                            );
                        }
                        break 2;
                    }
                }
            }
        }
    }
    
    // Check if properly configured
    $api_key = $this->options['api_key'] ?? '';
    $assistant_id = $assistant_options['assistant_id'] ?? '';
    
    if (empty($api_key) || empty($assistant_id)) {
        return array(
            'text' => __('OpenAI Assistant is enabled but not properly configured.', 'mxchat-assistant-api'),
            'html' => ''
        );
    }
    
    // Check for actions FIRST
    $action_result = apply_filters('mxchat_check_actions_only', false, $message, $user_id, $session_id);
    
    if ($action_result !== false) {
        return $message; // Action matched - let core plugin handle everything
    }
    
    // No action matched - we're using Assistant, so save user message
    $this->save_message_to_transcript($session_id, 'user', $message);
    
    try {
        $response = $this->openai->get_assistant_response($message, $session_id, $api_key, $assistant_id);
        
        return array(
            'text' => $response,
            'html' => ''
        );
        
    } catch (Exception $e) {
        error_log('MxChat Assistant API Error: ' . $e->getMessage());
        
        $error_message = __('Sorry, I encountered an error. Please try again.', 'mxchat-assistant-api');
        
        return array(
            'text' => $error_message,
            'html' => ''
        );
    }
}

    /**
     * Save message to transcript system (mimics core plugin functionality)
     */
    private function save_message_to_transcript($session_id, $role, $message) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'mxchat_chat_transcripts';
        
        // Check if transcripts table exists
        if ($wpdb->get_var("SHOW TABLES LIKE '{$table_name}'") != $table_name) {
            return false;
        }
        
        // Generate unique message ID
        $message_id = uniqid();
        
        // Determine user information
        $user_id = is_user_logged_in() ? get_current_user_id() : 0;
        $user_identifier = $this->get_user_identifier();
        $user_email = $this->get_user_email();
        
        // Check for saved email in options
        $email_option_key = "mxchat_email_{$session_id}";
        $saved_email = get_option($email_option_key);
        if ($saved_email) {
            $user_email = $saved_email;
        }
        
        // Determine displayed name
        $displayed_name = $user_email ?: $user_identifier;
        
        // Save to session history in wp_options (for chat continuity)
        $history_key = "mxchat_history_{$session_id}";
        $history = get_option($history_key, array());
        $history[] = array(
            'id' => $message_id,
            'role' => $role,
            'content' => $message,
            'timestamp' => round(microtime(true) * 1000),
            'agent_name' => $displayed_name,
        );
        update_option($history_key, $history, 'no');
        
        // Save to transcripts database
        $insert_data = array(
            'user_id' => $user_id,
            'user_identifier' => $user_identifier,
            'user_email' => $user_email,
            'session_id' => $session_id,
            'role' => $role,
            'message' => $message,
            'timestamp' => current_time('mysql', 1),
        );
        
        $result = $wpdb->insert($table_name, $insert_data);
        
        // Update user email in existing records if we have a saved email
        if ($saved_email && $result !== false) {
            $wpdb->update(
                $table_name,
                array('user_email' => $saved_email),
                array('session_id' => $session_id),
                array('%s'),
                array('%s')
            );
        }
        
        // Send notification email for first user message in new session
        if ($role === 'user' && $result !== false) {
            $existing_messages = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$table_name} WHERE session_id = %s AND role = 'user'",
                $session_id
            ));
            
            if ($existing_messages == 1) {
                $this->send_new_chat_notification($session_id, array(
                    'identifier' => $user_identifier,
                    'email' => $user_email,
                    'ip' => $_SERVER['REMOTE_ADDR'] ?? ''
                ));
            }
        }
        
        return $result !== false;
    }

    /**
     * Get user identifier (mimics core plugin method)
     */
    private function get_user_identifier() {
        if (is_user_logged_in()) {
            $current_user = wp_get_current_user();
            return $current_user->display_name ?: $current_user->user_login;
        }
        
        // For non-logged-in users, create identifier from IP and user agent
        $ip = $_SERVER['REMOTE_ADDR'] ?? '';
        $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        return 'Guest_' . substr(md5($ip . $user_agent), 0, 8);
    }

    /**
     * Get user email (mimics core plugin method)
     */
    private function get_user_email() {
        if (is_user_logged_in()) {
            $current_user = wp_get_current_user();
            return $current_user->user_email;
        }
        return '';
    }

    /**
     * Send new chat notification email (mimics core plugin method)
     */
    private function send_new_chat_notification($session_id, $user_info = array()) {
        $options = get_option('mxchat_transcripts_options');
        
        // Check if notifications are enabled
        if (empty($options['mxchat_enable_notifications'])) {
            return false;
        }
        
        // Get notification email
        $to = !empty($options['mxchat_notification_email']) ? 
              $options['mxchat_notification_email'] : 
              get_option('admin_email');
        
        if (!is_email($to)) {
            return false;
        }
        
        // Prepare email content
        $subject = sprintf('[%s] New Chat Session Started (Assistant API)', get_bloginfo('name'));
        
        $user_identifier = isset($user_info['identifier']) ? $user_info['identifier'] : 'Guest';
        $user_email = isset($user_info['email']) ? $user_info['email'] : 'Not provided';
        $user_ip = isset($user_info['ip']) ? $user_info['ip'] : ($_SERVER['REMOTE_ADDR'] ?? '');
        
        $message = sprintf(
            "A new chat session has started on your website using the Assistant API.\n\n" .
            "Session ID: %s\n" .
            "User: %s\n" .
            "Email: %s\n" .
            "IP Address: %s\n" .
            "Time: %s\n" .
            "Mode: OpenAI Assistant\n\n" .
            "View transcripts: %s",
            $session_id,
            $user_identifier,
            $user_email,
            $user_ip,
            current_time('mysql'),
            admin_url('admin.php?page=mxchat-transcripts')
        );
        
        // Send email
        return wp_mail($to, $subject, $message);
    }

    /**
     * AJAX handler for testing assistant connection
     */
    public function ajax_test_assistant() {
        check_ajax_referer('mxchat_assistant_api_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array(
                'message' => __('Unauthorized access.', 'mxchat-assistant-api')
            ));
        }

        $assistant_id = sanitize_text_field($_POST['assistant_id'] ?? '');
        
        if (empty($assistant_id)) {
            wp_send_json_error(array(
                'message' => __('Assistant ID is required.', 'mxchat-assistant-api')
            ));
        }

        if (!preg_match('/^asst_[a-zA-Z0-9]+$/', $assistant_id)) {
            wp_send_json_error(array(
                'message' => __('Invalid Assistant ID format. Must start with "asst_".', 'mxchat-assistant-api')
            ));
        }

        // Get API key from main plugin options
        $api_key = $this->options['api_key'] ?? '';
        
        if (empty($api_key)) {
            wp_send_json_error(array(
                'message' => __('OpenAI API key not configured. Please configure it in MxChat settings first.', 'mxchat-assistant-api')
            ));
        }

        // Test the connection
        $test_result = $this->openai->test_assistant_connection($api_key, $assistant_id);
        
        if ($test_result['success']) {
            wp_send_json_success(array(
                'message' => $test_result['message']
            ));
        } else {
            wp_send_json_error(array(
                'message' => $test_result['message']
            ));
        }
    }

    /**
     * AJAX handler for clearing assistant session
     */
    public function ajax_clear_session() {
        check_ajax_referer('mxchat_assistant_api_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array(
                'message' => __('Unauthorized access.', 'mxchat-assistant-api')
            ));
        }

        $session_id = sanitize_text_field($_POST['session_id'] ?? '');
        
        if (empty($session_id)) {
            wp_send_json_error(array(
                'message' => __('Session ID is required.', 'mxchat-assistant-api')
            ));
        }

        // Clear the session thread
        $cleared = $this->openai->clear_session_thread($session_id);
        
        if ($cleared) {
            wp_send_json_success(array(
                'message' => __('Assistant session cleared successfully.', 'mxchat-assistant-api')
            ));
        } else {
            wp_send_json_error(array(
                'message' => __('No active session found or failed to clear.', 'mxchat-assistant-api')
            ));
        }
    }

    public function __clone() {
        _doing_it_wrong(__FUNCTION__, __('Cheatin&#8217; huh?', 'mxchat-assistant-api'), '1.0.0');
    }

    public function __wakeup() {
        _doing_it_wrong(__FUNCTION__, __('Cheatin&#8217; huh?', 'mxchat-assistant-api'), '1.0.0');
    }
}

/**
 * Returns the main instance of MxChat_Assistant_API
 */
function MxChat_Assistant_API() {
    return MxChat_Assistant_API::instance();
}

// Initialize the plugin
add_action('plugins_loaded', 'MxChat_Assistant_API');