(function($) {
    'use strict';

    let fieldCounter = 0;

    // Initialize
    $(document).ready(function() {
        bindEvents();
        
        // Modal close handlers
        $('.mxforms-modal__close').on('click', closeModal);
        
        // Close when clicking outside the modal
        $(document).on('click', '.mxforms-modal', function(event) {
            if ($(event.target).hasClass('mxforms-modal')) {
                closeModal();
            }
        });
        
        // Message count testing tools initialization
        initMessageCountTools();
    });

    function initMessageCountTools() {
        // Handle reset button click
        $('#reset-message-count').on('click', function() {
            if (confirm(mxchatForms.i18n.confirmResetCount || 'Are you sure you want to reset your message count?')) {
                $.ajax({
                    url: mxchatForms.ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'mxchat_reset_message_count',
                        nonce: mxchatForms.nonce
                    },
                    success: function(response) {
                        if (response.success) {
                            $('#current-message-count').text('0');
                            alert(response.data.message);
                        } else {
                            alert(response.data.message || 'Error resetting message count');
                        }
                    }
                });
            }
        });
        
        // Set up auto-refresh of message count (every 10 seconds)
        if ($('#message-count-tools').length > 0) {
            setInterval(function() {
                $.ajax({
                    url: mxchatForms.ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'mxchat_get_message_count',
                        nonce: mxchatForms.nonce
                    },
                    success: function(response) {
                        if (response.success) {
                            $('#current-message-count').text(response.data.count);
                            if (response.data.session_id) {
                                $('#current-session-id').text(response.data.session_id);
                            }
                        }
                    }
                });
            }, 10000); // Refresh every 10 seconds
        }
    }

    function bindEvents() {
        // Form list actions
        $('#create-form').on('click', showFormEditor);
        $('.edit-form').on('click', editForm);
        $('.delete-form').on('click', deleteForm);
        
        // Form editor actions
        $('#mxchat-form-editor').on('submit', saveForm);
        $('#cancel-edit, #close-editor').on('click', hideFormEditor);
        $('.add-field').on('click', addField);
        
        // Field actions (delegated)
        $('#form-fields').on('click', '.field-remove', removeField);
        $('#form-fields').on('click', '.field-toggle', toggleField);
        $('#form-fields').on('input', '.field-label', updateFieldTitle);
        $('#form-fields').on('click', '.field-move-up', moveFieldUp);
        $('#form-fields').on('click', '.field-move-down', moveFieldDown);
        
        // Submission actions
        $('.delete-submission').on('click', deleteSubmission);
        $('#delete-all-submissions').on('click', deleteAllSubmissions); // Add this line
        $('.view-submission').on('click', viewSubmission);
        $('#export-submissions').on('click', exportSubmissions);
        $('.mxchat-modal-close').on('click', closeModal);
        
        $('#similarity_threshold').on('input', function() {
            $('#similarity_value').text(Math.round($(this).val() * 100));
        });
    }

    function moveFieldUp() {
        const currentField = $(this).closest('.form-field');
        const prevField = currentField.prev('.form-field');
        
        if (prevField.length) {
            currentField.insertBefore(prevField);
            currentField.addClass('field-moved');
            setTimeout(() => currentField.removeClass('field-moved'), 300);
        }
    }

    function moveFieldDown() {
        const currentField = $(this).closest('.form-field');
        const nextField = currentField.next('.form-field');
        
        if (nextField.length) {
            currentField.insertAfter(nextField);
            currentField.addClass('field-moved');
            setTimeout(() => currentField.removeClass('field-moved'), 300);
        }
    }

    function showFormEditor() {
        $('#forms-list').hide();
        $('#form-editor').show();
        if ($('#form_id').val() === '') {
            resetFormEditor();
        }
    }

    function hideFormEditor() {
        $('#form-editor').hide();
        $('#forms-list').show();
        resetFormEditor();
    }

    function resetFormEditor() {
        $('#form_id').val('');
        $('#form_title').val('');
        $('#trigger_phrases').val('');
        $('#response_message').val('');
        $('#form-fields').empty();
        $('#editor-title').text(mxchatForms.i18n.createForm);
        fieldCounter = 0;
    }

    function addField() {
        const template = $('#field-template').html();
        const newField = template.replace(/{id}/g, fieldCounter++);
        $('#form-fields').append(newField);
    }

    function removeField() {
        $(this).closest('.form-field').remove();
    }

    function toggleField() {
        const content = $(this).closest('.field-header').next('.field-content');
        content.slideToggle();
        $(this).find('.dashicons').toggleClass('dashicons-arrow-down-alt2 dashicons-arrow-up-alt2');
    }

    function updateFieldTitle() {
        const label = $(this).val() || mxchatForms.i18n.newField;
        $(this).closest('.form-field').find('.field-title').text(label);
    }

    function editForm() {
        const formId = $(this).data('id');
        
        $.ajax({
            url: mxchatForms.ajaxurl,
            type: 'POST',
            data: {
                action: 'mxchat_get_form',
                nonce: mxchatForms.nonce,
                form_id: formId
            },
            success: function(response) {
                if (response.success) {
                    populateFormEditor(response.data);
                    showFormEditor();
                } else {
                    alert(response.data.message);
                }
            }
        });
    }

function populateFormEditor(form) {
    $('#form_id').val(form.id);
    $('#form_title').val(form.title);
    $('#trigger_phrases').val(form.trigger_phrases);
    $('#response_message').val(form.response_message || '');
    $('#message_count').val(form.message_count || 0);
    $('#form_blocks_chat').prop('checked', form.blocks_chat == 1);
    $('#notification_email').val(form.notification_email || '');
    $('#regex_pattern').val(form.regex_pattern || ''); // New field for regex pattern
    
    // Set similarity threshold
    const similarity = form.similarity_threshold || 0.85;
    $('#similarity_threshold').val(similarity);
    $('#similarity_value').text(Math.round(similarity * 100));
    
    // Handle allowed roles
    if (form.allowed_roles) {
        if (form.allowed_roles === 'all') {
            $('#allowed_roles option[value="all"]').prop('selected', true);
        } else {
            const roleArray = form.allowed_roles.split(',');
            $('#allowed_roles option').prop('selected', false);
            roleArray.forEach(role => {
                $('#allowed_roles option[value="' + role + '"]').prop('selected', true);
            });
        }
    } else {
        // Default to "all" if no roles are specified
        $('#allowed_roles option[value="all"]').prop('selected', true);
    }
    
    $('#editor-title').text(mxchatForms.i18n.editForm);
    
    $('#form-fields').empty();
    if (form.fields) {
        form.fields.forEach(function(field) {
            const template = $('#field-template').html();
            const newField = template.replace(/{id}/g, fieldCounter++);
            const $field = $(newField);

            $field.find('.field-label').val(field.label);
            $field.find('.field-type').val(field.type);
            $field.find('.field-required').prop('checked', field.required);
            $field.find('.field-title').text(field.label);

            $('#form-fields').append($field);
        });
    }
}
function saveForm(e) {
    e.preventDefault();

    const formData = {
        form_id: $('#form_id').val(),
        title: $('#form_title').val(),
        trigger_phrases: $('#trigger_phrases').val(),
        response_message: $('#response_message').val(),
        message_count: $('#message_count').val() || 0,
        allowed_roles: $('#allowed_roles').val(),
        blocks_chat: $('#form_blocks_chat').is(':checked') ? 1 : 0,
        notification_email: $('#notification_email').val(),
        similarity_threshold: $('#similarity_threshold').val(),
        regex_pattern: $('#regex_pattern').val(), // New field for regex pattern
        fields: []
    };
    
    $('.form-field').each(function() {
        formData.fields.push({
            type: $(this).find('.field-type').val(),
            label: $(this).find('.field-label').val(),
            required: $(this).find('.field-required').is(':checked')
        });
    });

    $.ajax({
        url: mxchatForms.ajaxurl,
        type: 'POST',
        data: {
            action: 'mxchat_save_form',
            nonce: mxchatForms.nonce,
            form_id: formData.form_id,
            title: formData.title,
            fields: JSON.stringify(formData.fields),
            trigger_phrases: formData.trigger_phrases,
            response_message: formData.response_message,
            message_count: formData.message_count,
            allowed_roles: formData.allowed_roles,
            blocks_chat: $('#form_blocks_chat').is(':checked') ? '1' : '0',
            notification_email: formData.notification_email,
            similarity_threshold: formData.similarity_threshold,
            regex_pattern: formData.regex_pattern // New field for regex pattern
        },
        success: function(response) {
                if (response.success) {
                    location.reload();
                } else {
                    alert(response.data.message);
                }
            }
        });
}
    function deleteForm() {
        if (!confirm(mxchatForms.i18n.confirmDelete)) {
            return;
        }

        const formId = $(this).data('id');

        $.ajax({
            url: mxchatForms.ajaxurl,
            type: 'POST',
            data: {
                action: 'mxchat_delete_form',
                nonce: mxchatForms.nonce,
                form_id: formId
            },
            success: function(response) {
                if (response.success) {
                    location.reload();
                } else {
                    alert(response.data.message);
                }
            }
        });
    }

    function deleteSubmission() {
        if (!confirm(mxchatForms.i18n.confirmDelete)) {
            return;
        }

        const submissionId = $(this).data('id');

        $.ajax({
            url: mxchatForms.ajaxurl,
            type: 'POST',
            data: {
                action: 'mxchat_delete_submission',
                nonce: mxchatForms.nonce,
                submission_id: submissionId
            },
            success: function(response) {
                if (response.success) {
                    location.reload();
                } else {
                    alert(response.data.message);
                }
            }
        });
    }
    
    function deleteAllSubmissions() {
    if (!confirm('Are you sure you want to delete ALL submissions? This action cannot be undone!')) {
        return;
    }

    // Double confirmation for safety
    if (!confirm('This will permanently delete all form submissions. Are you absolutely sure?')) {
        return;
    }

    $.ajax({
        url: mxchatForms.ajaxurl,
        type: 'POST',
        data: {
            action: 'mxchat_delete_all_submissions',
            nonce: mxchatForms.nonce
        },
        success: function(response) {
            if (response.success) {
                alert(response.data.message);
                location.reload();
            } else {
                alert(response.data.message);
            }
        },
        error: function() {
            alert('An error occurred while deleting submissions.');
        }
    });
}
    
    function exportSubmissions() {
        const url = new URL(mxchatForms.ajaxurl);
        url.searchParams.append('action', 'mxchat_export_submissions');
        url.searchParams.append('nonce', mxchatForms.nonce);
        
        window.location.href = url.toString();
    }
    
    function viewSubmission() {
        const submissionData = $(this).closest('tr').find('.submission-data-preview').html();
        let modal = $('#submission-modal');
        
        if (!modal.parent().is('body')) {
            modal.appendTo('body');
        }
        
        modal.find('.mxforms-modal__body').html(submissionData);
        modal.show();
        $('body').addClass('mxforms-modal-open');
    }
        
    function closeModal() {
        const modal = $('#submission-modal');
        modal.hide();
        $('body').removeClass('mxforms-modal-open');
    }

})(jQuery);
