<?php
/**
 * Fired during plugin activation
 *
 * @link       https://mxchat.ai/
 * @since      1.0.0
 * @package    MxChat_Forms
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class MxChat_Forms_Activator
 *
 * Handles all activation functionality including database table creation
 * and initial plugin setup.
 *
 * @since      1.0.0
 * @package    MxChat_Forms
 */
class MxChat_Forms_Activator {

    /**
     * Database version
     */
    const DB_VERSION = '1.1.7';

    /**
     * Table names without prefix
     */
    const TABLE_FORMS = 'mxchat_forms';
    const TABLE_SUBMISSIONS = 'mxchat_form_submissions';

    /**
     * Fired during plugin activation.
     *
     * @since    1.0.0
     * @access   public
     * @static
     */
    public static function activate() {
        self::create_tables();
        self::set_version();
        self::maybe_set_defaults();
    }

    /**
     * Create necessary database tables
     *
     * @since    1.0.0
     * @access   private
     * @static
     */
/**
 * Create necessary database tables
 *
 * @since    1.0.0
 * @access   private
 * @static
 */
private static function create_tables() {
    global $wpdb;
    $charset_collate = $wpdb->get_charset_collate();

    // Forms table - updated to include the message_count column
    $forms_table = $wpdb->prefix . self::TABLE_FORMS;
    $forms_sql = "CREATE TABLE IF NOT EXISTS {$forms_table} (
        id bigint(20) NOT NULL AUTO_INCREMENT,
        title varchar(255) NOT NULL,
        fields longtext NOT NULL,
        trigger_phrases text NOT NULL,
        response_message text NOT NULL,
        enabled tinyint(1) DEFAULT 1,
        message_count int(11) DEFAULT 0, 
        created_at datetime DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY  (id)
    ) {$charset_collate};";

    // Submissions table (unchanged)
    $submissions_table = $wpdb->prefix . self::TABLE_SUBMISSIONS;
    $submissions_sql = "CREATE TABLE IF NOT EXISTS {$submissions_table} (
        id bigint(20) NOT NULL AUTO_INCREMENT,
        form_id bigint(20) NOT NULL,
        submission_data longtext NOT NULL,
        user_id bigint(20),
        session_id varchar(255),
        created_at datetime DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY  (id),
        KEY form_id (form_id)
    ) {$charset_collate};";

    // Execute table creation
    require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
    
    // Create or update tables
    dbDelta( $forms_sql );
    dbDelta( $submissions_sql );
    
    // Check if the response_message column exists; if not, add it
    $response_column = $wpdb->get_results( "SHOW COLUMNS FROM {$forms_table} LIKE 'response_message'" );
    if ( empty( $response_column ) ) {
        $alter_sql = "ALTER TABLE {$forms_table} ADD COLUMN response_message text NOT NULL AFTER trigger_phrases";
        $wpdb->query( $alter_sql );
        if ( $wpdb->last_error ) {
            //error_log( '[MxChat Forms] Error adding column response_message: ' . $wpdb->last_error );
        }
    }
    
    // Check if the allowed_roles column exists; if not, add it
    $allowed_roles_column = $wpdb->get_results("SHOW COLUMNS FROM {$forms_table} LIKE 'allowed_roles'");
    if (empty($allowed_roles_column)) {
        $alter_sql = "ALTER TABLE {$forms_table} ADD COLUMN allowed_roles text DEFAULT '' AFTER message_count";
        $wpdb->query($alter_sql);
        if ($wpdb->last_error) {
            //error_log('[MxChat Forms] Error adding column allowed_roles: ' . $wpdb->last_error);
        }
    }
    
    // Check if the blocks_chat column exists; if not, add it
    $blocks_chat_column = $wpdb->get_results("SHOW COLUMNS FROM {$forms_table} LIKE 'blocks_chat'");
    if (empty($blocks_chat_column)) {
        $alter_sql = "ALTER TABLE {$forms_table} ADD COLUMN blocks_chat tinyint(1) DEFAULT 0 AFTER enabled";
        $wpdb->query($alter_sql);
    }
    
    // Check if the notification_email column exists; if not, add it
    $notification_email_column = $wpdb->get_results("SHOW COLUMNS FROM {$forms_table} LIKE 'notification_email'");
    if (empty($notification_email_column)) {
        $alter_sql = "ALTER TABLE {$forms_table} ADD COLUMN notification_email varchar(255) DEFAULT '' AFTER blocks_chat";
        $wpdb->query($alter_sql);
        if ($wpdb->last_error) {
            //error_log('[MxChat Forms] Error adding column notification_email: ' . $wpdb->last_error);
        }
    }
    
    // Check if the similarity_threshold column exists; if not, add it
    $similarity_threshold_column = $wpdb->get_results("SHOW COLUMNS FROM {$forms_table} LIKE 'similarity_threshold'");
    if (empty($similarity_threshold_column)) {
        $alter_sql = "ALTER TABLE {$forms_table} ADD COLUMN similarity_threshold float DEFAULT 0.85 AFTER notification_email";
        $wpdb->query($alter_sql);
        if ($wpdb->last_error) {
            //error_log('[MxChat Forms] Error adding column similarity_threshold: ' . $wpdb->last_error);
        }
    }
    
    // Check if the regex_pattern column exists; if not, add it
    $regex_pattern_column = $wpdb->get_results("SHOW COLUMNS FROM {$forms_table} LIKE 'regex_pattern'");
    if (empty($regex_pattern_column)) {
        $alter_sql = "ALTER TABLE {$forms_table} ADD COLUMN regex_pattern text DEFAULT '' AFTER similarity_threshold";
        $wpdb->query($alter_sql);
        if ($wpdb->last_error) {
            //error_log('[MxChat Forms] Error adding column regex_pattern: ' . $wpdb->last_error);
        }
    }
    
    // Also need to add this column to the mxchat_intents table
    $intents_table = $wpdb->prefix . 'mxchat_intents';
    $intents_regex_pattern_column = $wpdb->get_results("SHOW COLUMNS FROM {$intents_table} LIKE 'regex_pattern'");
    if (empty($intents_regex_pattern_column)) {
        $alter_sql = "ALTER TABLE {$intents_table} ADD COLUMN regex_pattern text DEFAULT '' AFTER similarity_threshold";
        $wpdb->query($alter_sql);
        if ($wpdb->last_error) {
            //error_log('[MxChat Forms] Error adding column regex_pattern to intents table: ' . $wpdb->last_error);
        }
    }
    
    // Check if the message_count column exists; if not, add it
    $message_count_column = $wpdb->get_results( "SHOW COLUMNS FROM {$forms_table} LIKE 'message_count'" );
    if ( empty( $message_count_column ) ) {
        $alter_sql = "ALTER TABLE {$forms_table} ADD COLUMN message_count int(11) DEFAULT 0 AFTER enabled";
        $wpdb->query( $alter_sql );
        if ( $wpdb->last_error ) {
            //error_log( '[MxChat Forms] Error adding column message_count: ' . $wpdb->last_error );
        }
    }

    // Verify tables were created
    self::verify_tables();
}
    
    
    /**
 * Update database schema if needed - this should be called on plugin load
 * 
 * @since    1.1.0
 * @access   public
 * @static
 */
public static function maybe_update_db_schema() {
    $current_db_version = get_option('mxchat_forms_db_version', '1.0.0');
    $plugin_version = get_option('mxchat_forms_version', '1.0.0');
    
    // Check if plugin was updated or db schema needs update
    if (version_compare($plugin_version, MXCHAT_FORMS_VERSION, '<') || 
        version_compare($current_db_version, self::DB_VERSION, '<')) {
        
        // Run updates
        self::create_tables();
        self::set_version();
        
        // Update plugin version tracker
        update_option('mxchat_forms_version', MXCHAT_FORMS_VERSION);

    }
}


    /**
     * Verify that all required tables exist
     *
     * @since    1.0.0
     * @access   private
     * @static
     * @return   bool    True if all tables exist, false otherwise
     */
    private static function verify_tables() {
        global $wpdb;
        
        $tables = [
            $wpdb->prefix . self::TABLE_FORMS,
            $wpdb->prefix . self::TABLE_SUBMISSIONS
        ];

        foreach ( $tables as $table ) {
            if ( $wpdb->get_var( "SHOW TABLES LIKE '{$table}'" ) !== $table ) {
                //error_log( "[MxChat Forms] Table {$table} was not created successfully" );
                return false;
            }
        }

        return true;
    }

    /**
     * Set the database version
     *
     * @since    1.0.0
     * @access   private
     * @static
     */
    private static function set_version() {
        update_option( 'mxchat_forms_db_version', self::DB_VERSION );
    }

    /**
     * Set default options if they don't exist
     *
     * @since    1.0.0
     * @access   private
     * @static
     */
    private static function maybe_set_defaults() {
        // Set default options if they don't exist
        $defaults = array(
            'enable_form_collection' => true,
            'default_form_settings'  => array(
                'require_user_login' => false,
                'store_user_data'    => true,
                'notify_admin'       => true
            )
        );

        $existing_options = get_option( 'mxchat_forms_options', array() );
        $updated_options  = wp_parse_args( $existing_options, $defaults );
        update_option( 'mxchat_forms_options', $updated_options );
    }

    /**
     * Get table names with prefix
     *
     * @since    1.0.0
     * @access   public
     * @static
     * @return   array    Array of table names with prefix
     */
    public static function get_table_names() {
        global $wpdb;
        return array(
            'forms'       => $wpdb->prefix . self::TABLE_FORMS,
            'submissions' => $wpdb->prefix . self::TABLE_SUBMISSIONS
        );
    }
}
