<?php
/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://mxchat.ai/
 * @since      1.0.0
 * @package    MxChat_Forms
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class MxChat_Forms_Admin
 *
 * Handles all admin functionality including:
 * - Menu creation
 * - Asset management
 * - Admin page rendering
 * - AJAX handlers
 *
 * @since      1.0.0
 * @package    MxChat_Forms
 */
class MxChat_Forms_Admin {

    /**
     * The loader that's responsible for maintaining and registering all hooks.
     *
     * @since    1.0.0
     * @access   protected
     * @var      MxChat_Forms_Loader    $loader    Maintains and registers all hooks for the plugin.
     */
    protected $loader;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param    MxChat_Forms_Loader    $loader    The loader object.
     */
    public function __construct( $loader ) {
        $this->loader = $loader;
        $this->setup_hooks();
    }

/**
 * Register all hooks for the admin area.
 *
 * @since    1.0.0
 * @access   private
 */
private function setup_hooks() {
    // Add menu items
    $this->loader->add_action( 'admin_menu', $this, 'add_menu_pages' );

    // Register scripts and styles
    $this->loader->add_action( 'admin_enqueue_scripts', $this, 'enqueue_assets' );

    // Register AJAX handlers
    $this->loader->add_admin_ajax_action( 'mxchat_save_form', $this, 'ajax_save_form' );
    $this->loader->add_admin_ajax_action( 'mxchat_delete_form', $this, 'ajax_delete_form' );
    $this->loader->add_admin_ajax_action( 'mxchat_delete_submission', $this, 'ajax_delete_submission' );
    $this->loader->add_admin_ajax_action( 'mxchat_delete_all_submissions', $this, 'ajax_delete_all_submissions' ); // Add this line
    $this->loader->add_admin_ajax_action( 'mxchat_get_form', $this, 'ajax_get_form' );
    
    // Message count testing tools
    $this->loader->add_admin_ajax_action( 'mxchat_reset_message_count', $this, 'ajax_reset_message_count' );
    $this->loader->add_admin_ajax_action( 'mxchat_get_message_count', $this, 'ajax_get_message_count' );
}

    /**
     * Register the admin menu pages.
     *
     * @since    1.0.0
     * @access   public
     */
    public function add_menu_pages() {
        if (is_multisite() && is_network_admin()) {
            add_submenu_page(
                'settings.php',  // Network settings menu
                esc_html__('Form Collection', 'mxchat-forms'),
                esc_html__('Form Collection', 'mxchat-forms'),
                'manage_network_options',
                'mxchat-forms',
                array($this, 'render_forms_page')
            );
        } else {
            add_submenu_page(
                'mxchat-max',
                esc_html__('Form Collection', 'mxchat-forms'),
                esc_html__('Form Collection', 'mxchat-forms'),
                'manage_options',
                'mxchat-forms',
                array($this, 'render_forms_page')
            );
        }
    }
/**
 * Enqueue admin-specific scripts and styles.
 *
 * @since    1.0.0
 * @access   public
 * @param    string    $hook    The current admin page.
 */
public function enqueue_assets( $hook ) {
    // Only load on our plugin's page
    if ( 'mxchat_page_mxchat-forms' !== $hook ) {
        return;
    }

    // Enqueue CSS
    wp_enqueue_style(
        'mxchat-forms-admin',
        MXCHAT_FORMS_PLUGIN_URL . 'admin/css/admin.css',
        array(),
        MXCHAT_FORMS_VERSION
    );

    // Enqueue jQuery UI Sortable
    wp_enqueue_script('jquery-ui-sortable');

    // Enqueue JavaScript
    wp_enqueue_script(
        'mxchat-forms-admin',
        MXCHAT_FORMS_PLUGIN_URL . 'admin/js/admin.js',
        array( 'jquery', 'jquery-ui-sortable' ), // Add jquery-ui-sortable as a dependency
        MXCHAT_FORMS_VERSION,
        true
    );

    // Localize script
    wp_localize_script( 'mxchat-forms-admin', 'mxchatForms', array(
        'ajaxurl' => admin_url( 'admin-ajax.php' ),
        'nonce'   => wp_create_nonce( 'mxchat_forms_nonce' ),
        'i18n'    => array(
            'text'         => __( 'Text', 'mxchat-forms' ),
            'email'        => __( 'Email', 'mxchat-forms' ),
            'phone'        => __( 'Phone', 'mxchat-forms' ),
            'message'      => __( 'Message', 'mxchat-forms' ),
            'fieldLabel'   => __( 'Field Label', 'mxchat-forms' ),
            'required'     => __( 'Required', 'mxchat-forms' ),
            'remove'       => __( 'Remove', 'mxchat-forms' ),
            'confirmDelete'=> __( 'Are you sure you want to delete this?', 'mxchat-forms' ),
            'confirmResetCount' => __( 'Are you sure you want to reset your message count?', 'mxchat-forms' )
        )
    ) );
}

/**
 * AJAX handler to reset message count for testing
 */
public function ajax_reset_message_count() {
    check_ajax_referer('mxchat_forms_nonce', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => __('Unauthorized', 'mxchat-forms')]);
    }
    
    $user_id = get_current_user_id();
    $rate_limit_transient_key = 'mxchat_chat_limit_' . $user_id;
    
    // Delete the transient to reset count
    delete_transient($rate_limit_transient_key);
    
    wp_send_json_success([
        'message' => __('Message count reset successfully! Your next message will be counted as message #1.', 'mxchat-forms')
    ]);
}

/**
 * AJAX handler to get current message count
 */
public function ajax_get_message_count() {
    check_ajax_referer('mxchat_forms_nonce', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => __('Unauthorized', 'mxchat-forms')]);
    }
    
    $user_id = get_current_user_id();
    $rate_limit_transient_key = 'mxchat_chat_limit_' . $user_id;
    $chat_count = get_transient($rate_limit_transient_key);
    $chat_count = ($chat_count !== false) ? $chat_count : 0;
    
    // Get current session ID from cookie
    $session_id = '';
    if (isset($_COOKIE['mxchat_session_id'])) {
        $session_id = sanitize_text_field($_COOKIE['mxchat_session_id']);
    }
    
    wp_send_json_success([
        'count' => $chat_count,
        'session_id' => $session_id
    ]);
}


    /**
     * Render the main forms management page.
     *
     * @since    1.0.0
     * @access   public
     */
    public function render_forms_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        // Include the main forms page template
        require_once MXCHAT_FORMS_PLUGIN_DIR . 'admin/partials/forms-page.php';
    }
    
    
/**
 * AJAX handler for saving forms.
 *
 * @since    1.0.0
 * @access   public
 */
public function ajax_save_form() {
    // Check nonce and capabilities
    check_ajax_referer('mxchat_forms_nonce', 'nonce');
    if (!current_user_can('manage_options')) {
        wp_send_json_error(array('message' => __('Unauthorized', 'mxchat-forms')));
    }

    // Get form data
    $form_id           = isset($_POST['form_id']) ? intval($_POST['form_id']) : 0;
    $title             = sanitize_text_field($_POST['title']);
    $fields            = json_decode(stripslashes($_POST['fields']), true);
    $trigger_phrases   = wp_unslash($_POST['trigger_phrases']); 
    $response_message  = wp_unslash(sanitize_textarea_field($_POST['response_message']));
    $message_count     = isset($_POST['message_count']) ? intval($_POST['message_count']) : 0;
    $allowed_roles     = isset($_POST['allowed_roles']) ? $_POST['allowed_roles'] : 'all';
    $blocks_chat = (isset($_POST['blocks_chat']) && $_POST['blocks_chat'] === '1') ? 1 : 0;
    $notification_email = isset($_POST['notification_email']) ? sanitize_email($_POST['notification_email']) : '';
    $similarity_threshold = isset($_POST['similarity_threshold']) ? floatval($_POST['similarity_threshold']) : 0.85;
    $regex_pattern     = isset($_POST['regex_pattern']) ? sanitize_text_field($_POST['regex_pattern']) : ''; // Add this line
    
    // Validate similarity threshold
    if ($similarity_threshold < 0.1 || $similarity_threshold > 1) {
        $similarity_threshold = 0.85;
    }
    
    // Sanitize allowed roles
    if (is_array($allowed_roles)) {
        $allowed_roles = array_map('sanitize_text_field', $allowed_roles);
    }

    // Validate data
    if (empty($title) || empty($fields) || empty($trigger_phrases)) {
        wp_send_json_error(array('message' => __('Required fields are missing', 'mxchat-forms')));
    }

    // Save form using form handler
    $form_handler = new MxChat_Forms_Form_Handler();
    $result = $form_handler->save_form(
        $form_id, 
        $title, 
        $fields, 
        $trigger_phrases, 
        $response_message, 
        $message_count, 
        $allowed_roles, 
        $blocks_chat, 
        $notification_email,
        $similarity_threshold,
        $regex_pattern  // Add this parameter
    );
    
    if (is_wp_error($result)) {
        wp_send_json_error(array('message' => $result->get_error_message()));
    }

    wp_send_json_success(array(
        'message' => __('Form saved successfully', 'mxchat-forms'),
        'form_id' => $result
    ));
}
    /**
     * AJAX handler for deleting forms.
     *
     * @since    1.0.0
     * @access   public
     */
    public function ajax_delete_form() {
        // Check nonce and capabilities
        check_ajax_referer( 'mxchat_forms_nonce', 'nonce' );
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Unauthorized', 'mxchat-forms' ) ) );
        }

        $form_id = intval( $_POST['form_id'] );
        
        // Delete form using form handler
        $form_handler = new MxChat_Forms_Form_Handler();
        $result       = $form_handler->delete_form( $form_id );

        if ( is_wp_error( $result ) ) {
            wp_send_json_error( array( 'message' => $result->get_error_message() ) );
        }

        wp_send_json_success( array( 'message' => __( 'Form deleted successfully', 'mxchat-forms' ) ) );
    }

    /**
     * AJAX handler for deleting submissions.
     *
     * @since    1.0.0
     * @access   public
     */
    public function ajax_delete_submission() {
        // Check nonce and capabilities
        check_ajax_referer( 'mxchat_forms_nonce', 'nonce' );
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Unauthorized', 'mxchat-forms' ) ) );
        }

        $submission_id = intval( $_POST['submission_id'] );

        // Delete submission using submissions handler
        $submissions_handler = new MxChat_Forms_Submissions();
        $result              = $submissions_handler->delete_submission( $submission_id );

        if ( is_wp_error( $result ) ) {
            wp_send_json_error( array( 'message' => $result->get_error_message() ) );
        }

        wp_send_json_success( array( 'message' => __( 'Submission deleted successfully', 'mxchat-forms' ) ) );
    }
    
    /**
     * AJAX handler for fetching a form for editing.
     *
     * @since    1.0.0
     * @access   public
     */
    public function ajax_get_form() {
        // Check nonce and capabilities
        check_ajax_referer( 'mxchat_forms_nonce', 'nonce' );
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Unauthorized', 'mxchat-forms' ) ) );
        }
        
        // Get form ID from the AJAX request
        $form_id = isset( $_POST['form_id'] ) ? intval( $_POST['form_id'] ) : 0;
        if ( ! $form_id ) {
            wp_send_json_error( array( 'message' => __( 'Invalid form ID', 'mxchat-forms' ) ) );
        }
        
        // Get the form using the form handler
        $form_handler = new MxChat_Forms_Form_Handler();
        $form         = $form_handler->get_form( $form_id );
        if ( ! $form ) {
            wp_send_json_error( array( 'message' => __( 'Form not found', 'mxchat-forms' ) ) );
        }
        
        // Return the form data
        wp_send_json_success( $form );
    }

    /**
     * Get a list of all forms.
     *
     * @since    1.0.0
     * @access   public
     * @return   array    Array of form objects.
     */
    public function get_form($form_id) {
        global $wpdb;
        
        $form = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->tables['forms']} WHERE id = %d",
            $form_id
        ));
        if ($form) {
            // Decode fields as an associative array
            $form->fields = json_decode($form->fields, true);
        }
        return $form;
    }


    /**
     * Get a list of all submissions.
     *
     * @since    1.0.0
     * @access   public
     * @return   array    Array of submission objects.
     */
    public function get_submissions() {
        $submissions_handler = new MxChat_Forms_Submissions();
        return $submissions_handler->get_submissions();
    }
    
    
    /**
 * AJAX handler for deleting all submissions.
 *
 * @since    1.0.0
 * @access   public
 */
public function ajax_delete_all_submissions() {
    // Check nonce and capabilities
    check_ajax_referer( 'mxchat_forms_nonce', 'nonce' );
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_send_json_error( array( 'message' => __( 'Unauthorized', 'mxchat-forms' ) ) );
    }

    // Delete all submissions using submissions handler
    global $wpdb;
    $tables = MxChat_Forms_Activator::get_table_names();
    
    $result = $wpdb->query( "DELETE FROM {$tables['submissions']}" );

    if ( $result === false ) {
        wp_send_json_error( array( 'message' => __( 'Error deleting submissions', 'mxchat-forms' ) ) );
    }

    wp_send_json_success( array( 
        'message' => sprintf( __( '%d submission(s) deleted successfully', 'mxchat-forms' ), $result )
    ) );
}
    
}
