<?php
/**
 * Handles all form CRUD operations
 *
 * @link       https://mxchat.ai/
 * @since      1.0.0
 * @package    MxChat_Forms
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class MxChat_Forms_Form_Handler
 *
 * Handles all form operations including:
 * - Creating forms
 * - Reading form data
 * - Updating forms
 * - Deleting forms
 * - Managing form intents
 *
 * @since      1.0.0
 * @package    MxChat_Forms
 * @author     MxChat
 */
class MxChat_Forms_Form_Handler {

    /**
     * The table names
     *
     * @since    1.0.0
     * @access   private
     * @var      array    $tables    The database table names
     */
    private $tables;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     */
    public function __construct() {
        $this->tables = MxChat_Forms_Activator::get_table_names();
    }

/**
 * Save a form (create or update)
 *
 * @since    1.0.0
 * @access   public
 * @param    int      $form_id          The form ID (0 for new forms)
 * @param    string   $title            The form title
 * @param    array    $fields           The form fields
 * @param    string   $trigger_phrases  The trigger phrases
 * @param    string   $response_message The custom response message
 * @param    int      $message_count    The message count threshold (0 = no message count trigger)
 * @param    string   $allowed_roles    The allowed user roles
 * @param    int      $blocks_chat      Whether this form blocks further chat
 * @param    string   $notification_email Email for notifications
 * @param    float    $similarity_threshold The similarity threshold (0.1 to 1.0)
 * @param    string   $regex_pattern    Optional regex pattern for additional matching
 * @return   mixed    Form ID on success, WP_Error on failure
 */
public function save_form($form_id, $title, $fields, $trigger_phrases, $response_message, $message_count = 0, $allowed_roles = 'all', $blocks_chat = 0, $notification_email = '', $similarity_threshold = 0.85, $regex_pattern = '') {
    global $wpdb;

    // Validate input
    if (empty($title) || empty($fields) || empty($trigger_phrases)) {
        return new WP_Error('invalid_input', __('Required fields are missing', 'mxchat-forms'));
    }

    // Validate similarity threshold
    $similarity_threshold = floatval($similarity_threshold);
    if ($similarity_threshold < 0.1 || $similarity_threshold > 1) {
        $similarity_threshold = 0.85; // Default to 85% if invalid
    }

    // Handle roles data
    if (is_array($allowed_roles)) {
        // If 'all' is among the selected roles, just use 'all'
        if (in_array('all', $allowed_roles)) {
            $allowed_roles = 'all';
        } else {
            $allowed_roles = implode(',', $allowed_roles);
        }
    }

    // Prepare data
    $data = array(
        'title'              => $title,
        'fields'             => wp_json_encode($fields),
        'trigger_phrases'    => $trigger_phrases,
        'response_message'   => $response_message,
        'enabled'            => 1,
        'message_count'      => intval($message_count),
        'allowed_roles'      => $allowed_roles,
        'blocks_chat'        => $blocks_chat ? 1 : 0,
        'notification_email' => $notification_email,
        'similarity_threshold' => $similarity_threshold,
        'regex_pattern'      => $regex_pattern, // New field for regex pattern
    );

    // Set data formats
    $formats = array(
        '%s', // title
        '%s', // fields
        '%s', // trigger_phrases
        '%s', // response_message
        '%d', // enabled
        '%d', // message_count
        '%s', // allowed_roles
        '%d', // blocks_chat
        '%s', // notification_email
        '%f', // similarity_threshold
        '%s', // regex_pattern
    );

    if ($form_id > 0) {
        // Update existing form
        $result = $wpdb->update(
            $this->tables['forms'],
            $data,
            array('id' => $form_id),
            $formats,
            array('%d')
        );
    } else {
        // Insert new form
        $result = $wpdb->insert(
            $this->tables['forms'],
            $data,
            $formats
        );
        $form_id = $wpdb->insert_id;
    }

    if ($result === false) {
        return new WP_Error(
            'db_error',
            __('Error saving form to database', 'mxchat-forms'),
            $wpdb->last_error
        );
    }

    // Create or update the corresponding intent (if needed)
    $intent_result = $this->create_or_update_form_intent($form_id, $trigger_phrases, $similarity_threshold, $regex_pattern);
    if (is_wp_error($intent_result)) {
        //error_log('[MxChat Forms] Error creating/updating intent: ' . $intent_result->get_error_message());
    }

    return $form_id;
}

    /**
     * Delete a form
     *
     * @since    1.0.0
     * @access   public
     * @param    int      $form_id    The form ID
     * @return   bool|WP_Error        True on success, WP_Error on failure
     */
    public function delete_form($form_id) {
        global $wpdb;
    
        // Delete form
        $result = $wpdb->delete(
            $this->tables['forms'],
            array('id' => $form_id),
            array('%d')
        );
    
        if ($result === false) {
            return new WP_Error(
                'db_error',
                __('Error deleting form', 'mxchat-forms'),
                $wpdb->last_error
            );
        }
    
        // Delete associated intent
        $this->delete_form_intent($form_id);
    
        // Note: We are no longer deleting submissions when a form is deleted.
        // This preserves submission data even if the form is removed.
        
        return true;
    }

    /**
     * Get a single form
     *
     * @since    1.0.0
     * @access   public
     * @param    int      $form_id    The form ID
     * @return   object|null          Form object or null if not found
     */
    public function get_form($form_id) {
        global $wpdb;
        
        $form = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->tables['forms']} WHERE id = %d",
            $form_id
        ));
    
        if ($form) {
            $form->fields = json_decode($form->fields, true);
        }
    
        return $form;
    }

    /**
     * Get all forms
     *
     * @since    1.0.0
     * @access   public
     * @param    bool     $enabled_only    Whether to get only enabled forms
     * @return   array    Array of form objects
     */
    public function get_forms($enabled_only = false) {
        global $wpdb;

        $query = "SELECT * FROM {$this->tables['forms']}";
        if ($enabled_only) {
            $query .= " WHERE enabled = 1";
        }
        $query .= " ORDER BY created_at DESC";

        $forms = $wpdb->get_results($query);

        // Decode JSON fields
        foreach ($forms as $form) {
            $form->fields = json_decode($form->fields);
        }

        return $forms;
    }

/**
 * Create or update form intent
 *
 * @since    1.0.0
 * @access   private
 * @param    int      $form_id             The form ID
 * @param    string   $trigger_phrases     The trigger phrases
 * @param    float    $similarity_threshold The similarity threshold (0.1 to 1.0)
 * @param    string   $regex_pattern       Optional regex pattern for additional matching
 * @return   bool|WP_Error                 True on success, WP_Error on failure
 */
private function create_or_update_form_intent($form_id, $trigger_phrases, $similarity_threshold = 0.85, $regex_pattern = '') {
    global $wpdb;
    $intents_table = $wpdb->prefix . 'mxchat_intents';

    // Generate embedding for trigger phrases
    $embedding = $this->generate_embedding($trigger_phrases);
    if (is_wp_error($embedding)) {
        return $embedding;
    }

    // Check if intent already exists
    $existing_intent = $wpdb->get_row($wpdb->prepare(
        "SELECT * FROM {$intents_table} WHERE intent_label LIKE %s",
        'Form ' . $form_id . '%'
    ));

    // Validate similarity threshold
    $similarity_threshold = floatval($similarity_threshold);
    if ($similarity_threshold < 0.1 || $similarity_threshold > 1) {
        $similarity_threshold = 0.85; // Default to 85% if invalid
    }

    // Prepare intent data
    $intent_data = array(
        'intent_label' => 'Form ' . $form_id . ' - ' . substr($trigger_phrases, 0, 50),
        'phrases' => $trigger_phrases,
        'embedding_vector' => serialize($embedding),
        'callback_function' => 'mxchat_handle_form_collection',
        'similarity_threshold' => $similarity_threshold,
        'regex_pattern' => $regex_pattern // New field for regex pattern
    );

    if ($existing_intent) {
        // Update existing intent
        $result = $wpdb->update(
            $intents_table,
            $intent_data,
            array('id' => $existing_intent->id)
        );
    } else {
        // Create new intent
        $result = $wpdb->insert($intents_table, $intent_data);
    }

    if ($result === false) {
        return new WP_Error(
            'db_error',
            __('Error creating/updating intent', 'mxchat-forms'),
            $wpdb->last_error
        );
    }

    return true;
}


    /**
     * Delete form intent
     *
     * @since    1.0.0
     * @access   private
     * @param    int      $form_id    The form ID
     * @return   bool|WP_Error          True on success, WP_Error on failure
     */
    private function delete_form_intent($form_id) {
        global $wpdb;
        $intents_table = $wpdb->prefix . 'mxchat_intents';
        $like = 'Form ' . $form_id . '%';
        
        $result = $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$intents_table} WHERE intent_label LIKE %s",
                $like
            )
        );
        
        if ($result === false) {
            return new WP_Error('db_error', __('Error deleting form intent', 'mxchat-forms'), $wpdb->last_error);
        }
        return true;
    }


/**
 * Generate embedding for trigger phrases
 *
 * @since    1.0.0
 * @access   private
 * @param    string   $text    The text to generate embedding for
 * @return   array|WP_Error    Embedding array or WP_Error on failure
 */
private function generate_embedding($text) {
    // Get main plugin options
    $main_plugin_options = get_option('mxchat_options', array());
    
    // Get selected embedding model from main plugin
    $selected_model = $main_plugin_options['embedding_model'] ?? 'text-embedding-ada-002';

    // Determine provider and get appropriate API key
    if (strpos($selected_model, 'voyage') === 0) {
        $endpoint = 'https://api.voyageai.com/v1/embeddings';
        $api_key = $main_plugin_options['voyage_api_key'] ?? '';
    } elseif (strpos($selected_model, 'gemini-embedding') === 0) {
        $endpoint = 'https://generativelanguage.googleapis.com/v1beta/models/' . $selected_model . ':embedContent';
        $api_key = $main_plugin_options['gemini_api_key'] ?? '';
    } else {
        $endpoint = 'https://api.openai.com/v1/embeddings';
        $api_key = $main_plugin_options['api_key'] ?? '';
    }

    // Check for API key
    if (empty($api_key)) {
        return new WP_Error('missing_api_key', 
            __('API key is missing for ' . $selected_model, 'mxchat-forms')
        );
    }

    // Prepare request body and headers based on provider
    if (strpos($selected_model, 'gemini-embedding') === 0) {
        $request_body = array(
            'model' => 'models/' . $selected_model,
            'content' => array(
                'parts' => array(
                    array('text' => $text)
                )
            ),
            'outputDimensionality' => 1536
        );
        
        $endpoint .= '?key=' . $api_key;
        $headers = array(
            'Content-Type' => 'application/json'
        );
    } else {
        $request_body = array(
            'input' => $text,
            'model' => $selected_model
        );

        // Add output_dimension for voyage-3-large
        if ($selected_model === 'voyage-3-large') {
            $request_body['output_dimension'] = 2048;
        }
        
        $headers = array(
            'Authorization' => 'Bearer ' . $api_key,
            'Content-Type' => 'application/json'
        );
    }

    $response = wp_remote_post($endpoint, array(
        'headers' => $headers,
        'body' => wp_json_encode($request_body),
        'timeout' => 30
    ));

    if (is_wp_error($response)) {
        return $response;
    }

    $body = json_decode(wp_remote_retrieve_body($response), true);
    
    // Handle different response formats
    if (strpos($selected_model, 'gemini-embedding') === 0) {
        if (!isset($body['embedding']['values'])) {
            return new WP_Error(
                'invalid_response',
                __('Invalid response from Gemini API', 'mxchat-forms'),
                $body
            );
        }
        return $body['embedding']['values'];
    } else {
        if (!isset($body['data'][0]['embedding'])) {
            return new WP_Error(
                'invalid_response',
                __('Invalid response from API', 'mxchat-forms'),
                $body
            );
        }
        return $body['data'][0]['embedding'];
    }
}

    /**
     * Toggle form enabled status
     *
     * @since    1.0.0
     * @access   public
     * @param    int      $form_id    The form ID
     * @param    bool     $enabled    The new enabled status
     * @return   bool|WP_Error        True on success, WP_Error on failure
     */
    public function toggle_form_status($form_id, $enabled) {
        global $wpdb;

        $result = $wpdb->update(
            $this->tables['forms'],
            array('enabled' => $enabled ? 1 : 0),
            array('id' => $form_id),
            array('%d'),
            array('%d')
        );

        if ($result === false) {
            return new WP_Error(
                'db_error',
                __('Error updating form status', 'mxchat-forms'),
                $wpdb->last_error
            );
        }

        return true;
    }
}