<?php
/**
 * Handles integration with MxChat
 *
 * @package    MxChat_Forms
 */

if (!defined('ABSPATH')) {
    exit;
}

class MxChat_Forms_Integrations {
    private $loader;
    private $tables;
    private $fallback_response;

    public function __construct($loader) {
        $this->loader = $loader;
        $this->tables = MxChat_Forms_Activator::get_table_names();
        $this->setup_hooks();
    }

    private function setup_hooks() {
        $this->loader->add_filter('mxchat_available_callbacks', $this, 'register_form_callback');
        $this->loader->add_filter('mxchat_handle_form_collection', $this, 'handle_form_collection_callback', 10, 5);
        
        // Add a new filter that runs on every message to check for message count triggers
        $this->loader->add_filter('mxchat_pre_process_message', $this, 'check_message_count_triggers', 10, 3);
        
        // Add AJAX handlers
        $this->loader->add_ajax_action('mxchat_submit_form', $this, 'handle_form_submission');
        $this->loader->add_admin_ajax_action('mxchat_export_submissions', $this, 'handle_export_submissions');
    }
    
/**
 * Check if any forms should be triggered based on message count
 * 
 * @param string $message The user's message
 * @param string $user_id The user ID
 * @param string $session_id The session ID
 * @return mixed The original message if no forms trigger, or the form response if one triggers
 */
public function check_message_count_triggers($message, $user_id, $session_id) {
    global $wpdb;
    
    // Get the current message count for this user from the transient
    $rate_limit_transient_key = 'mxchat_chat_limit_' . $user_id;
    $chat_count = get_transient($rate_limit_transient_key);
    
    if ($chat_count === false) {
        $chat_count = 0; 
    }
    
    // ✅ INCREMENT IT OURSELVES - This simulates what the main plugin should do
    $next_count = intval($chat_count) + 1;
    set_transient($rate_limit_transient_key, $next_count, DAY_IN_SECONDS);
    
    error_log('[MxChat Forms Debug] Incremented message count from ' . $chat_count . ' to ' . $next_count);
    
    // Look for forms that should trigger on this new count
    $forms = $wpdb->get_results($wpdb->prepare(
        "SELECT * FROM {$this->tables['forms']} WHERE enabled = 1 AND message_count = %d AND message_count > 0",
        $next_count
    ));
    
    if (empty($forms)) {
        error_log('[MxChat Forms Debug] No forms found with message count: ' . $next_count);
        return $message;
    }
    
    error_log('[MxChat Forms Debug] Found ' . count($forms) . ' form(s) with matching message count ' . $next_count);
    
    // Rest of your method stays the same...
    
    // Get user role
    $user_role = 'visitor'; // Default for non-logged in users
    if (is_user_logged_in()) {
        $user = wp_get_current_user();
        $user_roles = $user->roles;
        $user_role = !empty($user_roles) ? $user_roles[0] : 'subscriber';
    }
    
    // Check each form for role permissions
    foreach ($forms as $form) {
        $allowed_roles = $form->allowed_roles;
        
        if ($allowed_roles === 'all' || empty($allowed_roles)) {
            $form_html = $this->generate_form_html($form, $user_id, $session_id);
            return [
                'text' => null,
                'html' => $form_html
            ];
        } else {
            $role_list = explode(',', $allowed_roles);
            if ((!is_user_logged_in() && in_array('visitor', $role_list)) || 
                (is_user_logged_in() && in_array($user_role, $role_list))) {
                $form_html = $this->generate_form_html($form, $user_id, $session_id);
                return [
                    'text' => null,
                    'html' => $form_html
                ];
            }
        }
    }
    
    return $message;
}

    public function handle_form_submission() {
        check_ajax_referer('mxchat_forms_nonce', 'nonce');
    
        $form_id = intval($_POST['form_id']);
        $user_id = intval($_POST['user_id']);
        $session_id = sanitize_text_field($_POST['session_id']);
        
        // Get form data (ignore action, nonce, and meta fields)
        $form_data = array();
        foreach ($_POST as $key => $value) {
            if (!in_array($key, ['action', 'nonce', 'form_id', 'user_id', 'session_id'])) {
                $form_data[$key] = sanitize_text_field($value);
            }
        }
    
        // Save submission
        $submissions = new MxChat_Forms_Submissions();
        $result = $submissions->save_submission($form_id, $form_data, $user_id, $session_id);
    
        if (is_wp_error($result)) {
            wp_send_json_error(['message' => $result->get_error_message()]);
        } else {
            // Get form details to check if this form blocks chat
            global $wpdb;
            $form = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM {$this->tables['forms']} WHERE id = %d",
                $form_id
            ));
            
            // If this form blocks chat, increment the message count
            if ($form && isset($form->blocks_chat) && $form->blocks_chat == 1) {
                $rate_limit_transient_key = 'mxchat_chat_limit_' . $user_id;
                $chat_count = get_transient($rate_limit_transient_key);
                if ($chat_count !== false) {
                    $chat_count++;
                    set_transient($rate_limit_transient_key, $chat_count, DAY_IN_SECONDS);
                    //error_log('[MxChat Forms Debug] Incremented message count after blocked form submission to: ' . $chat_count);
                }
            }
            
            wp_send_json_success(['message' => __('Form submitted successfully', 'mxchat-forms')]);
        }
    }
    
    public function handle_export_submissions() {
        check_ajax_referer('mxchat_forms_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Unauthorized', 'mxchat-forms')]);
        }
    
        $submissions = new MxChat_Forms_Submissions();
        $submissions->export_submissions_csv();
        exit;
    }

    public function register_form_callback($callbacks) {
        $callbacks['mxchat_handle_form_collection'] = [
            'label' => __('Form Collection', 'mxchat-forms'),
            'pro_only' => true,
            'group' => __('Customer Engagement', 'mxchat'),
        ];
        return $callbacks;
    }

public function handle_form_collection_callback($default, $message, $user_id, $session_id, $intent) {
    if (!$intent || !isset($intent->intent_label)) {
        //error_log('[MxChat Forms Debug] No valid intent found');
        return false;
    }

    if (!preg_match('/Form (\d+)/', $intent->intent_label, $matches)) {
        //error_log('[MxChat Forms Debug] Intent label does not match Form pattern: ' . $intent->intent_label);
        return false;
    }

    $form_id = intval($matches[1]);
    global $wpdb;

    //error_log('[MxChat Forms Debug] Checking form ID: ' . $form_id);
    
    $form = $wpdb->get_row($wpdb->prepare(
        "SELECT * FROM {$this->tables['forms']} WHERE id = %d AND enabled = 1",
        $form_id
    ));

    if (!$form) {
        //error_log('[MxChat Forms Debug] Form not found or not enabled');
        return false;
    }

    //error_log('[MxChat Forms Debug] Form found. Message count setting: ' . (isset($form->message_count) ? $form->message_count : 'not set'));

    // Check if this form has a message count threshold (for logging only)
    if (!empty($form->message_count) && intval($form->message_count) > 0) {
        $rate_limit_transient_key = 'mxchat_chat_limit_' . $user_id;
        $chat_count = get_transient($rate_limit_transient_key);
        $message_count_reached = ($chat_count !== false && intval($chat_count) >= intval($form->message_count));
        
        error_log('[MxChat Forms Debug] Form has message count setting (' . $form->message_count . '). Current count: ' . $chat_count . '. Triggering via similarity match.');
    }
        
    // Check regex pattern if present
    $regex_pattern = isset($intent->regex_pattern) ? trim($intent->regex_pattern) : '';
    if (!empty($regex_pattern)) {
        //error_log('[MxChat Forms Debug] Checking regex pattern: ' . $regex_pattern);
        
        // Attempt to run the regex match
        $regex_matched = false;
        $safe_regex = @preg_match('/' . $regex_pattern . '/i', $message);
        
        // If the regex is invalid, log the error but consider it a non-match
        if ($safe_regex === false) {
            //error_log('[MxChat Forms Debug] Invalid regex pattern: ' . $regex_pattern);
        } else {
            $regex_matched = ($safe_regex > 0);
        }
        
        if (!$regex_matched) {
            //error_log('[MxChat Forms Debug] Regex pattern did not match - form will not trigger');
            return false;
        }
        
        //error_log('[MxChat Forms Debug] Regex pattern matched - form will trigger');
    } else {
        //error_log('[MxChat Forms Debug] No regex pattern set - skipping regex check');
    }

    $form_html = $this->generate_form_html($form, $user_id, $session_id);

    $this->fallback_response = array(
        'text' => null,
        'html' => $form_html
    );

    //error_log('[MxChat Forms Debug] Form triggered successfully');
    return $this->fallback_response;
}

private function generate_form_html($form, $user_id, $session_id) {
    $fields = json_decode($form->fields, true);
    
    // Use the custom response message if set; otherwise, fall back to a default message.
    $response = !empty($form->response_message) 
        ? $form->response_message 
        : __('Thank you. General response time is within 24h. Please also check spam. What else can I help you with?', 'mxchat-forms');
    
    // Include the custom response, blocks_chat attributes, and session_id
    $html = sprintf(
        '<div class="mxchat-form" data-form-id="%d" data-response-message="%s" data-blocks-chat="%d" data-session-id="%s" data-user-id="%s">',
        esc_attr($form->id),
        esc_attr($response),
        esc_attr(isset($form->blocks_chat) ? $form->blocks_chat : 0),
        esc_attr($session_id),
        esc_attr($user_id)
    );
        
        if (is_array($fields)) {
            foreach ($fields as $field) {
                $html .= $this->generate_field_html($field);
            }
        }
        
        // Output only the submit button; the JS callback will be handled by public.js.
        $html .= '<button class="mxchat-submit-form">' . esc_html__('Submit', 'mxchat-forms') . '</button></div>';
        
        return $html;
    }

    private function generate_field_html($field) {
        if (empty($field['name'])) {
            $field['name'] = 'field_' . sanitize_title($field['label']);
        }

        $html = '<div class="mxchat-form-field">';
        $html .= '<label class="mxchat-form-label">';
        $html .= '<span class="mxchat-label-text">' . esc_html($field['label']) . '</span>';

        if (!empty($field['required'])) {
            $html .= '<span class="mxchat-required">*</span>';
        }

        $html .= '</label>';

        switch ($field['type']) {
            case 'text':
            case 'email':
            case 'tel':
                $html .= sprintf(
                    '<input type="%1$s" name="%2$s" %3$s class="mxchat-input">',
                    esc_attr($field['type']),
                    esc_attr($field['name']),
                    !empty($field['required']) ? 'required' : ''
                );
                break;
            case 'textarea':
                $html .= sprintf(
                    '<textarea name="%1$s" %2$s class="mxchat-textarea"></textarea>',
                    esc_attr($field['name']),
                    !empty($field['required']) ? 'required' : ''
                );
                break;
        }

        $html .= '</div>';
        return $html;
    }
}