<?php
/**
 * Handles form submissions
 *
 * @package    MxChat_Forms
 */

if (!defined('ABSPATH')) {
    exit;
}

class MxChat_Forms_Submissions {
    private $tables;

    public function __construct() {
        $this->tables = MxChat_Forms_Activator::get_table_names();
    }

public function save_submission($form_id, $data, $user_id = 0, $session_id = '') {
    global $wpdb;

    $result = $wpdb->insert(
        $this->tables['submissions'],
        array(
            'form_id' => $form_id,
            'submission_data' => wp_json_encode($data),
            'user_id' => $user_id,
            'session_id' => $session_id
        ),
        array('%d', '%s', '%d', '%s')
    );

    if ($result === false) {
        return new WP_Error('db_error', __('Error saving submission', 'mxchat-forms'));
    }

    // Get form details including notification_email if it exists
    $form = $wpdb->get_row($wpdb->prepare(
        "SELECT title, notification_email FROM {$this->tables['forms']} WHERE id = %d",
        $form_id
    ));

    // Send email notification
    $recipient_email = !empty($form->notification_email) ? $form->notification_email : get_option('admin_email');
    $site_name = get_bloginfo('name');
    
    $subject = $form->title . ' | MxChat';
    
    $message = sprintf(__("New submission received for form: %s\n\n", 'mxchat-forms'), $form->title);
    $message .= __("Submission Details:\n", 'mxchat-forms');
    
    foreach ($data as $field => $value) {
        $message .= sprintf("%s: %s\n", $field, $value);
    }
    
    $message .= "\n" . sprintf(
        __("View all submissions: %s", 'mxchat-forms'),
        admin_url('admin.php?page=mxchat-forms')
    );
    
    $message .= "\n\n" . __("Note: The complete chat transcript is attached to this email.", 'mxchat-forms');

    $headers = array('Content-Type: text/plain; charset=UTF-8');
    
    // Generate and attach chat transcript
    $transcript = $this->generate_chat_transcript($session_id);
    
    // Create a temporary file for the transcript
    $upload_dir = wp_upload_dir();
    $transcript_filename = 'chat-transcript-' . $session_id . '.txt';
    $transcript_file = $upload_dir['basedir'] . '/' . $transcript_filename;
    file_put_contents($transcript_file, $transcript);
    
    // Add the attachment
    $attachments = array($transcript_file);
    
    // Send email with attachment
    $mail_sent = wp_mail($recipient_email, $subject, $message, $headers, $attachments);
    
    // Delete the temporary file after sending
    @unlink($transcript_file);

    return $wpdb->insert_id;
}

/**
 * Generate a formatted chat transcript from the session history
 *
 * @param string $session_id The chat session ID
 * @return string Formatted transcript text
 */
private function generate_chat_transcript($session_id) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'mxchat_chat_transcripts';
    
    error_log('[MxChat Forms] Generating transcript for session: ' . $session_id);
    
    // If session_id is empty, we can't generate a transcript
    if (empty($session_id)) {
        error_log('[MxChat Forms] Error: Empty session_id provided');
        return __("Chat Transcript\n==============\n\nNo session ID provided. Cannot generate transcript.", 'mxchat-forms');
    }
    
    error_log('[MxChat Forms] Generating transcript for session: ' . $session_id);
    error_log('[MxChat Forms] Looking in table: ' . $table_name);
    
    // Check if table exists
    $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$table_name}'");
    if (!$table_exists) {
        error_log('[MxChat Forms] Error: Table does not exist: ' . $table_name);
        return __("Error: Chat transcript table not found.", 'mxchat-forms');
    }
    
    // Get messages for this session ordered by timestamp
    $query = $wpdb->prepare(
        "SELECT * FROM {$table_name}
        WHERE session_id = %s
        ORDER BY timestamp ASC",
        $session_id
    );
    
    error_log('[MxChat Forms] Executing query: ' . $query);
    
    $messages = $wpdb->get_results($query);
    
    error_log('[MxChat Forms] Query result: ' . ($messages ? count($messages) . ' messages found' : 'No messages found'));
    
    if (empty($messages)) {
        error_log('[MxChat Forms] No messages found for session: ' . $session_id);
        
        // Try looking in wp_options as fallback
        $history_key = "mxchat_history_{$session_id}";
        $history = get_option($history_key);
        error_log('[MxChat Forms] Checking wp_options key: ' . $history_key . ' - Result: ' . ($history ? 'Found' : 'Not found'));
        
        // No chat history found anywhere
        $transcript = __("Chat Transcript\n", 'mxchat-forms');
        $transcript .= "==============\n\n";
        $transcript .= __("Session ID: ", 'mxchat-forms') . $session_id . "\n";
        $transcript .= __("Date: ", 'mxchat-forms') . date_i18n(get_option('date_format') . ' ' . get_option('time_format')) . "\n\n";
        $transcript .= __("No chat history available for this session.", 'mxchat-forms');
        return $transcript;
    }
    
    // Initialize transcript with header
    $transcript = __("Chat Transcript\n", 'mxchat-forms');
    $transcript .= "==============\n\n";
    $transcript .= __("Session ID: ", 'mxchat-forms') . $session_id . "\n";
    $transcript .= __("Date: ", 'mxchat-forms') . date_i18n(get_option('date_format') . ' ' . get_option('time_format')) . "\n\n";
    
    // Get user email if available
    $user_email = '';
    foreach ($messages as $msg) {
        if (!empty($msg->user_email)) {
            $user_email = $msg->user_email;
            break;
        }
    }
    
    if (!empty($user_email)) {
        $transcript .= __("User Email: ", 'mxchat-forms') . $user_email . "\n\n";
    }
    
    // Log message details for debugging
    error_log('[MxChat Forms] Processing ' . count($messages) . ' messages for transcript');
    
    // Add each message to the transcript
    foreach ($messages as $msg) {
        // Debug log this message
        error_log('[MxChat Forms] Message - Role: ' . $msg->role . ', Timestamp: ' . $msg->timestamp);
        
        // Format timestamp
        $formatted_timestamp = date_i18n('F j, Y g:i a', strtotime($msg->timestamp));
        
        // Determine the sender display name
        switch ($msg->role) {
            case 'assistant':
            case 'bot':
                $display_role = __('Chatbot', 'mxchat-forms');
                break;
            case 'user':
                $display_role = !empty($msg->user_identifier)
                    ? $msg->user_identifier
                    : __('User', 'mxchat-forms');
                break;
            case 'agent':
                $display_role = __('Agent', 'mxchat-forms');
                break;
            case 'system':
                $display_role = __('System', 'mxchat-forms');
                break;
            default:
                $display_role = ucfirst($msg->role);
        }
        
        // Clean message content (remove HTML tags)
        $message_content = wp_strip_all_tags(stripslashes($msg->message));
        
        // Add to transcript
        $transcript .= "[{$formatted_timestamp}] {$display_role}: {$message_content}\n\n";
    }
    
    error_log('[MxChat Forms] Transcript generation complete - length: ' . strlen($transcript) . ' bytes');
    return $transcript;
}


/**
 * Format transcript from wp_options history array
 *
 * @param array $history The chat history from wp_options
 * @param string $session_id The session ID
 * @return string Formatted transcript
 */
private function format_transcript_from_history($history, $session_id) {
    $transcript = __("Chat Transcript\n", 'mxchat-forms');
    $transcript .= "==============\n\n";
    $transcript .= __("Session ID: ", 'mxchat-forms') . $session_id . "\n";
    $transcript .= __("Date: ", 'mxchat-forms') . date_i18n(get_option('date_format') . ' ' . get_option('time_format')) . "\n\n";
    
    foreach ($history as $entry) {
        // Format timestamp from milliseconds to readable time
        $time = isset($entry['timestamp']) 
            ? date_i18n(get_option('time_format'), $entry['timestamp'] / 1000) 
            : date_i18n(get_option('time_format'));
        
        // Determine role label
        $role_label = $this->get_role_label($entry['role']);
        
        // Add agent name if available
        if (!empty($entry['agent_name']) && $entry['role'] === 'assistant') {
            $role_label = $entry['agent_name'];
        }
        
        // Clean content (remove HTML tags)
        $content = strip_tags($entry['content']);
        
        // Add to transcript
        $transcript .= "[{$time}] {$role_label}: {$content}\n\n";
    }
    
    return $transcript;
}

/**
 * Format transcript from database records
 *
 * @param array $messages Database records from mxchat_chat_transcripts
 * @param string $session_id The session ID
 * @return string Formatted transcript
 */
private function format_transcript_from_db($messages, $session_id) {
    $transcript = __("Chat Transcript\n", 'mxchat-forms');
    $transcript .= "==============\n\n";
    $transcript .= __("Session ID: ", 'mxchat-forms') . $session_id . "\n";
    $transcript .= __("Date: ", 'mxchat-forms') . date_i18n(get_option('date_format') . ' ' . get_option('time_format')) . "\n\n";
    
    foreach ($messages as $message) {
        // Format timestamp
        $time = date_i18n(get_option('time_format'), strtotime($message->timestamp));
        
        // Determine role label
        $role_label = $this->get_role_label($message->role);
        
        // Use user_identifier if available for agent name
        if ($message->role === 'assistant' && !empty($message->user_identifier)) {
            $role_label = $message->user_identifier;
        }
        
        // Clean content (remove HTML tags)
        $content = strip_tags($message->message);
        
        // Add to transcript
        $transcript .= "[{$time}] {$role_label}: {$content}\n\n";
    }
    
    return $transcript;
}

/**
 * Get a human-readable label for the role
 *
 * @param string $role The role (user, assistant, bot, system)
 * @return string Human-readable label
 */
private function get_role_label($role) {
    switch ($role) {
        case 'user':
            return __('User', 'mxchat-forms');
        case 'assistant':
        case 'bot':
            return __('Bot', 'mxchat-forms');
        case 'system':
            return __('System', 'mxchat-forms');
        default:
            return ucfirst($role);
    }
}
    public function get_submissions($form_id = null) {
        global $wpdb;

        $query = "SELECT s.*, f.title as form_title 
                 FROM {$this->tables['submissions']} s
                 LEFT JOIN {$this->tables['forms']} f ON s.form_id = f.id";

        if ($form_id) {
            $query .= $wpdb->prepare(" WHERE s.form_id = %d", $form_id);
        }

        $query .= " ORDER BY s.created_at DESC";

        $submissions = $wpdb->get_results($query);

        foreach ($submissions as $submission) {
            $submission->submission_data = json_decode($submission->submission_data);
        }

        return $submissions;
    }

    public function get_submission($submission_id) {
        global $wpdb;

        $submission = $wpdb->get_row($wpdb->prepare(
            "SELECT s.*, f.title as form_title 
             FROM {$this->tables['submissions']} s
             LEFT JOIN {$this->tables['forms']} f ON s.form_id = f.id
             WHERE s.id = %d",
            $submission_id
        ));

        if ($submission) {
            $submission->submission_data = json_decode($submission->submission_data);
        }

        return $submission;
    }

    public function delete_submission($submission_id) {
        global $wpdb;

        $result = $wpdb->delete(
            $this->tables['submissions'],
            array('id' => $submission_id),
            array('%d')
        );

        if ($result === false) {
            return new WP_Error('db_error', __('Error deleting submission', 'mxchat-forms'));
        }

        return true;
    }

    public function delete_form_submissions($form_id) {
        global $wpdb;

        return $wpdb->delete(
            $this->tables['submissions'],
            array('form_id' => $form_id),
            array('%d')
        );
    }
    
    
    public function export_submissions_csv() {
        global $wpdb;
        
        // Get all submissions with form titles
        $submissions = $this->get_submissions();
        
        if (empty($submissions)) {
            return false;
        }
    
        // Set headers for CSV download
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment;filename=form-submissions-' . date('Y-m-d') . '.csv');
        $output = fopen('php://output', 'w');
    
        // CSV headers
        $headers = array(
            'Date',
            'Form',
            'User',
            'Session ID'
        );
    
        // Get field headers from first submission
        $first_submission = reset($submissions);
        if ($first_submission->submission_data) {
            foreach ($first_submission->submission_data as $key => $value) {
                $headers[] = $key;
            }
        }
    
        // Write headers
        fputcsv($output, $headers);
    
        // Write data
        foreach ($submissions as $submission) {
            $row = array(
                date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($submission->created_at)),
                $submission->form_title,
                $submission->user_id ? get_userdata($submission->user_id)->display_name : 'Guest',
                $submission->session_id
            );
    
            // Add submission data
            if ($submission->submission_data) {
                foreach ($first_submission->submission_data as $key => $value) {
                    $row[] = isset($submission->submission_data->$key) ? $submission->submission_data->$key : '';
                }
            }
    
            fputcsv($output, $row);
        }
    
        fclose($output);
        exit;
    }
}