<?php
/**
 * Plugin Name: MxChat Form Collection
 * Plugin URI: https://mxchat.ai/
 * Description: Form collection add-on for MxChat (Pro only)
 * Version: 1.1.2
 * Author: MxChat
 * Author URI: https://mxchat.ai/
 * Text Domain: mxchat-forms
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

function is_mxchat_active_forms() {
    if (is_multisite()) {
        $active_sitewide_plugins = get_site_option('active_sitewide_plugins');
        if (isset($active_sitewide_plugins['mxchat-basic/mxchat-basic.php'])) {
            return true;
        }
    }
    return in_array(
        'mxchat-basic/mxchat-basic.php',
        apply_filters('active_plugins', get_option('active_plugins'))
    );
}

// Check if MxChat is installed and active
if (!is_mxchat_active_forms()) {
    add_action('admin_notices', function() {
        ?>
        <div class="notice notice-error is-dismissible">
            <p><?php _e('<strong>MxChat Form Collection</strong> requires the MxChat plugin to be installed and active.', 'mxchat-forms'); ?></p>
        </div>
        <?php
    });
    return;
}

// Define plugin constants
define('MXCHAT_FORMS_VERSION', '1.1.2');
define('MXCHAT_FORMS_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('MXCHAT_FORMS_PLUGIN_URL', plugin_dir_url(__FILE__));
define('MXCHAT_FORMS_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * Class autoloader for the plugin
 */
spl_autoload_register(function ($class) {
    // Plugin namespace prefix
    $prefix = 'MxChat_Forms_';
    $base_dir = MXCHAT_FORMS_PLUGIN_DIR . 'includes/';

    // Check if the class uses our namespace
    $len = strlen($prefix);
    if (strncmp($prefix, $class, $len) !== 0) {
        return;
    }

    // Get the relative class name
    $relative_class = substr($class, $len);

    // Replace the namespace prefix with the base directory
    $file = $base_dir . 'class-mxchat-forms-' . str_replace('_', '-', strtolower($relative_class)) . '.php';

    // If the file exists, require it
    if (file_exists($file)) {
        require $file;
    }
});

require_once dirname(__FILE__) . '/update-checker.php';
PluginUpdateCheckerForms::init([
    'user_id' => 'wp_4_SGKQ2sGzkIGB',
    'plugin_slug' => 'mxchat-forms'
]);

class MxChat_Forms {
    private static $instance = null;
    private $options;
    
    public $loader;
    public $admin;
    public $form_handler;
    public $submissions;
    public $integrations;

    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        $this->options = get_option('mxchat_options', array());
        
        // Add admin notice for pro requirement
        add_action('admin_notices', array($this, 'show_pro_requirement_notice'));
        
        // Only load features if pro is active
        if ($this->is_license_active()) {
            $this->init_components();
        }
    }

    /**
     * Check if MxChat Pro is activated
     */
    private function is_license_active() {
        if (is_multisite()) {
            $network_license_status = get_site_option('mxchat_license_status');
            if ($network_license_status === 'active') {
                return true;
            }
        }
        $license_status = get_option('mxchat_license_status', 'inactive');
        return $license_status === 'active';
    }

    /**
     * Admin notice for pro requirement
     */
    public function show_pro_requirement_notice() {
        if (!$this->is_license_active()) {
            $class = 'notice notice-warning';
            $message = sprintf(
                __('MxChat Form Collection requires MxChat Pro to be activated. Please <a href="%s">activate your pro license</a> to use this addon.', 'mxchat-forms'),
                admin_url('admin.php?page=mxchat-activation')
            );
            printf('<div class="%1$s"><p>%2$s</p></div>', esc_attr($class), wp_kses_post($message));
        }
    }

private function init_components() {
    // Make sure loader class exists before initializing
    require_once MXCHAT_FORMS_PLUGIN_DIR . 'includes/class-mxchat-forms-loader.php';
    
    // Initialize the activator and update database schema if needed
    require_once MXCHAT_FORMS_PLUGIN_DIR . 'includes/class-mxchat-forms-activator.php';
    MxChat_Forms_Activator::maybe_update_db_schema();
    
    // Initialize components
    $this->loader = new MxChat_Forms_Loader();
    $this->form_handler = new MxChat_Forms_Form_Handler();
    $this->submissions = new MxChat_Forms_Submissions();
    $this->integrations = new MxChat_Forms_Integrations($this->loader);
    $this->admin = new MxChat_Forms_Admin($this->loader);

    // Add database integrity check for admin users
    add_action('admin_init', array($this, 'check_database_integrity'));
    
    // Run the loader
    $this->loader->run();

    // Enqueue scripts only if pro is active
    if (!is_admin()) {
        add_action('wp_enqueue_scripts', array($this, 'enqueue_public_scripts'));
    }
}

// Add to MxChat_Forms class
public function check_database_integrity() {
    // Only run for admin users
    if (!current_user_can('manage_options')) {
        return;
    }
    
    global $wpdb;
    $forms_table = $wpdb->prefix . MxChat_Forms_Activator::TABLE_FORMS;
    $intents_table = $wpdb->prefix . 'mxchat_intents';
    
    // Check for regex_pattern column in both tables
    $regex_column_forms = $wpdb->get_results("SHOW COLUMNS FROM {$forms_table} LIKE 'regex_pattern'");
    $regex_column_intents = $wpdb->get_results("SHOW COLUMNS FROM {$intents_table} LIKE 'regex_pattern'");
    
    // If either column is missing, show admin notice
    if (empty($regex_column_forms) || empty($regex_column_intents)) {
        add_action('admin_notices', function() {
            ?>
            <div class="notice notice-error is-dismissible">
                <p><?php _e('<strong>MxChat Form Collection</strong> database update issue detected. Please deactivate and reactivate the plugin to resolve this issue.', 'mxchat-forms'); ?></p>
            </div>
            <?php
        });
    }
}

    public function enqueue_public_scripts() {
        // Enqueue CSS
        wp_enqueue_style(
            'mxchat-forms-public',
            MXCHAT_FORMS_PLUGIN_URL . 'public/css/public.css',
            array(),
            MXCHAT_FORMS_VERSION
        );
    
        // Enqueue JavaScript
        wp_enqueue_script(
            'mxchat-forms-public',
            MXCHAT_FORMS_PLUGIN_URL . 'public/js/public.js',
            array('jquery'), 
            MXCHAT_FORMS_VERSION,
            true
        );
    
        wp_localize_script('mxchat-forms-public', 'mxchatForms', array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('mxchat_forms_nonce'),
        ));
    }

    public function __clone() {
        _doing_it_wrong(__FUNCTION__, __('Cheatin&#8217; huh?', 'mxchat-forms'), '1.0.6');
    }

    public function __wakeup() {
        _doing_it_wrong(__FUNCTION__, __('Cheatin&#8217; huh?', 'mxchat-forms'), '1.0.6');
    }
}

/**
 * Returns the main instance of MxChat_Forms
 */
function MxChat_Forms() {
    return MxChat_Forms::instance();
}

// Initialize the plugin
add_action('plugins_loaded', 'MxChat_Forms');

// Activation hook
register_activation_hook(__FILE__, function() {
    require_once MXCHAT_FORMS_PLUGIN_DIR . 'includes/class-mxchat-forms-activator.php';
    MxChat_Forms_Activator::activate();
});