(function() {
    'use strict';

    // Store form validation state
    const formValidation = {
        isValid: false,
        errors: new Set()
    };

    // Track whether input is currently blocked
    let inputIsBlocked = false;

    // Initialize form validation when forms are loaded
    document.addEventListener('DOMContentLoaded', function() {
        console.log("DOMContentLoaded: Initializing forms.");
        initializeForms();
    });

    // Initialize new forms that might be dynamically added
    const observer = new MutationObserver(function(mutations) {
        mutations.forEach(function(mutation) {
            mutation.addedNodes.forEach(function(node) {
                if (node.nodeType === 1) { // Only process Element nodes
                    // Check if the node itself is a form
                    if (node.classList && node.classList.contains('mxchat-form')) {
                        console.log("MutationObserver: New mxchat-form detected.");
                        initializeForm(node);
                        checkIfFormBlocksChat(node);
                    }
                    
                    // Also check if the node contains any forms
                    const forms = node.querySelectorAll('.mxchat-form');
                    forms.forEach(form => {
                        console.log("MutationObserver: New mxchat-form detected within node.");
                        initializeForm(form);
                        checkIfFormBlocksChat(form);
                    });
                }
            });
        });
    });

    // Start observing for dynamically added forms
    observer.observe(document.body, {
        childList: true,
        subtree: true
    });

    function initializeForms() {
        document.querySelectorAll('.mxchat-form').forEach(form => {
            initializeForm(form);
            checkIfFormBlocksChat(form);
        });
    }

    function initializeForm(form) {
        // Add input event listeners for validation
        form.querySelectorAll('input, textarea').forEach(input => {
            input.addEventListener('input', function() {
                validateField(this);
                updateSubmitButton(form);
            });
            input.addEventListener('blur', function() {
                validateField(this);
                updateSubmitButton(form);
            });
        });
    }

    function checkIfFormBlocksChat(form) {
        if (form.dataset.blocksChat === "1") {
            console.log("Form blocks chat input - disabling chat input.");
            blockChatInput();
        }
    }

    function blockChatInput() {
        const chatInput = document.getElementById('chat-input');
        const sendButton = document.getElementById('send-button');
        
        if (chatInput && sendButton) {
            chatInput.disabled = true;
            chatInput.placeholder = 'Please submit the form above to continue...';
            sendButton.disabled = true;
            sendButton.style.opacity = '0.5';
            inputIsBlocked = true;
        }
    }

    function unblockChatInput() {
        if (inputIsBlocked) {
            const chatInput = document.getElementById('chat-input');
            const sendButton = document.getElementById('send-button');
            
            if (chatInput && sendButton) {
                chatInput.disabled = false;
                chatInput.placeholder = 'How can I assist?';
                sendButton.disabled = false;
                sendButton.style.opacity = '1';
                inputIsBlocked = false;
            }
        }
    }

    function validateField(field) {
        const fieldWrapper = field.closest('.mxchat-form-field');
        const validationMessage = fieldWrapper ? fieldWrapper.querySelector('.mxchat-validation-message') : null;
        let isValid = true;
        let message = '';

        // Required field validation
        if (field.hasAttribute('required') && !field.value.trim()) {
            isValid = false;
            message = 'This field is required';
        }

        // Email validation
        if (field.type === 'email' && field.value.trim()) {
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            if (!emailRegex.test(field.value)) {
                isValid = false;
                message = 'Please enter a valid email address';
            }
        }

        // Phone validation
        if (field.type === 'tel' && field.value.trim()) {
            const phoneRegex = /^[\d\s\-+()]{7,}$/;
            if (!phoneRegex.test(field.value)) {
                isValid = false;
                message = 'Please enter a valid phone number';
            }
        }

        // Update validation state
        if (!isValid) {
            formValidation.errors.add(field.name);
            if (fieldWrapper) fieldWrapper.classList.add('error');
            if (validationMessage) {
                validationMessage.textContent = message;
            }
        } else {
            formValidation.errors.delete(field.name);
            if (fieldWrapper) fieldWrapper.classList.remove('error');
        }

        formValidation.isValid = formValidation.errors.size === 0;
    }

    function updateSubmitButton(form) {
        const submitButton = form.querySelector('.mxchat-submit-form');
        if (submitButton) {
            submitButton.disabled = !formValidation.isValid;
        }
    }

    // Global form submission handler
    window.mxchatSubmitForm = function(button) {
        console.log("mxchatSubmitForm triggered");
        const form = button.closest('.mxchat-form');
        
        // Validate all fields before submission
        form.querySelectorAll('input, textarea').forEach(validateField);
        
        if (!formValidation.isValid) {
            console.log("Form is not valid, aborting submission");
            return;
        }

        // Prepare form data
        const formData = new FormData();
        formData.append('action', 'mxchat_submit_form');
        formData.append('nonce', mxchatForms.nonce);
        formData.append('form_id', form.dataset.formId);
        formData.append('session_id', form.dataset.sessionId || '');
        formData.append('user_id', form.dataset.userId || '0');
        
        // Add user and session data if available
        if (typeof mxchatForms.userId !== 'undefined') {
            formData.append('user_id', mxchatForms.userId);
        }
        if (typeof mxchatForms.sessionId !== 'undefined') {
            formData.append('session_id', mxchatForms.sessionId);
        }

        // Add form fields
        form.querySelectorAll('input, textarea').forEach(input => {
            formData.append(input.name, input.value);
        });

        // Show loading state
        form.classList.add('loading');
        button.disabled = true;

        console.log("Submitting form with ID: " + form.dataset.formId);
        
        // Submit form via fetch
        fetch(mxchatForms.ajaxurl, {
            method: 'POST',
            body: formData,
            credentials: 'same-origin'
        })
        .then(response => {
            console.log("Fetch response received.");
            return response.json();
        })
        .then(data => {
            if (data.success) {
                console.log("Form submission successful.");
                // Retrieve the custom response message from the data attribute
                const responseMessage = form.dataset.responseMessage || 
                    'Thank you. General response time is within 24h. Please also check spam. What else can I help you with?';
                // Show success message
                form.innerHTML = '<p class="success">' + responseMessage + '</p>';
                
                // Unblock chat input if it was blocked by this form
                unblockChatInput();
                
                // Trigger success event
                const event = new CustomEvent('mxchatFormSubmitted', {
                    detail: { formId: form.dataset.formId }
                });
                document.dispatchEvent(event);
            } else {
                throw new Error(data.data.message || 'Submission failed');
            }
        })
        .catch(error => {
            console.error('Form submission error:', error);
            // Show error message
            const errorMessage = document.createElement('p');
            errorMessage.className = 'error';
            errorMessage.textContent = 'Error submitting form. Please try again.';
            form.insertBefore(errorMessage, button);
            
            // Reset loading state
            form.classList.remove('loading');
            button.disabled = false;
        });
    };

    // Attach a global click listener for submit buttons (removes inline onclick requirement)
    document.addEventListener('click', function(event) {
        if (event.target && event.target.classList.contains('mxchat-submit-form')) {
            event.preventDefault();
            console.log("Submit button clicked.");
            mxchatSubmitForm(event.target);
        }
    });
})();