/**
 * MxChat Vision Admin JavaScript
 * Handles admin interface interactions and settings management
 */

(function($) {
    'use strict';

    class MxChatVisionAdmin {
        constructor() {
            this.isTestingAPI = false;
            this.statsUpdateInterval = null;
            
            this.init();
        }

        init() {
            this.setupEventListeners();
            this.initializeToggles();
            this.loadAPIStatus();
            this.setupStatsRefresh();
            this.initializeTooltips();
        }

        setupEventListeners() {
            // Vision enabled toggle
            $('#mxchat_vision_enabled').on('change', () => {
                this.handleVisionToggle();
            });

            // Model selection change
            $('#mxchat_vision_model').on('change', () => {
                this.handleModelChange();
            });

            // API key test buttons
            $(document).on('click', '.test-api-key', (e) => {
                e.preventDefault();
                const model = $(e.target).data('model');
                this.testAPIKey(model);
            });

            // Refresh API status
            $(document).on('click', '#refresh-api-status', (e) => {
                e.preventDefault();
                this.loadAPIStatus(true);
            });

            // Refresh statistics
            $(document).on('click', '#refresh-stats', (e) => {
                e.preventDefault();
                this.loadStatistics(true);
            });

            // Clear statistics
            $(document).on('click', '#clear-stats', (e) => {
                e.preventDefault();
                this.clearStatistics();
            });

            // Settings form submission
            $('#mxchat-vision-settings-form').on('submit', (e) => {
                this.handleFormSubmission(e);
            });

            // Import/Export settings
            $('#export-settings').on('click', () => {
                this.exportSettings();
            });

            $('#import-settings').on('change', (e) => {
                this.importSettings(e.target.files[0]);
            });

            // Help toggle
            $('.help-toggle').on('click', function() {
                $(this).next('.help-content').slideToggle();
                $(this).find('.help-arrow').toggleClass('expanded');
            });

            // Advanced settings toggle
            $('#show-advanced-settings').on('change', () => {
                $('.advanced-setting').toggle();
            });
        }

        initializeToggles() {
            // Set initial state of dependent settings
            this.handleVisionToggle();
            
            // Initialize any custom toggles
            $('.mxchat-toggle-switch input').each(function() {
                const $toggle = $(this);
                const $dependent = $('.' + $toggle.attr('id') + '-dependent');
                
                if ($dependent.length) {
                    $dependent.toggle($toggle.is(':checked'));
                    
                    $toggle.on('change', function() {
                        $dependent.toggle($(this).is(':checked'));
                    });
                }
            });
        }

        handleVisionToggle() {
            const isEnabled = $('#mxchat_vision_enabled').is(':checked');
            
            // Show/hide dependent settings
            $('.vision-dependent-setting').toggle(isEnabled);
            
            // Update status indicators
            this.updateFeatureStatus(isEnabled);
            
            // Load API status if enabled
            if (isEnabled) {
                this.loadAPIStatus();
            }
        }

        handleModelChange() {
            const selectedModel = $('#mxchat_vision_model').val();
            
            // Update model-specific information
            this.updateModelInfo(selectedModel);
            
            // Update API status for selected model
            this.updateAPIStatusForModel(selectedModel);
        }

        updateModelInfo(model) {
            const modelInfo = {
                'openai': {
                    name: 'OpenAI Vision (GPT-4o)',
                    description: 'Advanced image analysis with high accuracy. Good for detailed descriptions and complex scenes.',
                    pricing: 'Pay per token usage',
                    strengths: ['Detailed analysis', 'Text recognition', 'Complex scenes']
                },
                'grok': {
                    name: 'Grok Vision',
                    description: 'Fast and efficient image analysis. Good for general purpose image understanding.',
                    pricing: 'Pay per request',
                    strengths: ['Fast processing', 'General analysis', 'Cost effective']
                }
            };

            const info = modelInfo[model];
            if (info) {
                $('#model-info-name').text(info.name);
                $('#model-info-description').text(info.description);
                $('#model-info-pricing').text(info.pricing);
                
                const strengthsList = $('#model-info-strengths');
                strengthsList.empty();
                info.strengths.forEach(strength => {
                    strengthsList.append(`<li>${strength}</li>`);
                });
            }
        }

        updateAPIStatusForModel(model) {
            const statusElements = {
                'openai': $('#openai-status'),
                'grok': $('#grok-status')
            };

            // Highlight selected model status
            Object.keys(statusElements).forEach(key => {
                const element = statusElements[key];
                element.toggleClass('selected-model', key === model);
            });
        }

        updateFeatureStatus(isEnabled) {
            const statusElement = $('#vision-feature-status');
            const statusText = isEnabled ? 'Enabled' : 'Disabled';
            const statusClass = isEnabled ? 'status-enabled' : 'status-disabled';
            
            statusElement
                .removeClass('status-enabled status-disabled')
                .addClass(statusClass)
                .text(statusText);
        }

        loadAPIStatus(showLoading = false) {
            if (showLoading) {
                $('#api-status-loading').show();
            }

            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'mxchat_vision_check_api_status',
                    nonce: mxchatVisionAdmin.nonce
                },
                success: (response) => {
                    if (response.success) {
                        this.displayAPIStatus(response.data);
                    } else {
                        this.showNotice('Error loading API status: ' + response.data.message, 'error');
                    }
                },
                error: () => {
                    this.showNotice('Failed to load API status', 'error');
                },
                complete: () => {
                    $('#api-status-loading').hide();
                }
            });
        }

        displayAPIStatus(status) {
            const providers = ['openai', 'grok'];
            
            providers.forEach(provider => {
                const isActive = status[provider];
                const element = $(`#${provider}-status`);
                const indicator = element.find('.status-indicator');
                const text = element.find('.status-text');
                const testBtn = element.find('.test-api-key');
                
                indicator
                    .removeClass('status-active status-inactive')
                    .addClass(isActive ? 'status-active' : 'status-inactive');
                
                text.text(isActive ? 'Active' : 'Not Set');
                testBtn.toggle(isActive);
            });
        }

        testAPIKey(model) {
            if (this.isTestingAPI) return;
            
            this.isTestingAPI = true;
            const button = $(`.test-api-key[data-model="${model}"]`);
            const originalText = button.text();
            
            button.text('Testing...').prop('disabled', true);

            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'mxchat_vision_test_api',
                    model: model,
                    nonce: mxchatVisionAdmin.nonce
                },
                success: (response) => {
                    if (response.success) {
                        this.showNotice(`${model.toUpperCase()} API test successful!`, 'success');
                        this.updateTestResult(model, true, response.data.message);
                    } else {
                        this.showNotice(`${model.toUpperCase()} API test failed: ${response.data.message}`, 'error');
                        this.updateTestResult(model, false, response.data.message);
                    }
                },
                error: () => {
                    this.showNotice(`${model.toUpperCase()} API test failed - connection error`, 'error');
                    this.updateTestResult(model, false, 'Connection error');
                },
                complete: () => {
                    this.isTestingAPI = false;
                    button.text(originalText).prop('disabled', false);
                }
            });
        }

        updateTestResult(model, success, message) {
            const resultElement = $(`#${model}-test-result`);
            resultElement
                .removeClass('test-success test-error')
                .addClass(success ? 'test-success' : 'test-error')
                .text(message)
                .show();
            
            // Auto-hide after 5 seconds
            setTimeout(() => {
                resultElement.fadeOut();
            }, 5000);
        }

        loadStatistics(showLoading = false) {
            if (showLoading) {
                $('#stats-loading').show();
            }

            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'mxchat_vision_get_stats',
                    nonce: mxchatVisionAdmin.nonce
                },
                success: (response) => {
                    if (response.success) {
                        this.displayStatistics(response.data);
                    }
                },
                complete: () => {
                    $('#stats-loading').hide();
                }
            });
        }

        displayStatistics(stats) {
            // Update main stats
            $('#total-analyses').text(stats.total_analyses || 0);
            $('#successful-analyses').text(stats.successful_analyses || 0);
            $('#failed-analyses').text(stats.failed_analyses || 0);
            
            // Calculate success rate
            const successRate = stats.total_analyses > 0 
                ? Math.round((stats.successful_analyses / stats.total_analyses) * 100)
                : 0;
            $('#success-rate').text(successRate + '%');
            
            // Update model usage
            this.displayModelUsage(stats.model_usage || {});
            
            // Update last updated time
            if (stats.last_updated) {
                $('#stats-last-updated').text(new Date(stats.last_updated).toLocaleString());
            }
        }

        displayModelUsage(modelUsage) {
            const modelsContainer = $('#model-usage-stats');
            modelsContainer.empty();
            
            Object.keys(modelUsage).forEach(model => {
                const usage = modelUsage[model];
                const percentage = usage.total > 0 
                    ? Math.round((usage.successful / usage.total) * 100)
                    : 0;
                
                const modelStats = $(`
                    <div class="model-usage-item">
                        <h4>${model.toUpperCase()}</h4>
                        <div class="usage-stats">
                            <span class="stat">Total: ${usage.total}</span>
                            <span class="stat">Success: ${usage.successful}</span>
                            <span class="stat">Failed: ${usage.failed}</span>
                            <span class="stat">Rate: ${percentage}%</span>
                        </div>
                        <div class="usage-bar">
                            <div class="usage-fill" style="width: ${percentage}%"></div>
                        </div>
                    </div>
                `);
                
                modelsContainer.append(modelStats);
            });
        }

        clearStatistics() {
            if (!confirm('Are you sure you want to clear all statistics? This action cannot be undone.')) {
                return;
            }

            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'mxchat_vision_clear_stats',
                    nonce: mxchatVisionAdmin.nonce
                },
                success: (response) => {
                    if (response.success) {
                        this.showNotice('Statistics cleared successfully', 'success');
                        this.loadStatistics();
                    } else {
                        this.showNotice('Failed to clear statistics', 'error');
                    }
                }
            });
        }

        setupStatsRefresh() {
            // Auto-refresh stats every 30 seconds if page is visible
            this.statsUpdateInterval = setInterval(() => {
                if (!document.hidden && $('#mxchat-vision-admin').is(':visible')) {
                    this.loadStatistics();
                }
            }, 30000);
        }

        handleFormSubmission(e) {
            // Don't prevent default - let WordPress handle the form submission
            // But we can add visual feedback
            const submitButton = $(e.target).find('input[type="submit"]');
            const originalValue = submitButton.val();
            
            submitButton.val('Saving...').prop('disabled', true);
            
            // Re-enable after a short delay (WordPress will redirect anyway)
            setTimeout(() => {
                submitButton.val(originalValue).prop('disabled', false);
            }, 2000);
        }

        exportSettings() {
            const settings = {
                vision_enabled: $('#mxchat_vision_enabled').is(':checked'),
                vision_model: $('#mxchat_vision_model').val(),
                max_images: $('#mxchat_vision_max_images').val(),
                max_file_size: $('#mxchat_vision_max_file_size').val(),
                export_timestamp: new Date().toISOString()
            };

            const dataStr = JSON.stringify(settings, null, 2);
            const dataUri = 'data:application/json;charset=utf-8,'+ encodeURIComponent(dataStr);
            
            const exportFileDefaultName = 'mxchat-vision-settings.json';
            
            const linkElement = document.createElement('a');
            linkElement.setAttribute('href', dataUri);
            linkElement.setAttribute('download', exportFileDefaultName);
            linkElement.click();
        }

        importSettings(file) {
            if (!file) return;
            
            const reader = new FileReader();
            reader.onload = (e) => {
                try {
                    const settings = JSON.parse(e.target.result);
                    
                    // Validate and apply settings
                    if (typeof settings.vision_enabled === 'boolean') {
                        $('#mxchat_vision_enabled').prop('checked', settings.vision_enabled);
                    }
                    
                    if (settings.vision_model) {
                        $('#mxchat_vision_model').val(settings.vision_model);
                    }
                    
                    if (settings.max_images) {
                        $('#mxchat_vision_max_images').val(settings.max_images);
                    }
                    
                    if (settings.max_file_size) {
                        $('#mxchat_vision_max_file_size').val(settings.max_file_size);
                    }
                    
                    // Trigger change events
                    $('#mxchat_vision_enabled').trigger('change');
                    $('#mxchat_vision_model').trigger('change');
                    
                    this.showNotice('Settings imported successfully', 'success');
                    
                } catch (error) {
                    this.showNotice('Invalid settings file', 'error');
                }
            };
            
            reader.readAsText(file);
        }

        initializeTooltips() {
            // Simple tooltip implementation
            $('[data-tooltip]').each(function() {
                const $element = $(this);
                const tooltipText = $element.data('tooltip');
                
                $element.on('mouseenter', function() {
                    const $tooltip = $('<div class="admin-tooltip">' + tooltipText + '</div>');
                    $('body').append($tooltip);
                    
                    const offset = $element.offset();
                    $tooltip.css({
                        top: offset.top - $tooltip.outerHeight() - 5,
                        left: offset.left + ($element.outerWidth() / 2) - ($tooltip.outerWidth() / 2)
                    });
                });
                
                $element.on('mouseleave', function() {
                    $('.admin-tooltip').remove();
                });
            });
        }

        showNotice(message, type = 'info') {
            const noticeClass = type === 'error' ? 'notice-error' : 
                              type === 'success' ? 'notice-success' : 
                              'notice-info';
            
            const notice = $(`
                <div class="notice ${noticeClass} is-dismissible">
                    <p>${message}</p>
                    <button type="button" class="notice-dismiss">
                        <span class="screen-reader-text">Dismiss this notice.</span>
                    </button>
                </div>
            `);
            
            // Remove existing notices
            $('.mxchat-vision-notice').remove();
            
            // Add new notice
            notice.addClass('mxchat-vision-notice');
            $('.wrap h1').after(notice);
            
            // Handle dismiss
            notice.find('.notice-dismiss').on('click', function() {
                notice.fadeOut(() => notice.remove());
            });
            
            // Auto-dismiss after 5 seconds for success/info
            if (type !== 'error') {
                setTimeout(() => {
                    notice.fadeOut(() => notice.remove());
                }, 5000);
            }
        }

        destroy() {
            if (this.statsUpdateInterval) {
                clearInterval(this.statsUpdateInterval);
            }
        }
    }

    // Initialize when document is ready
    $(document).ready(() => {
        // Only initialize on the vision admin page
        if ($('#mxchat-vision-admin').length > 0) {
            window.mxchatVisionAdmin = new MxChatVisionAdmin();
        }
    });

    // Cleanup on page unload
    $(window).on('beforeunload', () => {
        if (window.mxchatVisionAdmin) {
            window.mxchatVisionAdmin.destroy();
        }
    });

})(jQuery);