<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * MxChat Vision Core Class
 * 
 * Handles core functionality and integration with MxChat
 */
class MxChat_Vision_Core {

    /**
     * Constructor
     */
    public function __construct() {
        $this->init_hooks();
    }

    /**
     * Initialize WordPress hooks
     */
    private function init_hooks() {
        // AJAX handlers
        add_action('wp_ajax_mxchat_vision_upload', array($this, 'handle_image_upload'));
        add_action('wp_ajax_nopriv_mxchat_vision_upload', array($this, 'handle_image_upload'));
        add_action('wp_ajax_mxchat_vision_analyze', array($this, 'handle_image_analysis'));
        add_action('wp_ajax_nopriv_mxchat_vision_analyze', array($this, 'handle_image_analysis'));
        
        // Frontend hooks
        add_action('wp_footer', array($this, 'render_image_preview_styles'));
        add_action('wp_enqueue_scripts', array($this, 'maybe_enqueue_vision_assets'));
        
        // Integration with MxChat chat system
        add_filter('mxchat_process_message', array($this, 'process_vision_message'), 10, 2);
        add_filter('mxchat_message_data', array($this, 'add_vision_data_to_message'), 10, 1);
    }

    /**
     * Maybe enqueue vision assets if enabled
     */
    public function maybe_enqueue_vision_assets() {
        if (!MxChat_Vision_Settings::is_enabled()) {
            return;
        }

        // Only enqueue on pages where MxChat is active
        if (!$this->is_mxchat_page()) {
            return;
        }

        wp_enqueue_script(
            'mxchat-vision-frontend',
            MXCHAT_VISION_PLUGIN_URL . 'public/js/vision-frontend.js',
            array('jquery'),
            MXCHAT_VISION_VERSION,
            true
        );

        $settings = MxChat_Vision_Settings::get_settings();

        wp_localize_script('mxchat-vision-frontend', 'mxchatVision', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('mxchat_vision_nonce'),
            'maxFileSize' => ($settings['max_file_size'] * 1024 * 1024), // Convert MB to bytes
            'maxImages' => $settings['max_images'],
            'allowedTypes' => array('image/jpeg', 'image/png', 'image/gif', 'image/webp'),
            'strings' => array(
                'uploadError' => __('Upload failed. Please try again.', 'mxchat-vision'),
                'fileTooLarge' => __('File is too large. Maximum size is', 'mxchat-vision') . ' ' . $settings['max_file_size'] . 'MB',
                'invalidFileType' => __('Invalid file type. Please upload an image.', 'mxchat-vision'),
                'tooManyFiles' => __('Too many files. Maximum is', 'mxchat-vision') . ' ' . $settings['max_images'],
                'analyzing' => __('Analyzing image...', 'mxchat-vision'),
                'analysisError' => __('Analysis failed. Please try again.', 'mxchat-vision')
            )
        ));
    }

    /**
     * Check if current page should have MxChat
     */
    private function is_mxchat_page() {
        // Check if MxChat is set to appear on all pages
        $mxchat_options = get_option('mxchat_options', array());
        
        if (isset($mxchat_options['append_to_body']) && $mxchat_options['append_to_body'] === 'on') {
            return true;
        }

        // Check if page has MxChat shortcode
        global $post;
        if ($post && has_shortcode($post->post_content, 'mxchat_chatbot')) {
            return true;
        }

        return false;
    }

/**
 * Render image preview styles and structure
 */
public function render_image_preview_styles() {
    if (!MxChat_Vision_Settings::is_enabled() || !$this->is_mxchat_page()) {
        return;
    }
    // Add inline CSS for image preview
    echo '<style>
    .vision-image-preview {
        padding: 10px;
    }
    .vision-images-grid {
        display: flex;
        gap: 10px;
        flex-wrap: wrap;
        margin-bottom: 10px;
    }
    .vision-image-item {
        position: relative;
        width: 80px;
        height: 80px;
    }
    .vision-image-wrapper {
        position: relative;
        width: 100%;
        height: 100%;
        border-radius: 6px;
        overflow: hidden;
        box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        cursor: pointer;
    }
    .vision-image-wrapper img {
        width: 100%;
        height: 100%;
        object-fit: cover;
        transition: all 0.2s ease;
    }
    .remove-vision-image {
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: rgba(0, 0, 0, 0.7) !important;
        border: none !important;
        display: flex;
        align-items: center;
        justify-content: center;
        cursor: pointer;
        opacity: 0;
        transition: opacity 0.2s ease !important;
        font-size: 32px;
        font-weight: bold;
        color: white !important;
        outline: none !important;
        box-shadow: none !important;
        text-decoration: none !important;
    }
    .remove-vision-image:hover,
    .remove-vision-image:focus,
    .remove-vision-image:active {
        background: rgba(0, 0, 0, 0.7) !important;
        color: white !important;
        outline: none !important;
        box-shadow: none !important;
        text-decoration: none !important;
        border: none !important;
    }
    .vision-image-wrapper:hover .remove-vision-image {
        opacity: 1;
    }
    .vision-image-name {
        font-size: 10px;
        text-align: center;
        margin-top: 2px;
    }
    .clear-vision-btn {
        background: #6c757d;
        color: white;
        border: none;
        padding: 5px 10px;
        border-radius: 4px;
        font-size: 12px;
        cursor: pointer;
        display: flex;
        align-items: center;
        gap: 5px;
    }
    .vision-chat-image {
        max-width: 200px;
        max-height: 150px;
        border-radius: 8px;
        margin: 5px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    }
    .vision-images-in-chat {
        margin-bottom: 10px;
    }
    .vision-meta {
        font-size: 12px;
        color: #666;
        margin-top: 5px;
    }
    .vision-notification {
        font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
    }
    .typing-indicator {
        display: flex;
        gap: 4px;
        align-items: center;
    }
    .typing-indicator span {
        width: 8px;
        height: 8px;
        border-radius: 50%;
        background: #666;
        animation: typing 1.4s infinite ease-in-out;
    }
    .typing-indicator span:nth-child(2) { animation-delay: 0.2s; }
    .typing-indicator span:nth-child(3) { animation-delay: 0.4s; }
    @keyframes typing {
        0%, 80%, 100% { opacity: 0.3; }
        40% { opacity: 1; }
    }
    .typing-text {
        margin-left: 10px;
        color: #666;
        font-style: italic;
    }
    </style>';
}

    /**
     * Handle image upload via AJAX
     */
    public function handle_image_upload() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'mxchat_vision_nonce')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'mxchat-vision')));
        }

        // Check if vision is enabled
        if (!MxChat_Vision_Settings::is_enabled()) {
            wp_send_json_error(array('message' => __('Vision analysis is disabled.', 'mxchat-vision')));
        }

        // Check if files were uploaded
        if (empty($_FILES['images'])) {
            wp_send_json_error(array('message' => __('No files uploaded.', 'mxchat-vision')));
        }

        $settings = MxChat_Vision_Settings::get_settings();
        $uploaded_files = array();
        $errors = array();

        // Handle multiple files
        $files = $_FILES['images'];
        $file_count = is_array($files['name']) ? count($files['name']) : 1;

        // Check file count limit
        if ($file_count > $settings['max_images']) {
            wp_send_json_error(array('message' => sprintf(__('Too many files. Maximum is %d.', 'mxchat-vision'), $settings['max_images'])));
        }

        // Process each file
        for ($i = 0; $i < $file_count; $i++) {
            $file = array(
                'name' => is_array($files['name']) ? $files['name'][$i] : $files['name'],
                'type' => is_array($files['type']) ? $files['type'][$i] : $files['type'],
                'tmp_name' => is_array($files['tmp_name']) ? $files['tmp_name'][$i] : $files['tmp_name'],
                'error' => is_array($files['error']) ? $files['error'][$i] : $files['error'],
                'size' => is_array($files['size']) ? $files['size'][$i] : $files['size']
            );

            $result = $this->process_uploaded_file($file, $settings);
            
            if ($result['success']) {
                $uploaded_files[] = $result['data'];
            } else {
                $errors[] = $result['message'];
            }
        }

        if (!empty($uploaded_files)) {
            wp_send_json_success(array(
                'files' => $uploaded_files,
                'errors' => $errors
            ));
        } else {
            wp_send_json_error(array(
                'message' => __('All uploads failed.', 'mxchat-vision'),
                'errors' => $errors
            ));
        }
    }

    /**
     * Process individual uploaded file
     */
    private function process_uploaded_file($file, $settings) {
        // Check for upload errors
        if ($file['error'] !== UPLOAD_ERR_OK) {
            return array(
                'success' => false,
                'message' => __('Upload error occurred.', 'mxchat-vision')
            );
        }

        // Check file size
        $max_size = $settings['max_file_size'] * 1024 * 1024; // Convert MB to bytes
        if ($file['size'] > $max_size) {
            return array(
                'success' => false,
                'message' => sprintf(__('File is too large. Maximum size is %dMB.', 'mxchat-vision'), $settings['max_file_size'])
            );
        }

        // Check file type
        $allowed_types = array('image/jpeg', 'image/png', 'image/gif', 'image/webp');
        if (!in_array($file['type'], $allowed_types)) {
            return array(
                'success' => false,
                'message' => __('Invalid file type. Please upload an image.', 'mxchat-vision')
            );
        }

        // Verify it's actually an image
        $image_info = getimagesize($file['tmp_name']);
        if ($image_info === false) {
            return array(
                'success' => false,
                'message' => __('Invalid image file.', 'mxchat-vision')
            );
        }

        // Create upload directory if it doesn't exist
        $upload_dir = wp_upload_dir();
        $vision_dir = $upload_dir['basedir'] . '/mxchat-vision/';
        
        if (!file_exists($vision_dir)) {
            wp_mkdir_p($vision_dir);
        }

        // Generate unique filename
        $file_extension = pathinfo($file['name'], PATHINFO_EXTENSION);
        $unique_filename = wp_unique_filename($vision_dir, uniqid() . '.' . $file_extension);
        $file_path = $vision_dir . $unique_filename;

        // Move uploaded file
        if (!move_uploaded_file($file['tmp_name'], $file_path)) {
            return array(
                'success' => false,
                'message' => __('Failed to save uploaded file.', 'mxchat-vision')
            );
        }

        // Encode image to base64 for API transmission
        $image_data = file_get_contents($file_path);
        $base64_image = base64_encode($image_data);

        return array(
            'success' => true,
            'data' => array(
                'filename' => $unique_filename,
                'original_name' => $file['name'],
                'file_path' => $file_path,
                'file_url' => $upload_dir['baseurl'] . '/mxchat-vision/' . $unique_filename,
                'base64' => $base64_image,
                'mime_type' => $file['type'],
                'file_size' => $file['size'],
                'dimensions' => array(
                    'width' => $image_info[0],
                    'height' => $image_info[1]
                )
            )
        );
    }

    /**
     * Handle image analysis via AJAX
     */
    public function handle_image_analysis() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'mxchat_vision_nonce')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'mxchat-vision')));
        }

        // Check if vision is enabled
        if (!MxChat_Vision_Settings::is_enabled()) {
            wp_send_json_error(array('message' => __('Vision analysis is disabled.', 'mxchat-vision')));
        }

        // Get posted data
        $images = isset($_POST['images']) ? json_decode(stripslashes($_POST['images']), true) : array();
        $prompt = isset($_POST['prompt']) ? sanitize_textarea_field($_POST['prompt']) : '';

        if (empty($images)) {
            wp_send_json_error(array('message' => __('No images provided for analysis.', 'mxchat-vision')));
        }

        if (empty($prompt)) {
            $prompt = __('Please analyze this image and describe what you see.', 'mxchat-vision');
        }

        try {
            // Initialize processor and models
            $processor = new Vision_Processor();
            $models = new Vision_Models();

            // Get selected model
            $settings = MxChat_Vision_Settings::get_settings();
            $selected_model = $settings['model'];

            // Process images and get analysis
            $analysis_result = $processor->analyze_images($images, $prompt, $selected_model);

            if ($analysis_result['success']) {
                // Check if this is an edited image result
                if (isset($analysis_result['is_edited_image']) && $analysis_result['is_edited_image']) {
                    wp_send_json_success(array(
                        'is_edited_image' => true,
                        'image_url' => $analysis_result['image_url'],
                        'model_used' => $analysis_result['model'],
                        'processing_time' => $analysis_result['processing_time']
                    ));
                } else {
                    // Regular analysis result
                    wp_send_json_success(array(
                        'analysis' => $analysis_result['analysis'],
                        'model_used' => $selected_model,
                        'images_processed' => count($images)
                    ));
                }
            } else {
                wp_send_json_error(array(
                    'message' => $analysis_result['message']
                ));
            }

        } catch (Exception $e) {
            wp_send_json_error(array(
                'message' => __('Analysis failed: ', 'mxchat-vision') . $e->getMessage()
            ));
        }
    }

    /**
     * Process vision message and integrate with MxChat chat handler
     */
    public function process_vision_message($message_data, $user_message) {
        // Check if this message contains vision analysis (from our frontend)
        if (!isset($_POST['vision_processed']) || empty($_POST['vision_images_count'])) {
            return $message_data;
        }

        // This message already contains the vision analysis
        // Just let it pass through to the regular AI processing
        $message_data['has_vision_analysis'] = true;
        $message_data['vision_images_count'] = intval($_POST['vision_images_count']);
        
        return $message_data;
    }

    /**
     * Add vision data to message for frontend display
     */
    public function add_vision_data_to_message($message_data) {
        // Add vision-specific data that the frontend can use
        if (isset($message_data['vision_analysis'])) {
            $message_data['has_vision'] = true;
            $message_data['vision_display'] = true;
        }

        return $message_data;
    }

    /**
     * Clean up old uploaded files (called via cron or manually)
     */
    public static function cleanup_old_files() {
        $upload_dir = wp_upload_dir();
        $vision_dir = $upload_dir['basedir'] . '/mxchat-vision/';

        if (!file_exists($vision_dir)) {
            return;
        }

        $files = glob($vision_dir . '*');
        $cleanup_time = time() - (24 * 60 * 60); // 24 hours ago

        foreach ($files as $file) {
            if (is_file($file) && filemtime($file) < $cleanup_time) {
                unlink($file);
            }
        }
    }

    /**
     * Get vision statistics
     */
    public static function get_vision_stats() {
        $upload_dir = wp_upload_dir();
        $vision_dir = $upload_dir['basedir'] . '/mxchat-vision/';

        $stats = array(
            'files_count' => 0,
            'total_size' => 0,
            'enabled' => MxChat_Vision_Settings::is_enabled()
        );

        if (file_exists($vision_dir)) {
            $files = glob($vision_dir . '*');
            $stats['files_count'] = count($files);
            
            foreach ($files as $file) {
                if (is_file($file)) {
                    $stats['total_size'] += filesize($file);
                }
            }
        }

        return $stats;
    }
}
?>