<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * MxChat Vision Settings Class
 * 
 * Handles admin interface and settings management
 */
class MxChat_Vision_Settings {

    /**
     * Constructor
     */
    public function __construct() {
        add_action('admin_init', array($this, 'init_settings'));
        add_action('wp_ajax_mxchat_vision_save_settings', array($this, 'save_settings'));
    }

    /**
     * Initialize settings
     */
    public function init_settings() {
        register_setting('mxchat_vision_settings', 'mxchat_vision_enabled');
        register_setting('mxchat_vision_settings', 'mxchat_vision_model');
        register_setting('mxchat_vision_settings', 'mxchat_vision_mode');
        register_setting('mxchat_vision_settings', 'mxchat_vision_max_images');
        register_setting('mxchat_vision_settings', 'mxchat_vision_max_file_size');
        register_setting('mxchat_vision_settings', 'mxchat_vision_custom_prompt');
        register_setting('mxchat_vision_settings', 'mxchat_vision_editing_prompt');
        register_setting('mxchat_vision_settings', 'mxchat_vision_editing_model');
        register_setting('mxchat_vision_settings', 'mxchat_vision_image_size');
        register_setting('mxchat_vision_settings', 'mxchat_vision_aspect_ratio');
        register_setting('mxchat_vision_settings', 'mxchat_vision_disable_input_on_editing');
        register_setting('mxchat_vision_settings', 'mxchat_vision_edited_image_message');
    }

    /**
     * Display admin page
     */
    public function display_admin_page() {
        // Check user permissions
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have sufficient permissions to access this page.', 'mxchat-vision'));
        }

        // Get current settings
        $vision_enabled = get_option('mxchat_vision_enabled', false);
        $selected_model = get_option('mxchat_vision_model', 'openai');
        $vision_mode = get_option('mxchat_vision_mode', 'analysis');
        $max_images = get_option('mxchat_vision_max_images', 5);
        $max_file_size = get_option('mxchat_vision_max_file_size', 10);
        $custom_prompt = get_option('mxchat_vision_custom_prompt', 'Analyze this image in detail. Describe what you see, including any text, objects, people, colors, and overall context.');
        $editing_prompt = get_option('mxchat_vision_editing_prompt', 'Apply the following modifications to this image:');
        $editing_model = get_option('mxchat_vision_editing_model', 'gemini-2.5-flash-image');
        $image_size = get_option('mxchat_vision_image_size', '1K');
        $aspect_ratio = get_option('mxchat_vision_aspect_ratio', '1:1');
        $disable_input_on_editing = get_option('mxchat_vision_disable_input_on_editing', false);
        $edited_image_message = get_option('mxchat_vision_edited_image_message', "Here's your edited image:");

        // Check if API keys are available
        $api_keys_status = $this->check_api_keys_status();

        // Available models
        $available_models = array(
            'openai' => array(
                'name' => __('OpenAI Vision (GPT-4V)', 'mxchat-vision'),
                'available' => $api_keys_status['openai'],
                'supports_editing' => false
            ),
            'grok' => array(
                'name' => __('Grok Vision', 'mxchat-vision'),
                'available' => $api_keys_status['grok'],
                'supports_editing' => false
            ),
            'google_imagen' => array(
                'name' => __('Google Gemini 2.5 Flash (Editing)', 'mxchat-vision'),
                'available' => $api_keys_status['google'],
                'supports_editing' => true
            ),
            'google_imagen_pro' => array(
                'name' => __('Google Gemini 3 Pro (Editing)', 'mxchat-vision'),
                'available' => $api_keys_status['google'],
                'supports_editing' => true
            )
        );

        ?>
        <div class="mxchat-vision-wrapper">
            <!-- Hero Section -->
            <div class="mxchat-vision-hero">
                <div class="mxchat-vision-hero-content">
                    <h1 class="mxchat-vision-title">
                        <span class="mxchat-gradient-text">Image</span> Analysis
                    </h1>
                    <p class="mxchat-vision-subtitle">
                        <?php _e('Enhance your AI chatbot with powerful image analysis capabilities. Upload, analyze, and get intelligent insights from images.', 'mxchat-vision'); ?>
                    </p>
                </div>
            </div>

            <!-- Main Content -->
            <div class="mxchat-vision-content">
                <?php if (isset($_GET['settings-updated']) && $_GET['settings-updated']): ?>
                    <div class="mxchat-notice mxchat-notice-success">
                        <span class="dashicons dashicons-yes-alt"></span>
                        <p><?php _e('Settings saved successfully!', 'mxchat-vision'); ?></p>
                    </div>
                <?php endif; ?>

                <div class="mxchat-vision-admin-container">
                    <!-- Main Settings Column -->
                    <div class="mxchat-vision-main-content">
                        
                        <!-- Feature Status Card -->
                        <div class="mxchat-vision-card">
                            <div class="mxchat-card-header">
                                <h2><?php _e('Image Feature Status', 'mxchat-vision'); ?></h2>
                                <div class="feature-status-badge <?php echo $vision_enabled ? 'status-enabled' : 'status-disabled'; ?>">
                                    <span class="status-dot"></span>
                                    <?php echo $vision_enabled ? __('Enabled', 'mxchat-vision') : __('Disabled', 'mxchat-vision'); ?>
                                </div>
                            </div>
                            <div class="mxchat-card-content">
                                <?php if ($vision_enabled): ?>
                                    <div class="feature-overview-grid">
                                        <div class="overview-item">
                                            <div class="overview-icon">🤖</div>
                                            <div class="overview-details">
                                                <strong><?php _e('Active Model', 'mxchat-vision'); ?></strong>
                                                <span><?php echo esc_html($available_models[$selected_model]['name']); ?></span>
                                            </div>
                                        </div>
                                        <div class="overview-item">
                                            <div class="overview-icon">📸</div>
                                            <div class="overview-details">
                                                <strong><?php _e('Max Images', 'mxchat-vision'); ?></strong>
                                                <span><?php echo esc_html($max_images); ?> per message</span>
                                            </div>
                                        </div>
                                        <div class="overview-item">
                                            <div class="overview-icon">📏</div>
                                            <div class="overview-details">
                                                <strong><?php _e('Max File Size', 'mxchat-vision'); ?></strong>
                                                <span><?php echo esc_html($max_file_size); ?>MB</span>
                                            </div>
                                        </div>
                                    </div>
                                <?php else: ?>
                                    <div class="feature-disabled-state">
                                        <div class="disabled-icon">⚠️</div>
                                        <p><?php _e('Vision analysis is currently disabled. Enable it below to start analyzing images with AI.', 'mxchat-vision'); ?></p>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>

                        <!-- Settings Form -->
                        <div class="mxchat-vision-card">
                            <div class="mxchat-card-header">
                                <h2><?php _e('Configuration', 'mxchat-vision'); ?></h2>
                            </div>
                            <div class="mxchat-card-content">
                                <form method="post" action="options.php" class="mxchat-vision-form">
                                    <?php settings_fields('mxchat_vision_settings'); ?>
                                    
                                    <!-- Enable Vision Toggle -->
                                    <div class="mxchat-form-row">
                                        <div class="mxchat-form-label">
                                            <label for="mxchat_vision_enabled">
                                                <?php _e('Enable Vision Analysis', 'mxchat-vision'); ?>
                                            </label>
                                        </div>
                                        <div class="mxchat-form-control">
                                            <label class="mxchat-toggle-switch">
                                                <input type="checkbox" 
                                                       id="mxchat_vision_enabled" 
                                                       name="mxchat_vision_enabled" 
                                                       value="1" 
                                                       <?php checked($vision_enabled, true); ?>>
                                                <span class="mxchat-toggle-slider"></span>
                                            </label>
                                            <p class="mxchat-form-description">
                                                <?php _e('Enable image analysis functionality in the chatbot. When enabled, users will see a camera icon in the chat toolbar.', 'mxchat-vision'); ?>
                                            </p>
                                        </div>
                                    </div>

                                    <!-- Dependent Settings -->
                                    <div id="vision-dependent-settings" class="vision-dependent-settings" <?php echo !$vision_enabled ? 'style="display:none;"' : ''; ?>>

                                        <!-- Vision Mode Selection -->
                                        <div class="mxchat-form-row">
                                            <div class="mxchat-form-label">
                                                <label for="mxchat_vision_mode">
                                                    <?php _e('Vision Mode', 'mxchat-vision'); ?>
                                                </label>
                                            </div>
                                            <div class="mxchat-form-control">
                                                <div class="mode-selection-group">
                                                    <label class="mode-option">
                                                        <input type="radio"
                                                               name="mxchat_vision_mode"
                                                               value="analysis"
                                                               <?php checked($vision_mode, 'analysis'); ?>>
                                                        <span class="mode-label">
                                                            <strong><?php _e('Image Analysis', 'mxchat-vision'); ?></strong>
                                                            <small><?php _e('Analyze images and return text descriptions', 'mxchat-vision'); ?></small>
                                                        </span>
                                                    </label>
                                                    <label class="mode-option">
                                                        <input type="radio"
                                                               name="mxchat_vision_mode"
                                                               value="editing"
                                                               <?php checked($vision_mode, 'editing'); ?>>
                                                        <span class="mode-label">
                                                            <strong><?php _e('Image Editing', 'mxchat-vision'); ?></strong>
                                                            <small><?php _e('Edit images using Google Gemini 2.5 Flash and return modified images', 'mxchat-vision'); ?></small>
                                                        </span>
                                                    </label>
                                                </div>
                                                <p class="mxchat-form-description">
                                                    <?php _e('Choose whether to analyze images (text response) or edit images (returns modified image)', 'mxchat-vision'); ?>
                                                </p>
                                            </div>
                                        </div>

                                        <!-- === ANALYSIS MODE SETTINGS === -->

                                        <!-- AI Model Selection (Analysis Mode Only) -->
                                        <div class="mxchat-form-row analysis-only-setting" style="<?php echo $vision_mode !== 'analysis' ? 'display:none;' : ''; ?>">
                                            <div class="mxchat-form-label">
                                                <label for="mxchat_vision_model">
                                                    <?php _e('Analysis Model', 'mxchat-vision'); ?>
                                                </label>
                                            </div>
                                            <div class="mxchat-form-control">
                                                <select id="mxchat_vision_model" name="mxchat_vision_model" class="mxchat-select">
                                                    <?php
                                                    // Only show analysis models (non-editing models)
                                                    foreach ($available_models as $model_key => $model_info):
                                                        if ($model_info['supports_editing']) continue; // Skip editing models
                                                    ?>
                                                        <option value="<?php echo esc_attr($model_key); ?>"
                                                                <?php selected($selected_model, $model_key); ?>
                                                                <?php echo !$model_info['available'] ? 'disabled' : ''; ?>>
                                                            <?php echo esc_html($model_info['name']); ?>
                                                            <?php if (!$model_info['available']): ?>
                                                                (<?php _e('API Key Required', 'mxchat-vision'); ?>)
                                                            <?php endif; ?>
                                                        </option>
                                                    <?php endforeach; ?>
                                                </select>
                                                <p class="mxchat-form-description">
                                                    <?php _e('Select the AI model for analyzing images and returning text descriptions.', 'mxchat-vision'); ?>
                                                </p>
                                            </div>
                                        </div>

                                        <!-- Custom Analysis Prompt (Analysis Mode Only) -->
                                        <div class="mxchat-form-row analysis-only-setting" style="<?php echo $vision_mode !== 'analysis' ? 'display:none;' : ''; ?>">
                                            <div class="mxchat-form-label">
                                                <label for="mxchat_vision_custom_prompt">
                                                    <?php _e('Analysis Prompt', 'mxchat-vision'); ?>
                                                </label>
                                            </div>
                                            <div class="mxchat-form-control">
                                                <textarea
                                                    id="mxchat_vision_custom_prompt"
                                                    name="mxchat_vision_custom_prompt"
                                                    rows="4"
                                                    class="mxchat-textarea"
                                                    placeholder="<?php esc_attr_e('Enter your custom instructions for image analysis...', 'mxchat-vision'); ?>"
                                                ><?php echo esc_textarea($custom_prompt); ?></textarea>
                                                <p class="mxchat-form-description">
                                                    <?php _e('Customize how the AI analyzes images. The user\'s specific question will automatically be added after your instructions.', 'mxchat-vision'); ?>
                                                </p>

                                                <!-- Prompt Examples -->
                                                <details class="mxchat-prompt-examples" style="margin-top: 10px;">
                                                    <summary style="cursor: pointer; font-weight: bold; color: #0073aa;">
                                                        <?php _e('📝 View Example Prompts', 'mxchat-vision'); ?>
                                                    </summary>
                                                    <div style="margin-top: 10px; padding: 15px; background: #f8f9fa; border-radius: 6px; font-size: 13px;">
                                                        <p><strong><?php _e('Professional Analysis:', 'mxchat-vision'); ?></strong><br>
                                                        <code>You are an expert image analyst. Examine this image systematically and provide: 1) A comprehensive description, 2) Key visual elements, 3) Any text content, 4) Professional insights.</code></p>

                                                        <p><strong><?php _e('Simple & Friendly:', 'mxchat-vision'); ?></strong><br>
                                                        <code>Look at this image and be detailed but conversational in your response.</code></p>

                                                        <p><strong><?php _e('OCR Focus:', 'mxchat-vision'); ?></strong><br>
                                                        <code>First, extract and transcribe all readable text, numbers, or written content from this image. Then provide a detailed description of the visual elements.</code></p>
                                                    </div>
                                                </details>
                                            </div>
                                        </div>

                                        <!-- === EDITING MODE SETTINGS === -->

                                        <!-- Image Editing Model Selection -->
                                        <div class="mxchat-form-row editing-only-setting" style="<?php echo $vision_mode !== 'editing' ? 'display:none;' : ''; ?>">
                                            <div class="mxchat-form-label">
                                                <label for="mxchat_vision_editing_model">
                                                    <?php _e('Editing Model', 'mxchat-vision'); ?>
                                                </label>
                                            </div>
                                            <div class="mxchat-form-control">
                                                <select id="mxchat_vision_editing_model" name="mxchat_vision_editing_model" class="mxchat-select">
                                                    <option value="gemini-2.5-flash-image" <?php selected($editing_model, 'gemini-2.5-flash-image'); ?>>
                                                        <?php _e('Gemini 2.5 Flash Image (Fast)', 'mxchat-vision'); ?>
                                                    </option>
                                                    <option value="gemini-3-pro-image-preview" <?php selected($editing_model, 'gemini-3-pro-image-preview'); ?>>
                                                        <?php _e('Gemini 3 Pro Image (Advanced - 4K, Text Rendering)', 'mxchat-vision'); ?>
                                                    </option>
                                                </select>
                                                <p class="mxchat-form-description">
                                                    <?php _e('Select the image editing model. Gemini 3 Pro offers higher resolution (up to 4K) and advanced features.', 'mxchat-vision'); ?>
                                                </p>
                                            </div>
                                        </div>

                                        <!-- Image Size (Gemini 3 Pro Only) -->
                                        <div class="mxchat-form-row pro-only-setting" style="<?php echo ($vision_mode !== 'editing' || $editing_model !== 'gemini-3-pro-image-preview') ? 'display:none;' : ''; ?>">
                                            <div class="mxchat-form-label">
                                                <label for="mxchat_vision_image_size">
                                                    <?php _e('Resolution', 'mxchat-vision'); ?>
                                                </label>
                                            </div>
                                            <div class="mxchat-form-control">
                                                <select id="mxchat_vision_image_size" name="mxchat_vision_image_size" class="mxchat-select">
                                                    <option value="1K" <?php selected($image_size, '1K'); ?>>
                                                        <?php _e('1K - 1024x1024 (Fast)', 'mxchat-vision'); ?>
                                                    </option>
                                                    <option value="2K" <?php selected($image_size, '2K'); ?>>
                                                        <?php _e('2K - 2048x2048 (Balanced)', 'mxchat-vision'); ?>
                                                    </option>
                                                    <option value="4K" <?php selected($image_size, '4K'); ?>>
                                                        <?php _e('4K - 4096x4096 (High Quality)', 'mxchat-vision'); ?>
                                                    </option>
                                                </select>
                                                <p class="mxchat-form-description">
                                                    <?php _e('Higher resolutions produce better quality but take longer to generate.', 'mxchat-vision'); ?>
                                                </p>
                                            </div>
                                        </div>

                                        <!-- Aspect Ratio (Gemini 3 Pro Only) -->
                                        <div class="mxchat-form-row pro-only-setting" style="<?php echo ($vision_mode !== 'editing' || $editing_model !== 'gemini-3-pro-image-preview') ? 'display:none;' : ''; ?>">
                                            <div class="mxchat-form-label">
                                                <label for="mxchat_vision_aspect_ratio">
                                                    <?php _e('Aspect Ratio', 'mxchat-vision'); ?>
                                                </label>
                                            </div>
                                            <div class="mxchat-form-control">
                                                <select id="mxchat_vision_aspect_ratio" name="mxchat_vision_aspect_ratio" class="mxchat-select">
                                                    <option value="1:1" <?php selected($aspect_ratio, '1:1'); ?>>
                                                        <?php _e('1:1 - Square', 'mxchat-vision'); ?>
                                                    </option>
                                                    <option value="16:9" <?php selected($aspect_ratio, '16:9'); ?>>
                                                        <?php _e('16:9 - Widescreen', 'mxchat-vision'); ?>
                                                    </option>
                                                    <option value="9:16" <?php selected($aspect_ratio, '9:16'); ?>>
                                                        <?php _e('9:16 - Portrait', 'mxchat-vision'); ?>
                                                    </option>
                                                    <option value="4:3" <?php selected($aspect_ratio, '4:3'); ?>>
                                                        <?php _e('4:3 - Classic', 'mxchat-vision'); ?>
                                                    </option>
                                                    <option value="3:4" <?php selected($aspect_ratio, '3:4'); ?>>
                                                        <?php _e('3:4 - Portrait Classic', 'mxchat-vision'); ?>
                                                    </option>
                                                    <option value="5:4" <?php selected($aspect_ratio, '5:4'); ?>>
                                                        <?php _e('5:4 - Near Square', 'mxchat-vision'); ?>
                                                    </option>
                                                </select>
                                                <p class="mxchat-form-description">
                                                    <?php _e('Choose the aspect ratio for edited images.', 'mxchat-vision'); ?>
                                                </p>
                                            </div>
                                        </div>

                                        <!-- Custom Editing Prompt (Editing Mode Only) -->
                                        <div class="mxchat-form-row editing-only-setting" style="<?php echo $vision_mode !== 'editing' ? 'display:none;' : ''; ?>">
                                            <div class="mxchat-form-label">
                                                <label for="mxchat_vision_editing_prompt">
                                                    <?php _e('Editing Prompt', 'mxchat-vision'); ?>
                                                </label>
                                            </div>
                                            <div class="mxchat-form-control">
                                                <textarea
                                                    id="mxchat_vision_editing_prompt"
                                                    name="mxchat_vision_editing_prompt"
                                                    rows="4"
                                                    class="mxchat-textarea"
                                                    placeholder="<?php esc_attr_e('Enter your custom instructions for image editing...', 'mxchat-vision'); ?>"
                                                ><?php echo esc_textarea($editing_prompt); ?></textarea>
                                                <p class="mxchat-form-description">
                                                    <?php _e('Customize how images are edited. The user\'s editing instructions will be appended after this prompt.', 'mxchat-vision'); ?>
                                                </p>

                                                <!-- Prompt Examples -->
                                                <details class="mxchat-prompt-examples" style="margin-top: 10px;">
                                                    <summary style="cursor: pointer; font-weight: bold; color: #0073aa;">
                                                        <?php _e('📝 View Example Editing Prompts', 'mxchat-vision'); ?>
                                                    </summary>
                                                    <div style="margin-top: 10px; padding: 15px; background: #f8f9fa; border-radius: 6px; font-size: 13px;">
                                                        <p><strong><?php _e('Simple Editing:', 'mxchat-vision'); ?></strong><br>
                                                        <code>Edit this image based on the following instructions:</code></p>

                                                        <p><strong><?php _e('Professional Enhancement:', 'mxchat-vision'); ?></strong><br>
                                                        <code>Apply professional photo editing to this image with these modifications:</code></p>

                                                        <p><strong><?php _e('Creative Style:', 'mxchat-vision'); ?></strong><br>
                                                        <code>Transform this image creatively by applying these artistic changes:</code></p>
                                                    </div>
                                                </details>
                                            </div>
                                        </div>

                                        <!-- Disable Input on Upload (Editing Mode Only) -->
                                        <div class="mxchat-form-row editing-only-setting" style="<?php echo $vision_mode !== 'editing' ? 'display:none;' : ''; ?>">
                                            <div class="mxchat-form-label">
                                                <label for="mxchat_vision_disable_input_on_editing">
                                                    <?php _e('Disable User Input on Upload', 'mxchat-vision'); ?>
                                                </label>
                                            </div>
                                            <div class="mxchat-form-control">
                                                <label class="mxchat-toggle-switch">
                                                    <input type="checkbox"
                                                           id="mxchat_vision_disable_input_on_editing"
                                                           name="mxchat_vision_disable_input_on_editing"
                                                           value="1"
                                                           <?php checked($disable_input_on_editing, true); ?>>
                                                    <span class="mxchat-toggle-slider"></span>
                                                </label>
                                                <p class="mxchat-form-description">
                                                    <?php _e('When enabled, the chat input field will be disabled when users upload images for editing. This forces users to upload images without custom prompts.', 'mxchat-vision'); ?>
                                                </p>
                                            </div>
                                        </div>

                                        <!-- Edited Image Response Message (Editing Mode Only) -->
                                        <div class="mxchat-form-row editing-only-setting" style="<?php echo $vision_mode !== 'editing' ? 'display:none;' : ''; ?>">
                                            <div class="mxchat-form-label">
                                                <label for="mxchat_vision_edited_image_message">
                                                    <?php _e('Edited Image Message', 'mxchat-vision'); ?>
                                                </label>
                                            </div>
                                            <div class="mxchat-form-control">
                                                <input type="text"
                                                       id="mxchat_vision_edited_image_message"
                                                       name="mxchat_vision_edited_image_message"
                                                       value="<?php echo esc_attr($edited_image_message); ?>"
                                                       class="mxchat-text-input"
                                                       style="width: 100%;">
                                                <p class="mxchat-form-description">
                                                    <?php _e('The message displayed above the edited image when it is returned to the user.', 'mxchat-vision'); ?>
                                                </p>
                                            </div>
                                        </div>

                                        <!-- === GENERAL SETTINGS (Both Modes) === -->

                                        <!-- Max Images -->
                                        <div class="mxchat-form-row">
                                            <div class="mxchat-form-label">
                                                <label for="mxchat_vision_max_images">
                                                    <?php _e('Max Images Per Message', 'mxchat-vision'); ?>
                                                </label>
                                            </div>
                                            <div class="mxchat-form-control">
                                                <input type="number" 
                                                       id="mxchat_vision_max_images" 
                                                       name="mxchat_vision_max_images" 
                                                       value="<?php echo esc_attr($max_images); ?>" 
                                                       min="1" 
                                                       max="10" 
                                                       class="mxchat-number-input">
                                                <p class="mxchat-form-description">
                                                    <?php _e('Maximum number of images users can upload per message. Higher numbers increase processing time and costs.', 'mxchat-vision'); ?>
                                                </p>
                                            </div>
                                        </div>

                                        <!-- Max File Size -->
                                        <div class="mxchat-form-row">
                                            <div class="mxchat-form-label">
                                                <label for="mxchat_vision_max_file_size">
                                                    <?php _e('Max File Size (MB)', 'mxchat-vision'); ?>
                                                </label>
                                            </div>
                                            <div class="mxchat-form-control">
                                                <input type="number"
                                                       id="mxchat_vision_max_file_size"
                                                       name="mxchat_vision_max_file_size"
                                                       value="<?php echo esc_attr($max_file_size); ?>"
                                                       min="1"
                                                       max="20"
                                                       class="mxchat-number-input">
                                                <p class="mxchat-form-description">
                                                    <?php _e('Maximum file size for uploaded images. Larger images will be automatically resized for processing.', 'mxchat-vision'); ?>
                                                </p>
                                            </div>
                                        </div>

                                    </div>

                                    <!-- Submit Button -->
                                    <div class="mxchat-form-actions">
                                        <?php submit_button(__('Save Settings', 'mxchat-vision'), 'primary', 'submit', false, ['class' => 'mxchat-button-primary']); ?>
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>

                    <!-- Sidebar -->
                    <div class="mxchat-vision-sidebar">
                        
                        <!-- API Keys Status -->
                        <div class="mxchat-vision-card">
                            <div class="mxchat-card-header">
                                <h3><?php _e('API Keys Status', 'mxchat-vision'); ?></h3>
                            </div>
                            <div class="mxchat-card-content">
                                <div class="api-status-list">
                                    <div class="api-status-item">
                                        <div class="api-info">
                                            <span class="api-name"><?php _e('OpenAI Vision', 'mxchat-vision'); ?></span>
                                            <span class="api-status-badge <?php echo $api_keys_status['openai'] ? 'status-active' : 'status-inactive'; ?>">
                                                <?php echo $api_keys_status['openai'] ? __('Active', 'mxchat-vision') : __('Not Set', 'mxchat-vision'); ?>
                                            </span>
                                        </div>
                                    </div>
                                    <div class="api-status-item">
                                        <div class="api-info">
                                            <span class="api-name"><?php _e('Grok Vision', 'mxchat-vision'); ?></span>
                                            <span class="api-status-badge <?php echo $api_keys_status['grok'] ? 'status-active' : 'status-inactive'; ?>">
                                                <?php echo $api_keys_status['grok'] ? __('Active', 'mxchat-vision') : __('Not Set', 'mxchat-vision'); ?>
                                            </span>
                                        </div>
                                    </div>
                                    <div class="api-status-item">
                                        <div class="api-info">
                                            <span class="api-name"><?php _e('Google Gemini 2.5 Flash', 'mxchat-vision'); ?></span>
                                            <span class="api-status-badge <?php echo $api_keys_status['google'] ? 'status-active' : 'status-inactive'; ?>">
                                                <?php echo $api_keys_status['google'] ? __('Active', 'mxchat-vision') : __('Not Set', 'mxchat-vision'); ?>
                                            </span>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="api-help">
                                    <p class="api-help-text">
                                        <?php _e('Set API key by changing your chat model in the core MxChat plugin. Once you enter API key you can change your chat model back.', 'mxchat-vision'); ?>
                                    </p>
                                    <a href="<?php echo admin_url('admin.php?page=mxchat-max'); ?>" class="mxchat-button-secondary">
                                        <?php _e('Configure API Keys', 'mxchat-vision'); ?>
                                    </a>
                                </div>
                            </div>
                        </div>

                        <!-- Supported Features -->
                        <div class="mxchat-vision-card">
                            <div class="mxchat-card-header">
                                <h3><?php _e('Supported Features', 'mxchat-vision'); ?></h3>
                            </div>
                            <div class="mxchat-card-content">
                                <div class="features-grid">
                                    <div class="feature-category">
                                        <h4><?php _e('Image Formats', 'mxchat-vision'); ?></h4>
                                        <ul class="feature-list">
                                            <li>JPEG (.jpg, .jpeg)</li>
                                            <li>PNG (.png)</li>
                                            <li>GIF (.gif)</li>
                                            <li>WebP (.webp)</li>
                                        </ul>
                                    </div>
                                    
                                    <div class="feature-category">
                                        <h4><?php _e('Upload Methods', 'mxchat-vision'); ?></h4>
                                        <ul class="feature-list">
                                            <li><?php _e('File selection', 'mxchat-vision'); ?></li>
                                            <li><?php _e('Copy & paste', 'mxchat-vision'); ?></li>
                                            <li><?php _e('Drag & drop', 'mxchat-vision'); ?></li>
                                        </ul>
                                    </div>
                                    
                                    <div class="feature-category">
                                        <h4><?php _e('Analysis Types', 'mxchat-vision'); ?></h4>
                                        <ul class="feature-list">
                                            <li><?php _e('Object detection', 'mxchat-vision'); ?></li>
                                            <li><?php _e('Text recognition (OCR)', 'mxchat-vision'); ?></li>
                                            <li><?php _e('Scene description', 'mxchat-vision'); ?></li>
                                            <li><?php _e('Detail analysis', 'mxchat-vision'); ?></li>
                                        </ul>
                                    </div>
                                </div>
                            </div>
                        </div>

                    </div>
                </div>
            </div>
        </div>

        <script>
        // Pass data to JavaScript
        var mxchatVisionAdmin = {
            nonce: '<?php echo wp_create_nonce('mxchat_vision_admin'); ?>',
            ajaxUrl: '<?php echo admin_url('admin-ajax.php'); ?>',
            strings: {
                loading: '<?php echo esc_js(__('Loading...', 'mxchat-vision')); ?>',
                saved: '<?php echo esc_js(__('Settings saved!', 'mxchat-vision')); ?>',
                error: '<?php echo esc_js(__('An error occurred. Please try again.', 'mxchat-vision')); ?>'
            }
        };

        // Toggle dependent settings
        document.addEventListener('DOMContentLoaded', function() {
            const enabledCheckbox = document.getElementById('mxchat_vision_enabled');
            const dependentSettings = document.getElementById('vision-dependent-settings');
            const statusBadge = document.querySelector('.feature-status-badge');

            if (enabledCheckbox && dependentSettings && statusBadge) {
                enabledCheckbox.addEventListener('change', function() {
                    if (this.checked) {
                        dependentSettings.style.display = 'block';
                        statusBadge.className = 'feature-status-badge status-enabled';
                        statusBadge.innerHTML = '<span class="status-dot"></span><?php echo esc_js(__('Enabled', 'mxchat-vision')); ?>';
                    } else {
                        dependentSettings.style.display = 'none';
                        statusBadge.className = 'feature-status-badge status-disabled';
                        statusBadge.innerHTML = '<span class="status-dot"></span><?php echo esc_js(__('Disabled', 'mxchat-vision')); ?>';
                    }
                });
            }

            // Toggle mode-specific settings
            const modeRadios = document.querySelectorAll('input[name="mxchat_vision_mode"]');
            const analysisOnlySettings = document.querySelectorAll('.analysis-only-setting');
            const editingOnlySettings = document.querySelectorAll('.editing-only-setting');
            const proOnlySettings = document.querySelectorAll('.pro-only-setting');

            function updateSettingsVisibility() {
                const selectedMode = document.querySelector('input[name="mxchat_vision_mode"]:checked');
                if (!selectedMode) return;

                const isAnalysis = selectedMode.value === 'analysis';
                const isEditing = selectedMode.value === 'editing';

                // Show/hide based on mode
                analysisOnlySettings.forEach(function(setting) {
                    setting.style.display = isAnalysis ? '' : 'none';
                });

                editingOnlySettings.forEach(function(setting) {
                    setting.style.display = isEditing ? '' : 'none';
                });

                // If in editing mode, check which editing model is selected
                if (isEditing) {
                    const editingModelSelect = document.getElementById('mxchat_vision_editing_model');
                    if (editingModelSelect) {
                        const isPro = editingModelSelect.value === 'gemini-3-pro-image-preview';
                        proOnlySettings.forEach(function(setting) {
                            setting.style.display = isPro ? '' : 'none';
                        });
                    }
                } else {
                    // Hide pro settings when not in editing mode
                    proOnlySettings.forEach(function(setting) {
                        setting.style.display = 'none';
                    });
                }
            }

            // Add event listeners to mode radios
            modeRadios.forEach(function(radio) {
                radio.addEventListener('change', updateSettingsVisibility);
            });

            // Toggle pro-only settings based on editing model
            const editingModelSelect = document.getElementById('mxchat_vision_editing_model');
            if (editingModelSelect) {
                editingModelSelect.addEventListener('change', function() {
                    const isPro = this.value === 'gemini-3-pro-image-preview';
                    proOnlySettings.forEach(function(setting) {
                        setting.style.display = isPro ? '' : 'none';
                    });
                });
            }

            // Initial visibility update
            updateSettingsVisibility();
        });
        </script>
        <?php
    }

    /**
     * Check API keys status from MxChat core
     */
    private function check_api_keys_status() {
        // Get the main plugin options - same way as other add-ons
        $main_plugin_options = get_option('mxchat_options', array());

        return array(
            'openai' => !empty($main_plugin_options['api_key']), // OpenAI uses 'api_key'
            'grok' => !empty($main_plugin_options['xai_api_key']), // Grok uses xAI API key
            'google' => !empty($main_plugin_options['gemini_api_key']) // Google uses gemini_api_key
        );
    }

    /**
     * Save settings via AJAX
     */
public function save_settings() {
    // Verify nonce
    if (!wp_verify_nonce($_POST['nonce'], 'mxchat_vision_settings')) {
        wp_die(__('Security check failed.', 'mxchat-vision'));
    }

    // Check permissions
    if (!current_user_can('manage_options')) {
        wp_die(__('You do not have sufficient permissions.', 'mxchat-vision'));
    }

    // Sanitize and save settings
    $vision_enabled = isset($_POST['vision_enabled']) ? sanitize_text_field($_POST['vision_enabled']) : false;
    $vision_model = isset($_POST['vision_model']) ? sanitize_text_field($_POST['vision_model']) : 'openai';
    $vision_mode = isset($_POST['vision_mode']) ? sanitize_text_field($_POST['vision_mode']) : 'analysis';
    $max_images = isset($_POST['max_images']) ? intval($_POST['max_images']) : 5;
    $max_file_size = isset($_POST['max_file_size']) ? intval($_POST['max_file_size']) : 10;
    $custom_prompt = isset($_POST['custom_prompt']) ? sanitize_textarea_field($_POST['custom_prompt']) : '';
    $editing_prompt = isset($_POST['editing_prompt']) ? sanitize_textarea_field($_POST['editing_prompt']) : '';
    $editing_model = isset($_POST['editing_model']) ? sanitize_text_field($_POST['editing_model']) : 'gemini-2.5-flash-image';
    $image_size = isset($_POST['image_size']) ? sanitize_text_field($_POST['image_size']) : '1K';
    $aspect_ratio = isset($_POST['aspect_ratio']) ? sanitize_text_field($_POST['aspect_ratio']) : '1:1';
    $disable_input_on_editing = isset($_POST['mxchat_vision_disable_input_on_editing']) ? sanitize_text_field($_POST['mxchat_vision_disable_input_on_editing']) : false;
    $edited_image_message = isset($_POST['mxchat_vision_edited_image_message']) ? sanitize_text_field($_POST['mxchat_vision_edited_image_message']) : "Here's your edited image:";

    update_option('mxchat_vision_enabled', $vision_enabled);
    update_option('mxchat_vision_model', $vision_model);
    update_option('mxchat_vision_mode', $vision_mode);
    update_option('mxchat_vision_max_images', $max_images);
    update_option('mxchat_vision_max_file_size', $max_file_size);
    update_option('mxchat_vision_custom_prompt', $custom_prompt);
    update_option('mxchat_vision_editing_prompt', $editing_prompt);
    update_option('mxchat_vision_editing_model', $editing_model);
    update_option('mxchat_vision_image_size', $image_size);
    update_option('mxchat_vision_aspect_ratio', $aspect_ratio);
    update_option('mxchat_vision_disable_input_on_editing', $disable_input_on_editing);
    update_option('mxchat_vision_edited_image_message', $edited_image_message);

    wp_send_json_success(array(
        'message' => __('Settings saved successfully!', 'mxchat-vision')
    ));
}

    /**
     * Get vision settings
     */
public static function get_settings() {
    return array(
        'enabled' => get_option('mxchat_vision_enabled', false),
        'model' => get_option('mxchat_vision_model', 'openai'),
        'mode' => get_option('mxchat_vision_mode', 'analysis'),
        'max_images' => get_option('mxchat_vision_max_images', 5),
        'max_file_size' => get_option('mxchat_vision_max_file_size', 10),
        'custom_prompt' => get_option('mxchat_vision_custom_prompt', 'Analyze this image in detail. Describe what you see, including any text, objects, people, colors, and overall context.'),
        'editing_prompt' => get_option('mxchat_vision_editing_prompt', 'Apply the following modifications to this image:'),
        'editing_model' => get_option('mxchat_vision_editing_model', 'gemini-2.5-flash-image'),
        'image_size' => get_option('mxchat_vision_image_size', '1K'),
        'aspect_ratio' => get_option('mxchat_vision_aspect_ratio', '1:1'),
        'disable_input_on_editing' => get_option('mxchat_vision_disable_input_on_editing', false),
        'edited_image_message' => get_option('mxchat_vision_edited_image_message', "Here's your edited image:")
    );
}

    /**
     * Check if vision is enabled
     */
    public static function is_enabled() {
        return get_option('mxchat_vision_enabled', false);
    }
}
?>