<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Vision Models Class
 * 
 * Handles API integration with OpenAI Vision and Grok
 */
class Vision_Models {

    private $api_keys;
    private $request_timeout = 60; // 60 seconds timeout
    private $max_retries = 3;
    private $max_editing_retries = 3; // Specific retries for image editing

    /**
     * Constructor
     */
    public function __construct() {
        $this->init_api_keys();
    }

    /**
     * Initialize API keys from core plugin
     */
    private function init_api_keys() {
        // Get the main plugin options - same way as other addons
        $main_plugin_options = get_option('mxchat_options', array());

        $this->api_keys = array(
            'openai' => isset($main_plugin_options['api_key']) ? $main_plugin_options['api_key'] : '',
            'grok' => isset($main_plugin_options['xai_api_key']) ? $main_plugin_options['xai_api_key'] : '',
            'google' => isset($main_plugin_options['gemini_api_key']) ? $main_plugin_options['gemini_api_key'] : '',
        );
    }

    /**
     * Analyze images with OpenAI Vision API
     *
     * @param array $images Processed image data
     * @param string $prompt User prompt
     * @return array Result array
     */
    public function analyze_with_openai($images, $prompt) {
        $start_time = microtime(true);

        try {
            // Check API key
            if (empty($this->api_keys['openai'])) {
                return array(
                    'success' => false,
                    'message' => __('OpenAI API key is not configured.', 'mxchat-vision')
                );
            }

            // Prepare the messages array
            $messages = array(
                array(
                    'role' => 'user',
                    'content' => $this->prepare_openai_content($images, $prompt)
                )
            );

            // Prepare the request
            $request_data = array(
                'model' => 'gpt-4o', // Updated to latest vision model
                'messages' => $messages,
                'max_tokens' => 1000,
                'temperature' => 0.7
            );

            // Make API request
            $response = $this->make_openai_request($request_data);

            $processing_time = microtime(true) - $start_time;

            if ($response['success']) {
                return array(
                    'success' => true,
                    'analysis' => $response['data']['choices'][0]['message']['content'],
                    'processing_time' => $processing_time,
                    'tokens_used' => isset($response['data']['usage']) ? $response['data']['usage'] : null,
                    'model' => 'gpt-4o'
                );
            } else {
                return array(
                    'success' => false,
                    'message' => $response['message'],
                    'processing_time' => $processing_time
                );
            }

        } catch (Exception $e) {
            return array(
                'success' => false,
                'message' => sprintf(__('OpenAI API error: %s', 'mxchat-vision'), $e->getMessage()),
                'processing_time' => microtime(true) - $start_time
            );
        }
    }

    /**
     * Analyze images with Grok Vision API
     *
     * @param array $images Processed image data
     * @param string $prompt User prompt
     * @return array Result array
     */
    public function analyze_with_grok($images, $prompt) {
        $start_time = microtime(true);

        try {
            // Check API key
            if (empty($this->api_keys['grok'])) {
                return array(
                    'success' => false,
                    'message' => __('Grok API key is not configured.', 'mxchat-vision')
                );
            }

            // Prepare the messages array
            $messages = array(
                array(
                    'role' => 'user',
                    'content' => $this->prepare_grok_content($images, $prompt)
                )
            );

            // Prepare the request
            $request_data = array(
                'model' => 'grok-4',
                'messages' => $messages,
                'max_tokens' => 1000,
                'temperature' => 0.7
            );

            // Make API request
            $response = $this->make_grok_request($request_data);

            $processing_time = microtime(true) - $start_time;

            if ($response['success']) {
                return array(
                    'success' => true,
                    'analysis' => $response['data']['choices'][0]['message']['content'],
                    'processing_time' => $processing_time,
                    'tokens_used' => isset($response['data']['usage']) ? $response['data']['usage'] : null,
                    'model' => 'grok-vision-beta'
                );
            } else {
                return array(
                    'success' => false,
                    'message' => $response['message'],
                    'processing_time' => $processing_time
                );
            }

        } catch (Exception $e) {
            return array(
                'success' => false,
                'message' => sprintf(__('Grok API error: %s', 'mxchat-vision'), $e->getMessage()),
                'processing_time' => microtime(true) - $start_time
            );
        }
    }

    /**
     * Prepare content for OpenAI API
     *
     * @param array $images Image data
     * @param string $prompt User prompt
     * @return array Formatted content array
     */
    private function prepare_openai_content($images, $prompt) {
        $content = array();

        // Add text prompt
        $content[] = array(
            'type' => 'text',
            'text' => $prompt
        );

        // Add images
        foreach ($images as $image) {
            $content[] = array(
                'type' => 'image_url',
                'image_url' => array(
                    'url' => 'data:' . $image['mime_type'] . ';base64,' . $image['base64'],
                    'detail' => 'auto' // Can be 'low', 'high', or 'auto'
                )
            );
        }

        return $content;
    }

    /**
     * Prepare content for Grok API
     *
     * @param array $images Image data
     * @param string $prompt User prompt
     * @return array Formatted content array
     */
    private function prepare_grok_content($images, $prompt) {
    $content = array();

    // Add text prompt
    $content[] = array(
        'type' => 'text',
        'text' => $prompt
    );

    // Add images with detail parameter (as shown in xAI docs)
    foreach ($images as $image) {
        $content[] = array(
            'type' => 'image_url',
            'image_url' => array(
                'url' => 'data:' . $image['mime_type'] . ';base64,' . $image['base64'],
                'detail' => 'high' // Add the detail parameter as shown in xAI docs
            )
        );
    }

    return $content;
}

    /**
     * Edit image with Google Gemini 3 Pro Image (with retry mechanism)
     *
     * @param array $images Processed image data (supports up to 14 images)
     * @param string $prompt User editing instructions
     * @param array $config Image configuration (aspectRatio, imageSize)
     * @return array Result array with edited image
     */
    public function edit_with_google_imagen_pro($images, $prompt, $config = array()) {
        $start_time = microtime(true);
        $attempt = 0;
        $last_error = '';

        while ($attempt < $this->max_editing_retries) {
            $attempt++;
            
            try {
                // Check API key
                if (empty($this->api_keys['google'])) {
                    return array(
                        'success' => false,
                        'message' => __('Google Gemini API key is not configured.', 'mxchat-vision')
                    );
                }

                // Log retry attempt
                if ($attempt > 1) {
                    error_log(sprintf(
                        'MxChat Vision: Retrying image editing with Gemini 3 Pro (Attempt %d of %d)',
                        $attempt,
                        $this->max_editing_retries
                    ));
                }

                // Gemini 3 Pro Image can process up to 14 images
                $images_to_process = array_slice($images, 0, 14);

                // Prepare content array with text and images
                $content_parts = array(
                    array(
                        'text' => $prompt
                    )
                );

                // Add all images to content
                foreach ($images_to_process as $image) {
                    $content_parts[] = array(
                        'inline_data' => array(
                            'mime_type' => $image['mime_type'],
                            'data' => $image['base64']
                        )
                    );
                }

                // Set default image config if not provided
                $default_config = array(
                    'aspectRatio' => '1:1',
                    'imageSize' => '1K'
                );
                $image_config = array_merge($default_config, $config);

                // Prepare the request for Gemini 3 Pro Image
                $request_data = array(
                    'contents' => array(
                        array(
                            'parts' => $content_parts
                        )
                    ),
                    'generationConfig' => array(
                        'responseModalities' => array('IMAGE'),
                        'imageConfig' => $image_config
                    )
                );

                // Make API request
                $response = $this->make_google_imagen_pro_request($request_data);

                $processing_time = microtime(true) - $start_time;

                if ($response['success']) {
                    // Log the full response for debugging
                    error_log('Gemini 3 Pro API Response: ' . print_r($response['data'], true));

                    // Extract the edited image from response
                    $edited_image_data = null;
                    $thought_signature = null;

                    if (isset($response['data']['candidates'][0]['content']['parts'])) {
                        foreach ($response['data']['candidates'][0]['content']['parts'] as $part) {
                            // Extract thought signature if present
                            if (isset($part['thought']) && $part['thought']) {
                                $thought_signature = isset($part['text']) ? $part['text'] : null;
                            }

                            // Extract the final image (camelCase)
                            if (isset($part['inlineData']['data'])) {
                                $edited_image_data = $part['inlineData']['data'];
                            }
                        }
                    }

                    if ($edited_image_data) {
                        // Success! Log and return
                        if ($attempt > 1) {
                            error_log(sprintf(
                                'MxChat Vision: Image editing succeeded on attempt %d',
                                $attempt
                            ));
                        }

                        return array(
                            'success' => true,
                            'edited_image' => $edited_image_data,
                            'mime_type' => 'image/png',
                            'processing_time' => $processing_time,
                            'model' => 'gemini-3-pro-image',
                            'thought_signature' => $thought_signature,
                            'config_used' => $image_config,
                            'attempts' => $attempt
                        );
                    } else {
                        $last_error = __('No edited image returned from API', 'mxchat-vision');
                        error_log('No image found. Full response structure: ' . json_encode($response['data']));

                        // If this is not the last attempt, retry
                        if ($attempt < $this->max_editing_retries) {
                            sleep(pow(2, $attempt)); // Exponential backoff: 2s, 4s
                            continue;
                        }

                        return array(
                            'success' => false,
                            'message' => $last_error,
                            'processing_time' => $processing_time,
                            'debug_response' => json_encode($response['data']),
                            'attempts' => $attempt
                        );
                    }
                } else {
                    $last_error = $response['message'];
                    error_log('Gemini 3 Pro API Error: ' . $last_error);
                    
                    // If this is not the last attempt and error is retryable, retry
                    if ($attempt < $this->max_editing_retries && $this->is_retryable_error($last_error)) {
                        sleep(pow(2, $attempt)); // Exponential backoff
                        continue;
                    }

                    return array(
                        'success' => false,
                        'message' => $last_error,
                        'processing_time' => $processing_time,
                        'attempts' => $attempt
                    );
                }

            } catch (Exception $e) {
                $last_error = $e->getMessage();
                error_log('Gemini 3 Pro Exception: ' . $last_error);
                
                // If this is not the last attempt, retry
                if ($attempt < $this->max_editing_retries) {
                    sleep(pow(2, $attempt)); // Exponential backoff
                    continue;
                }

                return array(
                    'success' => false,
                    'message' => sprintf(__('Google Gemini 3 Pro API error: %s', 'mxchat-vision'), $last_error),
                    'processing_time' => microtime(true) - $start_time,
                    'attempts' => $attempt
                );
            }
        }

        // If we get here, all retries failed
        return array(
            'success' => false,
            'message' => sprintf(
                __('Image editing failed after %d attempts. Last error: %s', 'mxchat-vision'),
                $this->max_editing_retries,
                $last_error
            ),
            'processing_time' => microtime(true) - $start_time,
            'attempts' => $attempt
        );
    }

    /**
     * Edit image with Google Gemini 2.5 Flash Image (with retry mechanism)
     *
     * @param array $images Processed image data (only first image is used)
     * @param string $prompt User editing instructions
     * @return array Result array with edited image
     */
    public function edit_with_google_imagen($images, $prompt) {
        $start_time = microtime(true);
        $attempt = 0;
        $last_error = '';

        while ($attempt < $this->max_editing_retries) {
            $attempt++;
            
            try {
                // Check API key
                if (empty($this->api_keys['google'])) {
                    return array(
                        'success' => false,
                        'message' => __('Google Gemini API key is not configured.', 'mxchat-vision')
                    );
                }

                // Log retry attempt
                if ($attempt > 1) {
                    error_log(sprintf(
                        'MxChat Vision: Retrying image editing with Gemini 2.5 Flash (Attempt %d of %d)',
                        $attempt,
                        $this->max_editing_retries
                    ));
                }

                // Gemini 2.5 Flash processes one image at a time
                $image = $images[0];

                // Prepare content array with text and image
                $content_parts = array(
                    array(
                        'text' => $prompt
                    ),
                    array(
                        'inline_data' => array(
                            'mime_type' => $image['mime_type'],
                            'data' => $image['base64']
                        )
                    )
                );

                // Prepare the request for Gemini 2.5 Flash Image
                $request_data = array(
                    'contents' => array(
                        array(
                            'parts' => $content_parts
                        )
                    ),
                    'generationConfig' => array(
                        'responseModalities' => array('IMAGE')
                    )
                );

                // Make API request
                $response = $this->make_google_imagen_request($request_data);

                $processing_time = microtime(true) - $start_time;

                if ($response['success']) {
                    // Log the full response for debugging
                    error_log('Gemini API Response: ' . print_r($response['data'], true));

                    // Extract the edited image from response
                    $edited_image_data = null;

                    if (isset($response['data']['candidates'][0]['content']['parts'])) {
                        foreach ($response['data']['candidates'][0]['content']['parts'] as $part) {
                            // Gemini returns inlineData (camelCase), not inline_data (snake_case)
                            if (isset($part['inlineData']['data'])) {
                                $edited_image_data = $part['inlineData']['data'];
                                break;
                            }
                        }
                    }

                    if ($edited_image_data) {
                        // Success! Log and return
                        if ($attempt > 1) {
                            error_log(sprintf(
                                'MxChat Vision: Image editing succeeded on attempt %d',
                                $attempt
                            ));
                        }

                        return array(
                            'success' => true,
                            'edited_image' => $edited_image_data,
                            'mime_type' => 'image/png',
                            'processing_time' => $processing_time,
                            'model' => 'gemini-2.5-flash-image',
                            'attempts' => $attempt
                        );
                    } else {
                        $last_error = __('No edited image returned from API', 'mxchat-vision');
                        error_log('No image found. Full response structure: ' . json_encode($response['data']));

                        // If this is not the last attempt, retry
                        if ($attempt < $this->max_editing_retries) {
                            sleep(pow(2, $attempt)); // Exponential backoff: 2s, 4s
                            continue;
                        }

                        return array(
                            'success' => false,
                            'message' => $last_error,
                            'processing_time' => $processing_time,
                            'debug_response' => json_encode($response['data']),
                            'attempts' => $attempt
                        );
                    }
                } else {
                    $last_error = $response['message'];
                    error_log('Gemini API Error: ' . $last_error);
                    
                    // If this is not the last attempt and error is retryable, retry
                    if ($attempt < $this->max_editing_retries && $this->is_retryable_error($last_error)) {
                        sleep(pow(2, $attempt)); // Exponential backoff
                        continue;
                    }

                    return array(
                        'success' => false,
                        'message' => $last_error,
                        'processing_time' => $processing_time,
                        'attempts' => $attempt
                    );
                }

            } catch (Exception $e) {
                $last_error = $e->getMessage();
                error_log('Gemini Exception: ' . $last_error);
                
                // If this is not the last attempt, retry
                if ($attempt < $this->max_editing_retries) {
                    sleep(pow(2, $attempt)); // Exponential backoff
                    continue;
                }

                return array(
                    'success' => false,
                    'message' => sprintf(__('Google Gemini API error: %s', 'mxchat-vision'), $last_error),
                    'processing_time' => microtime(true) - $start_time,
                    'attempts' => $attempt
                );
            }
        }

        // If we get here, all retries failed
        return array(
            'success' => false,
            'message' => sprintf(
                __('Image editing failed after %d attempts. Last error: %s', 'mxchat-vision'),
                $this->max_editing_retries,
                $last_error
            ),
            'processing_time' => microtime(true) - $start_time,
            'attempts' => $attempt
        );
    }

    /**
     * Check if an error is retryable
     *
     * @param string $error_message Error message to check
     * @return bool True if error should be retried
     */
    private function is_retryable_error($error_message) {
        $retryable_patterns = array(
            'timeout',
            'timed out',
            'connection',
            'network',
            'temporary',
            'rate limit',
            'too many requests',
            '429',
            '500',
            '502',
            '503',
            '504',
            'gateway',
            'unavailable',
            'overloaded'
        );

        $error_lower = strtolower($error_message);
        
        foreach ($retryable_patterns as $pattern) {
            if (strpos($error_lower, $pattern) !== false) {
                return true;
            }
        }

        return false;
    }

    /**
     * Make OpenAI API request
     *
     * @param array $request_data Request payload
     * @return array Response array
     */
    private function make_openai_request($request_data) {
        $url = 'https://api.openai.com/v1/chat/completions';
        
        $headers = array(
            'Authorization' => 'Bearer ' . $this->api_keys['openai'],
            'Content-Type' => 'application/json',
            'User-Agent' => 'MxChat-Vision/' . MXCHAT_VISION_VERSION
        );

        return $this->make_http_request($url, $headers, $request_data, 'openai');
    }

    /**
     * Make Grok API request
     *
     * @param array $request_data Request payload
     * @return array Response array
     */
    private function make_grok_request($request_data) {
        $url = 'https://api.x.ai/v1/chat/completions';

        $headers = array(
            'Authorization' => 'Bearer ' . $this->api_keys['grok'],
            'Content-Type' => 'application/json',
            'User-Agent' => 'MxChat-Vision/' . MXCHAT_VISION_VERSION
        );

        return $this->make_http_request($url, $headers, $request_data, 'grok');
    }

    /**
     * Make Google Gemini 3 Pro Image API request
     *
     * @param array $request_data Request payload
     * @return array Response array
     */
    private function make_google_imagen_pro_request($request_data) {
        // Gemini 3 Pro Image endpoint
        $url = 'https://generativelanguage.googleapis.com/v1beta/models/gemini-3-pro-image-preview:generateContent';

        // Uses simple API key authentication
        $url .= '?key=' . $this->api_keys['google'];

        $headers = array(
            'Content-Type' => 'application/json',
            'User-Agent' => 'MxChat-Vision/' . MXCHAT_VISION_VERSION
        );

        return $this->make_http_request($url, $headers, $request_data, 'google');
    }

    /**
     * Make Google Gemini 2.5 Flash Image API request
     *
     * @param array $request_data Request payload
     * @return array Response array
     */
    private function make_google_imagen_request($request_data) {
        // Gemini 2.5 Flash Image endpoint with image generation support
        $url = 'https://generativelanguage.googleapis.com/v1beta/models/gemini-2.5-flash-image:generateContent';

        // Uses simple API key authentication (same as regular Gemini)
        $url .= '?key=' . $this->api_keys['google'];

        $headers = array(
            'Content-Type' => 'application/json',
            'User-Agent' => 'MxChat-Vision/' . MXCHAT_VISION_VERSION
        );

        return $this->make_http_request($url, $headers, $request_data, 'google');
    }

    /**
     * Make HTTP request with retry logic
     *
     * @param string $url API endpoint URL
     * @param array $headers Request headers
     * @param array $request_data Request payload
     * @param string $provider API provider name
     * @return array Response array
     */
    private function make_http_request($url, $headers, $request_data, $provider) {
        $retry_count = 0;
        $last_error = '';

        while ($retry_count < $this->max_retries) {
            $response = wp_remote_post($url, array(
                'headers' => $headers,
                'body' => json_encode($request_data),
                'timeout' => $this->request_timeout,
                'sslverify' => true
            ));

            // Check for WordPress HTTP errors
            if (is_wp_error($response)) {
                $last_error = $response->get_error_message();
                $retry_count++;
                
                if ($retry_count < $this->max_retries) {
                    sleep(pow(2, $retry_count)); // Exponential backoff
                    continue;
                }
                
                return array(
                    'success' => false,
                    'message' => sprintf(__('HTTP request failed: %s', 'mxchat-vision'), $last_error)
                );
            }

            $status_code = wp_remote_retrieve_response_code($response);
            $body = wp_remote_retrieve_body($response);
            $decoded_body = json_decode($body, true);

            // Handle successful response
            if ($status_code >= 200 && $status_code < 300) {
                if ($decoded_body && !isset($decoded_body['error'])) {
                    return array(
                        'success' => true,
                        'data' => $decoded_body
                    );
                } else {
                    $error_message = isset($decoded_body['error']['message']) 
                        ? $decoded_body['error']['message'] 
                        : __('Unknown API error', 'mxchat-vision');
                    
                    return array(
                        'success' => false,
                        'message' => $error_message
                    );
                }
            }

            // Handle rate limiting (429) and server errors (5xx) with retry
            if ($status_code == 429 || ($status_code >= 500 && $status_code < 600)) {
                $retry_count++;
                
                if ($retry_count < $this->max_retries) {
                    $retry_after = wp_remote_retrieve_header($response, 'retry-after');
                    $delay = $retry_after ? (int)$retry_after : pow(2, $retry_count);
                    sleep($delay);
                    continue;
                }
            }

            // Handle other client errors (4xx)
            if ($status_code >= 400 && $status_code < 500) {
                $error_message = isset($decoded_body['error']['message']) 
                    ? $decoded_body['error']['message'] 
                    : sprintf(__('API request failed with status %d', 'mxchat-vision'), $status_code);
                
                return array(
                    'success' => false,
                    'message' => $error_message
                );
            }

            $last_error = sprintf(__('Unexpected status code: %d', 'mxchat-vision'), $status_code);
            $retry_count++;
        }

        return array(
            'success' => false,
            'message' => sprintf(
                __('Request failed after %d retries. Last error: %s', 'mxchat-vision'),
                $this->max_retries,
                $last_error
            )
        );
    }

    /**
     * Test API key validity
     *
     * @param string $model Model to test ('openai' or 'grok')
     * @return array Test result
     */
    public function test_api_key($model) {
        $test_image = $this->create_test_image();
        $test_prompt = __('What do you see in this image?', 'mxchat-vision');

        switch ($model) {
            case 'openai':
                return $this->analyze_with_openai(array($test_image), $test_prompt);
            case 'grok':
                return $this->analyze_with_grok(array($test_image), $test_prompt);
            default:
                return array(
                    'success' => false,
                    'message' => __('Invalid model specified for testing.', 'mxchat-vision')
                );
        }
    }

    /**
     * Create a simple test image for API testing
     *
     * @return array Test image data
     */
    private function create_test_image() {
        // Create a simple 100x100 red square for testing
        $image = imagecreate(100, 100);
        $red = imagecolorallocate($image, 255, 0, 0);
        imagefill($image, 0, 0, $red);

        ob_start();
        imagepng($image);
        $image_data = ob_get_contents();
        ob_end_clean();
        imagedestroy($image);

        return array(
            'base64' => base64_encode($image_data),
            'mime_type' => 'image/png',
            'original_name' => 'test-image.png',
            'dimensions' => array('width' => 100, 'height' => 100),
            'file_size' => strlen($image_data)
        );
    }

    /**
     * Get available models
     *
     * @return array Available models with their status
     */
    public function get_available_models() {
        return array(
            'openai' => array(
                'name' => __('OpenAI Vision (GPT-4o)', 'mxchat-vision'),
                'available' => !empty($this->api_keys['openai']),
                'model_id' => 'gpt-4o',
                'supports_editing' => false
            ),
            'grok' => array(
                'name' => __('Grok Vision', 'mxchat-vision'),
                'available' => !empty($this->api_keys['grok']),
                'model_id' => 'grok-4',
                'supports_editing' => false
            ),
            'google_imagen' => array(
                'name' => __('Google Gemini 2.5 Flash Image (Editing)', 'mxchat-vision'),
                'available' => !empty($this->api_keys['google']),
                'model_id' => 'gemini-2.5-flash-image',
                'supports_editing' => true
            ),
            'google_imagen_pro' => array(
                'name' => __('Google Gemini 3 Pro Image (Editing)', 'mxchat-vision'),
                'available' => !empty($this->api_keys['google']),
                'model_id' => 'gemini-3-pro-image-preview',
                'supports_editing' => true
            )
        );
    }

    /**
     * Get API key status
     *
     * @return array API key status for each provider
     */
    public function get_api_key_status() {
        return array(
            'openai' => !empty($this->api_keys['openai']),
            'grok' => !empty($this->api_keys['grok']),
            'google' => !empty($this->api_keys['google'])
        );
    }

    /**
     * Estimate token usage for images
     *
     * @param array $images Image data
     * @param string $prompt Text prompt
     * @return array Token estimation
     */
    public function estimate_token_usage($images, $prompt) {
        // Basic token estimation
        $text_tokens = ceil(strlen($prompt) / 4); // Rough estimation: 4 chars per token
        
        $image_tokens = 0;
        foreach ($images as $image) {
            // OpenAI vision token calculation (simplified)
            $width = $image['dimensions']['width'];
            $height = $image['dimensions']['height'];
            
            // Base tokens for any image
            $base_tokens = 85;
            
            // Additional tokens based on image size
            $tile_tokens = ceil($width / 512) * ceil($height / 512) * 170;
            
            $image_tokens += $base_tokens + $tile_tokens;
        }

        return array(
            'text_tokens' => $text_tokens,
            'image_tokens' => $image_tokens,
            'total_estimated' => $text_tokens + $image_tokens,
            'images_count' => count($images)
        );
    }
}
?>