<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Vision Processor Class
 * 
 * Handles image processing and coordination with AI models
 */
class Vision_Processor {

    private $models;
    private $max_image_size = 1024; // Max dimension for processing

    /**
     * Constructor
     */
    public function __construct() {
        $this->models = new Vision_Models();
    }

    /**
     * Analyze images with the specified model
     *
     * @param array $images Array of image data
     * @param string $prompt User prompt/question
     * @param string $model Model to use ('openai' or 'grok')
     * @return array Result with success status and analysis
     */
public function analyze_images($images, $prompt, $model = 'openai') {
    try {
        // Validate inputs
        if (empty($images)) {
            return array(
                'success' => false,
                'message' => __('No images provided for analysis.', 'mxchat-vision')
            );
        }

        // Get settings to check mode
        $vision_settings = MxChat_Vision_Settings::get_settings();
        $vision_mode = $vision_settings['mode'];

        // Route to editing if mode is 'editing'
        if ($vision_mode === 'editing') {
            return $this->edit_images($images, $prompt);
        }

        // Continue with analysis mode (original functionality)
        $custom_prompt = $vision_settings['custom_prompt'];

        // Build the final prompt
        if (empty($prompt)) {
            $user_question = 'Please analyze this image and tell me what you see.';
        } else {
            $user_question = $prompt;
        }

        // Combine custom prompt with user question
        if (empty($custom_prompt)) {
            // No custom prompt - just use the user's question
            $final_prompt = $user_question;
        } else {
            // Append user question to custom prompt
            $final_prompt = $custom_prompt . "\n\nUser question: " . $user_question;
        }

        // Process images for API
        $processed_images = $this->process_images_for_api($images);

        if (empty($processed_images)) {
            return array(
                'success' => false,
                'message' => __('Failed to process images for analysis.', 'mxchat-vision')
            );
        }

        // Log the analysis attempt
        $this->log_analysis_attempt($model, count($processed_images), $final_prompt);

        // Call the appropriate model with the final prompt
        switch ($model) {
            case 'openai':
                $result = $this->models->analyze_with_openai($processed_images, $final_prompt);
                break;

            case 'grok':
                $result = $this->models->analyze_with_grok($processed_images, $final_prompt);
                break;

            default:
                return array(
                    'success' => false,
                    'message' => __('Invalid model specified.', 'mxchat-vision')
                );
        }

        // Process and enhance the result
        if ($result['success']) {
            $enhanced_result = $this->enhance_analysis_result($result, $processed_images, $model);
            $this->log_analysis_success($model, $enhanced_result);
            return $enhanced_result;
        } else {
            $this->log_analysis_error($model, $result['message']);
            return $result;
        }

    } catch (Exception $e) {
        $error_message = sprintf(
            __('Analysis failed: %s', 'mxchat-vision'),
            $e->getMessage()
        );

        $this->log_analysis_error($model, $error_message);

        return array(
            'success' => false,
            'message' => $error_message
        );
    }
}

    /**
     * Edit images using Google Gemini models
     *
     * @param array $images Array of image data
     * @param string $prompt User editing instructions
     * @return array Result with success status and edited image
     */
    public function edit_images($images, $prompt) {
        try {
            // Validate inputs
            if (empty($images)) {
                return array(
                    'success' => false,
                    'message' => __('No images provided for editing.', 'mxchat-vision')
                );
            }

            // Get the editing settings
            $vision_settings = MxChat_Vision_Settings::get_settings();
            $editing_prompt = $vision_settings['editing_prompt'];
            $editing_model = $vision_settings['editing_model'];
            $image_size = $vision_settings['image_size'];
            $aspect_ratio = $vision_settings['aspect_ratio'];

            // Build the final editing prompt
            if (empty($prompt)) {
                $user_instructions = 'Edit this image';
            } else {
                $user_instructions = $prompt;
            }

            // Combine editing prompt with user instructions
            if (empty($editing_prompt)) {
                $final_prompt = $user_instructions;
            } else {
                $final_prompt = $editing_prompt . " " . $user_instructions;
            }

            // Process images for API
            $processed_images = $this->process_images_for_api($images);

            if (empty($processed_images)) {
                return array(
                    'success' => false,
                    'message' => __('Failed to process images for editing.', 'mxchat-vision')
                );
            }

            // Log the editing attempt
            $model_name = ($editing_model === 'gemini-3-pro-image-preview') ? 'google_imagen_pro' : 'google_imagen';
            $this->log_analysis_attempt($model_name, count($processed_images), $final_prompt);

            // Call the appropriate Google Gemini model based on settings
            if ($editing_model === 'gemini-3-pro-image-preview') {
                // Use Gemini 3 Pro Image with advanced config
                $config = array(
                    'aspectRatio' => $aspect_ratio,
                    'imageSize' => $image_size
                );
                $result = $this->models->edit_with_google_imagen_pro($processed_images, $final_prompt, $config);
            } else {
                // Use Gemini 2.5 Flash Image (default)
                $result = $this->models->edit_with_google_imagen($processed_images, $final_prompt);
            }

            // Process the result
            if ($result['success']) {
                // Save the edited image
                $saved_image = $this->save_edited_image($result['edited_image'], $result['mime_type']);

                if ($saved_image['success']) {
                    $enhanced_result = array(
                        'success' => true,
                        'is_edited_image' => true,
                        'image_url' => $saved_image['url'],
                        'image_path' => $saved_image['path'],
                        'processing_time' => $result['processing_time'],
                        'model' => $result['model'],
                        'original_prompt' => $final_prompt
                    );

                    // Add thought signature if available (Gemini 3 Pro)
                    if (isset($result['thought_signature'])) {
                        $enhanced_result['thought_signature'] = $result['thought_signature'];
                    }

                    // Add config info if available (Gemini 3 Pro)
                    if (isset($result['config_used'])) {
                        $enhanced_result['config_used'] = $result['config_used'];
                    }

                    $this->log_analysis_success($model_name, $enhanced_result);
                    return $enhanced_result;
                } else {
                    return array(
                        'success' => false,
                        'message' => __('Failed to save edited image.', 'mxchat-vision')
                    );
                }
            } else {
                $this->log_analysis_error($model_name, $result['message']);
                return $result;
            }

        } catch (Exception $e) {
            $error_message = sprintf(
                __('Image editing failed: %s', 'mxchat-vision'),
                $e->getMessage()
            );

            $this->log_analysis_error('google_imagen', $error_message);

            return array(
                'success' => false,
                'message' => $error_message
            );
        }
    }

    /**
     * Save edited image to uploads directory
     *
     * @param string $base64_image Base64 encoded image
     * @param string $mime_type MIME type of image
     * @return array Result with path and URL
     */
    private function save_edited_image($base64_image, $mime_type) {
        try {
            $upload_dir = wp_upload_dir();
            $vision_dir = $upload_dir['basedir'] . '/mxchat-vision/';

            if (!file_exists($vision_dir)) {
                wp_mkdir_p($vision_dir);
            }

            // Determine file extension
            $extension = 'png';
            if ($mime_type === 'image/jpeg') {
                $extension = 'jpg';
            } elseif ($mime_type === 'image/webp') {
                $extension = 'webp';
            }

            // Generate unique filename
            $filename = 'edited_' . uniqid() . '.' . $extension;
            $file_path = $vision_dir . $filename;

            // Decode and save
            $image_data = base64_decode($base64_image);
            if (file_put_contents($file_path, $image_data)) {
                return array(
                    'success' => true,
                    'path' => $file_path,
                    'url' => $upload_dir['baseurl'] . '/mxchat-vision/' . $filename,
                    'filename' => $filename
                );
            } else {
                return array(
                    'success' => false,
                    'message' => __('Failed to write image file', 'mxchat-vision')
                );
            }

        } catch (Exception $e) {
            return array(
                'success' => false,
                'message' => $e->getMessage()
            );
        }
    }


    /**
     * Process images for API consumption
     *
     * @param array $images Raw image data
     * @return array Processed images ready for API
     */
    private function process_images_for_api($images) {
        $processed = array();

        foreach ($images as $image) {
            try {
                $processed_image = $this->process_single_image($image);
                if ($processed_image) {
                    $processed[] = $processed_image;
                }
            } catch (Exception $e) {
                //error_log('MxChat Vision: Failed to process image: ' . $e->getMessage());
                continue;
            }
        }

        return $processed;
    }

    /**
     * Process a single image
     *
     * @param array $image Image data
     * @return array|false Processed image data or false on failure
     */
    private function process_single_image($image) {
        // Validate required fields
        if (!isset($image['base64']) || !isset($image['mime_type'])) {
            return false;
        }

        // Resize image if necessary
        $resized_image = $this->resize_image_if_needed($image);
        
        if (!$resized_image) {
            return false;
        }

        return array(
            'base64' => $resized_image['base64'],
            'mime_type' => $resized_image['mime_type'],
            'original_name' => isset($image['original_name']) ? $image['original_name'] : 'image',
            'dimensions' => $resized_image['dimensions'],
            'file_size' => strlen(base64_decode($resized_image['base64']))
        );
    }

    /**
     * Fix image orientation based on EXIF data
     *
     * @param resource $image_resource GD image resource
     * @param string $image_data Raw image data
     * @param string $mime_type MIME type of image
     * @return resource Corrected image resource
     */
    private function fix_image_orientation($image_resource, $image_data, $mime_type) {
        // Only JPEG images have EXIF orientation data
        if ($mime_type !== 'image/jpeg') {
            return $image_resource;
        }

        // Try to read EXIF data from the image data
        try {
            // Create a temporary file to read EXIF data
            $temp_file = wp_tempnam('exif_');
            file_put_contents($temp_file, $image_data);

            if (function_exists('exif_read_data')) {
                $exif = @exif_read_data($temp_file);

                if ($exif && isset($exif['Orientation'])) {
                    $orientation = $exif['Orientation'];

                    switch ($orientation) {
                        case 2: // Horizontal flip
                            imageflip($image_resource, IMG_FLIP_HORIZONTAL);
                            break;
                        case 3: // 180 degree rotation
                            $image_resource = imagerotate($image_resource, 180, 0);
                            break;
                        case 4: // Vertical flip
                            imageflip($image_resource, IMG_FLIP_VERTICAL);
                            break;
                        case 5: // Vertical flip + 90 rotate clockwise
                            imageflip($image_resource, IMG_FLIP_VERTICAL);
                            $image_resource = imagerotate($image_resource, -90, 0);
                            break;
                        case 6: // 90 rotate clockwise
                            $image_resource = imagerotate($image_resource, -90, 0);
                            break;
                        case 7: // Horizontal flip + 90 rotate clockwise
                            imageflip($image_resource, IMG_FLIP_HORIZONTAL);
                            $image_resource = imagerotate($image_resource, -90, 0);
                            break;
                        case 8: // 90 rotate counter-clockwise
                            $image_resource = imagerotate($image_resource, 90, 0);
                            break;
                    }
                }
            }

            // Clean up temp file
            @unlink($temp_file);

        } catch (Exception $e) {
            // If EXIF reading fails, just return the original
            error_log('MxChat Vision: EXIF read error: ' . $e->getMessage());
        }

        return $image_resource;
    }

    /**
     * Resize image if it exceeds maximum dimensions
     *
     * @param array $image Image data
     * @return array|false Resized image data or false on failure
     */
    private function resize_image_if_needed($image) {
        try {
            // Decode base64 image
            $image_data = base64_decode($image['base64']);
            if (!$image_data) {
                return false;
            }

            // Create image resource based on MIME type
            $image_resource = null;
            switch ($image['mime_type']) {
                case 'image/jpeg':
                    $image_resource = imagecreatefromstring($image_data);
                    break;
                case 'image/png':
                    $image_resource = imagecreatefromstring($image_data);
                    break;
                case 'image/gif':
                    $image_resource = imagecreatefromstring($image_data);
                    break;
                case 'image/webp':
                    if (function_exists('imagecreatefromwebp')) {
                        $image_resource = imagecreatefromstring($image_data);
                    }
                    break;
            }

            if (!$image_resource) {
                return false;
            }

            // Fix orientation based on EXIF data (important for mobile photos)
            $image_resource = $this->fix_image_orientation($image_resource, $image_data, $image['mime_type']);

            $original_width = imagesx($image_resource);
            $original_height = imagesy($image_resource);

            // Even if no resize is needed, we need to re-encode the image
            // to apply any orientation corrections from EXIF data
            $needs_resize = ($original_width > $this->max_image_size || $original_height > $this->max_image_size);

            if (!$needs_resize) {
                // Re-encode the orientation-corrected image
                ob_start();
                switch ($image['mime_type']) {
                    case 'image/jpeg':
                        imagejpeg($image_resource, null, 90);
                        break;
                    case 'image/png':
                        imagepng($image_resource, null, 6);
                        break;
                    case 'image/gif':
                        imagegif($image_resource);
                        break;
                    case 'image/webp':
                        if (function_exists('imagewebp')) {
                            imagewebp($image_resource, null, 90);
                        } else {
                            imagejpeg($image_resource, null, 90);
                        }
                        break;
                }
                $corrected_data = ob_get_contents();
                ob_end_clean();

                imagedestroy($image_resource);

                return array(
                    'base64' => base64_encode($corrected_data),
                    'mime_type' => $image['mime_type'],
                    'dimensions' => array(
                        'width' => $original_width,
                        'height' => $original_height
                    )
                );
            }

            // Calculate new dimensions
            $ratio = min(
                $this->max_image_size / $original_width,
                $this->max_image_size / $original_height
            );

            $new_width = round($original_width * $ratio);
            $new_height = round($original_height * $ratio);

            // Create new image
            $new_image = imagecreatetruecolor($new_width, $new_height);

            // Preserve transparency for PNG and GIF
            if ($image['mime_type'] === 'image/png' || $image['mime_type'] === 'image/gif') {
                imagealphablending($new_image, false);
                imagesavealpha($new_image, true);
                $transparent = imagecolorallocatealpha($new_image, 255, 255, 255, 127);
                imagefill($new_image, 0, 0, $transparent);
            }

            // Resize the image
            imagecopyresampled(
                $new_image, $image_resource,
                0, 0, 0, 0,
                $new_width, $new_height,
                $original_width, $original_height
            );

            // Output to string
            ob_start();
            switch ($image['mime_type']) {
                case 'image/jpeg':
                    imagejpeg($new_image, null, 85);
                    break;
                case 'image/png':
                    imagepng($new_image, null, 6);
                    break;
                case 'image/gif':
                    imagegif($new_image);
                    break;
                case 'image/webp':
                    if (function_exists('imagewebp')) {
                        imagewebp($new_image, null, 85);
                    } else {
                        imagejpeg($new_image, null, 85);
                    }
                    break;
            }
            $resized_data = ob_get_contents();
            ob_end_clean();

            // Clean up
            imagedestroy($image_resource);
            imagedestroy($new_image);

            if (!$resized_data) {
                return false;
            }

            return array(
                'base64' => base64_encode($resized_data),
                'mime_type' => $image['mime_type'],
                'dimensions' => array(
                    'width' => $new_width,
                    'height' => $new_height
                )
            );

        } catch (Exception $e) {
            //error_log('MxChat Vision: Image resize error: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Enhance analysis result with additional metadata
     *
     * @param array $result Analysis result from AI model
     * @param array $images Processed images
     * @param string $model Model used
     * @return array Enhanced result
     */
    private function enhance_analysis_result($result, $images, $model) {
        $enhanced = $result;
        
        // Add metadata
        $enhanced['metadata'] = array(
            'model_used' => $model,
            'images_processed' => count($images),
            'processing_time' => isset($result['processing_time']) ? $result['processing_time'] : null,
            'timestamp' => current_time('mysql'),
            'total_image_size' => array_sum(array_column($images, 'file_size')),
            'image_details' => array()
        );

        // Add individual image details
        foreach ($images as $image) {
            $enhanced['metadata']['image_details'][] = array(
                'name' => $image['original_name'],
                'dimensions' => $image['dimensions'],
                'size' => $image['file_size'],
                'type' => $image['mime_type']
            );
        }

        // Format analysis for better readability
        $enhanced['formatted_analysis'] = $this->format_analysis_text($result['analysis']);

        return $enhanced;
    }

    /**
     * Format analysis text for better presentation
     *
     * @param string $analysis Raw analysis text
     * @return string Formatted analysis
     */
    private function format_analysis_text($analysis) {
        // Basic formatting improvements
        $formatted = trim($analysis);
        
        // Ensure proper paragraph breaks
        $formatted = preg_replace('/\n{3,}/', "\n\n", $formatted);
        
        // Add emphasis to key phrases (optional)
        $key_phrases = array(
            'I can see',
            'The image shows',
            'This appears to be',
            'Notable features include',
            'In summary',
            'Key observations'
        );

        foreach ($key_phrases as $phrase) {
            $formatted = str_ireplace($phrase, "**{$phrase}**", $formatted);
        }

        return $formatted;
    }

    /**
     * Log analysis attempt
     */
    private function log_analysis_attempt($model, $image_count, $prompt) {
        if (defined('WP_DEBUG') && WP_DEBUG) {

        }
    }

    /**
     * Log successful analysis
     */
    private function log_analysis_success($model, $result) {
        if (defined('WP_DEBUG') && WP_DEBUG) {

        }

        // Store analytics (optional)
        $this->store_analysis_stats($model, 'success');
    }

    /**
     * Log analysis error
     */
    private function log_analysis_error($model, $error_message) {


        // Store analytics (optional)
        $this->store_analysis_stats($model, 'error');
    }

    /**
     * Store analysis statistics
     */
    private function store_analysis_stats($model, $status) {
        $stats = get_option('mxchat_vision_stats', array(
            'total_analyses' => 0,
            'successful_analyses' => 0,
            'failed_analyses' => 0,
            'model_usage' => array(),
            'last_updated' => current_time('mysql')
        ));

        $stats['total_analyses']++;
        
        if ($status === 'success') {
            $stats['successful_analyses']++;
        } else {
            $stats['failed_analyses']++;
        }

        if (!isset($stats['model_usage'][$model])) {
            $stats['model_usage'][$model] = array(
                'total' => 0,
                'successful' => 0,
                'failed' => 0
            );
        }

        $stats['model_usage'][$model]['total']++;
        $stats['model_usage'][$model][$status === 'success' ? 'successful' : 'failed']++;
        $stats['last_updated'] = current_time('mysql');

        update_option('mxchat_vision_stats', $stats);
    }

    /**
     * Get analysis statistics
     *
     * @return array Statistics data
     */
    public static function get_analysis_stats() {
        return get_option('mxchat_vision_stats', array(
            'total_analyses' => 0,
            'successful_analyses' => 0,
            'failed_analyses' => 0,
            'model_usage' => array(),
            'last_updated' => null
        ));
    }

    /**
     * Validate image data structure
     *
     * @param array $image Image data to validate
     * @return bool True if valid, false otherwise
     */
    public static function validate_image_data($image) {
        $required_fields = array('base64', 'mime_type');
        
        foreach ($required_fields as $field) {
            if (!isset($image[$field]) || empty($image[$field])) {
                return false;
            }
        }

        // Validate MIME type
        $allowed_types = array('image/jpeg', 'image/png', 'image/gif', 'image/webp');
        if (!in_array($image['mime_type'], $allowed_types)) {
            return false;
        }

        // Validate base64 data
        $decoded = base64_decode($image['base64'], true);
        if ($decoded === false) {
            return false;
        }

        return true;
    }

    /**
     * Clean up processing resources
     */
    public function cleanup() {
        // Free up any resources if needed
        if (isset($this->models)) {
            unset($this->models);
        }
    }

    /**
     * Destructor
     */
    public function __destruct() {
        $this->cleanup();
    }
}
?>