<?php
/**
 * Plugin Name: MxChat Image Analysis
 * Plugin URI: https://mxchat.ai/
 * Description: Image analysis and editing add-on for MxChat using OpenAI Vision, Grok, and Google Gemini 2.5 Flash APIs
 * Version: 1.0.6
 * Author: MxChat
 * Author URI: https://mxchat.ai/
 * Text Domain: mxchat-vision
 * Requires at least: 5.8
 * Requires PHP: 7.4
 */

if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('MXCHAT_VISION_VERSION', '1.0.6');
define('MXCHAT_VISION_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('MXCHAT_VISION_PLUGIN_URL', plugin_dir_url(__FILE__));
define('MXCHAT_VISION_PLUGIN_BASENAME', plugin_basename(__FILE__));

// Autoloader for classes
spl_autoload_register(function ($class) {
    $prefix = 'MxChat_Vision_';
    $base_dir = MXCHAT_VISION_PLUGIN_DIR . 'includes/';

    $len = strlen($prefix);
    if (strncmp($prefix, $class, $len) !== 0) {
        // Also handle Vision_Processor and Vision_Models classes
        if (in_array($class, ['Vision_Processor', 'Vision_Models'])) {
            $file = $base_dir . 'class-' . str_replace('_', '-', strtolower($class)) . '.php';
            if (file_exists($file)) {
                require_once $file;
            }
        }
        return;
    }

    $relative_class = substr($class, $len);
    $file = $base_dir . 'class-mxchat-vision-' . str_replace('_', '-', strtolower($relative_class)) . '.php';

    if (file_exists($file)) {
        require_once $file;
    }
});

/**
 * Check if MxChat is active, supporting multisite environments
 */
function is_mxchat_active_vision() {
    if (is_multisite()) {
        $active_sitewide_plugins = get_site_option('active_sitewide_plugins');
        if (isset($active_sitewide_plugins['mxchat-basic/mxchat-basic.php'])) {
            return true;
        }
    }
    return in_array(
        'mxchat-basic/mxchat-basic.php',
        apply_filters('active_plugins', get_option('active_plugins'))
    );
}

/**
 * Check if MxChat Pro license is active
 */
function is_mxchat_pro_license_active_vision() {
    if (is_multisite()) {
        $network_license_status = get_site_option('mxchat_license_status');
        if ($network_license_status === 'active') {
            return true;
        }
    }
    $license_status = get_option('mxchat_license_status', 'inactive');
    return $license_status === 'active';
}

// Initialize update checker
require_once dirname(__FILE__) . '/update-checker.php';
PluginUpdateCheckerVision::init([
    'user_id' => 'wp_4_SGKQ2sGzkIGB',
    'plugin_slug' => 'mxchat-image-analysis'
]);


/**
 * Main plugin class
 */
final class MxChat_Vision {
    private static $instance = null;
    public $core;
    public $settings;
    public $processor;
    public $models;

    /**
     * Get singleton instance
     */
    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        try {
            if (!is_mxchat_active_vision()) {
                add_action('admin_notices', array($this, 'show_mxchat_required_notice'));
                return;
            }

            // Add admin notice for pro requirement
            add_action('admin_notices', array($this, 'show_pro_license_required_notice'));
            
            // Only load features if pro is active
            if (is_mxchat_pro_license_active_vision()) {
                $this->init_hooks();
                $this->init_components();
            }
            
        } catch (Exception $e) {
            add_action('admin_notices', function() use ($e) {
                echo '<div class="notice notice-error"><p>MxChat Vision Error: ' . esc_html($e->getMessage()) . '</p></div>';
            });
        }
    }

    /**
     * Initialize WordPress hooks
     */
    private function init_hooks() {
        add_action('admin_menu', array($this, 'add_admin_menu'), 20);
        add_action('network_admin_menu', array($this, 'add_network_admin_menu'), 20);
        add_action('wp_footer', array($this, 'control_image_button_visibility'));
        add_action('admin_footer', array($this, 'control_image_button_visibility'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_assets'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
    }

    /**
     * Initialize plugin components
     */
    private function init_components() {
        // Only initialize if classes exist
        if (class_exists('MxChat_Vision_Settings')) {
            $this->settings = new MxChat_Vision_Settings();
        }
        
        if (class_exists('MxChat_Vision_Core')) {
            $this->core = new MxChat_Vision_Core();
        }
        
        if (class_exists('Vision_Processor')) {
            $this->processor = new Vision_Processor();
        }
        
        if (class_exists('Vision_Models')) {
            $this->models = new Vision_Models();
        }
    }

    /**
     * Display notice if MxChat is not active
     */
    public function show_mxchat_required_notice() {
        ?>
        <div class="notice notice-error is-dismissible">
            <p><?php _e('<strong>MxChat Vision</strong> requires the MxChat plugin to be installed and active.', 'mxchat-vision'); ?></p>
        </div>
        <?php
    }

    /**
     * Display notice if MxChat Pro license is not active
     */
    public function show_pro_license_required_notice() {
        if (!is_mxchat_pro_license_active_vision()) {
            $class = 'notice notice-warning';
            $message = sprintf(
                __('MxChat Vision requires MxChat Pro to be activated. Please <a href="%s">activate your pro license</a> to use this addon.', 'mxchat-vision'),
                admin_url('admin.php?page=mxchat-activation')
            );
            printf('<div class="%1$s"><p>%2$s</p></div>', esc_attr($class), wp_kses_post($message));
        }
    }

    /**
     * Control the visibility of the image upload button in MxChat toolbar
     */
    public function control_image_button_visibility() {
        $vision_enabled = get_option('mxchat_vision_enabled', false);
        
        if ($vision_enabled) {
            ?>
            <script>
            document.addEventListener('DOMContentLoaded', function() {
                // Show the image upload button
                const imageButton = document.getElementById('image-upload-btn');
                if (imageButton) {
                    imageButton.style.display = 'block';
                }
            });
            </script>
            <?php
        }
    }

    /**
     * Enqueue frontend assets
     */
    public function enqueue_frontend_assets() {
        // Get current settings - use direct get_option calls to ensure values are retrieved
        $mode = get_option('mxchat_vision_mode', 'analysis');
        $disable_input = get_option('mxchat_vision_disable_input_on_editing', false);
        $edited_image_message = get_option('mxchat_vision_edited_image_message', "Here's your edited image:");

        // Convert boolean to string for JavaScript
        $disable_input_value = ($disable_input === true || $disable_input === '1' || $disable_input === 1) ? '1' : '0';

        // Enqueue the script
        wp_enqueue_script(
            'mxchat-vision-frontend',
            MXCHAT_VISION_PLUGIN_URL . 'public/js/vision-frontend.js',
            array('jquery'),
            MXCHAT_VISION_VERSION,
            true
        );

        // Add inline script BEFORE the main script to ensure mxchatVision object exists
        $inline_script = sprintf(
            'window.mxchatVision = %s;',
            json_encode(array(
                'ajaxUrl' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('mxchat_vision_nonce'),
                'maxFileSize' => 10 * 1024 * 1024, // 10MB
                'allowedTypes' => array('image/jpeg', 'image/png', 'image/gif', 'image/webp'),
                'mode' => $mode,
                'disableInputOnEditing' => $disable_input_value,
                'editedImageMessage' => $edited_image_message
            ))
        );

        wp_add_inline_script('mxchat-vision-frontend', $inline_script, 'before');
    }

    /**
     * Enqueue admin assets
     */
    public function enqueue_admin_assets($hook) {
        // Only load on our admin page
        if ($hook !== 'mxchat_page_mxchat-vision') {
            return;
        }

        wp_enqueue_style(
            'mxchat-vision-admin',
            MXCHAT_VISION_PLUGIN_URL . 'admin/css/vision-admin.css',
            array(),
            MXCHAT_VISION_VERSION
        );

        wp_enqueue_script(
            'mxchat-vision-admin',
            MXCHAT_VISION_PLUGIN_URL . 'admin/js/vision-admin.js',
            array('jquery'),
            MXCHAT_VISION_VERSION,
            true
        );
    }

    /**
     * Add menu to network admin under Settings
     */
    public function add_network_admin_menu() {
        add_submenu_page(
            'settings.php',
            esc_html__('MxChat Vision', 'mxchat-vision'),
            esc_html__('Vision', 'mxchat-vision'),
            'manage_network_options',
            'mxchat-vision',
            array($this->settings, 'display_admin_page')
        );
    }

    /**
     * Add menu to site admin under MxChat
     */
    public function add_admin_menu() {
        add_submenu_page(
            'mxchat-max',
            esc_html__('MxChat Vision', 'mxchat-vision'),
            esc_html__('Image Analysis', 'mxchat-vision'),
            'manage_options',
            'mxchat-vision',
            array($this->settings, 'display_admin_page')
        );
    }


}

/**
 * Initialize the plugin
 */
function MxChat_Vision() {
    return MxChat_Vision::instance();
}

add_action('plugins_loaded', 'MxChat_Vision', 20);

/**
 * Activation hook
 */
register_activation_hook(__FILE__, function() {
    // Set default options
    add_option('mxchat_vision_enabled', false);
    add_option('mxchat_vision_model', 'openai');
    add_option('mxchat_vision_mode', 'analysis');
    add_option('mxchat_vision_editing_prompt', 'Apply the following modifications to this image:');
    add_option('mxchat_vision_editing_model', 'gemini-2.5-flash-image');
    add_option('mxchat_vision_image_size', '1K');
    add_option('mxchat_vision_aspect_ratio', '1:1');
    add_option('mxchat_vision_disable_input_on_editing', false);
    add_option('mxchat_vision_edited_image_message', "Here's your edited image:");
});
?>
