/**
 * MxChat Vision Frontend JavaScript
 * Integrates with existing MxChat UI for seamless image analysis
 */

(function($) {
    'use strict';

    class MxChatVision {
        constructor() {
            this.uploadedImages = [];
            this.maxFileSize = mxchatVision.maxFileSize || (10 * 1024 * 1024); // 10MB default
            this.maxImages = mxchatVision.maxImages || 5;
            this.allowedTypes = mxchatVision.allowedTypes || ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
            this.isProcessingVision = false;
            this.originalSendMessage = null;
            
            this.init();
        }

        init() {
            this.setupEventListeners();
            this.setupImagePreviewContainer();
            this.setupPasteHandler();
            
            // Wait for the core MxChat to finish loading, then override
            setTimeout(() => {
                this.overrideSendMessage();
            }, 500);
        }

        escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }

        setupEventListeners() {
            // Image upload button click
            $(document).on('click', '#image-upload-btn', (e) => {
                e.preventDefault();
                $('#image-upload').click();
            });

            // File input change
            $(document).on('change', '#image-upload', (e) => {
                this.handleFileSelect(e.target.files);
                // Reset the file input to allow re-uploading the same file
                e.target.value = '';
            });

            // Remove individual images
            $(document).on('click', '.remove-vision-image', (e) => {
                e.preventDefault();
                const index = $(e.target).closest('.vision-image-item').data('index');
                this.removeImage(index);
            });

            // Clear all images
            $(document).on('click', '#clear-vision-images', (e) => {
                e.preventDefault();
                this.clearAllImages();
            });
        }

        setupImagePreviewContainer() {
            // Create image preview container above popular questions if it doesn't exist
            if ($('#vision-image-preview').length === 0) {
                const previewHtml = `
                    <div id="vision-image-preview" class="vision-image-preview" style="display: none;">
                        <div class="vision-images-container">
                            <div class="vision-images-grid" id="vision-images-grid"></div>
                        </div>
                    </div>
                `;
                
                // Try to insert before popular questions first, fallback to input container
                const $popularQuestions = $('#mxchat-popular-questions');
                if ($popularQuestions.length > 0) {
                    $popularQuestions.before(previewHtml);
                } else {
                    $('#input-container').before(previewHtml);
                }
            }
        }

        setupPasteHandler() {
            // Handle paste events anywhere in the chat
            $(document).on('paste', '#chat-input, #mxchat-chatbot', (e) => {
                const items = e.originalEvent.clipboardData?.items;
                if (!items) return;

                for (let i = 0; i < items.length; i++) {
                    const item = items[i];
                    if (item.type.indexOf('image') !== -1) {
                        e.preventDefault();
                        e.stopPropagation(); // Prevent event bubbling
                        const file = item.getAsFile();
                        if (file) {
                            this.handleFileSelect([file]);
                        }
                        break;
                    }
                }
            });
        }

        overrideSendMessage() {
            // Store the original sendMessage function if it exists
            if (typeof window.sendMessage === 'function') {
                this.originalSendMessage = window.sendMessage;
                //console.log('MxChat Vision: Stored original sendMessage function');
            } else {
                //console.log('MxChat Vision: sendMessage not found yet, will fallback to core logic');
            }

            // Create our override function
            const self = this;
            window.sendMessage = function() {
                //console.log('MxChat Vision: sendMessage called, images:', self.uploadedImages.length);
                
                if (self.uploadedImages.length > 0 && !self.isProcessingVision) {
                    //console.log('MxChat Vision: Processing with vision');
                    self.handleVisionMessage();
                } else {
                    //console.log('MxChat Vision: Using original sendMessage');
                    
                    // If we have the original function, use it
                    if (self.originalSendMessage && typeof self.originalSendMessage === 'function') {
                        self.originalSendMessage();
                    } else {
                        // Fallback: replicate core sendMessage logic
                        //console.log('MxChat Vision: Using fallback sendMessage logic');
                        self.fallbackSendMessage();
                    }
                }
            };

            // Also need to rebind the events since MxChat binds them after us
            this.rebindSendEvents();
        }

        fallbackSendMessage() {
            var message = $('#chat-input').val();
            if (message) {
                // Use core plugin functions if available
                if (typeof window.appendMessage === 'function') {
                    window.appendMessage("user", message);
                } else {
                    // Fallback DOM manipulation
                    const userBgColor = window.mxchatChat?.user_message_bg_color || '#fff';
                    const userFontColor = window.mxchatChat?.user_message_font_color || '#000';
                    const messageHtml = `
                        <div class="user-message" style="background: ${userBgColor}; color: ${userFontColor}; margin-bottom: 1em; padding: 10px; border-radius: 8px;" dir="auto">
                            ${this.escapeHtml(message)}
                        </div>
                    `;
                    $('#chat-box').append(messageHtml);
                }

                $('#chat-input').val('');
                $('#chat-input').css('height', 'auto');

                // Collapse quick questions
                if (typeof window.hasQuickQuestions === 'function' && typeof window.collapseQuickQuestions === 'function') {
                    if (window.hasQuickQuestions()) {
                        window.collapseQuickQuestions();
                    }
                }

                // Show thinking message
                if (typeof window.appendThinkingMessage === 'function') {
                    window.appendThinkingMessage();
                } else {
                    // Fallback thinking message
                    const botBgColor = window.mxchatChat?.bot_message_bg_color || '#f0f0f0';
                    const botFontColor = window.mxchatChat?.bot_message_font_color || '#000';
                    const thinkingHtml = `
                        <div class="bot-message temporary-message" style="background: ${botBgColor}; color: ${botFontColor}; margin-bottom: 1em; padding: 10px; border-radius: 8px;">
                            <div class="thinking-dots-container">
                                <div class="thinking-dots">
                                    <span class="dot" style="background-color: ${botFontColor};"></span>
                                    <span class="dot" style="background-color: ${botFontColor};"></span>
                                    <span class="dot" style="background-color: ${botFontColor};"></span>
                                </div>
                            </div>
                        </div>
                    `;
                    $('#chat-box').append(thinkingHtml);
                }

                // Scroll to bottom
                if (typeof window.scrollToBottom === 'function') {
                    window.scrollToBottom();
                } else {
                    // Fallback scroll
                    const chatBox = $('#chat-box');
                    chatBox.scrollTop(chatBox.prop("scrollHeight"));
                }

                // Send to server
                const currentModel = window.mxchatChat?.model || 'gpt-4o';
                const useStreaming = typeof window.shouldUseStreaming === 'function' ? 
                                     window.shouldUseStreaming(currentModel) : false;

                if (useStreaming && typeof window.callMxChatStream === 'function') {
                    window.callMxChatStream(message, function(response) {
                        $('.bot-message.temporary-message').removeClass('temporary-message');
                    });
                } else if (typeof window.callMxChat === 'function') {
                    window.callMxChat(message, function(response) {
                        if (typeof window.replaceLastMessage === 'function') {
                            window.replaceLastMessage("bot", response);
                        }
                    });
                } else {
                    console.error('MxChat Vision: No core chat functions available');
                }
            }
        }

        rebindSendEvents() {
            // Remove existing handlers and add our own
            $('#send-button').off('click').on('click', () => {
                window.sendMessage();
            });

            $('#chat-input').off('keypress').on('keypress', (e) => {
                if (e.which === 13 && !e.shiftKey) {
                    e.preventDefault();
                    window.sendMessage();
                }
            });
        }

        handleVisionMessage() {
            const message = $('#chat-input').val().trim();
            
            if (!message && this.uploadedImages.length === 0) {
                return;
            }

            //console.log('MxChat Vision: Starting vision processing');
            this.isProcessingVision = true;

            // Disable send button immediately
            $('#send-button').prop('disabled', true);

            // Only disable chat input during processing if it wasn't already disabled by the setting
            if (!(mxchatVision.mode === 'editing' && (mxchatVision.disableInputOnEditing === '1' || mxchatVision.disableInputOnEditing === 1 || mxchatVision.disableInputOnEditing === true))) {
                $('#chat-input').prop('disabled', true);
            }

            // Collapse quick questions if they exist - using core plugin functions
            if (typeof window.hasQuickQuestions === 'function' && typeof window.collapseQuickQuestions === 'function') {
                if (window.hasQuickQuestions()) {
                    window.collapseQuickQuestions();
                    //console.log('MxChat Vision: Quick questions collapsed');
                }
            }

            // Show user message with images
            //console.log('MxChat Vision: Adding user message to chat');

            // Only show text message if user actually typed something
            if (message) {
                // Use core appendMessage function if available
                if (typeof window.appendMessage === 'function') {
                    window.appendMessage("user", message);
                } else {
                    // Fallback - Create user message directly matching your MxChat styling
                    const userBgColor = window.mxchatChat?.user_message_bg_color || '#fff';
                    const userFontColor = window.mxchatChat?.user_message_font_color || '#000';

                    const textMessageHtml = `
                        <div class="user-message" style="background: ${userBgColor}; color: ${userFontColor}; margin-bottom: 1em; padding: 10px; border-radius: 8px;" dir="auto">
                            ${this.escapeHtml(message)}
                        </div>
                    `;
                    $('#chat-box').append(textMessageHtml);
                }
            }

            // Add images display
            const imageHtml = this.uploadedImages.map(img =>
                `<img src="${img.preview_url}" alt="${img.original_name}" class="vision-chat-image" style="max-width: 200px; margin: 5px; border-radius: 8px; box-shadow: 0 2px 8px rgba(0,0,0,0.1);" />`
            ).join('');

            if (imageHtml) {
                const userBgColor = window.mxchatChat?.user_message_bg_color || '#fff';
                const userFontColor = window.mxchatChat?.user_message_font_color || '#000';

                // Don't show caption when in editing mode
                const showCaption = mxchatVision.mode !== 'editing';
                const captionHtml = showCaption ? `
                    <div style="margin-top: 8px; font-size: 12px;">
                        ${this.uploadedImages.length} image(s) uploaded for analysis
                    </div>
                ` : '';

                const imageMessageHtml = `
                    <div class="user-message" style="background: ${userBgColor}; color: ${userFontColor}; margin-bottom: 1em; padding: 10px; border-radius: 8px;">
                        <div class="vision-images-display">
                            ${imageHtml}
                            ${captionHtml}
                        </div>
                    </div>
                `;
                $('#chat-box').append(imageMessageHtml);
            }

            // Clear input (matching core sendMessage flow exactly)
            $('#chat-input').val('');
            $('#chat-input').css('height', 'auto');

            // Store images for processing
            const imagesToProcess = [...this.uploadedImages];
            this.clearAllImages();

            // Show thinking message using MxChat's function
            if (typeof window.appendThinkingMessage === 'function') {
                window.appendThinkingMessage();
            } else {
                // Fallback thinking message
                const botBgColor = window.mxchatChat?.bot_message_bg_color || '#f0f0f0';
                const botFontColor = window.mxchatChat?.bot_message_font_color || '#000';
                const thinkingHtml = `
                    <div class="bot-message temporary-message" style="background: ${botBgColor}; color: ${botFontColor}; margin-bottom: 1em; padding: 10px; border-radius: 8px;">
                        <div class="thinking-dots-container">
                            <div class="thinking-dots">
                                <span class="dot" style="background-color: ${botFontColor};"></span>
                                <span class="dot" style="background-color: ${botFontColor};"></span>
                                <span class="dot" style="background-color: ${botFontColor};"></span>
                            </div>
                        </div>
                    </div>
                `;
                $('#chat-box').append(thinkingHtml);
            }

            // Scroll to bottom using MxChat's function
            if (typeof window.scrollToBottom === 'function') {
                window.scrollToBottom();
            } else {
                // Fallback scroll
                const chatBox = $('#chat-box');
                chatBox.scrollTop(chatBox.prop("scrollHeight"));
            }

            // Step 1: Analyze images first
            this.analyzeImages(imagesToProcess, message);
        }

        analyzeImages(images, userMessage) {
            //console.log('MxChat Vision: Analyzing images');

            const visionData = {
                action: 'mxchat_vision_analyze',
                nonce: mxchatVision.nonce,
                images: JSON.stringify(images.map(img => ({
                    base64: img.base64,
                    mime_type: img.mime_type,
                    original_name: img.original_name,
                    file_size: img.file_size
                }))),
                prompt: userMessage || ''
            };

            $.ajax({
                url: mxchatVision.ajaxUrl,
                type: 'POST',
                data: visionData,
                success: (response) => {
                    //console.log('MxChat Vision: Analysis response:', response);
                    if (response.success) {
                        // Check if this is an edited image result
                        if (response.data.is_edited_image) {
                            this.displayEditedImage(response.data.image_url, userMessage);
                        } else {
                            // Regular analysis - send to main chat
                            this.sendVisionToMainChat(response.data.analysis, userMessage, images);
                        }
                    } else {
                        this.handleVisionError(response.data?.message || 'Image processing failed');
                    }
                },
                error: (xhr, status, error) => {
                    console.error('MxChat Vision: Analysis failed:', error);
                    this.handleVisionError('Failed to analyze images. Please try again.');
                }
            });
        }

sendVisionToMainChat(imageAnalysis, userMessage, images) {
    //console.log('MxChat Vision: Sending to main chat handler');
    
    // Create the message that should be saved to chat history (original user message)
    const originalUserMessage = userMessage || 'Please analyze these images.';
    
    // Create the message for AI processing - make it seem like the AI analyzed the image
    let aiProcessingMessage = '';
    if (imageAnalysis) {
        aiProcessingMessage += `I have analyzed the uploaded image(s) and can see: ${imageAnalysis}`;
    }
    
    if (userMessage) {
        aiProcessingMessage += `\n\nNow, to answer your specific question "${userMessage}":`;
    } else {
        aiProcessingMessage += `\n\nBased on this analysis, I can provide you with detailed information about what I see in the image(s).`;
    }

    // IMPORTANT: Set global flags to help preserve testing data through the request
    // These will be picked up by the streaming/regular chat handlers
    window.mxchatVisionProcessed = true;
    window.mxchatOriginalMessage = originalUserMessage;
    window.mxchatVisionImagesCount = images.length;

    // Get the current model and check streaming
    const currentModel = window.mxchatChat?.model || 'gpt-4o';
    const useStreaming = typeof window.shouldUseStreaming === 'function' ? 
                         window.shouldUseStreaming(currentModel) : false;

    //console.log('MxChat Vision: Using streaming:', useStreaming);

    if (useStreaming) {
        this.callVisionStream(aiProcessingMessage, images, originalUserMessage);
    } else {
        this.callVisionChat(aiProcessingMessage, images, originalUserMessage);
    }
}

callVisionStream(message, images, originalUserMessage) {
    console.log('MxChat Vision: Using streaming through core function');
    
    // Use the core streaming function
    if (typeof window.callMxChatStream === 'function') {
        window.callMxChatStream(message, (response) => {
            console.log('MxChat Vision: Core stream callback received');
            this.finishVisionProcessing();
        });
    } else {
        // Fallback to direct streaming only if core function isn't available
        console.log('MxChat Vision: WARNING - Core callMxChatStream not found, using direct streaming');
        
        const pageContext = typeof window.getPageContext === 'function' ? window.getPageContext() : null;
        
        const formData = new FormData();
        formData.append('action', 'mxchat_stream_chat');
        formData.append('message', message); // AI processing message
        formData.append('original_user_message', originalUserMessage); // Original user message for history
        formData.append('session_id', this.getSessionId());
        formData.append('nonce', window.mxchatChat?.nonce || mxchatVision.nonce);
        formData.append('vision_processed', 'true');
        formData.append('vision_images_count', images.length.toString());
        
        if (pageContext) {
            formData.append('page_context', JSON.stringify(pageContext));
        }

        let accumulatedContent = '';

        fetch(window.mxchatChat?.ajax_url || mxchatVision.ajaxUrl, {
            method: 'POST',
            body: formData,
            credentials: 'same-origin'
        })
        .then(response => {
            if (!response.ok) {
                throw new Error('Network response was not ok');
            }

            // Check if response is JSON instead of streaming
            const contentType = response.headers.get('content-type');
            if (contentType && contentType.includes('application/json')) {
                return response.json().then(data => {
                    this.handleChatResponse(data);
                });
            }

            // Handle streaming response
            const reader = response.body.getReader();
            const decoder = new TextDecoder();
            let buffer = '';

            const processStream = () => {
                reader.read().then(({ done, value }) => {
                    if (done) {
                        //console.log('MxChat Vision: Streaming completed');
                        // Remove temporary-message class from the final message
                        $('.bot-message.temporary-message').removeClass('temporary-message');
                        this.finishVisionProcessing();
                        return;
                    }

                    buffer += decoder.decode(value, { stream: true });
                    const lines = buffer.split('\n');
                    buffer = lines.pop() || '';

                    for (const line of lines) {
                        if (line.startsWith('data: ')) {
                            const data = line.substring(6);

                            if (data === '[DONE]') {
                                //console.log('MxChat Vision: Received [DONE] signal');
                                // Remove temporary-message class from the final message
                                $('.bot-message.temporary-message').removeClass('temporary-message');
                                this.finishVisionProcessing();
                                return;
                            }

                            try {
                                const json = JSON.parse(data);
                                
                                if (json.content) {
                                    accumulatedContent += json.content;
                                    // Use core updateStreamingMessage function if available
                                    if (typeof window.updateStreamingMessage === 'function') {
                                        window.updateStreamingMessage(accumulatedContent);
                                    } else {
                                        // Fallback: update the temporary message directly
                                        this.updateStreamingMessageFallback(accumulatedContent);
                                    }
                                } else if (json.error) {
                                    this.handleVisionError(json.error);
                                    return;
                                }
                            } catch (e) {
                                console.error('MxChat Vision: Error parsing SSE data:', e);
                            }
                        }
                    }

                    processStream();
                });
            };

            processStream();
        })
        .catch(error => {
            console.error('MxChat Vision: Streaming error:', error);
            this.handleVisionError('Streaming failed, please try again.');
        });
    }
}
        updateStreamingMessageFallback(content) {
            // Fallback function to update streaming message if core function isn't available
            const formattedContent = this.formatMessageContent(content);
            
            // Find the temporary message
            const tempMessage = $('.bot-message.temporary-message').last();
            
            if (tempMessage.length) {
                // Update existing message
                tempMessage.html(formattedContent);
            } else {
                // Create new temporary message if it doesn't exist
                const botBgColor = window.mxchatChat?.bot_message_bg_color || '#f0f0f0';
                const botFontColor = window.mxchatChat?.bot_message_font_color || '#000';
                
                const messageHtml = `
                    <div class="bot-message temporary-message" style="background: ${botBgColor}; color: ${botFontColor}; margin-bottom: 1em; padding: 10px; border-radius: 8px;">
                        ${formattedContent}
                    </div>
                `;
                $('#chat-box').append(messageHtml);
            }
        }

        formatMessageContent(content) {
            // Basic formatting - you might want to enhance this
            return content
                .replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>')
                .replace(/\n/g, '<br>');
        }

callVisionChat(message, images, originalUserMessage) {
    console.log('MxChat Vision: Using regular chat through core function');
    
    // The vision addon should use the CORE callMxChat function, not make its own AJAX call
    if (typeof window.callMxChat === 'function') {
        // Call the core function which will pick up our global flags
        window.callMxChat(message, (response) => {
            console.log('MxChat Vision: Core chat callback received');
            // Don't need to handle response here - core function does it
            this.finishVisionProcessing();
        });
    } else {
        // Fallback to direct AJAX only if core function isn't available
        console.log('MxChat Vision: WARNING - Core callMxChat not found, using direct AJAX');
        
        const pageContext = typeof window.getPageContext === 'function' ? window.getPageContext() : null;
        
        const ajaxData = {
            action: 'mxchat_handle_chat_request',
            message: message,
            original_user_message: originalUserMessage,
            session_id: this.getSessionId(),
            nonce: window.mxchatChat?.nonce || mxchatVision.nonce,
            vision_processed: true,
            vision_images_count: images.length
        };
        
        console.log('Vision AJAX data being sent:', ajaxData);
        
        if (pageContext) {
            ajaxData.page_context = JSON.stringify(pageContext);
        }
        
        $.ajax({
            url: window.mxchatChat?.ajax_url || mxchatVision.ajaxUrl,
            type: 'POST',
            dataType: 'json',
            data: ajaxData,
            success: (response) => {
                console.log('MxChat Vision: Chat response received:', response);
                console.log('Testing data in response:', response.testing_data);
                this.handleChatResponse(response);
            },
            error: (xhr, status, error) => {
                console.error('MxChat Vision: Chat request failed:', error);
                this.handleVisionError('Failed to get AI response.');
            }
        });
    }
}
        handleChatResponse(response) {
            //console.log('MxChat Vision: Handling chat response:', response);
            
            // Use MxChat's existing response handling logic
            if (typeof window.replaceLastMessage === 'function') {
                let responseText = response.text || '';
                let responseHtml = response.html || '';
                let responseMessage = response.message || '';

                //console.log('MxChat Vision: Response parts:', { responseText, responseHtml, responseMessage });

                if (responseText && responseHtml) {
                    window.replaceLastMessage("bot", responseText, responseHtml);
                } else if (responseText) {
                    window.replaceLastMessage("bot", responseText);
                } else if (responseHtml) {
                    window.replaceLastMessage("bot", "", responseHtml);
                } else if (responseMessage) {
                    window.replaceLastMessage("bot", responseMessage);
                } else {
                    window.replaceLastMessage("bot", "I received an empty response. Please try again.");
                }
            } else {
                //console.log('MxChat Vision: replaceLastMessage not found, using fallback');
                // Fallback - add bot message directly
                if (response.text || response.html || response.message) {
                    const botMessage = response.text || response.html || response.message;
                    const botBgColor = window.mxchatChat?.bot_message_bg_color || '#f0f0f0';
                    const botFontColor = window.mxchatChat?.bot_message_font_color || '#000';
                    
                    const botDiv = $(`
                        <div class="bot-message" style="background: ${botBgColor}; color: ${botFontColor}; margin-bottom: 1em; padding: 10px; border-radius: 8px;">
                            <div class="message-content">${botMessage}</div>
                        </div>
                    `);
                    
                    // Remove thinking message first
                    $('.bot-message.temporary-message').remove();
                    $('#chat-box').append(botDiv);
                    
                    // Scroll to bottom
                    if (typeof window.scrollToBottom === 'function') {
                        window.scrollToBottom();
                    } else {
                        const chatBox = $('#chat-box');
                        chatBox.scrollTop(chatBox.prop("scrollHeight"));
                    }

                    // Scroll to top of last user message (mimic core behavior)
                    setTimeout(() => {
                        const lastUserMessage = $('#chat-box').find('.user-message').last();
                        if (lastUserMessage.length && typeof window.scrollElementToTop === 'function') {
                            window.scrollElementToTop(lastUserMessage);
                        }
                    }, 100);
                }
            }

            this.finishVisionProcessing();
        }

        displayEditedImage(imageUrl, userMessage) {
            //console.log('MxChat Vision: Displaying edited image:', imageUrl);

            // Remove the thinking message
            $('.bot-message.temporary-message').remove();

            // Create bot message with edited image
            const botBgColor = window.mxchatChat?.bot_message_bg_color || '#f0f0f0';
            const botFontColor = window.mxchatChat?.bot_message_font_color || '#000';

            // Get customizable message from settings
            const editedImageMessage = mxchatVision.editedImageMessage || "Here's your edited image:";

            const imageHtml = `
                <div class="bot-message" style="background: ${botBgColor}; color: ${botFontColor}; margin-bottom: 1em; padding: 15px; border-radius: 8px;">
                    <div style="display: flex; flex-direction: column; gap: 10px;">
                        <p style="margin: 0; font-weight: 500;">${this.escapeHtml(editedImageMessage)}</p>
                        <img src="${imageUrl}" alt="Edited Image" style="max-width: 100%; width: 100%; border-radius: 8px; box-shadow: 0 2px 8px rgba(0,0,0,0.1);" />
                        <a href="${imageUrl}" target="_blank" download style="color: #0073aa; font-size: 12px; text-decoration: none;">Download Edited Image</a>
                    </div>
                </div>
            `;

            $('#chat-box').append(imageHtml);

            // Scroll to bottom
            if (typeof window.scrollToBottom === 'function') {
                window.scrollToBottom();
            } else {
                const chatBox = $('#chat-box');
                chatBox.scrollTop(chatBox.prop("scrollHeight"));
            }

            this.finishVisionProcessing();
        }

        handleVisionError(errorMessage) {
            console.error('MxChat Vision: Error:', errorMessage);

            // Remove thinking message and show error
            if (typeof window.replaceLastMessage === 'function') {
                window.replaceLastMessage("bot", errorMessage);
            }

            this.finishVisionProcessing();
        }

        finishVisionProcessing() {
            //console.log('MxChat Vision: Finishing processing');
            this.isProcessingVision = false;
            $('#send-button').prop('disabled', false);

            // Only re-enable chat input if it wasn't disabled by the setting
            // (it will be re-enabled when images are cleared via updatePreview)
            if (!(mxchatVision.mode === 'editing' && (mxchatVision.disableInputOnEditing === '1' || mxchatVision.disableInputOnEditing === 1 || mxchatVision.disableInputOnEditing === true))) {
                $('#chat-input').prop('disabled', false);
            }
        }

        handleFileSelect(files) {
            if (!files || files.length === 0) return;

            const fileArray = Array.from(files);
            
            // Check total count including existing images
            if (this.uploadedImages.length + fileArray.length > this.maxImages) {
                this.showNotification(`Maximum ${this.maxImages} images allowed`, 'error');
                return;
            }

            // Process each file
            fileArray.forEach(file => {
                if (this.validateFile(file)) {
                    this.processFile(file);
                }
            });
        }

        validateFile(file) {
            // Check file type
            if (!this.allowedTypes.includes(file.type)) {
                this.showNotification(`${file.name}: Invalid file type`, 'error');
                return false;
            }

            // Check file size
            if (file.size > this.maxFileSize) {
                const sizeMB = Math.round(this.maxFileSize / 1024 / 1024);
                this.showNotification(`${file.name}: File too large (max ${sizeMB}MB)`, 'error');
                return false;
            }

            return true;
        }

        processFile(file) {
            const reader = new FileReader();

            reader.onload = (e) => {
                const imageData = {
                    file: file,
                    base64: e.target.result.split(',')[1], // Remove data URL prefix
                    mime_type: file.type,
                    original_name: file.name,
                    file_size: file.size,
                    preview_url: e.target.result,
                    id: Date.now() + '_' + Math.random() // Unique ID for each upload attempt
                };

                this.uploadedImages.push(imageData);
                this.showNotification(`Added ${file.name}`, 'success');

                // Collapse quick questions as soon as first image is uploaded (like other file uploads)
                if (this.uploadedImages.length === 1) { // Only on first image
                    //console.log('MxChat Vision: First image uploaded, collapsing quick questions...');

                    // Use the same method as core plugin
                    if (typeof window.hasQuickQuestions === 'function' && typeof window.collapseQuickQuestions === 'function') {
                        if (window.hasQuickQuestions()) {
                            window.collapseQuickQuestions();
                            //console.log('MxChat Vision: Quick questions collapsed successfully');
                        } else {
                            //console.log('MxChat Vision: No quick questions to collapse');
                        }
                    } else {
                        //console.log('MxChat Vision: Quick questions functions not available');
                    }
                }

                // Update preview AFTER collapsing questions (this will handle input disable)
                this.updatePreview();
            };

            reader.onerror = () => {
                this.showNotification(`Failed to read ${file.name}`, 'error');
            };

            reader.readAsDataURL(file);
        }

        removeImage(index) {
            if (index >= 0 && index < this.uploadedImages.length) {
                const removedImage = this.uploadedImages.splice(index, 1)[0];
                this.updatePreview();
                this.showNotification(`Removed ${removedImage.original_name}`, 'info');
            }
        }

        clearAllImages() {
            this.uploadedImages = [];
            this.updatePreview();
        }

        updatePreview() {
            const hasImages = this.uploadedImages.length > 0;
            const $preview = $('#vision-image-preview');
            const $grid = $('#vision-images-grid');

            if (hasImages) {
                $preview.show();
                $grid.empty();

                this.uploadedImages.forEach((image, index) => {
                    const imageItem = $(`
                        <div class="vision-image-item" data-index="${index}">
                            <div class="vision-image-wrapper">
                                <img src="${image.preview_url}" alt="${image.original_name}" />
                                <button type="button" class="remove-vision-image" title="Remove image">×</button>
                            </div>
                            <div class="vision-image-name" title="${image.original_name}">
                                ${this.truncateText(image.original_name, 15)}
                            </div>
                        </div>
                    `);

                    $grid.append(imageItem);
                });

                // Check if we should disable input in editing mode
                const shouldDisableInput = mxchatVision.mode === 'editing' &&
                                          (mxchatVision.disableInputOnEditing === '1' ||
                                           mxchatVision.disableInputOnEditing === 1 ||
                                           mxchatVision.disableInputOnEditing === true);

                console.log('MxChat Vision Debug:', {
                    mode: mxchatVision.mode,
                    disableInputOnEditing: mxchatVision.disableInputOnEditing,
                    shouldDisableInput: shouldDisableInput
                });

                if (shouldDisableInput) {
                    // Disable input, clear any existing text, and update placeholder in editing mode if setting is enabled
                    console.log('Disabling input field and clearing text');
                    $('#chat-input').val(''); // Clear any existing text
                    $('#chat-input').css('height', 'auto'); // Reset height
                    $('#chat-input').prop('disabled', true);
                    $('#chat-input').attr('placeholder', 'Input disabled - upload images for editing');
                } else {
                    // Normal mode - update placeholder but keep input enabled
                    $('#chat-input').attr('placeholder', `Ask about ${this.uploadedImages.length} image(s)...`);
                }

            } else {
                $preview.hide();

                // Re-enable input if it was disabled
                if (mxchatVision.mode === 'editing' && (mxchatVision.disableInputOnEditing === '1' || mxchatVision.disableInputOnEditing === 1 || mxchatVision.disableInputOnEditing === true)) {
                    $('#chat-input').prop('disabled', false);
                }

                // Restore original placeholder
                const originalPlaceholder = $('#chat-input').data('original-placeholder') ||
                                          window.mxchatChat?.input_copy || 'How can I assist?';
                $('#chat-input').attr('placeholder', originalPlaceholder);
            }
        }

        getSessionId() {
            // Use MxChat's existing session function if available
            if (typeof window.getChatSession === 'function') {
                return window.getChatSession();
            }
            
            // Fallback
            return 'session_' + Date.now();
        }

        showNotification(message, type = 'info') {
            // Create temporary notification
            const notification = $(`
                <div class="vision-notification vision-notification-${type}">
                    ${this.escapeHtml(message)}
                </div>
            `);

            // Add to page
            $('body').append(notification);

            // Position and show
            notification.css({
                position: 'fixed',
                top: '20px',
                right: '20px',
                zIndex: 10000,
                padding: '10px 15px',
                borderRadius: '4px',
                color: 'white',
                backgroundColor: type === 'error' ? '#dc3545' : 
                                type === 'success' ? '#28a745' : '#6c757d',
                fontSize: '14px',
                maxWidth: '300px',
                opacity: 0
            });

            // Animate in
            notification.animate({ opacity: 1 }, 200);

            // Auto remove
            setTimeout(() => {
                notification.animate({ opacity: 0 }, 200, () => {
                    notification.remove();
                });
            }, 3000);
        }

        truncateText(text, maxLength) {
            return text.length > maxLength ? text.substring(0, maxLength) + '...' : text;
        }
    }

    // Initialize when document is ready, but after MxChat finishes loading
    $(document).ready(() => {
        // Wait a bit for MxChat to fully initialize
        setTimeout(() => {
            if (typeof mxchatVision !== 'undefined' && $('#image-upload-btn').length > 0) {
                //console.log('MxChat Vision: Initializing');
                window.mxchatVisionInstance = new MxChatVision();
                
                // Store original placeholder
                const originalPlaceholder = $('#chat-input').attr('placeholder');
                $('#chat-input').data('original-placeholder', originalPlaceholder);
            }
        }, 1000); // Give MxChat time to fully load
    });

})(jQuery);
