/**
 * MxChat Migration Tool - Admin JavaScript
 */

(function($) {
    'use strict';

    let migrationInProgress = false;
    let migrationId = null;
    let processingInterval = null;

    $(document).ready(function() {
        // Toggle Pinecone configuration fields
        $('#target_database').on('change', function() {
            if ($(this).val() === 'pinecone') {
                $('.pinecone-config').slideDown();
            } else {
                $('.pinecone-config').slideUp();
            }
        }).trigger('change');

        // Toggle API key fields based on embedding model
        $('#target_embedding_model').on('change', function() {
            const model = $(this).val();
            $('.api-key-row').hide();

            if (model.startsWith('voyage')) {
                $('.voyage-key').show();
            } else if (model.startsWith('gemini-embedding')) {
                $('.gemini-key').show();
            } else {
                $('.openai-key').show();
            }
        }).trigger('change');

        // Handle migration form submission
        $('#mxchat-migration-form').on('submit', function(e) {
            e.preventDefault();

            if (migrationInProgress) {
                showModal('A migration is already in progress', 'warning');
                return;
            }

            // Confirm migration
            const confirmMessage = mxchatMigration.strings.confirm_migration +
                '<br><br><small style="color: #666; font-style: italic;">Note: While we attempt to automatically adjust action similarity thresholds to match the new model, this will likely require manual review and adjustment.</small>';

            showConfirmModal(
                confirmMessage,
                function() {
                    startMigration();
                }
            );
        });

        // Cancel migration
        $('#cancel-migration-btn').on('click', function() {
            showConfirmModal(
                'Are you sure you want to cancel the migration?',
                function() {
                    cancelMigration();
                }
            );
        });

        // Close progress panel
        $('#close-progress-btn').on('click', function() {
            $('#migration-progress').slideUp();
            location.reload(); // Reload to update history
        });

        // Modal close handlers
        $(document).on('click', '.mxchat-mig-modal-close, .mxchat-mig-modal-overlay', function() {
            closeModal();
        });

        $(document).on('click', '.mxchat-mig-modal-content', function(e) {
            e.stopPropagation();
        });
    });

    /**
     * Start migration
     */
    function startMigration() {
        const formData = {
            action: 'mxchat_start_migration',
            nonce: mxchatMigration.nonce,
            migration_type: $('#migration_type').val(),
            target_database: $('#target_database').val(),
            target_embedding_model: $('#target_embedding_model').val(),
            batch_size: $('#batch_size').val(),
            openai_api_key: $('#openai_api_key').val(),
            voyage_api_key: $('#voyage_api_key').val(),
            gemini_api_key: $('#gemini_api_key').val(),
            pinecone_api_key: $('#pinecone_api_key').val(),
            pinecone_host: $('#pinecone_host').val(),
            pinecone_namespace: $('#pinecone_namespace').val()
        };

        // Show progress section
        $('#migration-progress').slideDown();
        $('#migration-status-text').text(mxchatMigration.strings.migration_in_progress);
        $('#migration-log').html('<div class="log-entry">Initializing migration...</div>');
        $('#start-migration-btn').prop('disabled', true);
        $('#cancel-migration-btn').show();

        $.ajax({
            url: mxchatMigration.ajax_url,
            type: 'POST',
            data: formData,
            success: function(response) {
                if (response.success) {
                    migrationInProgress = true;
                    migrationId = response.data.migration_id;

                    addLogEntry('Migration started successfully');
                    addLogEntry('Total items to migrate: ' + response.data.total_items);

                    // Start processing batches
                    processNextBatch();
                } else {
                    showError('Failed to start migration: ' + response.data.message);
                }
            },
            error: function(xhr, status, error) {
                showError('AJAX error: ' + error);
            }
        });
    }

    /**
     * Process next batch
     */
    function processNextBatch() {
        if (!migrationInProgress || !migrationId) {
            return;
        }

        $.ajax({
            url: mxchatMigration.ajax_url,
            type: 'POST',
            data: {
                action: 'mxchat_process_migration_batch',
                nonce: mxchatMigration.nonce,
                migration_id: migrationId
            },
            success: function(response) {
                if (response.success) {
                    const data = response.data;

                    // Show cleanup message if present
                    if (data.cleanup_message) {
                        addLogEntry(data.cleanup_message, 'info');
                    }

                    // Update progress
                    updateProgress(data.total_processed, data.total_items);

                    // Add log entry
                    if (data.processed > 0) {
                        addLogEntry(`Processed ${data.processed} items in ${data.current_stage} migration`);
                    }

                    if (data.failed > 0) {
                        addLogEntry(`Failed: ${data.failed} items`, 'error');
                    }

                    // Add error details
                    if (data.errors && data.errors.length > 0) {
                        data.errors.forEach(function(error) {
                            addLogEntry('Error: ' + error, 'error');
                        });
                    }

                    // Check if we should continue
                    if (data.continue) {
                        // Continue processing
                        setTimeout(processNextBatch, 1000); // 1 second delay between batches
                    } else {
                        // Migration complete
                        migrationComplete(data.total_processed, data.total_failed);
                    }
                } else {
                    showError('Batch processing failed: ' + response.data.message);
                }
            },
            error: function(xhr, status, error) {
                showError('AJAX error during batch processing: ' + error);
            }
        });
    }

    /**
     * Migration complete
     */
    function migrationComplete(totalProcessed, totalFailed) {
        migrationInProgress = false;

        $('#migration-status-text').text('Migration Completed');
        $('#cancel-migration-btn').hide();
        $('#close-progress-btn').show();

        updateProgress(totalProcessed, totalProcessed);

        if (totalFailed > 0) {
            addLogEntry(`Migration completed with ${totalFailed} failures`, 'warning');
        } else {
            addLogEntry('Migration completed successfully!', 'success');
        }

        // Show finalization modal with separate checkboxes
        showFinalizationModal();
    }

    /**
     * Cancel migration
     */
    function cancelMigration() {
        if (!migrationId) {
            return;
        }

        $.ajax({
            url: mxchatMigration.ajax_url,
            type: 'POST',
            data: {
                action: 'mxchat_cancel_migration',
                nonce: mxchatMigration.nonce,
                migration_id: migrationId
            },
            success: function(response) {
                if (response.success) {
                    migrationInProgress = false;
                    migrationId = null;

                    $('#migration-status-text').text('Migration Cancelled');
                    $('#cancel-migration-btn').hide();
                    $('#close-progress-btn').show();
                    addLogEntry('Migration cancelled by user', 'warning');
                } else {
                    showError('Failed to cancel migration: ' + response.data.message);
                }
            },
            error: function(xhr, status, error) {
                showError('AJAX error: ' + error);
            }
        });
    }

    /**
     * Finalize migration
     */
    function finalizeMigration(updateEmbeddingModel, updateDatabase) {
        $.ajax({
            url: mxchatMigration.ajax_url,
            type: 'POST',
            data: {
                action: 'mxchat_finalize_migration',
                nonce: mxchatMigration.nonce,
                migration_id: migrationId,
                update_embedding_model: updateEmbeddingModel,
                update_database: updateDatabase
            },
            success: function(response) {
                if (response.success) {
                    let message = 'Migration finalized successfully!';
                    if (updateEmbeddingModel || updateDatabase) {
                        message += ' Settings have been updated.';
                        addLogEntry('Plugin settings updated successfully', 'success');
                    } else {
                        addLogEntry('Migration finalized without updating settings', 'info');
                    }

                    showModal(message + ' Please refresh the page.', 'success', function() {
                        location.reload();
                    });
                } else {
                    showError('Failed to finalize migration: ' + response.data.message);
                }
            },
            error: function(xhr, status, error) {
                showError('AJAX error: ' + error);
            }
        });
    }

    /**
     * Update progress bar
     */
    function updateProgress(processed, total) {
        if (total === 0) {
            return;
        }

        const percentage = Math.round((processed / total) * 100);
        $('#progress-bar-fill').css('width', percentage + '%');
        $('#progress-percentage').text(percentage + '%');
        $('#migration-progress-text').text(processed + ' / ' + total);
    }

    /**
     * Add log entry
     */
    function addLogEntry(message, type) {
        type = type || 'info';
        const timestamp = new Date().toLocaleTimeString();
        const logEntry = $('<div>')
            .addClass('log-entry log-' + type)
            .html('<span class="log-time">[' + timestamp + ']</span> ' + message);

        $('#migration-log').append(logEntry);

        // Auto-scroll to bottom
        const logContainer = $('#migration-log');
        logContainer.scrollTop(logContainer[0].scrollHeight);
    }

    /**
     * Show error
     */
    function showError(message) {
        migrationInProgress = false;
        $('#migration-status-text').text('Error');
        $('#cancel-migration-btn').hide();
        $('#close-progress-btn').show();
        $('#start-migration-btn').prop('disabled', false);
        addLogEntry(message, 'error');
    }

    /**
     * Show modal
     */
    function showModal(message, type, callback) {
        type = type || 'info';

        const iconMap = {
            'success': '✓',
            'error': '✕',
            'warning': '⚠',
            'info': 'ℹ'
        };

        const modalHtml = `
            <div class="mxchat-mig-modal-overlay">
                <div class="mxchat-mig-modal-content">
                    <div class="mxchat-mig-modal-header mxchat-mig-modal-${type}">
                        <span class="mxchat-mig-modal-icon">${iconMap[type] || iconMap['info']}</span>
                        <span class="mxchat-mig-modal-close">&times;</span>
                    </div>
                    <div class="mxchat-mig-modal-body">
                        <p>${message}</p>
                    </div>
                    <div class="mxchat-mig-modal-footer">
                        <button type="button" class="mxchat-mig-modal-btn mxchat-mig-modal-btn-primary mxchat-mig-modal-ok">OK</button>
                    </div>
                </div>
            </div>
        `;

        $('body').append(modalHtml);

        $('.mxchat-mig-modal-ok').on('click', function() {
            closeModal();
            if (callback) callback();
        });
    }

    /**
     * Show confirm modal
     */
    function showConfirmModal(message, confirmCallback) {
        const modalHtml = `
            <div class="mxchat-mig-modal-overlay">
                <div class="mxchat-mig-modal-content">
                    <div class="mxchat-mig-modal-header">
                        <span class="mxchat-mig-modal-icon">?</span>
                        <span class="mxchat-mig-modal-close">&times;</span>
                    </div>
                    <div class="mxchat-mig-modal-body">
                        <p>${message}</p>
                    </div>
                    <div class="mxchat-mig-modal-footer">
                        <button type="button" class="mxchat-mig-modal-btn mxchat-mig-modal-btn-secondary mxchat-mig-modal-cancel">Cancel</button>
                        <button type="button" class="mxchat-mig-modal-btn mxchat-mig-modal-btn-primary mxchat-mig-modal-confirm">Confirm</button>
                    </div>
                </div>
            </div>
        `;

        $('body').append(modalHtml);

        $('.mxchat-mig-modal-confirm').on('click', function() {
            closeModal();
            if (confirmCallback) confirmCallback();
        });

        $('.mxchat-mig-modal-cancel').on('click', function() {
            closeModal();
        });
    }

    /**
     * Show finalization modal
     */
    function showFinalizationModal() {
        const targetModel = $('#target_embedding_model option:selected').text();
        const targetDatabase = $('#target_database option:selected').text();

        const modalHtml = `
            <div class="mxchat-mig-modal-overlay">
                <div class="mxchat-mig-modal-content mxchat-mig-finalize-modal">
                    <div class="mxchat-mig-modal-header mxchat-mig-modal-success">
                        <span class="mxchat-mig-modal-icon">✓</span>
                        <h3>Migration Complete!</h3>
                        <span class="mxchat-mig-modal-close">&times;</span>
                    </div>
                    <div class="mxchat-mig-modal-body">
                        <p class="mxchat-mig-finalize-intro">Would you like to update your plugin settings to use the new configuration?</p>

                        <div class="mxchat-mig-finalize-options">
                            <label class="mxchat-mig-checkbox-label">
                                <input type="checkbox" id="mxchat-mig-update-model" checked>
                                <span>Update embedding model to <strong>${targetModel}</strong></span>
                            </label>

                            <label class="mxchat-mig-checkbox-label">
                                <input type="checkbox" id="mxchat-mig-update-database" checked>
                                <span>Update default database to <strong>${targetDatabase}</strong></span>
                            </label>
                        </div>
                    </div>
                    <div class="mxchat-mig-modal-footer">
                        <button type="button" class="mxchat-mig-modal-btn mxchat-mig-modal-btn-secondary mxchat-mig-finalize-skip">Skip Updates</button>
                        <button type="button" class="mxchat-mig-modal-btn mxchat-mig-modal-btn-primary mxchat-mig-finalize-apply">Apply Changes</button>
                    </div>
                </div>
            </div>
        `;

        $('body').append(modalHtml);

        $('.mxchat-mig-finalize-apply').on('click', function() {
            const updateModel = $('#mxchat-mig-update-model').is(':checked');
            const updateDatabase = $('#mxchat-mig-update-database').is(':checked');
            closeModal();
            finalizeMigration(updateModel, updateDatabase);
        });

        $('.mxchat-mig-finalize-skip').on('click', function() {
            closeModal();
            finalizeMigration(false, false);
        });
    }

    /**
     * Close modal
     */
    function closeModal() {
        $('.mxchat-mig-modal-overlay').fadeOut(200, function() {
            $(this).remove();
        });
    }

})(jQuery);
