<?php
/**
 * Actions/Intents Migrator
 *
 * Handles migration of actions/intents when switching embedding models
 *
 * @package MxChat_Migration_Tool
 */

if (!defined('ABSPATH')) {
    exit;
}

class MxChat_Actions_Migrator {

    /**
     * Source embedding model
     */
    private $source_model;

    /**
     * Target embedding model
     */
    private $target_model;

    /**
     * API key for embedding generation
     */
    private $api_key;

    /**
     * Migration ID
     */
    private $migration_id;

    /**
     * Constructor
     */
    public function __construct($config) {
        $this->source_model = $config['source_model'];
        $this->target_model = $config['target_model'];
        $this->api_key = $config['api_key'];
        $this->migration_id = $config['migration_id'];
    }

    /**
     * Get similarity threshold adjustment factor based on model migration
     * Returns a multiplier to adjust the similarity threshold
     */
    private function get_similarity_adjustment_factor() {
        // Adjustment matrix: source_model_type => target_model_type => adjustment_factor
        // Factor < 1.0 means lower threshold, > 1.0 means higher threshold

        $source_type = $this->get_model_type($this->source_model);
        $target_type = $this->get_model_type($this->target_model);

        // Empirical adjustment factors based on embedding model characteristics
        // Based on real-world testing and OpenAI community reports:
        // - ada-002 typically gives 85-95% similarity for good matches
        // - text-embedding-3-small gives 50-70% similarity for the SAME matches
        // - This is a ~30% absolute drop, requiring significant threshold reduction
        $adjustment_matrix = array(
            'openai-ada-002' => array(
                'openai-ada-002' => 1.0,
                'openai-3-small' => 0.70,  // 3-small produces MUCH lower scores (30% reduction)
                'openai-3-large' => 0.68,  // 3-large also produces lower scores (32% reduction)
                'voyage' => 0.75,          // Voyage different distribution
                'gemini' => 0.73           // Gemini different distribution
            ),
            'openai-3-small' => array(
                'openai-ada-002' => 1.43,  // ada-002 gives much higher scores (43% increase)
                'openai-3-small' => 1.0,
                'openai-3-large' => 0.97,  // Similar models, slight adjustment
                'voyage' => 1.07,
                'gemini' => 1.04
            ),
            'openai-3-large' => array(
                'openai-ada-002' => 1.47,  // ada-002 gives much higher scores
                'openai-3-small' => 1.03,
                'openai-3-large' => 1.0,
                'voyage' => 1.10,
                'gemini' => 1.07
            ),
            'voyage' => array(
                'openai-ada-002' => 1.33,  // Moving to ada-002 requires higher threshold
                'openai-3-small' => 0.93,
                'openai-3-large' => 0.91,
                'voyage' => 1.0,
                'gemini' => 0.97
            ),
            'gemini' => array(
                'openai-ada-002' => 1.37,
                'openai-3-small' => 0.96,
                'openai-3-large' => 0.93,
                'voyage' => 1.03,
                'gemini' => 1.0
            )
        );

        // Get adjustment factor, default to 1.0 if not found
        if (isset($adjustment_matrix[$source_type][$target_type])) {
            return $adjustment_matrix[$source_type][$target_type];
        }

        // If same type, no adjustment
        if ($source_type === $target_type) {
            return 1.0;
        }

        // Default conservative adjustment
        return 0.95;
    }

    /**
     * Determine model type from model name
     */
    private function get_model_type($model) {
        if (strpos($model, 'text-embedding-ada-002') !== false) {
            return 'openai-ada-002';
        } elseif (strpos($model, 'text-embedding-3-small') !== false) {
            return 'openai-3-small';
        } elseif (strpos($model, 'text-embedding-3-large') !== false) {
            return 'openai-3-large';
        } elseif (strpos($model, 'voyage') !== false) {
            return 'voyage';
        } elseif (strpos($model, 'gemini') !== false) {
            return 'gemini';
        }

        // Default to treating as OpenAI
        return 'openai-ada-002';
    }

    /**
     * Adjust similarity threshold for new model
     */
    private function adjust_similarity_threshold($original_threshold) {
        $adjustment_factor = $this->get_similarity_adjustment_factor();
        $adjusted_threshold = $original_threshold * $adjustment_factor;

        // Ensure threshold stays within reasonable bounds
        $adjusted_threshold = max(0.60, min(0.95, $adjusted_threshold));

        error_log(sprintf(
            'MIGRATION: Adjusted threshold from %.2f to %.2f (factor: %.2f) for %s -> %s',
            $original_threshold,
            $adjusted_threshold,
            $adjustment_factor,
            $this->source_model,
            $this->target_model
        ));

        return $adjusted_threshold;
    }

    /**
     * Get total number of actions/intents
     */
    public function get_total_items() {
        global $wpdb;
        return (int) $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}mxchat_intents");
    }

    /**
     * Migrate actions/intents in batches
     */
    public function migrate_batch($offset, $limit) {
        global $wpdb;

        $intents = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM {$wpdb->prefix}mxchat_intents ORDER BY id LIMIT %d OFFSET %d",
                $limit,
                $offset
            ),
            ARRAY_A
        );

        if (empty($intents)) {
            return array(
                'success' => true,
                'processed' => 0,
                'failed' => 0,
                'message' => 'No more items to process'
            );
        }

        $processed = 0;
        $failed = 0;
        $errors = array();

        foreach ($intents as $intent) {
            try {
                // Parse training phrases - they are comma-separated text, not JSON
                $phrases_text = $intent['phrases'];

                error_log('MIGRATION DEBUG - Intent: ' . $intent['intent_label'] . ', Phrases: ' . $phrases_text);

                // Try JSON decode first (in case format changed)
                $phrases = json_decode($phrases_text, true);

                // If not JSON, treat as comma-separated text
                if (!is_array($phrases)) {
                    $phrases = array_map('trim', explode(',', $phrases_text));
                    $phrases = array_filter($phrases); // Remove empty values
                }

                error_log('MIGRATION DEBUG - Parsed phrases: ' . print_r($phrases, true));

                if (empty($phrases) || (count($phrases) === 1 && empty($phrases[0]))) {
                    throw new Exception('No training phrases found for intent: ' . $intent['intent_label']);
                }

                // Generate new embedding from combined phrases
                $combined_text = implode(' ', $phrases);
                $new_embedding = $this->generate_embedding($combined_text, $this->target_model, $this->api_key);

                if (!$new_embedding) {
                    throw new Exception('Failed to generate embedding for intent: ' . $intent['intent_label']);
                }

                // Adjust similarity threshold for the new model
                $original_threshold = floatval($intent['similarity_threshold']);
                $adjusted_threshold = $this->adjust_similarity_threshold($original_threshold);

                // Update the intent with new embedding and adjusted threshold
                $result = $wpdb->update(
                    $wpdb->prefix . 'mxchat_intents',
                    array(
                        'embedding_vector' => serialize($new_embedding),
                        'similarity_threshold' => $adjusted_threshold
                    ),
                    array('id' => $intent['id']),
                    array('%s', '%f'),
                    array('%d')
                );

                if ($result !== false) {
                    $processed++;
                    error_log(sprintf(
                        'MIGRATION SUCCESS: Updated intent "%s" - Threshold: %.3f -> %.3f',
                        $intent['intent_label'],
                        $original_threshold,
                        $adjusted_threshold
                    ));
                } else {
                    $failed++;
                    $errors[] = 'Failed to update intent: ' . $intent['intent_label'];
                }

            } catch (Exception $e) {
                $failed++;
                $errors[] = $e->getMessage();
                error_log('MxChat Migration Error: ' . $e->getMessage());
            }
        }

        $adjustment_factor = $this->get_similarity_adjustment_factor();
        $adjustment_percentage = round(($adjustment_factor - 1.0) * 100);

        $threshold_message = '';
        if ($adjustment_factor != 1.0) {
            if ($adjustment_percentage > 0) {
                $threshold_message = sprintf(' (thresholds increased by %d%%)', abs($adjustment_percentage));
            } else {
                $threshold_message = sprintf(' (thresholds decreased by %d%%)', abs($adjustment_percentage));
            }
        }

        return array(
            'success' => true,
            'processed' => $processed,
            'failed' => $failed,
            'errors' => $errors,
            'message' => sprintf('Processed %d actions, %d failed%s', $processed, $failed, $threshold_message),
            'threshold_adjustment' => $adjustment_factor
        );
    }

    /**
     * Generate embedding for content
     */
    private function generate_embedding($content, $model, $api_key) {
        // Use the core plugin's embedding generation method from MxChat_Admin
        if (class_exists('MxChat_Admin')) {
            // Get the knowledge manager instance
            if (class_exists('MxChat_Knowledge_Manager')) {
                $knowledge_manager = MxChat_Knowledge_Manager::get_instance();
                $admin = new MxChat_Admin($knowledge_manager);

                // Temporarily set the embedding model and API key
                $options = get_option('mxchat_options');
                $original_model = isset($options['embedding_model']) ? $options['embedding_model'] : '';
                $original_api_key = isset($options['api_key']) ? $options['api_key'] : '';

                $options['embedding_model'] = $model;

                // Set the appropriate API key based on model
                if (strpos($model, 'voyage') === 0) {
                    $original_voyage_key = isset($options['voyage_api_key']) ? $options['voyage_api_key'] : '';
                    $options['voyage_api_key'] = $api_key;
                } elseif (strpos($model, 'gemini-embedding') === 0) {
                    $original_gemini_key = isset($options['gemini_api_key']) ? $options['gemini_api_key'] : '';
                    $options['gemini_api_key'] = $api_key;
                } else {
                    $options['api_key'] = $api_key;
                }

                update_option('mxchat_options', $options);

                // Generate embedding
                $embedding = $admin->mxchat_generate_embedding($content);

                // Restore original settings
                $options['embedding_model'] = $original_model;
                if (strpos($model, 'voyage') === 0) {
                    $options['voyage_api_key'] = $original_voyage_key;
                } elseif (strpos($model, 'gemini-embedding') === 0) {
                    $options['gemini_api_key'] = $original_gemini_key;
                } else {
                    $options['api_key'] = $original_api_key;
                }
                update_option('mxchat_options', $options);

                // Check if embedding generation was successful
                if (is_array($embedding)) {
                    return $embedding;
                }

                error_log('MxChat Migration: Embedding generation failed: ' . print_r($embedding, true));
                return false;
            }
        }

        return false;
    }

    /**
     * Validate migrated actions
     */
    public function validate_migration() {
        global $wpdb;

        $total = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}mxchat_intents");
        $valid = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}mxchat_intents WHERE embedding_vector IS NOT NULL AND embedding_vector != ''");

        return array(
            'total' => (int) $total,
            'valid' => (int) $valid,
            'success' => ($total === $valid)
        );
    }
}
