<?php
/**
 * MxChat Migration Tool - Admin Interface
 *
 * @package MxChat_Migration_Tool
 */

if (!defined('ABSPATH')) {
    exit;
}

class MxChat_Migration_Admin {

    /**
     * Constructor
     */
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
    }

    /**
     * Add submenu page under MxChat
     */
    public function add_admin_menu() {
        add_submenu_page(
            'mxchat-max',
            __('Database Migration', 'mxchat-migration-tool'),
            __('Migration Tool', 'mxchat-migration-tool'),
            'manage_options',
            'mxchat-migration',
            array($this, 'render_admin_page')
        );
    }

    /**
     * Enqueue admin scripts and styles
     */
    public function enqueue_admin_scripts($hook) {
        if ($hook !== 'mxchat_page_mxchat-migration') {
            return;
        }

        wp_enqueue_style(
            'mxchat-migration-admin',
            MXCHAT_MIGRATION_URL . 'assets/css/admin.css',
            array(),
            MXCHAT_MIGRATION_VERSION
        );

        wp_enqueue_script(
            'mxchat-migration-admin',
            MXCHAT_MIGRATION_URL . 'assets/js/admin.js',
            array('jquery'),
            MXCHAT_MIGRATION_VERSION,
            true
        );

        wp_localize_script('mxchat-migration-admin', 'mxchatMigration', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('mxchat_migration_nonce'),
            'strings' => array(
                'confirm_migration' => __('Are you sure you want to start the migration? This process cannot be undone.', 'mxchat-migration-tool'),
                'migration_in_progress' => __('Migration in progress...', 'mxchat-migration-tool'),
                'migration_complete' => __('Migration completed successfully!', 'mxchat-migration-tool'),
                'migration_failed' => __('Migration failed. Please check the error log.', 'mxchat-migration-tool'),
            )
        ));
    }

    /**
     * Render admin page
     */
    public function render_admin_page() {
        // Get current MxChat settings
        $options = get_option('mxchat_options');
        $current_embedding_model = isset($options['embedding_model']) ? $options['embedding_model'] : 'text-embedding-ada-002';

        // Check if Pinecone is enabled
        $pinecone_options = get_option('mxchat_pinecone_addon_options', array());
        $current_database = (isset($pinecone_options['mxchat_use_pinecone']) && $pinecone_options['mxchat_use_pinecone'] === '1') ? 'pinecone' : 'wordpress';

        error_log('=== MXCHAT MIGRATION ADMIN PAGE ===');
        error_log('Current database: ' . $current_database);
        error_log('use_pinecone setting: ' . ($pinecone_options['mxchat_use_pinecone'] ?? 'NOT SET'));

        // Get database statistics
        global $wpdb;

        // Get knowledge count based on current database type
        if ($current_database === 'pinecone') {
            // Get count from Pinecone
            error_log('Getting knowledge count from Pinecone...');
            $knowledge_count = $this->get_pinecone_knowledge_count($pinecone_options);
            error_log('Knowledge count from Pinecone: ' . $knowledge_count);
        } else {
            // Get count from WordPress database
            error_log('Getting knowledge count from WordPress database...');
            $knowledge_count = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}mxchat_system_prompt_content");
            error_log('Knowledge count from WordPress: ' . $knowledge_count);
        }

        $actions_count = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}mxchat_intents");

        ?>
        <div class="wrap mxchat-migration-wrap">
            <!-- Hero Section -->
            <div class="mxchat-migration-hero">
                <h1>
                    <span class="mxchat-gradient-text">MxChat</span> Migration Tool
                </h1>
                <p><?php esc_html_e('Seamlessly migrate your knowledge base and actions between databases and embedding models', 'mxchat-migration-tool'); ?></p>
            </div>

            <!-- Main Content -->
            <div class="mxchat-migration-content">
                <div class="mxchat-migration-intro">
                    <p><?php esc_html_e('This tool helps you migrate your knowledge base and actions when switching embedding models or vector databases.', 'mxchat-migration-tool'); ?></p>
                    <p><strong><?php esc_html_e('Important:', 'mxchat-migration-tool'); ?></strong> <?php esc_html_e('The migration process will re-generate embeddings for all your content using the new model. This may take some time depending on the amount of data.', 'mxchat-migration-tool'); ?></p>
                </div>

            <!-- Current Configuration -->
            <div class="mxchat-migration-card mxchat-current-config">
                <h2><?php esc_html_e('Current Configuration', 'mxchat-migration-tool'); ?></h2>
                <table class="form-table">
                    <tr>
                        <th><?php esc_html_e('Vector Database:', 'mxchat-migration-tool'); ?></th>
                        <td>
                            <strong><?php echo esc_html(ucfirst($current_database)); ?></strong>
                            <?php if ($current_database === 'pinecone' && isset($pinecone_options['mxchat_pinecone_host'])): ?>
                                <br><span class="description"><?php echo esc_html($pinecone_options['mxchat_pinecone_host']); ?></span>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <tr>
                        <th><?php esc_html_e('Embedding Model:', 'mxchat-migration-tool'); ?></th>
                        <td><strong><?php echo esc_html($current_embedding_model); ?></strong></td>
                    </tr>
                    <tr>
                        <th><?php esc_html_e('Knowledge Base Entries:', 'mxchat-migration-tool'); ?></th>
                        <td><strong><?php echo number_format_i18n($knowledge_count); ?></strong></td>
                    </tr>
                    <tr>
                        <th><?php esc_html_e('Actions/Intents:', 'mxchat-migration-tool'); ?></th>
                        <td><strong><?php echo number_format_i18n($actions_count); ?></strong></td>
                    </tr>
                </table>
            </div>

            <!-- Migration Form -->
            <form id="mxchat-migration-form" class="mxchat-migration-card">
                <h2><?php esc_html_e('Migration Configuration', 'mxchat-migration-tool'); ?></h2>

                <!-- Migration Type -->
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="migration_type"><?php esc_html_e('What to Migrate', 'mxchat-migration-tool'); ?></label>
                        </th>
                        <td>
                            <select name="migration_type" id="migration_type" class="regular-text">
                                <option value="all"><?php esc_html_e('Knowledge Base + Actions', 'mxchat-migration-tool'); ?></option>
                                <option value="knowledge"><?php esc_html_e('Knowledge Base Only', 'mxchat-migration-tool'); ?></option>
                                <option value="actions"><?php esc_html_e('Actions Only', 'mxchat-migration-tool'); ?></option>
                            </select>
                            <p class="description"><?php esc_html_e('Select what you want to migrate.', 'mxchat-migration-tool'); ?></p>
                        </td>
                    </tr>

                    <!-- Target Database -->
                    <tr>
                        <th scope="row">
                            <label for="target_database"><?php esc_html_e('Target Database', 'mxchat-migration-tool'); ?></label>
                        </th>
                        <td>
                            <select name="target_database" id="target_database" class="regular-text">
                                <option value="wordpress" <?php selected($current_database, 'pinecone'); ?>><?php esc_html_e('WordPress Database', 'mxchat-migration-tool'); ?></option>
                                <option value="pinecone" <?php selected($current_database, 'wordpress'); ?>><?php esc_html_e('Pinecone Vector Database', 'mxchat-migration-tool'); ?></option>
                            </select>
                            <p class="description"><?php esc_html_e('Select the target database for migration.', 'mxchat-migration-tool'); ?></p>
                        </td>
                    </tr>

                    <!-- Pinecone Configuration (shown when Pinecone is selected) -->
                    <tr class="pinecone-config" style="display: none;">
                        <th scope="row">
                            <label for="pinecone_api_key"><?php esc_html_e('Pinecone API Key', 'mxchat-migration-tool'); ?></label>
                        </th>
                        <td>
                            <input type="password" name="pinecone_api_key" id="pinecone_api_key" class="regular-text" value="">
                            <p class="description"><?php esc_html_e('Your Pinecone API key.', 'mxchat-migration-tool'); ?></p>
                        </td>
                    </tr>

                    <tr class="pinecone-config" style="display: none;">
                        <th scope="row">
                            <label for="pinecone_host"><?php esc_html_e('Pinecone Host', 'mxchat-migration-tool'); ?></label>
                        </th>
                        <td>
                            <input type="text" name="pinecone_host" id="pinecone_host" class="regular-text" placeholder="your-index-abc123.svc.pinecone.io" value="">
                            <p class="description"><?php esc_html_e('Your Pinecone index host (with or without https://).', 'mxchat-migration-tool'); ?></p>
                        </td>
                    </tr>

                    <!-- REMOVED: Namespace field - Core plugin doesn't use namespaces, always uses __default__ -->

                    <!-- Target Embedding Model -->
                    <tr>
                        <th scope="row">
                            <label for="target_embedding_model"><?php esc_html_e('Target Embedding Model', 'mxchat-migration-tool'); ?></label>
                        </th>
                        <td>
                            <select name="target_embedding_model" id="target_embedding_model" class="regular-text">
                                <optgroup label="OpenAI">
                                    <option value="text-embedding-ada-002"><?php esc_html_e('text-embedding-ada-002 (1536 dimensions)', 'mxchat-migration-tool'); ?></option>
                                    <option value="text-embedding-3-small"><?php esc_html_e('text-embedding-3-small (1536 dimensions)', 'mxchat-migration-tool'); ?></option>
                                    <option value="text-embedding-3-large"><?php esc_html_e('text-embedding-3-large (3072 dimensions)', 'mxchat-migration-tool'); ?></option>
                                </optgroup>
                                <optgroup label="Voyage AI">
                                    <option value="voyage-2"><?php esc_html_e('voyage-2 (1024 dimensions)', 'mxchat-migration-tool'); ?></option>
                                    <option value="voyage-large-2"><?php esc_html_e('voyage-large-2 (1536 dimensions)', 'mxchat-migration-tool'); ?></option>
                                    <option value="voyage-3-large"><?php esc_html_e('voyage-3-large (2048 dimensions)', 'mxchat-migration-tool'); ?></option>
                                </optgroup>
                                <optgroup label="Google Gemini">
                                    <option value="gemini-embedding-001"><?php esc_html_e('gemini-embedding-001 (1536 dimensions)', 'mxchat-migration-tool'); ?></option>
                                </optgroup>
                            </select>
                            <p class="description"><?php esc_html_e('Select the embedding model to use for the migration.', 'mxchat-migration-tool'); ?></p>
                        </td>
                    </tr>

                    <!-- API Keys (based on selected model) -->
                    <tr class="api-key-row openai-key">
                        <th scope="row">
                            <label for="openai_api_key"><?php esc_html_e('OpenAI API Key', 'mxchat-migration-tool'); ?></label>
                        </th>
                        <td>
                            <input type="password" name="openai_api_key" id="openai_api_key" class="regular-text" value="<?php echo esc_attr($options['api_key'] ?? ''); ?>">
                            <p class="description"><?php esc_html_e('Your OpenAI API key (from main settings).', 'mxchat-migration-tool'); ?></p>
                        </td>
                    </tr>

                    <tr class="api-key-row voyage-key" style="display: none;">
                        <th scope="row">
                            <label for="voyage_api_key"><?php esc_html_e('Voyage AI API Key', 'mxchat-migration-tool'); ?></label>
                        </th>
                        <td>
                            <input type="password" name="voyage_api_key" id="voyage_api_key" class="regular-text" value="<?php echo esc_attr($options['voyage_api_key'] ?? ''); ?>">
                            <p class="description"><?php esc_html_e('Your Voyage AI API key (from main settings).', 'mxchat-migration-tool'); ?></p>
                        </td>
                    </tr>

                    <tr class="api-key-row gemini-key" style="display: none;">
                        <th scope="row">
                            <label for="gemini_api_key"><?php esc_html_e('Google Gemini API Key', 'mxchat-migration-tool'); ?></label>
                        </th>
                        <td>
                            <input type="password" name="gemini_api_key" id="gemini_api_key" class="regular-text" value="<?php echo esc_attr($options['gemini_api_key'] ?? ''); ?>">
                            <p class="description"><?php esc_html_e('Your Google Gemini API key (from main settings).', 'mxchat-migration-tool'); ?></p>
                        </td>
                    </tr>

                    <!-- Batch Size -->
                    <tr>
                        <th scope="row">
                            <label for="batch_size"><?php esc_html_e('Batch Size', 'mxchat-migration-tool'); ?></label>
                        </th>
                        <td>
                            <input type="number" name="batch_size" id="batch_size" class="small-text" value="10" min="1" max="50">
                            <p class="description"><?php esc_html_e('Number of items to process per batch (1-50). Lower values are more reliable but slower.', 'mxchat-migration-tool'); ?></p>
                        </td>
                    </tr>
                </table>

                <p class="submit">
                    <button type="submit" class="button button-primary button-large" id="start-migration-btn">
                        <?php esc_html_e('Start Migration', 'mxchat-migration-tool'); ?>
                    </button>
                </p>
            </form>

            <!-- Progress Section -->
            <div id="migration-progress" class="mxchat-migration-card" style="display: none;">
                <h2><?php esc_html_e('Migration Progress', 'mxchat-migration-tool'); ?></h2>

                <div class="migration-status">
                    <div class="status-item">
                        <span class="status-label"><?php esc_html_e('Status:', 'mxchat-migration-tool'); ?></span>
                        <span class="status-value" id="migration-status-text"><?php esc_html_e('Initializing...', 'mxchat-migration-tool'); ?></span>
                    </div>
                    <div class="status-item">
                        <span class="status-label"><?php esc_html_e('Progress:', 'mxchat-migration-tool'); ?></span>
                        <span class="status-value" id="migration-progress-text">0 / 0</span>
                    </div>
                </div>

                <div class="progress-bar-wrapper">
                    <div class="progress-bar">
                        <div class="progress-bar-fill" id="progress-bar-fill" style="width: 0%;"></div>
                    </div>
                    <div class="progress-percentage" id="progress-percentage">0%</div>
                </div>

                <div class="migration-details">
                    <h3><?php esc_html_e('Details', 'mxchat-migration-tool'); ?></h3>
                    <div id="migration-log" class="migration-log"></div>
                </div>

                <div class="migration-actions">
                    <button type="button" class="button button-secondary" id="cancel-migration-btn" style="display: none;">
                        <?php esc_html_e('Cancel Migration', 'mxchat-migration-tool'); ?>
                    </button>
                    <button type="button" class="button button-primary" id="close-progress-btn" style="display: none;">
                        <?php esc_html_e('Close', 'mxchat-migration-tool'); ?>
                    </button>
                </div>
            </div>

            <!-- Migration History -->
            <div class="mxchat-migration-card mxchat-migration-history">
                <h2><?php esc_html_e('Migration History', 'mxchat-migration-tool'); ?></h2>
                <?php $this->render_migration_history(); ?>
            </div>
            </div><!-- .mxchat-migration-content -->
        </div><!-- .mxchat-migration-wrap -->
        <?php
    }

    /**
     * Render migration history
     */
    private function render_migration_history() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'mxchat_migration_logs';

        $migrations = $wpdb->get_results(
            "SELECT * FROM $table_name ORDER BY started_at DESC LIMIT 3",
            ARRAY_A
        );

        if (empty($migrations)) {
            echo '<p>' . esc_html__('No migration history found.', 'mxchat-migration-tool') . '</p>';
            return;
        }

        ?>
        <table class="wp-list-table widefat fixed striped">
            <thead>
                <tr>
                    <th><?php esc_html_e('Date', 'mxchat-migration-tool'); ?></th>
                    <th><?php esc_html_e('Type', 'mxchat-migration-tool'); ?></th>
                    <th><?php esc_html_e('Source', 'mxchat-migration-tool'); ?></th>
                    <th><?php esc_html_e('Target', 'mxchat-migration-tool'); ?></th>
                    <th><?php esc_html_e('Items', 'mxchat-migration-tool'); ?></th>
                    <th><?php esc_html_e('Status', 'mxchat-migration-tool'); ?></th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($migrations as $migration): ?>
                    <tr>
                        <td><?php echo esc_html(mysql2date(get_option('date_format') . ' ' . get_option('time_format'), $migration['started_at'])); ?></td>
                        <td><?php echo esc_html(ucfirst($migration['migration_type'])); ?></td>
                        <td>
                            <?php echo esc_html(ucfirst($migration['source_database'])); ?>
                            <br><small><?php echo esc_html($migration['source_model']); ?></small>
                        </td>
                        <td>
                            <?php echo esc_html(ucfirst($migration['target_database'])); ?>
                            <br><small><?php echo esc_html($migration['target_model']); ?></small>
                        </td>
                        <td>
                            <?php
                            echo esc_html(sprintf(
                                '%d / %d',
                                $migration['processed_items'],
                                $migration['total_items']
                            ));
                            ?>
                            <?php if ($migration['failed_items'] > 0): ?>
                                <br><small class="error"><?php echo esc_html(sprintf(__('%d failed', 'mxchat-migration-tool'), $migration['failed_items'])); ?></small>
                            <?php endif; ?>
                        </td>
                        <td>
                            <span class="migration-status-badge status-<?php echo esc_attr($migration['status']); ?>">
                                <?php echo esc_html(ucfirst($migration['status'])); ?>
                            </span>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        <?php
    }

    /**
     * Get knowledge entry count from Pinecone
     */
    private function get_pinecone_knowledge_count($pinecone_options) {
        error_log('=== MXCHAT MIGRATION: Getting Pinecone knowledge count ===');
        error_log('Pinecone options: ' . print_r($pinecone_options, true));

        $api_key = $pinecone_options['mxchat_pinecone_api_key'] ?? '';
        $host = $pinecone_options['mxchat_pinecone_host'] ?? '';
        $namespace = $pinecone_options['mxchat_pinecone_namespace'] ?? '';

        error_log('API Key exists: ' . (!empty($api_key) ? 'yes' : 'no'));
        error_log('Host: ' . $host);
        error_log('Namespace: ' . $namespace);

        if (empty($api_key) || empty($host)) {
            error_log('MXCHAT MIGRATION: Missing API key or host, returning 0');
            return 0;
        }

        try {
            // Remove https:// if it's already in the host
            $host = str_replace(array('https://', 'http://'), '', $host);
            $stats_url = "https://{$host}/describe_index_stats";

            error_log('Stats URL: ' . $stats_url);

            $request_data = array();

            // Add namespace if provided - NOTE: Empty array if no namespace
            if (!empty($namespace)) {
                $request_data['filter'] = array();
                error_log('Using namespace: ' . $namespace);
            }

            $body_data = !empty($request_data) ? json_encode($request_data) : '{}';
            error_log('Request body: ' . $body_data);

            $response = wp_remote_post($stats_url, array(
                'headers' => array(
                    'Api-Key' => $api_key,
                    'Content-Type' => 'application/json'
                ),
                'body' => $body_data,
                'timeout' => 15
            ));

            error_log('Response code: ' . wp_remote_retrieve_response_code($response));

            if (is_wp_error($response)) {
                error_log('MXCHAT MIGRATION: WP Error: ' . $response->get_error_message());
                return 0;
            }

            $response_code = wp_remote_retrieve_response_code($response);
            $body = wp_remote_retrieve_body($response);

            error_log('Response body: ' . $body);

            if ($response_code === 200) {
                $stats_data = json_decode($body, true);
                error_log('Decoded stats data: ' . print_r($stats_data, true));

                // Check different possible locations for the count
                $total_count = 0;

                if (isset($stats_data['totalVectorCount'])) {
                    $total_count = $stats_data['totalVectorCount'];
                } elseif (isset($stats_data['namespaces']) && !empty($namespace)) {
                    // Check namespace-specific count
                    if (isset($stats_data['namespaces'][$namespace]['vectorCount'])) {
                        $total_count = $stats_data['namespaces'][$namespace]['vectorCount'];
                    }
                } elseif (isset($stats_data['namespaces'])) {
                    // Sum up all namespaces
                    foreach ($stats_data['namespaces'] as $ns => $ns_data) {
                        if (isset($ns_data['vectorCount'])) {
                            $total_count += $ns_data['vectorCount'];
                        }
                    }
                }

                error_log('Final total count: ' . $total_count);
                return intval($total_count);
            } else {
                error_log('MXCHAT MIGRATION: Non-200 response code: ' . $response_code);
                return 0;
            }

        } catch (Exception $e) {
            error_log('MXCHAT MIGRATION: Exception: ' . $e->getMessage());
            return 0;
        }
    }
}
