<?php
/**
 * Migration AJAX Handler
 *
 * Handles AJAX requests for migration operations
 *
 * @package MxChat_Migration_Tool
 */

if (!defined('ABSPATH')) {
    exit;
}

class MxChat_Migration_Handler {

    /**
     * Constructor
     */
    public function __construct() {
        add_action('wp_ajax_mxchat_start_migration', array($this, 'start_migration'));
        add_action('wp_ajax_mxchat_process_migration_batch', array($this, 'process_migration_batch'));
        add_action('wp_ajax_mxchat_get_migration_status', array($this, 'get_migration_status'));
        add_action('wp_ajax_mxchat_cancel_migration', array($this, 'cancel_migration'));
        add_action('wp_ajax_mxchat_finalize_migration', array($this, 'finalize_migration'));
    }

    /**
     * Start migration
     */
    public function start_migration() {
        check_ajax_referer('mxchat_migration_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Insufficient permissions'));
        }

        $migration_type = isset($_POST['migration_type']) ? sanitize_text_field($_POST['migration_type']) : 'all';
        $target_database = isset($_POST['target_database']) ? sanitize_text_field($_POST['target_database']) : '';
        $target_model = isset($_POST['target_embedding_model']) ? sanitize_text_field($_POST['target_embedding_model']) : '';
        $batch_size = isset($_POST['batch_size']) ? intval($_POST['batch_size']) : 10;

        // Validate inputs
        if (empty($target_database) || empty($target_model)) {
            wp_send_json_error(array('message' => 'Missing required parameters'));
        }

        // Get API key based on model
        $api_key = '';
        if (strpos($target_model, 'voyage') === 0) {
            $api_key = isset($_POST['voyage_api_key']) ? sanitize_text_field($_POST['voyage_api_key']) : '';
        } elseif (strpos($target_model, 'gemini-embedding') === 0) {
            $api_key = isset($_POST['gemini_api_key']) ? sanitize_text_field($_POST['gemini_api_key']) : '';
        } else {
            $api_key = isset($_POST['openai_api_key']) ? sanitize_text_field($_POST['openai_api_key']) : '';
        }

        if (empty($api_key)) {
            wp_send_json_error(array('message' => 'API key is required'));
        }

        // Get current configuration
        $options = get_option('mxchat_options');
        $pinecone_options = get_option('mxchat_pinecone_addon_options', array());

        $source_database = (isset($pinecone_options['mxchat_use_pinecone']) && $pinecone_options['mxchat_use_pinecone'] === '1') ? 'pinecone' : 'wordpress';
        $source_model = isset($options['embedding_model']) ? $options['embedding_model'] : 'text-embedding-ada-002';

        // Generate unique migration ID
        $migration_id = uniqid('migration_', true);

        // Prepare Pinecone config if needed
        $pinecone_config = array();
        if ($target_database === 'pinecone') {
            // Normalize Pinecone host - strip protocol if present
            $raw_host = isset($_POST['pinecone_host']) ? trim($_POST['pinecone_host']) : '';
            $normalized_host = str_replace(['https://', 'http://'], '', $raw_host);

            $pinecone_config = array(
                'api_key' => isset($_POST['pinecone_api_key']) ? sanitize_text_field($_POST['pinecone_api_key']) : '',
                'host' => $normalized_host,
                'namespace' => '' // Always use default namespace (empty = __default__)
            );

            if (empty($pinecone_config['api_key']) || empty($pinecone_config['host'])) {
                wp_send_json_error(array('message' => 'Pinecone API key and host are required'));
            }
        }

        // Calculate total items
        $total_items = 0;

        if ($migration_type === 'all' || $migration_type === 'knowledge') {
            $knowledge_migrator = new MxChat_Knowledge_Migrator(array(
                'source_database' => $source_database,
                'target_database' => $target_database,
                'source_model' => $source_model,
                'target_model' => $target_model,
                'api_key' => $api_key,
                'pinecone_config' => $pinecone_config,
                'migration_id' => $migration_id
            ));
            $total_items += $knowledge_migrator->get_total_items();
        }

        if ($migration_type === 'all' || $migration_type === 'actions') {
            $actions_migrator = new MxChat_Actions_Migrator(array(
                'source_model' => $source_model,
                'target_model' => $target_model,
                'api_key' => $api_key,
                'migration_id' => $migration_id
            ));
            $total_items += $actions_migrator->get_total_items();
        }

        // Save migration log to database
        global $wpdb;
        $wpdb->insert(
            $wpdb->prefix . 'mxchat_migration_logs',
            array(
                'migration_id' => $migration_id,
                'migration_type' => $migration_type,
                'source_database' => $source_database,
                'target_database' => $target_database,
                'source_model' => $source_model,
                'target_model' => $target_model,
                'total_items' => $total_items,
                'processed_items' => 0,
                'failed_items' => 0,
                'status' => 'in_progress',
                'started_at' => current_time('mysql')
            ),
            array('%s', '%s', '%s', '%s', '%s', '%s', '%d', '%d', '%d', '%s', '%s')
        );

        // Store migration configuration in transient
        set_transient('mxchat_migration_' . $migration_id, array(
            'migration_type' => $migration_type,
            'source_database' => $source_database,
            'target_database' => $target_database,
            'source_model' => $source_model,
            'target_model' => $target_model,
            'api_key' => $api_key,
            'pinecone_config' => $pinecone_config,
            'batch_size' => $batch_size,
            'total_items' => $total_items,
            'processed_items' => 0,
            'failed_items' => 0,
            'current_offset' => 0,
            'current_stage' => 'knowledge', // knowledge or actions
            'target_cleaned' => false // Track if target database has been cleaned
        ), 3600); // 1 hour expiration

        wp_send_json_success(array(
            'migration_id' => $migration_id,
            'total_items' => $total_items,
            'message' => 'Migration started successfully'
        ));
    }

    /**
     * Process migration batch
     */
    public function process_migration_batch() {
        check_ajax_referer('mxchat_migration_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Insufficient permissions'));
        }

        $migration_id = isset($_POST['migration_id']) ? sanitize_text_field($_POST['migration_id']) : '';

        if (empty($migration_id)) {
            wp_send_json_error(array('message' => 'Migration ID is required'));
        }

        // Get migration configuration
        $config = get_transient('mxchat_migration_' . $migration_id);

        if (!$config) {
            wp_send_json_error(array('message' => 'Migration configuration not found or expired'));
        }

        $result = array();
        $continue = true;
        $cleanup_message = '';

        // Clean target database on first batch (if not same database migration)
        if (!$config['target_cleaned']) {
            $should_clean = true;

            // Don't clean if migrating within same database (just changing model)
            if ($config['source_database'] === $config['target_database']) {
                if ($config['source_database'] === 'wordpress') {
                    $should_clean = false; // WordPress -> WordPress: UPDATE existing entries
                }
            }

            if ($should_clean) {
                $this->clean_target_database($config);
                $cleanup_message = 'Cleaned target database to ensure fresh migration';
            }

            $config['target_cleaned'] = true;
            set_transient('mxchat_migration_' . $migration_id, $config, 3600);
        }

        // Process based on current stage
        if ($config['current_stage'] === 'knowledge' &&
            ($config['migration_type'] === 'all' || $config['migration_type'] === 'knowledge')) {

            $knowledge_migrator = new MxChat_Knowledge_Migrator(array(
                'source_database' => $config['source_database'],
                'target_database' => $config['target_database'],
                'source_model' => $config['source_model'],
                'target_model' => $config['target_model'],
                'api_key' => $config['api_key'],
                'pinecone_config' => $config['pinecone_config'],
                'migration_id' => $migration_id
            ));

            $result = $knowledge_migrator->migrate_batch($config['current_offset'], $config['batch_size']);

            $config['processed_items'] += $result['processed'];
            $config['failed_items'] += $result['failed'];
            $config['current_offset'] += $config['batch_size'];

            // Check if knowledge migration is complete
            if ($result['processed'] === 0) {
                // Move to actions stage
                $config['current_stage'] = 'actions';
                $config['current_offset'] = 0;
            }

        } elseif ($config['current_stage'] === 'actions' &&
                  ($config['migration_type'] === 'all' || $config['migration_type'] === 'actions')) {

            $actions_migrator = new MxChat_Actions_Migrator(array(
                'source_model' => $config['source_model'],
                'target_model' => $config['target_model'],
                'api_key' => $config['api_key'],
                'migration_id' => $migration_id
            ));

            $result = $actions_migrator->migrate_batch($config['current_offset'], $config['batch_size']);

            $config['processed_items'] += $result['processed'];
            $config['failed_items'] += $result['failed'];
            $config['current_offset'] += $config['batch_size'];

            // Check if actions migration is complete
            if ($result['processed'] === 0) {
                $continue = false;
            }

        } else {
            // Migration type is only actions and we're done with knowledge
            if ($config['migration_type'] === 'actions' && $config['current_stage'] === 'knowledge') {
                $config['current_stage'] = 'actions';
                $config['current_offset'] = 0;
            } else {
                $continue = false;
            }
        }

        // Update migration configuration
        set_transient('mxchat_migration_' . $migration_id, $config, 3600);

        // Update database log
        global $wpdb;
        $wpdb->update(
            $wpdb->prefix . 'mxchat_migration_logs',
            array(
                'processed_items' => $config['processed_items'],
                'failed_items' => $config['failed_items'],
                'status' => $continue ? 'in_progress' : 'completed',
                'completed_at' => $continue ? null : current_time('mysql')
            ),
            array('migration_id' => $migration_id),
            array('%d', '%d', '%s', '%s'),
            array('%s')
        );

        wp_send_json_success(array(
            'processed' => $result['processed'],
            'failed' => $result['failed'],
            'total_processed' => $config['processed_items'],
            'total_failed' => $config['failed_items'],
            'total_items' => $config['total_items'],
            'continue' => $continue,
            'current_stage' => $config['current_stage'],
            'message' => $result['message'],
            'errors' => isset($result['errors']) ? $result['errors'] : array(),
            'cleanup_message' => $cleanup_message
        ));
    }

    /**
     * Get migration status
     */
    public function get_migration_status() {
        check_ajax_referer('mxchat_migration_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Insufficient permissions'));
        }

        $migration_id = isset($_POST['migration_id']) ? sanitize_text_field($_POST['migration_id']) : '';

        if (empty($migration_id)) {
            wp_send_json_error(array('message' => 'Migration ID is required'));
        }

        $config = get_transient('mxchat_migration_' . $migration_id);

        if (!$config) {
            wp_send_json_error(array('message' => 'Migration not found'));
        }

        wp_send_json_success(array(
            'total_items' => $config['total_items'],
            'processed_items' => $config['processed_items'],
            'failed_items' => $config['failed_items'],
            'current_stage' => $config['current_stage']
        ));
    }

    /**
     * Cancel migration
     */
    public function cancel_migration() {
        check_ajax_referer('mxchat_migration_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Insufficient permissions'));
        }

        $migration_id = isset($_POST['migration_id']) ? sanitize_text_field($_POST['migration_id']) : '';

        if (empty($migration_id)) {
            wp_send_json_error(array('message' => 'Migration ID is required'));
        }

        // Delete transient
        delete_transient('mxchat_migration_' . $migration_id);

        // Update database log
        global $wpdb;
        $wpdb->update(
            $wpdb->prefix . 'mxchat_migration_logs',
            array(
                'status' => 'cancelled',
                'completed_at' => current_time('mysql')
            ),
            array('migration_id' => $migration_id),
            array('%s', '%s'),
            array('%s')
        );

        wp_send_json_success(array('message' => 'Migration cancelled'));
    }

    /**
     * Finalize migration - update main plugin settings
     */
    public function finalize_migration() {
        check_ajax_referer('mxchat_migration_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Insufficient permissions'));
        }

        $migration_id = isset($_POST['migration_id']) ? sanitize_text_field($_POST['migration_id']) : '';
        $update_embedding_model = isset($_POST['update_embedding_model']) ? filter_var($_POST['update_embedding_model'], FILTER_VALIDATE_BOOLEAN) : false;
        $update_database = isset($_POST['update_database']) ? filter_var($_POST['update_database'], FILTER_VALIDATE_BOOLEAN) : false;

        if (empty($migration_id)) {
            wp_send_json_error(array('message' => 'Migration ID is required'));
        }

        $config = get_transient('mxchat_migration_' . $migration_id);

        if (!$config) {
            wp_send_json_error(array('message' => 'Migration not found'));
        }

        $settings_updated = array();

        // Update embedding model if requested
        if ($update_embedding_model) {
            $options = get_option('mxchat_options');
            $options['embedding_model'] = $config['target_model'];
            update_option('mxchat_options', $options);
            $settings_updated[] = 'embedding_model';
        }

        // Update database settings if requested
        if ($update_database) {
            global $wpdb;

            // Get current Pinecone options directly from database (bypassing WordPress filters)
            $current_options_raw = $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT option_value FROM {$wpdb->options} WHERE option_name = %s",
                    'mxchat_pinecone_addon_options'
                )
            );

            // Initialize current options
            if ($current_options_raw === null) {
                $current_options = array(
                    'mxchat_use_pinecone' => '0',
                    'mxchat_pinecone_api_key' => '',
                    'mxchat_pinecone_host' => '',
                    'mxchat_pinecone_namespace' => ''
                );
            } else {
                $current_options = maybe_unserialize($current_options_raw);
                if (!is_array($current_options)) {
                    $current_options = array(
                        'mxchat_use_pinecone' => '0',
                        'mxchat_pinecone_api_key' => '',
                        'mxchat_pinecone_host' => '',
                        'mxchat_pinecone_namespace' => ''
                    );
                }
            }

            if ($config['target_database'] === 'pinecone') {
                // Enable Pinecone and update connection details
                $current_options['mxchat_use_pinecone'] = '1';
                $current_options['mxchat_pinecone_api_key'] = $config['pinecone_config']['api_key'];
                $current_options['mxchat_pinecone_host'] = str_replace(['https://', 'http://'], '', $config['pinecone_config']['host']);
                $current_options['mxchat_pinecone_namespace'] = ''; // Always empty = use __default__ namespace
                $settings_updated[] = 'database_pinecone';
            } elseif ($config['target_database'] === 'wordpress') {
                // Disable Pinecone
                $current_options['mxchat_use_pinecone'] = '0';
                $settings_updated[] = 'database_wordpress';
            }

            // Save directly to database (same method as main plugin)
            $serialized_options = maybe_serialize($current_options);

            $option_exists = $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT COUNT(*) FROM {$wpdb->options} WHERE option_name = %s",
                    'mxchat_pinecone_addon_options'
                )
            );

            if ($option_exists > 0) {
                // Update existing option
                $wpdb->update(
                    $wpdb->options,
                    array('option_value' => $serialized_options),
                    array('option_name' => 'mxchat_pinecone_addon_options'),
                    array('%s'),
                    array('%s')
                );
            } else {
                // Insert new option
                $wpdb->insert(
                    $wpdb->options,
                    array(
                        'option_name' => 'mxchat_pinecone_addon_options',
                        'option_value' => $serialized_options,
                        'autoload' => 'yes'
                    ),
                    array('%s', '%s', '%s')
                );
            }

            error_log('MIGRATION: Updated Pinecone settings - use_pinecone: ' . $current_options['mxchat_use_pinecone']);
        }

        // Clean up transient
        delete_transient('mxchat_migration_' . $migration_id);

        wp_send_json_success(array(
            'message' => 'Migration finalized successfully',
            'settings_updated' => $settings_updated,
            'update_embedding_model' => $update_embedding_model,
            'update_database' => $update_database
        ));
    }

    /**
     * Clean target database before migration
     */
    private function clean_target_database($config) {
        global $wpdb;

        error_log('MIGRATION: Cleaning target database - ' . $config['target_database']);

        if ($config['target_database'] === 'wordpress') {
            // Clear WordPress knowledge base table
            $deleted = $wpdb->query("TRUNCATE TABLE {$wpdb->prefix}mxchat_system_prompt_content");
            error_log("MIGRATION: Cleared WordPress knowledge base - deleted rows: " . ($deleted !== false ? 'success' : 'failed'));

        } elseif ($config['target_database'] === 'pinecone') {
            // Clear Pinecone index (always __default__ namespace)
            $api_key = isset($config['pinecone_config']['api_key']) ? $config['pinecone_config']['api_key'] : '';
            $host = isset($config['pinecone_config']['host']) ? $config['pinecone_config']['host'] : '';

            if (!empty($api_key) && !empty($host)) {
                // Normalize host
                $host = str_replace(array('https://', 'http://'), '', $host);
                $url = 'https://' . rtrim($host, '/') . '/vectors/delete';

                // Don't include namespace parameter - this will delete from __default__ namespace
                $body = array(
                    'deleteAll' => true
                );

                $response = wp_remote_post($url, array(
                    'headers' => array(
                        'Api-Key' => $api_key,
                        'Content-Type' => 'application/json'
                    ),
                    'body' => wp_json_encode($body),
                    'method' => 'POST',
                    'timeout' => 30
                ));

                if (is_wp_error($response)) {
                    error_log('MIGRATION: Failed to clear Pinecone: ' . $response->get_error_message());
                } else {
                    $status_code = wp_remote_retrieve_response_code($response);
                    error_log('MIGRATION: Cleared Pinecone __default__ namespace - status: ' . $status_code);
                }
            }
        }
    }
}
