<?php
/**
 * Plugin Name: MxChat Migration Tool
 * Plugin URI: https://mxchat.ai
 * Description: Migrate your knowledge base and actions when switching between embedding models or vector databases
 * Version: 1.2.1
 * Author: MxChat
 * Author URI: https://mxchat.ai
 * Text Domain: mxchat-migration-tool
 * Requires at least: 5.0
 * Requires PHP: 7.4
 */

// If this file is called directly, abort.
if (!defined('WPINC')) {
    die;
}

define('MXCHAT_MIGRATION_VERSION', '1.2.1');
define('MXCHAT_MIGRATION_PATH', plugin_dir_path(__FILE__));
define('MXCHAT_MIGRATION_URL', plugin_dir_url(__FILE__));

// Initialize update checker
require_once dirname(__FILE__) . '/update-checker.php';
PluginUpdateCheckerMigration::init([
    'user_id' => 'wp_4_SGKQ2sGzkIGB',
    'plugin_slug' => 'mxchat-migration-tool'
]);

/**
 * Check if MxChat Basic is active
 */
function mxchat_migration_check_dependencies() {
    if (!class_exists('MxChat_Integrator')) {
        add_action('admin_notices', 'mxchat_migration_dependency_notice');
        deactivate_plugins(plugin_basename(__FILE__));
        return false;
    }
    return true;
}

/**
 * Display dependency notice
 */
function mxchat_migration_dependency_notice() {
    ?>
    <div class="notice notice-error">
        <p><?php esc_html_e('MxChat Migration Tool requires MxChat Basic plugin to be installed and activated.', 'mxchat-migration-tool'); ?></p>
    </div>
    <?php
}

/**
 * Check if MxChat Pro license is active
 */
function mxchat_migration_is_license_active() {
    // Check network-wide license for multisite
    if (is_multisite()) {
        $network_license_status = get_site_option('mxchat_license_status');
        if ($network_license_status === 'active') {
            return true;
        }
    }

    // Check site-specific license
    $license_status = get_option('mxchat_license_status', 'inactive');
    return $license_status === 'active';
}

/**
 * Display Pro requirement notice
 */
function mxchat_migration_show_pro_requirement_notice() {
    $class = 'notice notice-warning';
    $message = sprintf(
        __('MxChat Migration Tool requires MxChat Pro to be activated. Please <a href="%s">activate your pro license</a> to use this addon.', 'mxchat-migration-tool'),
        admin_url('admin.php?page=mxchat-activation')
    );
    printf('<div class="%1$s"><p>%2$s</p></div>', esc_attr($class), wp_kses_post($message));
}

/**
 * Initialize the plugin
 */
function mxchat_migration_init() {
    if (!mxchat_migration_check_dependencies()) {
        return;
    }

    // Check if Pro license is active
    if (!mxchat_migration_is_license_active()) {
        add_action('admin_notices', 'mxchat_migration_show_pro_requirement_notice');
        return;
    }

    // Load plugin classes
    require_once MXCHAT_MIGRATION_PATH . 'includes/class-migration-admin.php';
    require_once MXCHAT_MIGRATION_PATH . 'includes/class-knowledge-migrator.php';
    require_once MXCHAT_MIGRATION_PATH . 'includes/class-actions-migrator.php';
    require_once MXCHAT_MIGRATION_PATH . 'includes/class-migration-handler.php';

    // Initialize admin interface
    if (is_admin()) {
        new MxChat_Migration_Admin();
    }

    // Initialize AJAX handler
    new MxChat_Migration_Handler();
}
add_action('plugins_loaded', 'mxchat_migration_init');

/**
 * Activation hook
 */
function mxchat_migration_activate() {
    // Check dependencies on activation
    if (!class_exists('MxChat_Integrator')) {
        deactivate_plugins(plugin_basename(__FILE__));
        wp_die(
            esc_html__('MxChat Migration Tool requires MxChat Basic plugin to be installed and activated.', 'mxchat-migration-tool'),
            esc_html__('Plugin Activation Error', 'mxchat-migration-tool'),
            array('back_link' => true)
        );
    }

    // Create migration log table
    global $wpdb;
    $table_name = $wpdb->prefix . 'mxchat_migration_logs';
    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE IF NOT EXISTS $table_name (
        id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
        migration_id VARCHAR(64) NOT NULL,
        migration_type VARCHAR(50) NOT NULL,
        source_database VARCHAR(50) NOT NULL,
        target_database VARCHAR(50) NOT NULL,
        source_model VARCHAR(100) NOT NULL,
        target_model VARCHAR(100) NOT NULL,
        total_items INT(11) DEFAULT 0,
        processed_items INT(11) DEFAULT 0,
        failed_items INT(11) DEFAULT 0,
        status VARCHAR(20) DEFAULT 'pending',
        started_at DATETIME,
        completed_at DATETIME,
        error_log LONGTEXT,
        PRIMARY KEY (id),
        KEY migration_id (migration_id),
        KEY status (status)
    ) $charset_collate;";

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);

    // Set default options
    add_option('mxchat_migration_tool_version', MXCHAT_MIGRATION_VERSION);
}
register_activation_hook(__FILE__, 'mxchat_migration_activate');

/**
 * Deactivation hook
 */
function mxchat_migration_deactivate() {
    // Clean up any pending migrations
    delete_transient('mxchat_migration_in_progress');
}
register_deactivation_hook(__FILE__, 'mxchat_migration_deactivate');
