<?php
/**
 * MxChat Moderation Admin Class
 *
 * Handles all admin functionality for the moderation plugin
 * with modern sidebar-based UI design.
 */
class MX_Chat_Admin {
    private $ban_handler;
    private $plugin_screen_hook_suffix;

    public function __construct() {
        $this->ban_handler = new MX_Chat_Ban_Handler();
    }

    public function init() {
        // Register our submenu page under the core plugin menu
        add_action('admin_menu', array($this, 'register_submenu_page'));

        // Other hooks remain the same
        add_action('admin_init', array($this, 'handle_ban_actions'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));

        // AJAX handlers
        add_action('wp_ajax_mx_chat_moderation_delete_ban', array($this, 'handle_ajax_ban_deletion'));
        add_action('wp_ajax_mx_chat_moderation_quick_ban', array($this, 'handle_ajax_quick_ban'));

        // Inject ban button into core plugin's transcripts page
        add_action('admin_enqueue_scripts', array($this, 'maybe_inject_ban_button'));
    }

    public function enqueue_admin_assets($hook) {
        // Check if we're on our submenu page
        if ($hook !== $this->plugin_screen_hook_suffix) {
            return;
        }

        // Enqueue the new sidebar CSS
        wp_enqueue_style(
            'mx-chat-moderation-admin-sidebar',
            MX_CHAT_MODERATION_PLUGIN_URL . 'admin/css/admin-sidebar.css',
            array(),
            MX_CHAT_MODERATION_VERSION
        );

        // Enqueue admin JS
        wp_enqueue_script(
            'mx-chat-moderation-admin',
            MX_CHAT_MODERATION_PLUGIN_URL . 'admin/js/admin.js',
            array('jquery'),
            MX_CHAT_MODERATION_VERSION,
            true
        );

        wp_localize_script('mx-chat-moderation-admin', 'mxChatMod', array(
            'nonce' => wp_create_nonce('mx_chat_moderation_nonce'),
            'ajaxurl' => admin_url('admin-ajax.php')
        ));
    }

    public function register_submenu_page() {
        // Use your core plugin's menu slug as the first parameter.
        $this->plugin_screen_hook_suffix = add_submenu_page(
            'mxchat-max', // Parent slug
            __('Chat Moderation', 'mx-chat-moderation'), // Page title
            __('Chat Moderation', 'mx-chat-moderation'), // Menu title
            'manage_options', // Capability required
            'mx-chat-moderation', // Menu slug
            array($this, 'render_admin_page') // Callback function
        );
    }

    public function render_admin_page() {
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have sufficient permissions to access this page.'));
        }

        // Get current section and action
        $section = isset($_GET['section']) ? sanitize_text_field($_GET['section']) : 'bans';
        $action = isset($_GET['action']) ? sanitize_text_field($_GET['action']) : '';
        $id = isset($_GET['id']) ? absint($_GET['id']) : 0;

        // Get ban counts for sidebar badge
        $total_bans = $this->ban_handler->get_bans(array('per_page' => 1));
        $ban_count = isset($total_bans['total']) ? $total_bans['total'] : 0;

        // Start output
        ?>
        <div class="mxch-admin-wrapper">
            <!-- Desktop Sidebar -->
            <aside class="mxch-sidebar">
                <div class="mxch-sidebar-header">
                    <a href="<?php echo esc_url(admin_url('admin.php?page=mx-chat-moderation')); ?>" class="mxch-sidebar-logo">
                        <span class="mxch-sidebar-logo-text">Moderation</span>
                        <span class="mxch-sidebar-version">v<?php echo esc_html(MX_CHAT_MODERATION_VERSION); ?></span>
                    </a>
                </div>

                <nav class="mxch-sidebar-nav">
                    <!-- Manage Section -->
                    <div class="mxch-nav-section">
                        <div class="mxch-nav-section-title"><?php esc_html_e('Manage', 'mx-chat-moderation'); ?></div>
                        <div class="mxch-nav-item">
                            <button type="button" class="mxch-nav-link <?php echo ($section === 'bans' || $section === '') ? 'active' : ''; ?>" data-section="bans">
                                <span class="mxch-nav-link-icon">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <circle cx="12" cy="12" r="10"></circle>
                                        <line x1="4.93" y1="4.93" x2="19.07" y2="19.07"></line>
                                    </svg>
                                </span>
                                <span class="mxch-nav-link-text"><?php esc_html_e('Banned Users', 'mx-chat-moderation'); ?></span>
                                <?php if ($ban_count > 0): ?>
                                    <span class="mxch-nav-link-badge"><?php echo esc_html($ban_count); ?></span>
                                <?php endif; ?>
                            </button>
                        </div>
                        <div class="mxch-nav-item">
                            <button type="button" class="mxch-nav-link <?php echo ($action === 'add') ? 'active' : ''; ?>" data-section="add-ban">
                                <span class="mxch-nav-link-icon">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <circle cx="12" cy="12" r="10"></circle>
                                        <line x1="12" y1="8" x2="12" y2="16"></line>
                                        <line x1="8" y1="12" x2="16" y2="12"></line>
                                    </svg>
                                </span>
                                <span class="mxch-nav-link-text"><?php esc_html_e('Add New Ban', 'mx-chat-moderation'); ?></span>
                            </button>
                        </div>
                    </div>

                    <!-- Settings Section -->
                    <div class="mxch-nav-section">
                        <div class="mxch-nav-section-title"><?php esc_html_e('Settings', 'mx-chat-moderation'); ?></div>
                        <div class="mxch-nav-item">
                            <button type="button" class="mxch-nav-link <?php echo ($section === 'settings') ? 'active' : ''; ?>" data-section="settings">
                                <span class="mxch-nav-link-icon">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <circle cx="12" cy="12" r="3"></circle>
                                        <path d="M19.4 15a1.65 1.65 0 0 0 .33 1.82l.06.06a2 2 0 0 1 0 2.83 2 2 0 0 1-2.83 0l-.06-.06a1.65 1.65 0 0 0-1.82-.33 1.65 1.65 0 0 0-1 1.51V21a2 2 0 0 1-2 2 2 2 0 0 1-2-2v-.09A1.65 1.65 0 0 0 9 19.4a1.65 1.65 0 0 0-1.82.33l-.06.06a2 2 0 0 1-2.83 0 2 2 0 0 1 0-2.83l.06-.06a1.65 1.65 0 0 0 .33-1.82 1.65 1.65 0 0 0-1.51-1H3a2 2 0 0 1-2-2 2 2 0 0 1 2-2h.09A1.65 1.65 0 0 0 4.6 9a1.65 1.65 0 0 0-.33-1.82l-.06-.06a2 2 0 0 1 0-2.83 2 2 0 0 1 2.83 0l.06.06a1.65 1.65 0 0 0 1.82.33H9a1.65 1.65 0 0 0 1-1.51V3a2 2 0 0 1 2-2 2 2 0 0 1 2 2v.09a1.65 1.65 0 0 0 1 1.51 1.65 1.65 0 0 0 1.82-.33l.06-.06a2 2 0 0 1 2.83 0 2 2 0 0 1 0 2.83l-.06.06a1.65 1.65 0 0 0-.33 1.82V9a1.65 1.65 0 0 0 1.51 1H21a2 2 0 0 1 2 2 2 2 0 0 1-2 2h-.09a1.65 1.65 0 0 0-1.51 1z"></path>
                                    </svg>
                                </span>
                                <span class="mxch-nav-link-text"><?php esc_html_e('Options', 'mx-chat-moderation'); ?></span>
                            </button>
                        </div>
                    </div>
                </nav>
            </aside>

            <!-- Mobile Header -->
            <header class="mxch-mobile-header">
                <a href="<?php echo esc_url(admin_url('admin.php?page=mx-chat-moderation')); ?>" class="mxch-mobile-logo">
                    <span class="mxch-mobile-logo-text">Moderation</span>
                </a>
                <button type="button" class="mxch-mobile-menu-btn" aria-label="<?php esc_attr_e('Open menu', 'mx-chat-moderation'); ?>">
                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <line x1="3" y1="12" x2="21" y2="12"></line>
                        <line x1="3" y1="6" x2="21" y2="6"></line>
                        <line x1="3" y1="18" x2="21" y2="18"></line>
                    </svg>
                </button>
            </header>

            <!-- Mobile Menu Overlay -->
            <div class="mxch-mobile-overlay"></div>

            <!-- Mobile Menu -->
            <div class="mxch-mobile-menu">
                <div class="mxch-mobile-menu-header">
                    <span class="mxch-mobile-menu-title"><?php esc_html_e('Navigation', 'mx-chat-moderation'); ?></span>
                    <button type="button" class="mxch-mobile-menu-close" aria-label="<?php esc_attr_e('Close menu', 'mx-chat-moderation'); ?>">
                        <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <line x1="18" y1="6" x2="6" y2="18"></line>
                            <line x1="6" y1="6" x2="18" y2="18"></line>
                        </svg>
                    </button>
                </div>
                <nav class="mxch-mobile-menu-nav">
                    <div class="mxch-mobile-nav-section">
                        <div class="mxch-mobile-nav-section-title"><?php esc_html_e('Manage', 'mx-chat-moderation'); ?></div>
                        <button type="button" class="mxch-mobile-nav-link <?php echo ($section === 'bans' || $section === '') ? 'active' : ''; ?>" data-section="bans">
                            <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <circle cx="12" cy="12" r="10"></circle>
                                <line x1="4.93" y1="4.93" x2="19.07" y2="19.07"></line>
                            </svg>
                            <span><?php esc_html_e('Banned Users', 'mx-chat-moderation'); ?></span>
                            <?php if ($ban_count > 0): ?>
                                <span class="mxch-nav-link-badge"><?php echo esc_html($ban_count); ?></span>
                            <?php endif; ?>
                        </button>
                        <button type="button" class="mxch-mobile-nav-link <?php echo ($action === 'add') ? 'active' : ''; ?>" data-section="add-ban">
                            <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <circle cx="12" cy="12" r="10"></circle>
                                <line x1="12" y1="8" x2="12" y2="16"></line>
                                <line x1="8" y1="12" x2="16" y2="12"></line>
                            </svg>
                            <span><?php esc_html_e('Add New Ban', 'mx-chat-moderation'); ?></span>
                        </button>
                    </div>
                    <div class="mxch-mobile-nav-section">
                        <div class="mxch-mobile-nav-section-title"><?php esc_html_e('Settings', 'mx-chat-moderation'); ?></div>
                        <button type="button" class="mxch-mobile-nav-link <?php echo ($section === 'settings') ? 'active' : ''; ?>" data-section="settings">
                            <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <circle cx="12" cy="12" r="3"></circle>
                                <path d="M19.4 15a1.65 1.65 0 0 0 .33 1.82l.06.06a2 2 0 0 1 0 2.83 2 2 0 0 1-2.83 0l-.06-.06a1.65 1.65 0 0 0-1.82-.33 1.65 1.65 0 0 0-1 1.51V21a2 2 0 0 1-2 2 2 2 0 0 1-2-2v-.09A1.65 1.65 0 0 0 9 19.4a1.65 1.65 0 0 0-1.82.33l-.06.06a2 2 0 0 1-2.83 0 2 2 0 0 1 0-2.83l.06-.06a1.65 1.65 0 0 0 .33-1.82 1.65 1.65 0 0 0-1.51-1H3a2 2 0 0 1-2-2 2 2 0 0 1 2-2h.09A1.65 1.65 0 0 0 4.6 9a1.65 1.65 0 0 0-.33-1.82l-.06-.06a2 2 0 0 1 0-2.83 2 2 0 0 1 2.83 0l.06.06a1.65 1.65 0 0 0 1.82.33H9a1.65 1.65 0 0 0 1-1.51V3a2 2 0 0 1 2-2 2 2 0 0 1 2 2v.09a1.65 1.65 0 0 0 1 1.51 1.65 1.65 0 0 0 1.82-.33l.06-.06a2 2 0 0 1 2.83 0 2 2 0 0 1 0 2.83l-.06.06a1.65 1.65 0 0 0-.33 1.82V9a1.65 1.65 0 0 0 1.51 1H21a2 2 0 0 1 2 2 2 2 0 0 1-2 2h-.09a1.65 1.65 0 0 0-1.51 1z"></path>
                            </svg>
                            <span><?php esc_html_e('Options', 'mx-chat-moderation'); ?></span>
                        </button>
                    </div>
                </nav>
            </div>

            <!-- Main Content Area -->
            <main class="mxch-content">
                <?php $this->render_content_sections($section, $action, $id); ?>
            </main>
        </div>
        <?php
    }

    /**
     * Render the content sections
     */
    private function render_content_sections($section, $action, $id) {
        // Get ban data for list view
        $search = isset($_GET['s']) ? sanitize_text_field($_GET['s']) : '';
        $type = isset($_GET['type']) ? sanitize_text_field($_GET['type']) : '';
        $paged = isset($_GET['paged']) ? absint($_GET['paged']) : 1;

        $args = array(
            'search' => $search,
            'type' => $type,
            'paged' => $paged
        );
        $bans = $this->ban_handler->get_bans($args);

        // Get ban for edit
        $ban = null;
        if ($action === 'edit' && $id > 0) {
            $ban = $this->ban_handler->get_ban($id);
        }

        // Determine active section
        $active_section = $section;
        if ($action === 'add') {
            $active_section = 'add-ban';
        } elseif ($action === 'edit') {
            $active_section = 'edit-ban';
        } elseif (empty($section) || $section === 'bans') {
            $active_section = 'bans';
        }
        ?>

        <!-- Bans List Section -->
        <section class="mxch-section <?php echo ($active_section === 'bans') ? 'active' : ''; ?>" id="mxch-section-bans">
            <div class="mxch-content-header">
                <h1 class="mxch-content-title"><?php esc_html_e('Banned Users', 'mx-chat-moderation'); ?></h1>
                <p class="mxch-content-subtitle"><?php esc_html_e('Manage banned emails and IP addresses', 'mx-chat-moderation'); ?></p>
            </div>

            <?php include MX_CHAT_MODERATION_PLUGIN_DIR . 'admin/partials/ban-list.php'; ?>
        </section>

        <!-- Edit Ban Section -->
        <section class="mxch-section <?php echo ($active_section === 'edit-ban') ? 'active' : ''; ?>" id="mxch-section-edit-ban">
            <div class="mxch-content-header">
                <h1 class="mxch-content-title"><?php esc_html_e('Edit Ban', 'mx-chat-moderation'); ?></h1>
                <p class="mxch-content-subtitle"><?php esc_html_e('Update ban details', 'mx-chat-moderation'); ?></p>
            </div>

            <?php
            if ($ban) {
                include MX_CHAT_MODERATION_PLUGIN_DIR . 'admin/partials/ban-form.php';
            }
            ?>
        </section>

        <!-- Add Ban Section -->
        <section class="mxch-section <?php echo ($active_section === 'add-ban') ? 'active' : ''; ?>" id="mxch-section-add-ban">
            <div class="mxch-content-header">
                <h1 class="mxch-content-title"><?php esc_html_e('Add New Ban', 'mx-chat-moderation'); ?></h1>
                <p class="mxch-content-subtitle"><?php esc_html_e('Ban an email address or IP from using the chat', 'mx-chat-moderation'); ?></p>
            </div>

            <?php
            $ban = null; // Ensure we're in add mode
            include MX_CHAT_MODERATION_PLUGIN_DIR . 'admin/partials/ban-form.php';
            ?>
        </section>

        <!-- Settings Section -->
        <section class="mxch-section <?php echo ($active_section === 'settings') ? 'active' : ''; ?>" id="mxch-section-settings">
            <div class="mxch-content-header">
                <h1 class="mxch-content-title"><?php esc_html_e('Settings', 'mx-chat-moderation'); ?></h1>
                <p class="mxch-content-subtitle"><?php esc_html_e('Configure moderation options', 'mx-chat-moderation'); ?></p>
            </div>

            <div class="mxch-card">
                <div class="mxch-card-header">
                    <h3 class="mxch-card-title">
                        <span class="mxch-card-title-icon">
                            <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <circle cx="12" cy="12" r="10"></circle>
                                <line x1="12" y1="16" x2="12" y2="12"></line>
                                <line x1="12" y1="8" x2="12.01" y2="8"></line>
                            </svg>
                        </span>
                        <?php esc_html_e('About Moderation', 'mx-chat-moderation'); ?>
                    </h3>
                </div>
                <div class="mxch-card-body">
                    <div class="mxch-notice mxch-notice-info">
                        <span class="mxch-notice-icon">
                            <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <circle cx="12" cy="12" r="10"></circle>
                                <line x1="12" y1="16" x2="12" y2="12"></line>
                                <line x1="12" y1="8" x2="12.01" y2="8"></line>
                            </svg>
                        </span>
                        <div>
                            <strong><?php esc_html_e('How Moderation Works', 'mx-chat-moderation'); ?></strong>
                            <p style="margin: 8px 0 0 0;"><?php esc_html_e('When a user is banned by email or IP address, they will be prevented from using the MxChat chatbot. Bans are checked in real-time when users attempt to start a conversation.', 'mx-chat-moderation'); ?></p>
                        </div>
                    </div>

                    <div class="mxch-field" style="margin-top: 24px;">
                        <p class="mxch-field-description">
                            <strong><?php esc_html_e('Version:', 'mx-chat-moderation'); ?></strong> <?php echo esc_html(MX_CHAT_MODERATION_VERSION); ?>
                        </p>
                    </div>
                </div>
            </div>
        </section>
        <?php
    }

    public function handle_ban_actions() {
        if (!isset($_POST['mx_chat_moderation_action'])) {
            return;
        }
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have sufficient permissions to perform this action.'));
        }
        check_admin_referer('mx_chat_moderation_action', 'mx_chat_moderation_nonce');
        $action = sanitize_text_field($_POST['mx_chat_moderation_action']);
        switch ($action) {
            case 'add_ban':
                $this->handle_ban_submission();
                break;
            case 'edit_ban':
                $this->handle_ban_update();
                break;
            case 'delete_ban':
                $this->handle_ban_deletion();
                break;
        }
    }

    private function handle_ban_submission() {
        $identifier = isset($_POST['identifier']) ? sanitize_text_field($_POST['identifier']) : '';
        $type = isset($_POST['type']) ? sanitize_text_field($_POST['type']) : '';
        $notes = isset($_POST['notes']) ? sanitize_textarea_field($_POST['notes']) : '';

        if (empty($identifier) || empty($type)) {
            wp_redirect(add_query_arg(array('error' => 'missing_fields'), wp_get_referer()));
            exit;
        }
        if ($type === 'email' && !is_email($identifier)) {
            wp_redirect(add_query_arg(array('error' => 'invalid_email'), wp_get_referer()));
            exit;
        }
        if ($type === 'ip' && !filter_var($identifier, FILTER_VALIDATE_IP)) {
            wp_redirect(add_query_arg(array('error' => 'invalid_ip'), wp_get_referer()));
            exit;
        }
        $result = $this->ban_handler->add_ban($identifier, $type, $notes);
        if ($result) {
            wp_redirect(add_query_arg(array('message' => 'ban_added', 'section' => 'bans'), admin_url('admin.php?page=mx-chat-moderation')));
        } else {
            wp_redirect(add_query_arg(array('error' => 'add_failed'), wp_get_referer()));
        }
        exit;
    }

    private function handle_ban_update() {
        $id = isset($_POST['ban_id']) ? absint($_POST['ban_id']) : 0;
        if (!$id) {
            wp_redirect(add_query_arg(array('error' => 'invalid_id'), wp_get_referer()));
            exit;
        }
        $data = array(
            'notes' => isset($_POST['notes']) ? sanitize_textarea_field($_POST['notes']) : ''
        );
        $result = $this->ban_handler->update_ban($id, $data);
        if ($result) {
            wp_redirect(add_query_arg(array('message' => 'ban_updated', 'section' => 'bans'), admin_url('admin.php?page=mx-chat-moderation')));
        } else {
            wp_redirect(add_query_arg(array('error' => 'update_failed'), wp_get_referer()));
        }
        exit;
    }

    public function handle_ajax_ban_deletion() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'mx_chat_moderation_nonce')) {
            wp_send_json_error(array('message' => 'Invalid security token.'));
        }

        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'You do not have permission to perform this action.'));
        }

        // Get and validate ban ID
        $ban_id = isset($_POST['ban_id']) ? absint($_POST['ban_id']) : 0;
        if (!$ban_id) {
            wp_send_json_error(array('message' => 'Invalid ban ID.'));
        }

        // Attempt to remove the ban
        $result = $this->ban_handler->remove_ban($ban_id);

        if ($result) {
            wp_send_json_success(array('message' => 'Ban successfully removed.'));
        } else {
            wp_send_json_error(array('message' => 'Failed to remove ban.'));
        }
    }

    /**
     * Handle AJAX quick ban from transcripts page
     */
    public function handle_ajax_quick_ban() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'mx_chat_moderation_quick_ban')) {
            wp_send_json_error(array('message' => 'Invalid security token.'));
        }

        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'You do not have permission to perform this action.'));
        }

        // Get and validate data
        $identifier = isset($_POST['identifier']) ? sanitize_text_field($_POST['identifier']) : '';
        $type = isset($_POST['type']) ? sanitize_text_field($_POST['type']) : '';
        $notes = isset($_POST['notes']) ? sanitize_textarea_field($_POST['notes']) : '';

        if (empty($identifier) || empty($type)) {
            wp_send_json_error(array('message' => 'Missing required fields.'));
        }

        // Validate based on type
        if ($type === 'email' && !is_email($identifier)) {
            wp_send_json_error(array('message' => 'Invalid email address.'));
        }

        if ($type === 'ip' && !filter_var($identifier, FILTER_VALIDATE_IP)) {
            wp_send_json_error(array('message' => 'Invalid IP address.'));
        }

        // Check if already banned
        if ($this->ban_handler->check_ban($identifier, $type)) {
            wp_send_json_error(array('message' => 'This ' . $type . ' is already banned.'));
        }

        // Add the ban
        $result = $this->ban_handler->add_ban($identifier, $type, $notes);

        if ($result) {
            wp_send_json_success(array(
                'message' => ucfirst($type) . ' successfully banned.',
                'redirect' => admin_url('admin.php?page=mx-chat-moderation&section=bans')
            ));
        } else {
            wp_send_json_error(array('message' => 'Failed to add ban. It may already exist.'));
        }
    }

    /**
     * Inject ban button into core plugin's transcripts page
     */
    public function maybe_inject_ban_button($hook) {
        // Only load on the transcripts page
        if (strpos($hook, 'mxchat-transcripts') === false) {
            return;
        }

        // Add inline CSS for the ban button
        wp_add_inline_style('mxchat-admin-sidebar', $this->get_ban_button_css());

        // Add inline JS to inject the button
        add_action('admin_footer', array($this, 'render_ban_button_script'));
    }

    /**
     * Get CSS for the ban button
     */
    private function get_ban_button_css() {
        return '
            .mxch-btn-ban {
                background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%) !important;
                color: white !important;
                border: none !important;
                transition: all 0.2s ease !important;
            }
            .mxch-btn-ban:hover:not(:disabled) {
                background: linear-gradient(135deg, #dc2626 0%, #b91c1c 100%) !important;
                transform: translateY(-1px);
                box-shadow: 0 4px 12px rgba(239, 68, 68, 0.3);
            }
            .mxch-btn-ban:disabled {
                opacity: 0.7;
                cursor: not-allowed !important;
                transform: none !important;
            }
            .mxch-btn-ban.loading {
                pointer-events: none;
            }
            .mxch-btn-ban.banned {
                background: linear-gradient(135deg, #6b7280 0%, #4b5563 100%) !important;
                cursor: default !important;
                transform: none !important;
                box-shadow: none !important;
            }
            .mxch-btn-ban .mxch-ban-spinner {
                animation: mxch-spin 1s linear infinite;
            }
            @keyframes mxch-spin {
                from { transform: rotate(0deg); }
                to { transform: rotate(360deg); }
            }
            .mxch-ban-toast {
                position: fixed;
                top: 50px;
                right: 20px;
                z-index: 100001;
                padding: 12px 20px;
                color: white;
                border-radius: 8px;
                font-size: 14px;
                font-weight: 500;
                box-shadow: 0 4px 12px rgba(0,0,0,0.15);
                animation: mxch-toast-in 0.2s ease;
            }
            .mxch-ban-toast.success {
                background: #10b981;
            }
            .mxch-ban-toast.error {
                background: #ef4444;
            }
            @keyframes mxch-toast-in {
                from { opacity: 0; transform: translateY(-10px); }
                to { opacity: 1; transform: translateY(0); }
            }
        ';
    }

    /**
     * Render the JavaScript for injecting the ban button (direct one-click ban)
     */
    public function render_ban_button_script() {
        $nonce = wp_create_nonce('mx_chat_moderation_quick_ban');
        ?>
        <script type="text/javascript">
        (function($) {
            'use strict';

            var MxChatModIntegration = {
                nonce: '<?php echo esc_js($nonce); ?>',
                ajaxurl: '<?php echo esc_js(admin_url('admin-ajax.php')); ?>',

                init: function() {
                    this.injectBanButton();
                    this.bindEvents();
                },

                injectBanButton: function() {
                    var self = this;
                    // Wait for the details drawer to be available
                    var checkDrawer = setInterval(function() {
                        var $actionsDiv = $('.mxch-details-actions');
                        // Check if button already exists anywhere in the document
                        if ($actionsDiv.length && !$('#mxch-ban-user-btn').length) {
                            // Add the ban button inside the actions div (next to other buttons)
                            var banButtonHtml = '<button type="button" id="mxch-ban-user-btn" class="mxch-btn mxch-btn-ban mxch-btn-sm">' +
                                '<svg class="mxch-ban-icon" xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">' +
                                '<circle cx="12" cy="12" r="10"></circle>' +
                                '<line x1="4.93" y1="4.93" x2="19.07" y2="19.07"></line>' +
                                '</svg> <span class="mxch-ban-text">Ban</span></button>';

                            $actionsDiv.append(banButtonHtml);
                            console.log('MxChat Moderation: Ban button injected');

                            // Stop the interval once button is injected
                            clearInterval(checkDrawer);
                        }
                    }, 500);

                    // Stop checking after 30 seconds as a fallback
                    setTimeout(function() {
                        clearInterval(checkDrawer);
                    }, 30000);
                },

                bindEvents: function() {
                    var self = this;

                    // Direct ban on click - using event delegation for dynamically added button
                    $(document).on('click', '#mxch-ban-user-btn', function(e) {
                        e.preventDefault();
                        e.stopPropagation();

                        var $btn = $(this);

                        // Don't do anything if already banned or loading
                        if ($btn.hasClass('banned') || $btn.hasClass('loading')) {
                            return;
                        }

                        self.banUser($btn);
                    });
                },

                banUser: function($btn) {
                    var self = this;
                    var identifier = '';
                    var type = '';

                    // Get the user identifier from the header (this is the main display name)
                    var userIdentifier = $('#mxch-user-name').text().trim();
                    console.log('MxChat Moderation: User identifier from header:', userIdentifier);

                    // Check if it's an email or IP address
                    if (userIdentifier && userIdentifier !== 'User Name') {
                        // Simple email check - contains @ and .
                        if (userIdentifier.indexOf('@') !== -1 && userIdentifier.indexOf('.') !== -1) {
                            identifier = userIdentifier;
                            type = 'email';
                        }
                        // Check if it looks like an IP (IPv4 or IPv6)
                        else if (this.isValidIP(userIdentifier)) {
                            identifier = userIdentifier;
                            type = 'ip';
                        }
                    }

                    // Also check the detail drawer fields as backup
                    if (!identifier) {
                        var $emailRow = $('#mxch-detail-email-row');
                        if ($emailRow.is(':visible')) {
                            var emailText = $('#mxch-detail-email').text().trim();
                            if (emailText && emailText !== '-') {
                                identifier = emailText;
                                type = 'email';
                            }
                        }
                    }

                    if (!identifier) {
                        var $ipRow = $('#mxch-detail-ip-row');
                        if ($ipRow.is(':visible')) {
                            var ipText = $('#mxch-detail-ip').text().trim();
                            if (ipText && ipText !== '-') {
                                identifier = ipText;
                                type = 'ip';
                            }
                        }
                    }

                    console.log('MxChat Moderation: Banning', type, ':', identifier);

                    // If no identifier found, show error
                    if (!identifier || !type) {
                        self.showNotice('No email or IP address found to ban.', 'error');
                        return;
                    }

                    // Show loading state
                    $btn.addClass('loading').prop('disabled', true);
                    $btn.find('.mxch-ban-icon').addClass('mxch-ban-spinner');
                    $btn.find('.mxch-ban-text').text('Banning...');

                    // Send AJAX request
                    $.ajax({
                        url: this.ajaxurl,
                        type: 'POST',
                        data: {
                            action: 'mx_chat_moderation_quick_ban',
                            nonce: this.nonce,
                            identifier: identifier,
                            type: type,
                            notes: 'Banned from transcripts page'
                        },
                        success: function(response) {
                            $btn.removeClass('loading');
                            $btn.find('.mxch-ban-icon').removeClass('mxch-ban-spinner');

                            if (response.success) {
                                // Show banned state
                                $btn.addClass('banned').prop('disabled', true);
                                $btn.find('.mxch-ban-text').text('Banned');
                                self.showNotice(response.data.message, 'success');
                            } else {
                                // Reset button on error
                                $btn.prop('disabled', false);
                                $btn.find('.mxch-ban-text').text('Ban');
                                self.showNotice(response.data.message || 'Failed to ban user.', 'error');
                            }
                        },
                        error: function() {
                            // Reset button on error
                            $btn.removeClass('loading').prop('disabled', false);
                            $btn.find('.mxch-ban-icon').removeClass('mxch-ban-spinner');
                            $btn.find('.mxch-ban-text').text('Ban');
                            self.showNotice('Server error. Please try again.', 'error');
                        }
                    });
                },

                showNotice: function(message, type) {
                    // Remove any existing toasts
                    $('.mxch-ban-toast').remove();

                    // Create toast notification
                    var $toast = $('<div class="mxch-ban-toast ' + type + '"></div>').text(message);
                    $('body').append($toast);

                    setTimeout(function() {
                        $toast.fadeOut(300, function() {
                            $(this).remove();
                        });
                    }, 3000);
                },

                isValidIP: function(ip) {
                    // IPv4 pattern
                    var ipv4Pattern = /^(\d{1,3}\.){3}\d{1,3}$/;
                    // IPv6 pattern (simplified - matches most common formats)
                    var ipv6Pattern = /^([0-9a-fA-F]{1,4}:){7}[0-9a-fA-F]{1,4}$/;
                    // IPv6 compressed pattern
                    var ipv6CompressedPattern = /^(([0-9a-fA-F]{1,4}:)*[0-9a-fA-F]{1,4})?::(([0-9a-fA-F]{1,4}:)*[0-9a-fA-F]{1,4})?$/;
                    // IPv6 with mixed notation
                    var ipv6MixedPattern = /^([0-9a-fA-F]{1,4}:){6}(\d{1,3}\.){3}\d{1,3}$/;

                    if (ipv4Pattern.test(ip)) {
                        // Validate each octet is 0-255
                        var octets = ip.split('.');
                        return octets.every(function(octet) {
                            var num = parseInt(octet, 10);
                            return num >= 0 && num <= 255;
                        });
                    }

                    return ipv6Pattern.test(ip) || ipv6CompressedPattern.test(ip) || ipv6MixedPattern.test(ip);
                }
            };

            $(document).ready(function() {
                MxChatModIntegration.init();
            });

        })(jQuery);
        </script>
        <?php
    }
}
