/**
 * MxChat Moderation Admin JavaScript
 *
 * Handles sidebar navigation, mobile menu, and ban management functionality.
 */
(function ($) {
    'use strict';

    const MxChatMod = {
        /**
         * Initialize the admin interface
         */
        init: function () {
            this.bindEvents();
            this.initializeFormValidation();
            this.initializeMobileMenu();
        },

        /**
         * Bind event handlers
         */
        bindEvents: function () {
            // Section navigation (sidebar and content buttons)
            $(document).on('click', '[data-section]', this.handleSectionNavigation.bind(this));

            // Ban deletion
            $(document).on('click', '.mx-chat-mod-delete', this.handleBanDeletion.bind(this));

            // Type filter change
            $(document).on('change', '#mx-chat-mod-type-filter', this.handleTypeFilter);

            // Search form
            $(document).on('submit', '#mx-chat-mod-search', this.handleSearch);
        },

        /**
         * Initialize mobile menu functionality
         */
        initializeMobileMenu: function () {
            const $menuBtn = $('.mxch-mobile-menu-btn');
            const $closeBtn = $('.mxch-mobile-menu-close');
            const $overlay = $('.mxch-mobile-overlay');
            const $menu = $('.mxch-mobile-menu');

            // Open menu
            $menuBtn.on('click', function () {
                $menu.addClass('open');
                $overlay.addClass('open');
                $('body').css('overflow', 'hidden');
            });

            // Close menu
            const closeMenu = function () {
                $menu.removeClass('open');
                $overlay.removeClass('open');
                $('body').css('overflow', '');
            };

            $closeBtn.on('click', closeMenu);
            $overlay.on('click', closeMenu);

            // Close on escape key
            $(document).on('keydown', function (e) {
                if (e.key === 'Escape' && $menu.hasClass('open')) {
                    closeMenu();
                }
            });

            // Close menu when clicking navigation items on mobile
            $('.mxch-mobile-nav-link').on('click', function () {
                closeMenu();
            });
        },

        /**
         * Handle section navigation
         */
        handleSectionNavigation: function (e) {
            const $trigger = $(e.currentTarget);
            const targetSection = $trigger.data('section');

            if (!targetSection) return;

            e.preventDefault();

            // Update URL without reloading (for direct links)
            const newUrl = new URL(window.location.href);
            newUrl.searchParams.set('section', targetSection);
            newUrl.searchParams.delete('action');
            newUrl.searchParams.delete('id');
            newUrl.searchParams.delete('paged');
            newUrl.searchParams.delete('s');
            newUrl.searchParams.delete('type');
            window.history.pushState({}, '', newUrl.toString());

            // Hide all sections
            $('.mxch-section').removeClass('active');

            // Show target section (using ID)
            const $targetSection = $('#mxch-section-' + targetSection);
            $targetSection.addClass('active');

            // Update sidebar active states
            $('.mxch-nav-link, .mxch-mobile-nav-link').removeClass('active');
            $('[data-section="' + targetSection + '"]').filter('.mxch-nav-link, .mxch-mobile-nav-link').addClass('active');

            // Scroll to top of content
            $('.mxch-content').scrollTop(0);
        },

        /**
         * Initialize form validation
         */
        initializeFormValidation: function () {
            // Use event delegation for forms with the class
            $(document).on('submit', '.mx-chat-mod-form', function (e) {
                const $form = $(this);
                const identifier = $form.find('input[name="identifier"]').val();
                const type = $form.find('select[name="type"]').val() || $form.find('input[name="type"]').val();
                const isEdit = $form.find('input[name="mx_chat_moderation_action"]').val() === 'edit_ban';

                // Skip identifier validation on edit (it's readonly)
                if (!isEdit) {
                    if (!identifier || !type) {
                        e.preventDefault();
                        MxChatMod.showNotice('Please fill in all required fields.', 'error');
                        return false;
                    }

                    if (type === 'email' && !MxChatMod.isValidEmail(identifier)) {
                        e.preventDefault();
                        MxChatMod.showNotice('Please enter a valid email address.', 'error');
                        return false;
                    }

                    if (type === 'ip' && !MxChatMod.isValidIP(identifier)) {
                        e.preventDefault();
                        MxChatMod.showNotice('Please enter a valid IP address.', 'error');
                        return false;
                    }
                }

                // Show loading state
                $form.addClass('mxch-loading');
                $form.find('button[type="submit"]').prop('disabled', true);
            });
        },

        /**
         * Handle ban deletion
         */
        handleBanDeletion: function (e) {
            e.preventDefault();

            if (!confirm('Are you sure you want to remove this ban? This action cannot be undone.')) {
                return;
            }

            const $button = $(e.currentTarget);
            const banId = $button.data('id');
            const $row = $button.closest('tr');

            $.ajax({
                url: mxChatMod.ajaxurl,
                type: 'POST',
                data: {
                    action: 'mx_chat_moderation_delete_ban',
                    ban_id: banId,
                    nonce: mxChatMod.nonce,
                },
                beforeSend: function () {
                    $row.addClass('mxch-loading');
                    $button.prop('disabled', true);
                },
                success: function (response) {
                    if (response.success) {
                        $row.fadeOut(300, function () {
                            $(this).remove();

                            // Check if table is empty
                            if ($('.mxch-table tbody tr:not(.mxch-empty-row)').length === 0) {
                                $('.mxch-table tbody').html(
                                    '<tr class="mxch-empty-row">' +
                                    '<td colspan="6">' +
                                    '<div class="mxch-empty-state">' +
                                    '<div class="mxch-empty-state-icon">' +
                                    '<svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round">' +
                                    '<circle cx="12" cy="12" r="10"></circle>' +
                                    '<line x1="4.93" y1="4.93" x2="19.07" y2="19.07"></line>' +
                                    '</svg>' +
                                    '</div>' +
                                    '<h3 class="mxch-empty-state-title">No bans found</h3>' +
                                    '<p class="mxch-empty-state-text">You haven\'t banned any users yet. Use the "Add Ban" button to get started.</p>' +
                                    '</div>' +
                                    '</td>' +
                                    '</tr>'
                                );
                            }

                            // Update badge count in sidebar
                            MxChatMod.updateBadgeCount();
                        });

                        MxChatMod.showNotice('Ban successfully removed.', 'success');
                    } else {
                        MxChatMod.showNotice(
                            response.data.message || 'Error deleting ban.',
                            'error'
                        );
                    }
                },
                error: function () {
                    MxChatMod.showNotice('Server error occurred. Please try again.', 'error');
                },
                complete: function () {
                    $row.removeClass('mxch-loading');
                    $button.prop('disabled', false);
                },
            });
        },

        /**
         * Update the badge count in sidebar
         */
        updateBadgeCount: function () {
            const count = $('.mxch-table tbody tr:not(.mxch-empty-row)').length;
            const $badge = $('.mxch-nav-link[data-section="bans"] .mxch-nav-link-badge, .mxch-mobile-nav-link[data-section="bans"] .mxch-nav-link-badge');

            if (count > 0) {
                if ($badge.length) {
                    $badge.text(count);
                } else {
                    $('.mxch-nav-link[data-section="bans"] .mxch-nav-link-text').after('<span class="mxch-nav-link-badge">' + count + '</span>');
                }
            } else {
                $badge.remove();
            }
        },

        /**
         * Handle search form submission
         */
        handleSearch: function (e) {
            // Let the form submit naturally for full page reload
            // This ensures proper pagination and filtering
        },

        /**
         * Handle type filter change
         */
        handleTypeFilter: function () {
            const type = $(this).val();
            const currentUrl = new URL(window.location.href);

            if (type) {
                currentUrl.searchParams.set('type', type);
            } else {
                currentUrl.searchParams.delete('type');
            }

            currentUrl.searchParams.delete('paged');
            currentUrl.searchParams.set('section', 'bans');
            window.location.href = currentUrl.toString();
        },

        /**
         * Show a notification toast
         */
        showNotice: function (message, type = 'success') {
            // Remove any existing notices
            $('.mxch-toast').remove();

            const iconSvg = type === 'success'
                ? '<svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"></path><polyline points="22 4 12 14.01 9 11.01"></polyline></svg>'
                : '<svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"></circle><line x1="12" y1="8" x2="12" y2="12"></line><line x1="12" y1="16" x2="12.01" y2="16"></line></svg>';

            // Create new notice
            const $notice = $('<div></div>')
                .addClass('mxch-toast mxch-notice mxch-notice-' + type)
                .html('<span class="mxch-notice-icon">' + iconSvg + '</span><span>' + message + '</span>');

            // Insert notice
            $('body').append($notice);

            // Auto-remove notice after delay
            setTimeout(function () {
                $notice.fadeOut(300, function () {
                    $(this).remove();
                });
            }, 4000);
        },

        /**
         * Validate email format
         */
        isValidEmail: function (email) {
            return /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email);
        },

        /**
         * Validate IP address format (IPv4 and IPv6)
         */
        isValidIP: function (ip) {
            // IPv4 pattern
            const ipv4Pattern = /^(\d{1,3}\.){3}\d{1,3}$/;
            // IPv6 pattern (simplified)
            const ipv6Pattern = /^([0-9a-fA-F]{1,4}:){7}[0-9a-fA-F]{1,4}$/;
            // IPv6 compressed pattern
            const ipv6CompressedPattern = /^(([0-9a-fA-F]{1,4}:)*[0-9a-fA-F]{1,4})?::(([0-9a-fA-F]{1,4}:)*[0-9a-fA-F]{1,4})?$/;

            if (ipv4Pattern.test(ip)) {
                // Validate each octet is 0-255
                const octets = ip.split('.');
                return octets.every(function (octet) {
                    const num = parseInt(octet, 10);
                    return num >= 0 && num <= 255;
                });
            }

            return ipv6Pattern.test(ip) || ipv6CompressedPattern.test(ip);
        },
    };

    // Initialize when document is ready
    $(document).ready(function () {
        if (typeof jQuery !== 'undefined') {
            MxChatMod.init();
        }
    });

})(jQuery);
