<?php
/**
 * Plugin Name: MxChat Moderation
 * Plugin URI:        https://mxchat.ai
 * Description: User moderation and ban management addon for MxChat (Pro Only)
 * Version: 2.1.2
 * Author:           MxChat
 * Author URI:        https://mxchat.ai
 * License: GPL-2.0+
 * License URI: http://www.gnu.org/licenses/gpl-2.0.txt
 * Text Domain: mx-chat-moderation
 */

if (!defined('ABSPATH')) {
    exit;
}

function is_mxchat_active_moderation() {
    if (is_multisite()) {
        $active_sitewide_plugins = get_site_option('active_sitewide_plugins');
        if (isset($active_sitewide_plugins['mxchat-basic/mxchat-basic.php'])) {
            return true;
        }
    }
    return in_array(
        'mxchat-basic/mxchat-basic.php',
        apply_filters('active_plugins', get_option('active_plugins'))
    );
}

// Check if MxChat is installed and active
if (!is_mxchat_active_moderation()) {
    add_action('admin_notices', function() {
        ?>
        <div class="notice notice-error is-dismissible">
            <p><?php _e('<strong>MxChat Moderation</strong> requires the MxChat plugin to be installed and active.', 'mx-chat-moderation'); ?></p>
        </div>
        <?php
    });
    return;
}

require_once dirname(__FILE__) . '/update-checker.php';
PluginUpdateCheckerModeration::init([
    'user_id' => 'wp_4_SGKQ2sGzkIGB',
    'plugin_slug' => 'mxchat-moderation'
]);

define('MX_CHAT_MODERATION_VERSION', '2.1.2');
define('MX_CHAT_MODERATION_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('MX_CHAT_MODERATION_PLUGIN_URL', plugin_dir_url(__FILE__));
define('MX_CHAT_MODERATION_PLUGIN_BASENAME', plugin_basename(__FILE__));

class MX_Chat_Moderation {
    private $version;
    private $table_name;
    private $admin_instance;

    public function __construct() {
        global $wpdb;
        $this->version = MX_CHAT_MODERATION_VERSION;
        $this->table_name = $wpdb->prefix . 'mxchat_bans';
        
        // Add admin notice for pro requirement
        add_action('admin_notices', array($this, 'show_pro_requirement_notice'));
        
        // Only load features if pro is active
        if ($this->is_license_active()) {
            $this->load_dependencies();
            $this->init_hooks();
            
            if (is_admin()) {
                add_action('admin_menu', array($this, 'add_admin_menu'));
                add_action('admin_init', array($this, 'init_admin'));
            }
        }
    }

    private function is_license_active() {
        if (is_multisite()) {
            $network_license_status = get_site_option('mxchat_license_status');
            if ($network_license_status === 'active') {
                return true;
            }
        }
        $license_status = get_option('mxchat_license_status', 'inactive');
        return $license_status === 'active';
    }

    public function show_pro_requirement_notice() {
        if (!$this->is_license_active()) {
            $class = 'notice notice-warning';
            $message = sprintf(
                __('MxChat Moderation requires MxChat Pro to be activated. Please <a href="%s">activate your pro license</a> to use this addon.', 'mx-chat-moderation'),
                admin_url('admin.php?page=mxchat-activation')
            );
            printf('<div class="%1$s"><p>%2$s</p></div>', esc_attr($class), wp_kses_post($message));
        }
    }

    private function load_dependencies() {
        require_once MX_CHAT_MODERATION_PLUGIN_DIR . 'includes/class-ban-handler.php';
        require_once MX_CHAT_MODERATION_PLUGIN_DIR . 'admin/class-admin.php';
        $this->admin_instance = new MX_Chat_Admin();
    }

    private function init_hooks() {
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
        add_action('plugins_loaded', array($this, 'init'));
    }

    public function activate() {
        $this->create_tables();
        flush_rewrite_rules();
    }

    public function deactivate() {
        flush_rewrite_rules();
    }

    private function create_tables() {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE IF NOT EXISTS {$this->table_name} (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            identifier varchar(255) NOT NULL,
            type varchar(10) NOT NULL,
            banned_at datetime NOT NULL,
            banned_by bigint(20) unsigned NOT NULL,
            notes text,
            PRIMARY KEY (id),
            UNIQUE KEY identifier (identifier),
            KEY type (type)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }

    public function init() {
        load_plugin_textdomain('mx-chat-moderation', false, dirname(MX_CHAT_MODERATION_PLUGIN_BASENAME) . '/languages');
        
        if (is_admin() && $this->is_license_active()) {
            $this->admin_instance->init();
        }
    }

    public function init_admin() {
        // Register any admin-specific initialization here
        // For example: register settings, add meta boxes, etc.
        register_setting('mxchat_moderation_options', 'mxchat_moderation_settings');
        
        // Add any other admin initialization code as needed
    }

    public function add_admin_menu() {
        // This method should be implemented to add the admin menu items
        add_submenu_page(
            'mxchat',
            __('Moderation', 'mx-chat-moderation'),
            __('Moderation', 'mx-chat-moderation'),
            'manage_options',
            'mxchat-moderation',
            array($this->admin_instance, 'render_admin_page')
        );
    }

    public function get_table_name() {
        return $this->table_name;
    }

    public function get_version() {
        return $this->version;
    }
}

// Initialize the plugin
new MX_Chat_Moderation();