/**
 * MxChat Multi-Bot Manager Admin Scripts
 */

(function($) {
    'use strict';

    // Toast notification system
    window.MxChatToast = {
        container: null,
        toasts: [],
        
        init: function() {
            if (!this.container) {
                this.container = $('<div class="mxchat-toast-container"></div>');
                $('body').append(this.container);
            }
        },
        
        show: function(message, type = 'info', duration = 5000) {
            this.init();
            
            const icons = {
                success: '✓',
                error: '✕',
                warning: '⚠',
                info: 'ℹ'
            };
            
            const toastId = 'toast-' + Date.now() + '-' + Math.random().toString(36).substr(2, 9);
            
            const $toast = $(`
                <div class="mxchat-toast mxchat-toast-${type}" data-toast-id="${toastId}">
                    <div class="mxchat-toast-content">
                        <span class="mxchat-toast-icon">${icons[type] || icons.info}</span>
                        <p class="mxchat-toast-message">${message}</p>
                    </div>
                    <button class="mxchat-toast-close" aria-label="Close">×</button>
                    <div class="mxchat-toast-progress"></div>
                </div>
            `);
            
            // Add to container
            this.container.append($toast);
            this.toasts.push({
                id: toastId,
                element: $toast,
                timer: null
            });
            
            // Trigger show animation
            setTimeout(() => {
                $toast.addClass('show');
            }, 10);
            
            // Progress bar animation
            if (duration > 0) {
                const $progress = $toast.find('.mxchat-toast-progress');
                $progress.css('width', '100%');
                
                setTimeout(() => {
                    $progress.css({
                        'width': '0%',
                        'transition': `width ${duration}ms linear`
                    });
                }, 100);
            }
            
            // Auto dismiss
            if (duration > 0) {
                const toast = this.toasts.find(t => t.id === toastId);
                if (toast) {
                    toast.timer = setTimeout(() => {
                        this.hide(toastId);
                    }, duration);
                }
            }
            
            // Handle close button
            $toast.find('.mxchat-toast-close').on('click', () => {
                this.hide(toastId);
            });
            
            // Handle click to dismiss (optional)
            $toast.on('click', (e) => {
                if (!$(e.target).hasClass('mxchat-toast-close')) {
                    this.hide(toastId);
                }
            });
            
            return toastId;
        },
        
        hide: function(toastId) {
            const toastIndex = this.toasts.findIndex(t => t.id === toastId);
            if (toastIndex === -1) return;
            
            const toast = this.toasts[toastIndex];
            
            // Clear timer
            if (toast.timer) {
                clearTimeout(toast.timer);
            }
            
            // Hide animation
            toast.element.removeClass('show').addClass('hide');
            
            // Remove from DOM after animation
            setTimeout(() => {
                toast.element.remove();
                this.toasts.splice(toastIndex, 1);
            }, 400);
        },
        
        hideAll: function() {
            this.toasts.forEach(toast => {
                this.hide(toast.id);
            });
        },
        
        success: function(message, duration = 5000) {
            return this.show(message, 'success', duration);
        },
        
        error: function(message, duration = 7000) {
            return this.show(message, 'error', duration);
        },
        
        warning: function(message, duration = 6000) {
            return this.show(message, 'warning', duration);
        },
        
        info: function(message, duration = 5000) {
            return this.show(message, 'info', duration);
        }
    };

    $(document).ready(function() {
        
        // Initialize color pickers if present
        if ($.fn.wpColorPicker) {
            $('.color-picker').wpColorPicker();
        }

        // Function to handle bot form submission
        function handleBotFormSubmit(e) {
            e.preventDefault();
            
            var $form = $('#mxchat-bot-form');
            var $submitBtnBottom = $('#save-bot');
            var $submitBtnTop = $('#save-bot-top');
            var originalTextBottom = $submitBtnBottom.text();
            var originalTextTop = $submitBtnTop.text();
            
            // Get form data
            var formData = new FormData($form[0]);
            
            // Add WordPress editor content if present
            if (typeof tinyMCE !== 'undefined' && tinyMCE.get('intro_message')) {
                formData.set('intro_message', tinyMCE.get('intro_message').getContent());
            }
            
            // Add AJAX action and nonce
            formData.append('action', 'mxchat_multi_bot_save');
            formData.append('nonce', mxchatMultiBot.nonce);
            
            // Disable both submit buttons and show loading
            $submitBtnBottom.prop('disabled', true).text(mxchatMultiBot.strings.saving || 'Saving...');
            $submitBtnTop.prop('disabled', true).text(mxchatMultiBot.strings.saving || 'Saving...');
            
            $.ajax({
                url: mxchatMultiBot.ajax_url,
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                success: function(response) {
                    if (response.success) {
                        // Show success toast
                        MxChatToast.success(response.data.message);
                        
                        // If creating new bot, redirect to bot list
                        if ($('#is_new').val() === 'true') {
                            setTimeout(function() {
                                window.location.href = response.data.redirect || mxchatMultiBot.admin_url + 'admin.php?page=mxchat-multi-bot';
                            }, 1500);
                        } else {
                            // For edit mode, trigger form saved event
                            $(document).trigger('mxchat-form-saved');
                        }
                    } else {
                        MxChatToast.error(response.data || mxchatMultiBot.strings.error_occurred);
                    }
                },
                error: function() {
                    MxChatToast.error(mxchatMultiBot.strings.error_occurred);
                },
                complete: function() {
                    // Re-enable both buttons and restore original text
                    $submitBtnBottom.prop('disabled', false).text(originalTextBottom);
                    $submitBtnTop.prop('disabled', false).text(originalTextTop);
                }
            });
        }

        // Separate event handlers for form submit and button clicks
        $('#mxchat-bot-form').on('submit', handleBotFormSubmit);
        $('#save-bot-top').on('click', handleBotFormSubmit);

        // Handle bot deletion
        $(document).on('click', '.mxchat-delete-bot', function(e) {
            e.preventDefault();
            
            if (!confirm(mxchatMultiBot.strings.confirm_delete)) {
                return;
            }
            
            var $button = $(this);
            var botId = $button.data('bot-id');
            var $card = $button.closest('.mxchat-bot-card');
            
            $button.prop('disabled', true);
            
            $.ajax({
                url: mxchatMultiBot.ajax_url,
                type: 'POST',
                data: {
                    action: 'mxchat_multi_bot_delete',
                    bot_id: botId,
                    nonce: mxchatMultiBot.nonce
                },
                success: function(response) {
                    if (response.success) {
                        $card.fadeOut(400, function() {
                            $(this).remove();
                            
                            // Check if grid is empty
                            if ($('.mxchat-bot-card').length === 0) {
                                location.reload();
                            }
                        });
                        MxChatToast.success(response.data.message);
                    } else {
                        MxChatToast.error(response.data.message || mxchatMultiBot.strings.error_occurred);
                    }
                },
                error: function() {
                    MxChatToast.error(mxchatMultiBot.strings.error_occurred);
                },
                complete: function() {
                    $button.prop('disabled', false);
                }
            });
        });

        // Handle bot duplication
        $(document).on('click', '.mxchat-duplicate-bot', function(e) {
            e.preventDefault();
            
            var sourceBotId = $(this).data('bot-id');
            var sourceBotName = $(this).data('bot-name');
            
            var newBotName = prompt(mxchatMultiBot.strings.duplicate_name_prompt, sourceBotName + ' (Copy)');
            
            if (!newBotName) {
                return;
            }
            
            var newBotId = newBotName.toLowerCase().replace(/[^a-z0-9]+/g, '_').replace(/^_+|_+$/g, '');
            
            var $button = $(this);
            $button.prop('disabled', true);
            
            $.ajax({
                url: mxchatMultiBot.ajax_url,
                type: 'POST',
                data: {
                    action: 'mxchat_multi_bot_duplicate',
                    source_bot_id: sourceBotId,
                    new_bot_id: newBotId,
                    new_bot_name: newBotName,
                    nonce: mxchatMultiBot.nonce
                },
                success: function(response) {
                    if (response.success) {
                        MxChatToast.success(response.data.message);
                        setTimeout(function() {
                            location.reload();
                        }, 1500);
                    } else {
                        MxChatToast.error(response.data || mxchatMultiBot.strings.error_occurred);
                    }
                },
                error: function() {
                    MxChatToast.error(mxchatMultiBot.strings.error_occurred);
                },
                complete: function() {
                    $button.prop('disabled', false);
                }
            });
        });

        // Handle bot ID validation (for create form)
        var botIdTimer;
        $('#bot_id').on('input', function() {
            var $input = $(this);
            var $status = $('#mxchat-bot-id-status');
            var botId = $input.val();
            var currentId = $('#original_bot_id').val();
            
            clearTimeout(botIdTimer);
            
            if (!botId) {
                $status.removeClass('mxchat-status-available mxchat-status-taken mxchat-status-checking').text('');
                return;
            }
            
            // Validate format
            if (!/^[a-z0-9_]+$/.test(botId)) {
                $status.removeClass('mxchat-status-available mxchat-status-checking').addClass('mxchat-status-taken')
                    .text(mxchatMultiBot.strings.bot_id_format);
                return;
            }
            
            // If editing and it's the same ID, don't check
            if (currentId && botId === currentId) {
                $status.removeClass('mxchat-status-taken mxchat-status-checking').addClass('mxchat-status-available')
                    .text(mxchatMultiBot.strings.bot_id_current);
                return;
            }
            
            $status.removeClass('mxchat-status-available mxchat-status-taken').addClass('mxchat-status-checking')
                .text('Checking...');
            
            botIdTimer = setTimeout(function() {
                $.ajax({
                    url: mxchatMultiBot.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'mxchat_multi_bot_check_id',
                        bot_id: botId,
                        current_id: currentId,
                        nonce: mxchatMultiBot.nonce
                    },
                    success: function(response) {
                        if (response.success) {
                            $status.removeClass('mxchat-status-taken mxchat-status-checking').addClass('mxchat-status-available')
                                .text(mxchatMultiBot.strings.bot_id_available);
                        } else {
                            $status.removeClass('mxchat-status-available mxchat-status-checking').addClass('mxchat-status-taken')
                                .text(mxchatMultiBot.strings.bot_id_taken);
                        }
                    },
                    error: function() {
                        $status.removeClass('mxchat-status-available mxchat-status-checking mxchat-status-taken').text('');
                    }
                });
            }, 500);
        });

        // Handle shortcode copying
        $(document).on('click', '.mxchat-copy-shortcode', function(e) {
            e.preventDefault();
            
            var $button = $(this);
            var shortcode = $button.data('shortcode');
            var originalText = $button.text();
            
            // Create temporary input element
            var $temp = $('<input>');
            $('body').append($temp);
            $temp.val(shortcode).select();
            
            try {
                document.execCommand('copy');
                $button.text(mxchatMultiBot.strings.copied);
                MxChatToast.success('Shortcode copied to clipboard!');
                
                setTimeout(function() {
                    $button.text(originalText);
                }, 2000);
            } catch(err) {
                console.error('Failed to copy:', err);
                MxChatToast.error('Failed to copy shortcode');
            }
            
            $temp.remove();
        });
        
        // Handle adding new questions - UPDATED to match core plugin numbering
        $(document).off('click', '.mxchat-multi-quick-question-add-btn').on('click', '.mxchat-multi-quick-question-add-btn', function(e) {
            e.preventDefault();
            e.stopPropagation();
            
            var $container = $('#mxchat-additional-questions-container');
            var existingRows = $container.find('.mxchat-multi-question-row').length;
            var nextQuestionNumber = 4 + existingRows; // Start from 4 since we have questions 1, 2, 3 already
            
            var $newRow = $('<div class="mxchat-multi-question-row">' +
                '<input type="text" name="additional_popular_questions[]" value="" ' +
                'placeholder="Enter Additional Quick Question ' + nextQuestionNumber + '" ' +
                'class="mxchat-multi-question-input" />' +
                '<button type="button" class="mxchat-multi-quick-question-remove-btn">Remove</button>' +
                '</div>');
            
            $container.append($newRow);
        });

        // Handle removing questions
        $(document).off('click', '.mxchat-multi-quick-question-remove-btn').on('click', '.mxchat-multi-quick-question-remove-btn', function(e) {
            e.preventDefault();
            e.stopPropagation();
            
            var $container = $('#mxchat-additional-questions-container');
            
            // Don't remove if it's the last row
            if ($container.find('.mxchat-multi-question-row').length > 1) {
                $(this).closest('.mxchat-multi-question-row').remove();
            } else {
                // Clear the input instead
                $(this).siblings('.mxchat-multi-question-input').val('');
            }
        });

        // Handle tab navigation if present
        $('.nav-tab').on('click', function(e) {
            e.preventDefault();
            
            var $tab = $(this);
            var target = $tab.data('tab');
            
            // Update active tab
            $('.nav-tab').removeClass('nav-tab-active');
            $tab.addClass('nav-tab-active');
            
            // Show/hide tab content
            $('.tab-content').hide();
            $('#' + target).show();
            
            // Update URL without reload
            if (history.pushState) {
                var newUrl = window.location.protocol + "//" + window.location.host + 
                            window.location.pathname + '?page=mxchat-multi-bot&tab=' + target;
                window.history.pushState({path: newUrl}, '', newUrl);
            }
        });

        // Show/hide advanced settings
        $('#toggle-advanced-settings').on('click', function(e) {
            e.preventDefault();
            var $button = $(this);
            var $settings = $('.advanced-settings');
            
            if ($settings.is(':visible')) {
                $settings.slideUp();
                $button.text(mxchatMultiBot.strings.show_advanced || 'Show Advanced Settings');
            } else {
                $settings.slideDown();
                $button.text(mxchatMultiBot.strings.hide_advanced || 'Hide Advanced Settings');
            }
        });

        // Track form changes for unsaved changes indicator
        var formChanged = false;
        $('#mxchat-bot-form input, #mxchat-bot-form textarea, #mxchat-bot-form select').on('change input', function() {
            if (!formChanged) {
                formChanged = true;
                // Add visual indicator that form has unsaved changes
                $('#save-bot, #save-bot-top').addClass('mxchat-save-indicator');
            }
        });

        // Reset change indicator after successful save
        $(document).on('mxchat-form-saved', function() {
            formChanged = false;
            $('#save-bot, #save-bot-top').removeClass('mxchat-save-indicator');
        });

        // Add keyboard shortcut for save (Ctrl+S or Cmd+S)
        $(document).on('keydown', function(e) {
            if ((e.ctrlKey || e.metaKey) && e.which === 83) { // Ctrl+S or Cmd+S
                e.preventDefault();
                if ($('#save-bot-top').length) {
                    $('#save-bot-top').click();
                } else {
                    $('#save-bot').click();
                }
                return false;
            }
        });

        /**
         * Helper function to show notices (legacy support and backup)
         * This maintains backward compatibility while primarily using toasts
         */
        function showNotice(message, type) {
            // Remove any existing old-style notices
            $('.mxchat-notice').remove();
            
            // Show toast notification
            switch(type) {
                case 'success':
                    MxChatToast.success(message);
                    break;
                case 'error':
                    MxChatToast.error(message);
                    break;
                case 'warning':
                    MxChatToast.warning(message);
                    break;
                default:
                    MxChatToast.info(message);
            }
        }

        // Make showNotice available globally for backward compatibility
        window.showNotice = showNotice;

        /**
         * Initialize tooltips if present
         */
        if ($.fn.tooltip) {
            $('.help-tip').tooltip();
        }

    });

})(jQuery);