<?php
/**
 * Bot List View for MxChat Multi-Bot Manager
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Get bots from database
$bots = isset($bots) ? $bots : array();

// Get default model from core plugin for reference
$core_options = get_option('mxchat_options', array());
$default_model = isset($core_options['model']) ? $core_options['model'] : 'gpt-4o';
?>

<style>
/* FAQ Section Styles */
.mxchat-faq-section {
    padding: 6rem 0;
    background: white;
    border-radius: 16px;
    box-shadow: 0 4px 20px rgba(120, 115, 245, 0.08);
    margin-top: 40px;
}

.mxchat-faq-header {
  text-align: center;
  margin-bottom: 4rem;
  max-width: 600px;
  margin-left: auto;
  margin-right: auto;
}

.mxchat-faq-header h2 {
  font-size: 2.5rem;
  margin-bottom: 1rem;
  font-weight: 700;
  line-height: 1.2;
  color: #1e293b;
}

.mxchat-faq-header p {
  font-size: 1.125rem;
  color: #64748b;
  margin-bottom: 0;
  line-height: 1.7;
}

.mxchat-faq-grid {
  max-width: 800px;
  margin: 0 auto;
}

.mxchat-faq-item {
  border-bottom: 1px solid #e2e8f0;
  margin-bottom: 1rem;
}

.mxchat-faq-question {
  display: flex;
  justify-content: space-between;
  align-items: center;
  padding: 1.5rem 0;
  cursor: pointer;
  transition: color 0.3s ease;
}

.mxchat-faq-question:hover {
  color: #8b5cf6;
}

.mxchat-faq-question h3 {
  margin-bottom: 0;
  font-size: 1.125rem;
  font-weight: 700;
  line-height: 1.2;
  color: #1e293b;
}

.mxchat-faq-question:hover h3 {
  color: #8b5cf6;
}

.mxchat-faq-toggle {
  font-size: 1.5rem;
  color: #8b5cf6;
  transition: transform 0.3s ease;
  font-weight: 600;
  line-height: 1;
}

.mxchat-faq-item.active .mxchat-faq-toggle {
  transform: rotate(45deg);
}

.mxchat-faq-answer {
  padding-bottom: 1.5rem;
  display: none;
  animation: fadeIn 0.3s ease-in-out;
}

.mxchat-faq-item.active .mxchat-faq-answer {
  display: block;
}

.mxchat-faq-answer p {
  color: #64748b;
  margin-bottom: 0;
  line-height: 1.7;
  font-size: 1rem;
}

@keyframes fadeIn {
  from {
    opacity: 0;
    transform: translateY(-10px);
  }
  to {
    opacity: 1;
    transform: translateY(0);
  }
}

/* Bot configuration info styles */
.mxchat-bot-config {
    margin: 12px 0;
    padding: 12px;
    background: #f8fafc;
    border-radius: 6px;
    border-left: 3px solid #6366f1;
}

.mxchat-bot-config-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 6px;
}

.mxchat-bot-config-row:last-child {
    margin-bottom: 0;
}

.mxchat-bot-config-label {
    font-size: 12px;
    font-weight: 600;
    color: #64748b;
    text-transform: uppercase;
    letter-spacing: 0.05em;
}

.mxchat-bot-config-value {
    font-size: 13px;
    font-weight: 500;
    color: #1e293b;
}

.mxchat-model-default {
    color: #64748b;
    font-style: italic;
}

.mxchat-streaming-enabled {
    color: #059669;
}

.mxchat-streaming-disabled {
    color: #dc2626;
}

/* Responsive Design for FAQ */
@media (max-width: 768px) {
  .mxchat-faq-section {
    padding: 4rem 0;
  }
  
  .mxchat-faq-header h2 {
    font-size: 2rem;
  }
  
  .mxchat-faq-question {
    padding: 1rem 0;
  }
  
  .mxchat-faq-question h3 {
    font-size: 1rem;
    padding-right: 1rem;
  }
  
  .mxchat-bot-config-row {
    flex-direction: column;
    align-items: flex-start;
    gap: 2px;
  }
}

@media (max-width: 480px) {
  .mxchat-faq-header h2 {
    font-size: 1.75rem;
  }
  
  .mxchat-faq-header p {
    font-size: 1rem;
  }
  
  .mxchat-faq-question h3 {
    font-size: 0.95rem;
  }
  
  .mxchat-faq-answer p {
    font-size: 0.9rem;
  }
}    
</style>

<div class="mxchat-multi-bot-admin">
    <!-- Header Section -->
    <div class="mxchat-header">
        <div class="mxchat-header-content">
            <h1 class="mxchat-title">
                <span class="mxchat-title-gradient">Multi-Bot</span> Manager
            </h1>
            <p class="mxchat-subtitle">
                Create and manage specialized chatbots for different use cases. Each bot can have its own knowledge base and configuration.
            </p>
        </div>
    </div>

    <!-- Main Content -->
    <div class="mxchat-content">
        <!-- Action Bar -->
        <div class="mxchat-action-bar">
            <div class="mxchat-stats">
                <span class="mxchat-stat">
                    <strong><?php echo count($bots); ?></strong> 
                    <?php echo count($bots) === 1 ? __('Bot', 'mxchat-multi-bot') : __('Bots', 'mxchat-multi-bot'); ?>
                </span>
            </div>
            <div class="mxchat-actions">
                <a href="<?php echo admin_url('admin.php?page=mxchat-multi-bot&action=create'); ?>" class="mxchat-button-multi mxchat-button-primary">
                    <span class="mxchat-button-icon-multi">+</span>
                    <?php _e('Add New Bot', 'mxchat-multi-bot'); ?>
                </a>
            </div>
        </div>

        <?php if (empty($bots)) : ?>
            <!-- Empty State -->
            <div class="mxchat-empty-state">
                <div class="mxchat-empty-icon">🤖</div>
                <h2><?php _e('No bots created yet', 'mxchat-multi-bot'); ?></h2>
                <p><?php _e('Create your first specialized chatbot to get started with multi-bot functionality.', 'mxchat-multi-bot'); ?></p>
                <a href="<?php echo admin_url('admin.php?page=mxchat-multi-bot&action=create'); ?>" class="mxchat-button-multi mxchat-button-primary mxchat-button-large">
                    <?php _e('Create Your First Bot', 'mxchat-multi-bot'); ?>
                </a>
            </div>
        <?php else : ?>
            <!-- Bot Grid -->
            <div class="mxchat-bot-grid">
                <?php foreach ($bots as $bot) : ?>
                    <?php
                    // Decode bot settings to get model and streaming info
                    $bot_settings = array();
                    if (!empty($bot->bot_settings)) {
                        $decoded_settings = json_decode($bot->bot_settings, true);
                        if (is_array($decoded_settings)) {
                            $bot_settings = $decoded_settings;
                        }
                    }
                    
                    // Get model info
                    $bot_model = isset($bot_settings['model']) && !empty($bot_settings['model']) 
                        ? $bot_settings['model'] 
                        : $default_model;
                    $is_using_default_model = !isset($bot_settings['model']) || empty($bot_settings['model']);
                    
                    // Get streaming info (default to 'on' if not set)
                    $streaming_enabled = isset($bot_settings['enable_streaming_toggle']) 
                        ? ($bot_settings['enable_streaming_toggle'] === 'on') 
                        : true; // Default to enabled
                    ?>
                    <div class="mxchat-bot-card" data-bot-id="<?php echo esc_attr($bot->bot_id); ?>">
                        <div class="mxchat-bot-header">
                            <div class="mxchat-bot-info">
                                <h3 class="mxchat-bot-name">
                                    <a href="<?php echo admin_url('admin.php?page=mxchat-multi-bot&action=edit&bot_id=' . $bot->bot_id); ?>">
                                        <?php echo esc_html($bot->bot_name); ?>
                                    </a>
                                </h3>
                                <div class="mxchat-bot-id">
                                    <code><?php echo esc_html($bot->bot_id); ?></code>
                                </div>
                            </div>
                            <div class="mxchat-bot-status">
                                <span class="mxchat-status-badge <?php echo !empty($bot->pinecone_host) ? 'mxchat-status-active' : 'mxchat-status-inactive'; ?>">
                                    <?php echo !empty($bot->pinecone_host) ? __('Active', 'mxchat-multi-bot') : __('Inactive', 'mxchat-multi-bot'); ?>
                                </span>
                            </div>
                        </div>

                        <!-- Bot Configuration Info -->
                        <div class="mxchat-bot-config">
                            <div class="mxchat-bot-config-row">
                                <span class="mxchat-bot-config-label"><?php _e('Model', 'mxchat-multi-bot'); ?></span>
                                <span class="mxchat-bot-config-value <?php echo $is_using_default_model ? 'mxchat-model-default' : ''; ?>">
                                    <?php 
                                    if ($is_using_default_model) {
                                        echo esc_html($bot_model) . ' ' . __('(default)', 'mxchat-multi-bot');
                                    } else {
                                        echo esc_html($bot_model);
                                    }
                                    ?>
                                </span>
                            </div>
                            <div class="mxchat-bot-config-row">
                                <span class="mxchat-bot-config-label"><?php _e('Streaming', 'mxchat-multi-bot'); ?></span>
                                <span class="mxchat-bot-config-value <?php echo $streaming_enabled ? 'mxchat-streaming-enabled' : 'mxchat-streaming-disabled'; ?>">
                                    <?php echo $streaming_enabled ? __('Enabled', 'mxchat-multi-bot') : __('Disabled', 'mxchat-multi-bot'); ?>
                                </span>
                            </div>
                        </div>

                        <div class="mxchat-shortcode-section">
                            <label class="mxchat-shortcode-label"><?php _e('Shortcode:', 'mxchat-multi-bot'); ?></label>
                            <div class="mxchat-shortcode-container">
                                <input type="text" 
                                       class="mxchat-shortcode-input" 
                                       value='[mxchat_chatbot floating="yes" bot_id="<?php echo esc_attr($bot->bot_id); ?>"]' 
                                       readonly>
                                <button class="mxchat-button-multi mxchat-button-secondary mxchat-copy-shortcode" 
                                        data-shortcode='[mxchat_chatbot floating="yes" bot_id="<?php echo esc_attr($bot->bot_id); ?>"]'>
                                    <?php _e('Copy', 'mxchat-multi-bot'); ?>
                                </button>
                            </div>
                        </div>

                        <div class="mxchat-bot-actions">
                            <a href="<?php echo admin_url('admin.php?page=mxchat-multi-bot&action=edit&bot_id=' . $bot->bot_id); ?>" 
                               class="mxchat-button-multi mxchat-button-primary mxchat-button-small">
                                <?php _e('Edit', 'mxchat-multi-bot'); ?>
                            </a>
                            <button class="mxchat-button-multi mxchat-button-secondary mxchat-button-small mxchat-duplicate-bot" 
                                    data-bot-id="<?php echo esc_attr($bot->bot_id); ?>" 
                                    data-bot-name="<?php echo esc_attr($bot->bot_name); ?>">
                                <?php _e('Duplicate', 'mxchat-multi-bot'); ?>
                            </button>
                            <button class="mxchat-button-multi mxchat-button-danger mxchat-button-small mxchat-delete-bot" 
                                    data-bot-id="<?php echo esc_attr($bot->bot_id); ?>">
                                <?php _e('Delete', 'mxchat-multi-bot'); ?>
                            </button>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>

        <!-- FAQ Section -->
        <div class="mxchat-faq-section">
            <div class="mxchat-faq-header">
                <h2><?php _e('Frequently Asked Questions', 'mxchat-multi-bot'); ?></h2>
                <p><?php _e('Everything you need to know about MxChat Multi-Bot Manager', 'mxchat-multi-bot'); ?></p>
            </div>
            
            <div class="mxchat-faq-grid">
                <div class="mxchat-faq-item">
                    <div class="mxchat-faq-question">
                        <h3><?php _e('How do I manage knowledge base content for different bots?', 'mxchat-multi-bot'); ?></h3>
                        <span class="mxchat-faq-toggle">+</span>
                    </div>
                    <div class="mxchat-faq-answer">
                        <p><?php _e('Navigate to MxChat → Knowledge Base and use the bot dropdown selector at the top. This allows you to switch between bots and manage each one\'s knowledge base independently.', 'mxchat-multi-bot'); ?></p>
                    </div>
                </div>
                
                <div class="mxchat-faq-item">
                    <div class="mxchat-faq-question">
                        <h3><?php _e('Can I enable different actions for different bots?', 'mxchat-multi-bot'); ?></h3>
                        <span class="mxchat-faq-toggle">+</span>
                    </div>
                    <div class="mxchat-faq-answer">
                        <p><?php _e('Yes! Go to MxChat → Actions and you\'ll find options to enable or disable specific actions for each bot individually. This allows you to customize what each bot can do based on its intended purpose.', 'mxchat-multi-bot'); ?></p>
                    </div>
                </div>
                
                <div class="mxchat-faq-item">
                    <div class="mxchat-faq-question">
                        <h3><?php _e('How do I set different bots for different pages?', 'mxchat-multi-bot'); ?></h3>
                        <span class="mxchat-faq-toggle">+</span>
                    </div>
                    <div class="mxchat-faq-answer">
                        <p><?php _e('When editing any page or post, look for the "MxChat Settings" meta box in the editor. Here you can select which specific bot should display on that page, and choose whether it should be floating or embedded.', 'mxchat-multi-bot'); ?></p>
                    </div>
                </div>
                
                <div class="mxchat-faq-item">
                    <div class="mxchat-faq-question">
                        <h3><?php _e('Can I disable the global floating bot and use page-specific bots instead?', 'mxchat-multi-bot'); ?></h3>
                        <span class="mxchat-faq-toggle">+</span>
                    </div>
                    <div class="mxchat-faq-answer">
                        <p><?php _e('Absolutely! You can disable the global "append to body" setting in your main MxChat settings, then use either the MxChat Settings meta box on individual pages or shortcodes to control exactly where and which bots appear.', 'mxchat-multi-bot'); ?></p>
                    </div>
                </div>
                
                <div class="mxchat-faq-item">
                    <div class="mxchat-faq-question">
                        <h3><?php _e('Do I need separate Pinecone accounts for each bot?', 'mxchat-multi-bot'); ?></h3>
                        <span class="mxchat-faq-toggle">+</span>
                    </div>
                    <div class="mxchat-faq-answer">
                        <p><?php _e('No, you can use the same Pinecone account with different indexes. Separate indexes provide complete isolation between bots\' knowledge bases.', 'mxchat-multi-bot'); ?></p>
                    </div>
                </div>
                
                <div class="mxchat-faq-item">
                    <div class="mxchat-faq-question">
                        <h3><?php _e('Can I copy settings from one bot to another?', 'mxchat-multi-bot'); ?></h3>
                        <span class="mxchat-faq-toggle">+</span>
                    </div>
                    <div class="mxchat-faq-answer">
                        <p><?php _e('Yes, use the "Duplicate" feature in the bot list. This creates an exact copy with a new ID and name, which you can then modify as needed. Perfect for creating variations.', 'mxchat-multi-bot'); ?></p>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// FAQ Toggle Functionality
document.addEventListener('DOMContentLoaded', function() {
    const faqItems = document.querySelectorAll('.mxchat-faq-item');
    
    faqItems.forEach(item => {
        const question = item.querySelector('.mxchat-faq-question');
        
        question.addEventListener('click', () => {
            const isActive = item.classList.contains('active');
            
            // Close all other FAQ items
            faqItems.forEach(otherItem => {
                otherItem.classList.remove('active');
            });
            
            // Toggle current item
            if (!isActive) {
                item.classList.add('active');
            }
        });
    });
});
</script>