<?php
/**
 * Admin Interface for MxChat Multi-Bot Manager
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class MxChat_Multi_Bot_Admin {
    
    private static $instance = null;
    private $database;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        $this->database = MxChat_Multi_Bot_Database::get_instance();
        $this->init_hooks();
    }
    
    private function init_hooks() {
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_action('wp_ajax_mxchat_multi_bot_save', array($this, 'handle_save_bot'));
        add_action('wp_ajax_mxchat_multi_bot_delete', array($this, 'handle_delete_bot'));
        add_action('wp_ajax_mxchat_multi_bot_duplicate', array($this, 'handle_duplicate_bot'));
        add_action('wp_ajax_mxchat_multi_bot_check_id', array($this, 'handle_check_bot_id'));
    }
    
    public function enqueue_admin_scripts($hook) {
        if ($hook !== 'mxchat_page_mxchat-multi-bot') {
            return;
        }
        
        wp_enqueue_style(
            'mxchat-multi-bot-admin',
            MXCHAT_MULTI_BOT_PLUGIN_URL . 'admin/css/admin-styles.css',
            array(),
            MXCHAT_MULTI_BOT_VERSION
        );
        
        wp_enqueue_style('wp-color-picker');
        
        wp_enqueue_script(
            'mxchat-multi-bot-admin',
            MXCHAT_MULTI_BOT_PLUGIN_URL . 'admin/js/admin-scripts.js',
            array('jquery', 'wp-color-picker'),
            MXCHAT_MULTI_BOT_VERSION,
            true
        );
        
        wp_localize_script('mxchat-multi-bot-admin', 'mxchatMultiBot', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('mxchat_multi_bot_nonce'),
            'strings' => array(
                'confirm_delete' => __('Are you sure you want to delete this bot? This action cannot be undone.', 'mxchat-multi-bot'),
                'bot_saved' => __('Bot saved successfully!', 'mxchat-multi-bot'),
                'error_occurred' => __('An error occurred. Please try again.', 'mxchat-multi-bot'),
                'saving' => __('Saving...', 'mxchat-multi-bot'),
                'save_bot' => __('Save Bot', 'mxchat-multi-bot'),
                'copied' => __('Copied!', 'mxchat-multi-bot'),
                'bot_id_required' => __('Bot ID is required', 'mxchat-multi-bot'),
                'bot_id_format' => __('Bot ID can only contain lowercase letters, numbers, and underscores', 'mxchat-multi-bot'),
                'bot_id_current' => __('Current bot ID', 'mxchat-multi-bot'),
                'bot_id_available' => __('Bot ID is available', 'mxchat-multi-bot'),
                'bot_id_taken' => __('Bot ID already exists', 'mxchat-multi-bot'),
                'validation_failed' => __('Please fix the validation errors', 'mxchat-multi-bot'),
                'duplicate_name_prompt' => __('Enter name for the duplicated bot:', 'mxchat-multi-bot'),
                'confirm_bulk_delete' => __('Are you sure you want to delete %d bots?', 'mxchat-multi-bot'),
            )
        ));
    }
    
    public function bot_list_page() {
        $action = isset($_GET['action']) ? sanitize_text_field($_GET['action']) : 'list';
        $bot_id = isset($_GET['bot_id']) ? sanitize_key($_GET['bot_id']) : '';
        
        switch ($action) {
            case 'edit':
                $this->render_bot_edit_page($bot_id);
                break;
            case 'create':
                $this->render_bot_create_page();
                break;
            default:
                $this->render_bot_list_page();
                break;
        }
    }
    
    private function render_bot_list_page() {
        $bots = $this->database->get_all_bots();
        include MXCHAT_MULTI_BOT_PLUGIN_DIR . 'admin/views/bot-list.php';
    }
    
    private function render_bot_edit_page($bot_id) {
        $bot = $this->database->get_bot($bot_id);
        if (!$bot) {
            wp_die(__('Bot not found.', 'mxchat-multi-bot'));
        }
        include MXCHAT_MULTI_BOT_PLUGIN_DIR . 'admin/views/bot-edit.php';
    }
    
    private function render_bot_create_page() {
        $bot = null;
        include MXCHAT_MULTI_BOT_PLUGIN_DIR . 'admin/views/bot-create.php';
    }
    
    public function handle_check_bot_id() {
        check_ajax_referer('mxchat_multi_bot_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Insufficient permissions.', 'mxchat-multi-bot'));
        }
        
        $bot_id = isset($_POST['bot_id']) ? sanitize_key($_POST['bot_id']) : '';
        $current_id = isset($_POST['current_id']) ? sanitize_key($_POST['current_id']) : '';
        
        if (empty($bot_id)) {
            wp_send_json_error(__('Bot ID is required.', 'mxchat-multi-bot'));
        }
        
        if ($bot_id === $current_id) {
            wp_send_json_success(__('Current bot ID.', 'mxchat-multi-bot'));
        }
        
        $exists = $this->database->bot_id_exists($bot_id);
        
        if ($exists) {
            wp_send_json_error(__('Bot ID already exists.', 'mxchat-multi-bot'));
        } else {
            wp_send_json_success(__('Bot ID is available.', 'mxchat-multi-bot'));
        }
    }
    
    public function handle_save_bot() {
        check_ajax_referer('mxchat_multi_bot_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Insufficient permissions.', 'mxchat-multi-bot'));
        }
        
        $bot_id = isset($_POST['bot_id']) ? sanitize_key($_POST['bot_id']) : '';
        $bot_name = isset($_POST['bot_name']) ? sanitize_text_field($_POST['bot_name']) : '';
        $is_new = isset($_POST['is_new']) && $_POST['is_new'] === 'true';
        
        if (empty($bot_id) || empty($bot_name)) {
            wp_send_json_error(__('Bot ID and Name are required.', 'mxchat-multi-bot'));
        }
        
        if ($is_new && $this->database->get_bot($bot_id)) {
            wp_send_json_error(__('Bot ID already exists. Please choose a different ID.', 'mxchat-multi-bot'));
        }
        
        $bot_data = array(
            'bot_id' => $bot_id,
            'bot_name' => $bot_name,
            'pinecone_host' => isset($_POST['pinecone_host']) ? sanitize_text_field($_POST['pinecone_host']) : '',
            'pinecone_namespace' => isset($_POST['pinecone_namespace']) ? sanitize_text_field($_POST['pinecone_namespace']) : '',
            'similarity_threshold' => isset($_POST['similarity_threshold']) ? intval($_POST['similarity_threshold']) : 75,
            'intro_message' => isset($_POST['intro_message']) ? wp_kses_post($_POST['intro_message']) : '',
            'top_bar_title' => isset($_POST['top_bar_title']) ? sanitize_text_field($_POST['top_bar_title']) : '',
            'system_prompt_instructions' => isset($_POST['system_prompt_instructions']) ? wp_kses_post($_POST['system_prompt_instructions']) : '',
            'bot_settings' => json_encode($this->sanitize_bot_settings($_POST))
        );
        
        if ($is_new) {
            $result = $this->database->create_bot($bot_data);
        } else {
            $result = $this->database->update_bot($bot_id, $bot_data);
        }
        
        if ($result) {
            wp_send_json_success(array(
                'message' => $is_new ? __('Bot created successfully!', 'mxchat-multi-bot') : __('Bot updated successfully!', 'mxchat-multi-bot'),
                'bot_id' => $bot_id,
                'redirect' => admin_url('admin.php?page=mxchat-multi-bot')
            ));
        } else {
            wp_send_json_error(__('Failed to save bot. Please try again.', 'mxchat-multi-bot'));
        }
    }
    
    public function handle_delete_bot() {
        check_ajax_referer('mxchat_multi_bot_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'mxchat-multi-bot')));
        }
        
        $bot_id = isset($_POST['bot_id']) ? sanitize_key($_POST['bot_id']) : '';
        
        if (empty($bot_id)) {
            wp_send_json_error(array('message' => __('Bot ID is required.', 'mxchat-multi-bot')));
        }
        
        $result = $this->database->delete_bot($bot_id);
        
        if ($result) {
            wp_send_json_success(array(
                'message' => __('Bot deleted successfully!', 'mxchat-multi-bot')
            ));
        } else {
            wp_send_json_error(array('message' => __('Failed to delete bot. Please try again.', 'mxchat-multi-bot')));
        }
    }
    
    public function handle_duplicate_bot() {
        check_ajax_referer('mxchat_multi_bot_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Insufficient permissions.', 'mxchat-multi-bot'));
        }
        
        $source_bot_id = isset($_POST['source_bot_id']) ? sanitize_key($_POST['source_bot_id']) : '';
        $new_bot_id = isset($_POST['new_bot_id']) ? sanitize_key($_POST['new_bot_id']) : '';
        $new_bot_name = isset($_POST['new_bot_name']) ? sanitize_text_field($_POST['new_bot_name']) : '';
        
        if (empty($source_bot_id) || empty($new_bot_id) || empty($new_bot_name)) {
            wp_send_json_error(__('All fields are required for duplication.', 'mxchat-multi-bot'));
        }
        
        if ($this->database->get_bot($new_bot_id)) {
            wp_send_json_error(__('New Bot ID already exists. Please choose a different ID.', 'mxchat-multi-bot'));
        }
        
        $result = $this->database->duplicate_bot($source_bot_id, $new_bot_id, $new_bot_name);
        
        if ($result) {
            wp_send_json_success(array(
                'message' => __('Bot duplicated successfully!', 'mxchat-multi-bot'),
                'redirect' => admin_url('admin.php?page=mxchat-multi-bot')
            ));
        } else {
            wp_send_json_error(__('Failed to duplicate bot. Please try again.', 'mxchat-multi-bot'));
        }
    }
    
    private function sanitize_bot_settings($post_data) {
    $settings = array();
    
    $allowed_settings = array(
        'model' => 'sanitize_text_field',
        'chatbot_background_color' => 'sanitize_hex_color',
        'user_message_bg_color' => 'sanitize_hex_color',
        'user_message_font_color' => 'sanitize_hex_color',
        'bot_message_bg_color' => 'sanitize_hex_color',
        'bot_message_font_color' => 'sanitize_hex_color',
        'top_bar_bg_color' => 'sanitize_hex_color',
        'send_button_font_color' => 'sanitize_hex_color',
        'icon_color' => 'sanitize_hex_color',
        'chat_input_font_color' => 'sanitize_hex_color',
        'close_button_color' => 'sanitize_hex_color',
        'chatbot_bg_color' => 'sanitize_hex_color',
        'input_copy' => 'sanitize_text_field',
        'rate_limit_message' => 'sanitize_text_field',
        'ai_agent_text' => 'sanitize_text_field',
        'popular_question_1' => 'sanitize_text_field',
        'popular_question_2' => 'sanitize_text_field',
        'popular_question_3' => 'sanitize_text_field',
        'custom_icon' => 'esc_url_raw',
        'title_icon' => 'esc_url_raw',
        'privacy_toggle' => 'sanitize_text_field',
        'privacy_text' => 'wp_kses_post',
        'enable_email_block' => 'sanitize_text_field',
        'enable_name_field' => 'sanitize_text_field',
        'name_field_placeholder' => 'sanitize_text_field',
        'email_blocker_header_content' => 'wp_kses_post',
        'email_blocker_button_text' => 'sanitize_text_field',
    );
    
    foreach ($allowed_settings as $setting => $sanitize_function) {
        if (isset($post_data[$setting])) {
            $settings[$setting] = call_user_func($sanitize_function, $post_data[$setting]);
        }
    }
    
    // Handle enable_streaming_toggle checkbox
    if (isset($post_data['enable_streaming_toggle'])) {
        $settings['enable_streaming_toggle'] = ($post_data['enable_streaming_toggle'] === 'on') ? 'on' : 'off';
    } else {
        $settings['enable_streaming_toggle'] = 'off';
    }
    
    // Handle use_custom_rate_limits checkbox
    if (isset($post_data['use_custom_rate_limits'])) {
        $settings['use_custom_rate_limits'] = ($post_data['use_custom_rate_limits'] === 'on') ? 'on' : 'off';
    } else {
        $settings['use_custom_rate_limits'] = 'off';
    }
    
    // Handle custom rate limits array
    if (isset($post_data['rate_limits']) && is_array($post_data['rate_limits'])) {
        $rate_limits = [];
        foreach ($post_data['rate_limits'] as $role_id => $rate_data) {
            $rate_limits[sanitize_key($role_id)] = [
                'limit' => sanitize_text_field($rate_data['limit'] ?? '10'),
                'timeframe' => sanitize_text_field($rate_data['timeframe'] ?? 'daily'),
                'message' => sanitize_textarea_field($rate_data['message'] ?? '')
            ];
        }
        $settings['rate_limits'] = $rate_limits;
    }
    
    if (isset($post_data['additional_popular_questions']) && is_array($post_data['additional_popular_questions'])) {
        $settings['additional_popular_questions'] = array_map('sanitize_text_field', $post_data['additional_popular_questions']);
    }
    
    return $settings;
}
    
    public function get_shortcode_for_bot($bot_id) {
        return '[mxchat_chatbot floating="yes" bot_id="' . esc_attr($bot_id) . '"]';
    }
}