<?php
/**
 * Database operations for MxChat Multi-Bot Manager
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class MxChat_Multi_Bot_Database {
    
    private static $instance = null;
    private $table_name;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        global $wpdb;
        $this->table_name = $wpdb->prefix . 'mxchat_multi_bots';
        $this->maybe_create_tables();
    }
    
    /**
     * Create database tables
     */
    public static function create_tables() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'mxchat_multi_bots';
        
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE $table_name (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            bot_id varchar(50) NOT NULL,
            bot_name varchar(255) NOT NULL,
            pinecone_host varchar(255) DEFAULT '',
            similarity_threshold int(3) DEFAULT 75,
            intro_message text,
            top_bar_title varchar(255) DEFAULT '',
            system_prompt_instructions longtext,
            bot_settings longtext,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY bot_id (bot_id),
            KEY bot_name (bot_name)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
        
        // Add version option
        add_option('mxchat_multi_bot_db_version', '1.0.0');
    }
        
    /**
     * Maybe create tables if they don't exist
     */
    private function maybe_create_tables() {
        global $wpdb;
        
        $installed_version = get_option('mxchat_multi_bot_db_version');
        
        if ($installed_version !== MXCHAT_MULTI_BOT_VERSION) {
            self::create_tables();
            update_option('mxchat_multi_bot_db_version', MXCHAT_MULTI_BOT_VERSION);
        }
        
        // Check if system_prompt_instructions column exists
        $column_exists = $wpdb->get_results($wpdb->prepare(
            "SHOW COLUMNS FROM {$this->table_name} LIKE %s",
            'system_prompt_instructions'
        ));
        
        if (empty($column_exists)) {
            $wpdb->query("ALTER TABLE {$this->table_name} ADD COLUMN system_prompt_instructions longtext DEFAULT ''");
        }
    }
        
    /**
     * Create a new bot
     */
    public function create_bot($bot_data) {
    global $wpdb;
    
    $defaults = array(
        'pinecone_host' => '',
        'similarity_threshold' => 75,
        'intro_message' => '',
        'top_bar_title' => '',
        'bot_settings' => '{}',
        'system_prompt_instructions' => ''
    );
    
    $bot_data = wp_parse_args($bot_data, $defaults);
    
    $result = $wpdb->insert(
        $this->table_name,
        array(
            'bot_id' => $bot_data['bot_id'],
            'bot_name' => $bot_data['bot_name'],
            'pinecone_host' => $bot_data['pinecone_host'],
            'similarity_threshold' => intval($bot_data['similarity_threshold']),
            'intro_message' => $bot_data['intro_message'] ?: '', // Handle empty values
            'top_bar_title' => $bot_data['top_bar_title'],
            'bot_settings' => $bot_data['bot_settings'] ?: '{}',
            'system_prompt_instructions' => $bot_data['system_prompt_instructions'] ?: ''
        ),
        array(
            '%s', '%s', '%s', '%d', '%s', '%s', '%s', '%s'
        )
    );
    
    return $result !== false;
}
    
    /**
     * Get a bot by ID
     */
    public function get_bot($bot_id) {
        global $wpdb;
        
        $bot = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->table_name} WHERE bot_id = %s",
            $bot_id
        ));
        
        return $bot;
    }
    
    /**
     * Get all bots
     */
    public function get_all_bots() {
        global $wpdb;
        
        $bots = $wpdb->get_results(
            "SELECT * FROM {$this->table_name} ORDER BY bot_name ASC"
        );
        
        return $bots;
    }
    
    /**
     * Update a bot
     */
    public function update_bot($bot_id, $bot_data) {
    global $wpdb;
    
    $update_data = array();
    $format = array();
    
    // Only update provided fields
    $allowed_fields = array(
        'bot_name' => '%s',
        'pinecone_host' => '%s',
        'similarity_threshold' => '%d',
        'intro_message' => '%s',
        'top_bar_title' => '%s',
        'bot_settings' => '%s',
        'system_prompt_instructions' => '%s'
    );
    
    foreach ($allowed_fields as $field => $field_format) {
        if (isset($bot_data[$field])) {
            $update_data[$field] = $bot_data[$field];
            $format[] = $field_format;
        }
    }
    
    if (empty($update_data)) {
        return false;
    }
    
    $result = $wpdb->update(
        $this->table_name,
        $update_data,
        array('bot_id' => $bot_id),
        $format,
        array('%s')
    );
    
    return $result !== false;
}
    
    /**
     * Delete a bot
     */
    public function delete_bot($bot_id) {
        global $wpdb;
        
        $result = $wpdb->delete(
            $this->table_name,
            array('bot_id' => $bot_id),
            array('%s')
        );
        
        return $result !== false;
    }
    
    /**
     * Duplicate a bot
     */
public function duplicate_bot($source_bot_id, $new_bot_id, $new_bot_name) {
    $source_bot = $this->get_bot($source_bot_id);
    
    if (!$source_bot) {
        return false;
    }
    
    $bot_data = array(
        'bot_id' => $new_bot_id,
        'bot_name' => $new_bot_name,
        'pinecone_host' => $source_bot->pinecone_host,
        'similarity_threshold' => $source_bot->similarity_threshold,
        'intro_message' => $source_bot->intro_message,
        'top_bar_title' => $source_bot->top_bar_title,
        'bot_settings' => $source_bot->bot_settings,
        'system_prompt_instructions' => $source_bot->system_prompt_instructions ?? ''
    );
    
    return $this->create_bot($bot_data);
}

    /**
     * Get bots count
     */
    public function get_bots_count() {
        global $wpdb;
        
        $count = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$this->table_name}"
        );
        
        return intval($count);
    }
    
    /**
     * Search bots
     */
    public function search_bots($search_term) {
        global $wpdb;
        
        $search_term = '%' . $wpdb->esc_like($search_term) . '%';
        
        $bots = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$this->table_name} 
             WHERE bot_name LIKE %s OR bot_id LIKE %s 
             ORDER BY bot_name ASC",
            $search_term,
            $search_term
        ));
        
        return $bots;
    }
    
    /**
     * Get bots with pagination
     */
    public function get_bots_paginated($limit = 20, $offset = 0) {
        global $wpdb;
        
        $bots = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$this->table_name} 
             ORDER BY updated_at DESC 
             LIMIT %d OFFSET %d",
            $limit,
            $offset
        ));
        
        return $bots;
    }
    
    /**
     * Check if bot ID exists
     */
    public function bot_id_exists($bot_id) {
        global $wpdb;
        
        $exists = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$this->table_name} WHERE bot_id = %s",
            $bot_id
        ));
        
        return intval($exists) > 0;
    }
    
    /**
     * Get bot settings as array
     */
    public function get_bot_settings($bot_id) {
        $bot = $this->get_bot($bot_id);
        
        if (!$bot || empty($bot->bot_settings)) {
            return array();
        }
        
        $settings = json_decode($bot->bot_settings, true);
        
        return is_array($settings) ? $settings : array();
    }
    
    /**
     * Update bot settings
     */
    public function update_bot_settings($bot_id, $settings) {
        $bot_data = array(
            'bot_settings' => json_encode($settings)
        );
        
        return $this->update_bot($bot_id, $bot_data);
    }
    
    /**
     * Cleanup orphaned data
     */
    public function cleanup_orphaned_data() {
        global $wpdb;
        
        // This method can be used to clean up any orphaned data
        // related to deleted bots in other tables
        
        $deleted_count = 0;
        
        // Example: Clean up chat transcripts with non-existent bot_metadata
        $chat_table = $wpdb->prefix . 'mxchat_chat_transcripts';
        
        // Check if bot_metadata column exists
        $column_exists = $wpdb->get_var("SHOW COLUMNS FROM {$chat_table} LIKE 'bot_metadata'");
        
        if ($column_exists) {
            $orphaned_transcripts = $wpdb->get_results(
                "SELECT DISTINCT bot_metadata FROM {$chat_table} 
                 WHERE bot_metadata IS NOT NULL 
                 AND bot_metadata != '' 
                 AND bot_metadata != 'default'"
            );
            
            foreach ($orphaned_transcripts as $transcript) {
                $bot_exists = $this->bot_id_exists($transcript->bot_metadata);
                
                if (!$bot_exists) {
                    // Optionally delete or update orphaned transcripts
                    // For now, we'll just count them
                    $count = $wpdb->get_var($wpdb->prepare(
                        "SELECT COUNT(*) FROM {$chat_table} WHERE bot_metadata = %s",
                        $transcript->bot_metadata
                    ));
                    $deleted_count += intval($count);
                }
            }
        }
        
        return $deleted_count;
    }
    
    /**
     * Drop tables (for uninstall)
     */
    public static function drop_tables() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'mxchat_multi_bots';
        $wpdb->query("DROP TABLE IF EXISTS $table_name");
        
        delete_option('mxchat_multi_bot_db_version');
    }
}