<?php
/**
 * Core Manager for MxChat Multi-Bot Manager
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class MxChat_Multi_Bot_Core_Manager {
    
    private static $instance = null;
    private $database;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        $this->database = MxChat_Multi_Bot_Database::get_instance();
        $this->init_hooks();
    }
    
    private function init_hooks() {
        add_filter('mxchat_get_bot_options', array($this, 'get_bot_options'), 10, 2);
        add_filter('mxchat_get_bot_pinecone_config', array($this, 'get_bot_pinecone_config'), 10, 2);
        add_filter('mxchat_shortcode_atts', array($this, 'modify_shortcode_atts'), 10, 2);
        add_filter('mxchat_knowledge_bot_filter', array($this, 'filter_knowledge_by_bot'), 10, 2);
    }
    
    public function get_bot_options($default_options, $bot_id) {
        if ($bot_id === 'default' || empty($bot_id)) {
            return $default_options;
        }
        
        $bot = $this->database->get_bot($bot_id);
        if (!$bot) {
            return $default_options;
        }
        
        $bot_options = get_option('mxchat_options', array());
        
        if (!empty($bot->bot_settings)) {
            $bot_settings = json_decode($bot->bot_settings, true);
            if (is_array($bot_settings)) {
                $bot_options = array_merge($bot_options, $bot_settings);
                
                $has_custom_questions = !empty($bot_settings['popular_question_1']) || 
                                      !empty($bot_settings['popular_question_2']) || 
                                      !empty($bot_settings['popular_question_3']) || 
                                      !empty($bot_settings['additional_popular_questions']);
                
                if ($has_custom_questions) {
                    $bot_options['popular_question_1'] = $bot_settings['popular_question_1'] ?? '';
                    $bot_options['popular_question_2'] = $bot_settings['popular_question_2'] ?? '';
                    $bot_options['popular_question_3'] = $bot_settings['popular_question_3'] ?? '';
                    $bot_options['additional_popular_questions'] = $bot_settings['additional_popular_questions'] ?? array();
                }
                
                if (isset($bot_settings['model']) && !empty($bot_settings['model'])) {
                    $bot_options['model'] = $bot_settings['model'];
                }
            }
        }
        
        if (!empty($bot->similarity_threshold)) {
            $bot_options['similarity_threshold'] = $bot->similarity_threshold;
        }
        
        if (!empty($bot->intro_message)) {
            $bot_options['intro_message'] = wp_unslash($bot->intro_message);
        }
        
        if (!empty($bot->top_bar_title)) {
            $bot_options['top_bar_title'] = wp_unslash($bot->top_bar_title);
        }
        
        if (!empty($bot->system_prompt_instructions)) {
            $bot_options['system_prompt_instructions'] = wp_unslash($bot->system_prompt_instructions);
        }
                
        // Handle rate limits override
        if (isset($bot_settings['use_custom_rate_limits']) && $bot_settings['use_custom_rate_limits'] === 'on') {
            if (isset($bot_settings['rate_limits'])) {
                $bot_options['rate_limits'] = $bot_settings['rate_limits'];
            }
        }
        
        return $bot_options;
    }
    
    public function get_bot_pinecone_config($default_config, $bot_id) {
        if ($bot_id === 'default' || empty($bot_id)) {
            return $default_config;
        }
        
        $bot = $this->database->get_bot($bot_id);
        if (!$bot) {
            return $default_config;
        }
        
        $addon_options = get_option('mxchat_pinecone_addon_options', array());
        $core_api_key = $addon_options['mxchat_pinecone_api_key'] ?? '';
        
        $bot_config = array(
            'use_pinecone' => !empty($bot->pinecone_host) && !empty($core_api_key),
            'api_key' => $core_api_key,
            'host' => $bot->pinecone_host ?? ''
        );
        
        return $bot_config;
    }
    
    public function modify_shortcode_atts($atts, $defaults) {
        if (isset($atts['bot_id']) && $atts['bot_id'] !== 'default') {
            $bot = $this->database->get_bot($atts['bot_id']);
            if (!$bot) {
                $atts['bot_id'] = 'default';
            }
        }
        
        return $atts;
    }
    
    public function filter_knowledge_by_bot($content, $bot_id) {
        if ($bot_id === 'default' || empty($bot_id)) {
            return $content;
        }
        
        return $content;
    }
    
    public function get_available_bots() {
        $bots = $this->database->get_all_bots();
        
        $options = array(
            'default' => __('Default Bot', 'mxchat-multi-bot')
        );
        
        foreach ($bots as $bot) {
            $options[$bot->bot_id] = $bot->bot_name;
        }
        
        return $options;
    }
    
    public function get_bot_name($bot_id) {
        if ($bot_id === 'default') {
            return __('Default Bot', 'mxchat-multi-bot');
        }
        
        $bot = $this->database->get_bot($bot_id);
        return $bot ? $bot->bot_name : $bot_id;
    }
    
    public function bot_exists($bot_id) {
        if ($bot_id === 'default') {
            return true;
        }
        
        return $this->database->get_bot($bot_id) !== null;
    }
    
    public function get_bot_stats($bot_id = null) {
        global $wpdb;
        
        $stats = array();
        
        if ($bot_id) {
            $chat_table = $wpdb->prefix . 'mxchat_chat_transcripts';
            
            $column_exists = $wpdb->get_var("SHOW COLUMNS FROM {$chat_table} LIKE 'bot_metadata'");
            
            if ($column_exists) {
                $stats['total_messages'] = $wpdb->get_var($wpdb->prepare(
                    "SELECT COUNT(*) FROM {$chat_table} WHERE bot_metadata = %s",
                    $bot_id
                ));
                
                $stats['unique_sessions'] = $wpdb->get_var($wpdb->prepare(
                    "SELECT COUNT(DISTINCT session_id) FROM {$chat_table} WHERE bot_metadata = %s",
                    $bot_id
                ));
            } else {
                $stats['total_messages'] = 0;
                $stats['unique_sessions'] = 0;
            }
        } else {
            $bots = $this->database->get_all_bots();
            $stats['total_bots'] = count($bots);
            
            $active_bots = 0;
            foreach ($bots as $bot) {
                if (!empty($bot->pinecone_host) || !empty($bot->bot_settings)) {
                    $active_bots++;
                }
            }
            $stats['active_bots'] = $active_bots;
        }
        
        return $stats;
    }
    
    public function export_bot_config($bot_id) {
        $bot = $this->database->get_bot($bot_id);
        if (!$bot) {
            return false;
        }
        
        $config = array(
            'bot_id' => $bot->bot_id,
            'bot_name' => $bot->bot_name,
            'pinecone_host' => $bot->pinecone_host,
            'similarity_threshold' => $bot->similarity_threshold,
            'intro_message' => $bot->intro_message,
            'top_bar_title' => $bot->top_bar_title,
            'bot_settings' => $bot->bot_settings,
            'export_date' => current_time('mysql'),
            'export_version' => MXCHAT_MULTI_BOT_VERSION
        );
        
        return $config;
    }
    
    public function import_bot_config($config, $new_bot_id = null) {
        if (!is_array($config) || empty($config['bot_id'])) {
            return false;
        }
        
        $bot_id = $new_bot_id ?: $config['bot_id'];
        
        if ($this->database->get_bot($bot_id)) {
            return false;
        }
        
        $bot_data = array(
            'bot_id' => $bot_id,
            'bot_name' => $config['bot_name'] ?? '',
            'pinecone_host' => $config['pinecone_host'] ?? '',
            'similarity_threshold' => $config['similarity_threshold'] ?? 75,
            'intro_message' => $config['intro_message'] ?? '',
            'top_bar_title' => $config['top_bar_title'] ?? '',
            'bot_settings' => $config['bot_settings'] ?? '{}'
        );
        
        return $this->database->create_bot($bot_data);
    }
    
    public function get_bot_shortcode($bot_id, $floating = true) {
        $shortcode = '[mxchat_chatbot';
        
        if ($floating) {
            $shortcode .= ' floating="yes"';
        }
        
        if ($bot_id !== 'default') {
            $shortcode .= ' bot_id="' . esc_attr($bot_id) . '"';
        }
        
        $shortcode .= ']';
        
        return $shortcode;
    }
}